# -*-perl-*-
# Keitairc::IrcCallback
# $Id: IrcCallback.pm,v 1.1 2008/01/08 05:52:14 morimoto Exp $
# $Source: /cvsroot/keitairc/lib/Keitairc/IrcCallback.pm,v $
#
# Copyright (c) 2008 Jun Morimoto <morimoto@mrmt.net>
# This program is covered by the GNU General Public License 2

package Keitairc::IrcCallback;
use POE;
use Encode;
use strict;

################################################################
sub irc_start{
	my ($heap) = @_[HEAP];
	$heap->{Irc}->yield(register => 'all');
	$heap->{Irc}->yield(connect => {});
}

################################################################
sub irc_001{
	my ($kernel, $heap, $sender) = @_[KERNEL, HEAP, SENDER];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};

	for my $channel ($ib->channels()){
		$ib->add_message($channel, 'Connected to irc server!');
		$ib->part($channel);
	}
	$heap->{disconnect_msg} = 1;
	$kernel->delay(autoping => $cf->ping_delay());
}

################################################################
sub irc_public{
	my ($kernel, $heap, $who, $channel, $msg) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	$who =~ s/!.*//;
	$channel = $channel->[0];
	Encode::from_to($msg, 'jis', 'euc-jp');
	$heap->{IrcBuffer}->add_message($channel, $msg, $who);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_join{
	my ($kernel, $heap, $who, $channel) = @_[KERNEL, HEAP, ARG0, ARG1];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};

	$who =~ s/!.*//;

	# chop off after the gap to workaround madoka's bug
	$channel =~ s/ .*//;

	$ib->join($channel);
	if($who ne $cf->irc_nick()){
		$ib->add_message($channel, "$who joined");
	}

	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_part{
	my ($kernel, $heap, $who, $channel) = @_[KERNEL, HEAP, ARG0, ARG1];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};

	$who =~ s/!.*//;
	# chop off after the gap (bug workaround of POE::Filter::IRC)
	$channel =~ s/ .*//;

	if($who eq $cf->irc_nick()){
		$ib->part($channel);
	} else {
		$ib->add_message($channel, "$who leaves");
	}

	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_notice{
	my ($kernel, $heap, $who, $channel, $msg) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	my $ib = $heap->{IrcBuffer};

	$who =~ s/!.*//;
	$channel = $channel->[0];
	Encode::from_to($msg, 'jis', 'euc-jp');
	$ib->add_message($channel, $msg, $who);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_topic{
	my ($kernel, $heap, $who, $channel, $topic) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	my $ib = $heap->{IrcBuffer};

	$who =~ s/!.*//;
	Encode::from_to($topic, 'jis', 'euc-jp');
	$ib->add_message($channel, "$who set topic: $topic");
	$ib->topic($channel, $topic);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_topicraw{
	my ($kernel, $heap, $raw) = @_[KERNEL, HEAP, ARG1];
	my ($channel, $topic) = split(/ :/, $raw, 2);
	my $ib = $heap->{IrcBuffer};

	$ib->topic($channel, $topic);
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_ctcp_action{
	my ($kernel, $heap, $who, $channel, $msg) = @_[KERNEL, HEAP, ARG0 .. ARG2];
	my $ib = $heap->{IrcBuffer};

	$who =~ s/!.*//;
	$channel = $channel->[0];
	Encode::from_to($msg, 'jis', 'euc-jp');
	$ib->add_message($channel, sprintf('* %s %s', $who, $msg));
	$heap->{seen_traffic} = 1;
	$heap->{disconnect_msg} = 1;
}

################################################################
sub irc_reconnect{
	my ($kernel, $heap) = @_[KERNEL, HEAP];
	my $ib = $heap->{IrcBuffer};
	my $cf = $heap->{Config};

	if ($heap->{disconnect_msg}) {
		for my $channel ($ib->channels()){
			$ib->add_message($channel, 'Disconnected from irc server, trying to reconnect...');
		}
	}
	$heap->{disconnect_msg} = 0;
	$kernel->delay(connect => $cf->reconnect_delay());
}

################################################################
sub irc_connect{
	my ($kernel, $heap) = @_[KERNEL, HEAP];
	$kernel->post(keitairc_irc => connect => {});
}

################################################################
sub irc_autoping{
	my ($kernel, $heap) = @_[KERNEL, HEAP];
	my $cf = $heap->{Config};

	$kernel->post(keitairc_irc => time) unless $heap->{seen_traffic};
	$heap->{seen_traffic} = 0;
	$kernel->delay(autoping => $cf->ping_delay());
}

1;
