/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "gobject-utils.h"
#include "rss-menu.h"
#include "gtk-utils.h"
/* XPM */
#include "pixmaps/update_button.xpm"

static void rss_menu_class_init (RSSMenuClass *class);
static void rss_menu_init       (RSSMenu *rssmenu);
static void rss_menu_destroy    (GtkObject *object);

static void rss_menu_update(KzRSS *rss, RSSMenu *rssmenu);
static void rss_menu_item_button_press_cb(GtkWidget *widget,
					  GdkEventButton *event,
					  KzRSSItem *item);
static void rss_menu_items_new(KzWindow *kz, RSSMenu *rssmenu);

static gboolean rss_menu_press_cb(GtkWidget *widget, GdkEvent *event,
			      RSSMenu *rssmenu);
static void rss_menu_update_button_clicked(GtkWidget *widget,
					   RSSMenu *rssmenu);

static GtkHBoxClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(rss_menu, "RSSMenu", RSSMenu,
		   rss_menu_class_init, rss_menu_init,
		   GTK_TYPE_HBOX)

static void
rss_menu_class_init (RSSMenuClass *class)
{
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent (class);

	object_class = (GtkObjectClass *) class;

	/* GtkObject signals */
	object_class->destroy = rss_menu_destroy;
}

static void
rss_menu_init (RSSMenu *rssmenu)
{
	rssmenu->kz = NULL;
	rssmenu->rss = NULL;
	rssmenu->SubMenu = NULL;
	rssmenu->MenuItem = NULL;
	rssmenu->update_button = NULL;
}

static void
rss_menu_destroy (GtkObject *object)
{
	RSSMenu *rssmenu;

	g_return_if_fail (RSS_IS_MENU(object));

	rssmenu = RSS_MENU(object);

	rssmenu->kz = NULL;
	g_signal_handlers_disconnect_by_func(G_OBJECT(rssmenu->rss),
					     rss_menu_update, rssmenu);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy(object);
}

GtkWidget *
rss_menu_new(KzWindow *kz, KzRSS *rss)
{
	RSSMenu *rssmenu;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkWidget *button, *image, *hbox, *label, *eventbox;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	g_return_val_if_fail(KZ_IS_RSS(rss), NULL);

	rssmenu = RSS_MENU(g_object_new(RSS_TYPE_MENU, NULL));
	rssmenu->kz = kz;
	rssmenu->rss = rss;

	hbox = gtk_hbox_new(FALSE, 0);

	/* label */
	label = gtk_label_new(rss->title);
	eventbox = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(eventbox), label);
	g_signal_connect(G_OBJECT(eventbox), "button_press_event",
			 G_CALLBACK(rss_menu_press_cb), rssmenu);
	gtk_widget_show (label);

	/* button */
	rssmenu->update_button = button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	gtk_container_set_border_width(GTK_CONTAINER(button), 0);
	g_signal_connect(G_OBJECT(button), "clicked", 
			 G_CALLBACK(rss_menu_update_button_clicked), rssmenu);

	pixmap = gdk_pixmap_colormap_create_from_xpm_d(GTK_WIDGET(kz)->window,
						       gtk_widget_get_colormap(GTK_WIDGET(rssmenu)),
						       &mask,
						       NULL,
						       update_button);
	image = gtk_image_new_from_pixmap(pixmap,mask);
	gdk_pixmap_unref(pixmap);
	gdk_bitmap_unref(mask);
	gtk_container_add(GTK_CONTAINER(button), image);
	gtk_widget_set_size_request(button, 16, 16);
	gtk_widget_show(image);
	gtk_widget_show(button);

	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), eventbox, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(rssmenu), hbox);
	gtk_container_set_border_width(GTK_CONTAINER(rssmenu), 2);

	g_signal_connect(G_OBJECT(rssmenu->rss), "update_completed", 
			 G_CALLBACK(rss_menu_update), rssmenu);

	gtk_widget_show_all(GTK_WIDGET(rssmenu));

	return GTK_WIDGET(rssmenu);
}

static void
rss_menu_items_new(KzWindow *kz, RSSMenu *rssmenu)
{
	GtkWidget *menu_item;
	GSList *items;
	guint item_num, n;
	KzRSS *rss = NULL;
	KzRSSItem *item = NULL;
	GtkTooltips *tooltip = NULL;

	g_return_if_fail(RSS_IS_MENU(rssmenu));

	rss = rssmenu->rss;
	if (rssmenu->SubMenu != NULL)
	{
		gtk_widget_destroy(rssmenu->SubMenu);
	}

	rssmenu->SubMenu = gtk_menu_new();
	
	items = rss->items;
	item_num = g_slist_length(rss->items);

	for (n = 0; n < item_num; n++)
	{
		item = (KzRSSItem*) g_slist_nth_data(items, n);

		menu_item = gtk_menu_item_new_with_label(item->title);
		gtk_menu_shell_append(GTK_MENU_SHELL(rssmenu->SubMenu),
				     menu_item);
		item->rss = rss;
		item->kz = rssmenu->kz;
		g_signal_connect(G_OBJECT(menu_item), "button_press_event",
				 G_CALLBACK(rss_menu_item_button_press_cb), item);
		tooltip = gtk_tooltips_new();
		gtk_tooltips_set_tip(tooltip, menu_item, item->description, NULL);
	}
	gtk_widget_show_all(rssmenu->SubMenu);
}

static void 
rss_menu_update(KzRSS *rss, RSSMenu *rssmenu)
{
	g_return_if_fail(KZ_IS_RSS(rss));
	g_return_if_fail(RSS_IS_MENU(rssmenu));
	KzWindow *kz = rssmenu->kz;
	rss_menu_items_new(kz, rssmenu);
	gtk_widget_show_all(GTK_WIDGET(rssmenu));
}

static void
rss_menu_item_button_press_cb(GtkWidget *widget, GdkEventButton *event,
			      KzRSSItem *item)
{
	KzWindow *kz;
	kz = item->kz;
	
	if (!item->link) return;

	switch (event->button) {
	case 1:
	case 3:
		kz_window_load_url(kz, item->link);
		break;
	case 2:
		kz_window_open_new_tab(kz, item->link);
		break;
	}
}

static gboolean
rss_menu_press_cb(GtkWidget *widget, GdkEvent *event,
		  RSSMenu *rssmenu)
{
	KzWindow *kz;
	GdkEventButton *event_button;
	GSList *items;
	guint item_num, n;
	KzRSSItem *item = NULL;

	kz = rssmenu->kz;

	if (event->type == GDK_BUTTON_PRESS) 
	{
		event_button = (GdkEventButton *)event;
		switch(event_button->button)
		{
		 case 1:
			g_return_val_if_fail(GTK_IS_MENU(rssmenu->SubMenu), FALSE);
			gtk_menu_popup(GTK_MENU(rssmenu->SubMenu), NULL, NULL, 
				       menu_position_under_widget,
				       GTK_WIDGET(rssmenu),
				       event_button->button, event_button->time);
			break;
		 case 2:
			items = rssmenu->rss->items;
			item_num = g_slist_length(rssmenu->rss->items);
			for (n = 0; n < item_num; n++)
			{
				item = (KzRSSItem*) g_slist_nth_data(items, n);
				if (item->link)
					kz_window_open_new_tab(kz, item->link);
			}
			break;
		 case 3:
			kz_window_load_url(kz, rssmenu->rss->rss_link);
			break;
		}
	}
	return FALSE;
}

static void
rss_menu_update_button_clicked(GtkWidget *widget, RSSMenu *rssmenu)
{
	g_return_if_fail(RSS_IS_MENU(rssmenu));
	
	kz_rss_force_update(rssmenu->rss);
}
