/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_lang.h"

#include "mozilla.h"
#include "utils/intl.h"


#define DATA_KEY "PrefsLang::info"


static GtkWidget     *prefs_lang_create          (void);
static void           prefs_lang_response        (GtkWidget *widget,
						  gint       response);


static KzPrefsWinPageEntry prefs_entry =
{
	path:            "/Language",
	priority_hint:   0,
	create:          prefs_lang_create,
	response:        prefs_lang_response,
};


typedef struct _PrefsLang
{
	GtkWidget     *main_vbox;
	GtkWidget     *option_menu_lang;
	GtkWidget     *option_menu_autodetect;
} PrefsLang;


/*
 *  These arrays are picked from epiphany-0.7 (src/prefs-diarlog.c).
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 */
static const
struct
{
	char *name;
	char *code;
}
languages [] =
{
	{ N_("System language"), "system" },
	{ N_("Afrikaans"), "ak" },
	{ N_("Albanian"), "sq" },
	{ N_("Arabic"), "ar" },
	{ N_("Azerbaijani"), "az" },
	{ N_("Basque"), "eu" },
	{ N_("Breton"), "br" },
	{ N_("Bulgarian"), "bg" },
	{ N_("Byelorussian"), "be" },
	{ N_("Catalan"), "ca" },
	{ N_("Chinese"), "zh" },
	{ N_("Croatian"), "hr" },
	{ N_("Czech"), "cs" },
	{ N_("Danish"), "da" },
	{ N_("Dutch"), "nl" },
	{ N_("English"), "en" },
	{ N_("Esperanto"), "eo" },
	{ N_("Estonian"), "et" },
	{ N_("Faeroese"), "fo" },
	{ N_("Finnish"), "fi" },
	{ N_("French"), "fr" },
	{ N_("Galician"), "gl" },
	{ N_("German"), "de" },
	{ N_("Greek"), "el" },
	{ N_("Hebrew"), "he" },
	{ N_("Hungarian"), "hu" },
	{ N_("Icelandic"), "is" },
	{ N_("Indonesian"), "id" },
	{ N_("Irish"), "ga" },
	{ N_("Italian"), "it" },
	{ N_("Japanese"), "ja" },
	{ N_("Korean"), "ko" },
	{ N_("Latvian"), "lv" },
	{ N_("Lithuanian"), "lt" },
	{ N_("Macedonian"), "mk" },
	{ N_("Malay"), "ms" },
	{ N_("Norwegian/Nynorsk"), "nn" },
	{ N_("Norwegian/Bokmaal"), "nb" },
	{ N_("Norwegian"), "no" },
	{ N_("Polish"), "pl" },
	{ N_("Portuguese"), "pt" },
	{ N_("Portuguese of Brazil"), "pt-BR" },
	{ N_("Romanian"), "ro" },
	{ N_("Russian"), "ru" },
	{ N_("Scottish"), "gd" },
	{ N_("Serbian"), "sr" },
	{ N_("Slovak"), "sk" },
	{ N_("Slovenian"), "sl" },
	{ N_("Spanish"), "es" },
	{ N_("Swedish"), "sv" },
	{ N_("Tamil"), "ta" },
	{ N_("Turkish"), "tr" },
	{ N_("Ukrainian"), "uk" },
	{ N_("Vietnamian"), "vi" },
	{ N_("Walloon"), "wa" }
};
static guint n_languages = G_N_ELEMENTS (languages);

typedef struct
{
	gchar *title;
	gchar *name;
} EncodingAutodetectorInfo;

static EncodingAutodetectorInfo encoding_autodetectors[] =
{
	{ N_("Off"),			"" },
	{ N_("Chinese"),		"zh_parallel_state_machine" },
	{ N_("East asian"),		"cjk_parallel_state_machine" },
	{ N_("Japanese"),		"ja_parallel_state_machine" },
	{ N_("Korean"),			"ko_parallel_state_machine" },
	{ N_("Russian"),		"ruprob" },
	{ N_("Simplified Chinese"),	"zhcn_parallel_state_machine" },
	{ N_("Traditional Chinese"),	"zhtw_parallel_state_machine" },
	{ N_("Ukrainian"),		"ukprob" }
};
static guint n_encoding_autodetectors = G_N_ELEMENTS (encoding_autodetectors);


#define CREATE_MENU(menu, array, n_array, field)                         \
{                                                                        \
	gint i;                                                          \
                                                                         \
	menu = gtk_menu_new ();                                          \
                                                                         \
	for (i = 0; i < n_array; i++)                                    \
	{                                                                \
		GtkWidget *item;                                         \
                                                                         \
		item = gtk_menu_item_new_with_label (_(array[i].field)); \
		gtk_menu_shell_append (GTK_MENU_SHELL(menu), item);      \
		gtk_widget_show (item);                                  \
	}                                                                \
}


static void
prefs_lang_destroy (gpointer data)
{
	PrefsLang *prefsui = data;

	g_free(prefsui);
}


static GtkWidget *
prefs_lang_create (void)
{
	PrefsLang *prefsui = g_new0(PrefsLang, 1);
	GtkWidget *main_vbox, *vbox, *hbox;
	GtkWidget *label, *table, *option_menu, *menu, *frame;

	main_vbox = gtk_vbox_new (FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full (G_OBJECT(main_vbox),
				DATA_KEY, prefsui,
				(GDestroyNotify) prefs_lang_destroy);

	label = kz_prefs_ui_utils_create_title("Language");
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), vbox,
			   TRUE, TRUE, 0);
	gtk_widget_show(vbox);

	table = gtk_table_new(2, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), table,
			   FALSE, FALSE, 0);
	gtk_widget_show(table);

	/*
	 *  Default Encoding
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 0, 1,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic("_Default Encoding: ");
	gtk_label_set_use_underline(GTK_LABEL(label), TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	option_menu = gtk_option_menu_new();
	prefsui->option_menu_lang = option_menu;
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), option_menu);
	gtk_table_attach(GTK_TABLE(table), option_menu,
			 1, 2, 0, 1,
			 GTK_EXPAND | GTK_FILL, 0,
			 2, 2);
	gtk_widget_show(option_menu);

	CREATE_MENU(menu, languages, n_languages, name);
	gtk_option_menu_set_menu (GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(menu);

	/*
	 *  Auto detect
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 1, 2,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic("A_utodetect Encoding: ");
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	option_menu = gtk_option_menu_new();
	prefsui->option_menu_autodetect = option_menu;
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), option_menu);
	gtk_table_attach(GTK_TABLE(table), option_menu,
			 1, 2, 1, 2,
			 GTK_EXPAND | GTK_FILL, 0,
			 2, 2);
	gtk_widget_show(option_menu);

	CREATE_MENU(menu, encoding_autodetectors, n_encoding_autodetectors, title);
	gtk_option_menu_set_menu (GTK_OPTION_MENU(option_menu), menu);
	gtk_widget_show(menu);

	/*
	 *  Accept Language
	 */
	frame = gtk_frame_new("Accept Language");
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
	gtk_widget_show(frame);

	return main_vbox;
}


#define GET_HISTORY(option_menu) \
	(gtk_option_menu_get_history(GTK_OPTION_MENU(option_menu)))


static void
prefs_lang_response (GtkWidget *widget, gint response)
{
	PrefsLang *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);
	gint i, j;

	g_return_if_fail(prefsui);


	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
		i = GET_HISTORY(prefsui->option_menu_lang);
		j = GET_HISTORY(prefsui->option_menu_autodetect);
		if (i >= 0 && i < n_languages) {
			mozilla_preference_set("intl.charset.default",
					       languages[i].code);
		}
		if (j >= 0 && j < n_encoding_autodetectors) {
			if (mozilla_preference_set("intl.charset.detector",
						   encoding_autodetectors[i].name));
		}
		break;
	case GTK_RESPONSE_REJECT:
		break;
	default:
		break;
	}
}


KzPrefsWinPageEntry *
prefs_lang_get_entry (void)
{
	return &prefs_entry;
}
