/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2002 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-window.h"

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gtkmozembed.h>

#include "egg-entryaction.h"
#include "egg-toolbar.h"
#include "gobject-utils.h"
#include "gtk-utils.h"
#include "kz-bookmarkbar.h"
#include "kz-mozembed.h"
#include "kz-tablabel.h"
#include "kz-home.h"
#include "kz-prefswin.h"
#include "google.h"
#include "kazehakase.h"
#include "intl.h"
#include "mozilla.h"
#include "prefs.h"

struct _KzWindowPriv
{
	KzHome     *kzhome;
	const char *status_msg;
	char       *temp_msg;
	gboolean   did_find; /* for find keyword */
};

enum {
	STATUS_LINK_MESSAGE,
	STATUS_GESTURE
};

enum {
	TARGET_KAZEHAKASE_TAB,
	TARGET_NETSCAPE_URL,
	TARGET_TEXT_URI_LIST,
	TARGET_TEXT_PLAIN,
	TARGET_STRING
};

static GtkTargetEntry url_drag_types [] =
{
        { "_KAZEHAKASE_TAB", 0, TARGET_KAZEHAKASE_TAB},
        { "_NETSCAPE_URL",   0, TARGET_NETSCAPE_URL},
	{ "text/uri-list",   0, TARGET_TEXT_URI_LIST},
	{ "text/plain",      0, TARGET_TEXT_PLAIN},
	{ "STRING",          0, TARGET_STRING}
};

static guint n_url_drag_types = G_N_ELEMENTS (url_drag_types);

static void     kz_window_class_init     (KzWindowClass *class);
static void     kz_window_init           (KzWindow *kz);
static gboolean kz_window_delete_event   (GtkWidget *widget,
					  GdkEventAny *event);
static void     kz_window_destroy        (GtkObject *object);
static void     kz_window_finalize       (GObject *object);

static void     kz_window_menu_merge_set_widget (KzWindow *kz);
static void     kz_window_gesture_update        (KzWindow *kz);
static void     kz_window_set_moz_callbacks     (KzWindow *kz,
					         KzMozEmbed *kzembed);
static void     kz_window_unset_moz_callbacks   (KzWindow *kz,
					         KzMozEmbed *kzembed);
static void     kz_window_update_nav_buttons    (KzWindow *kz,
					         KzMozEmbed *kzembed);

/* actions */
static void     act_close_window         (EggAction *action, KzWindow *kz);
static void     act_close_tab            (EggAction *action, KzWindow *kz);
static void     act_new_window           (EggAction *action, KzWindow *kz);
static void     act_new_tab              (EggAction *action, KzWindow *kz);
static void     act_home                 (EggAction *action, KzWindow *kz);
static void     act_back                 (EggAction *action, KzWindow *kz);
static void     act_forward              (EggAction *action, KzWindow *kz);
static void     act_stop                 (EggAction *action, KzWindow *kz);
static void     act_reload               (EggAction *action, KzWindow *kz);
static void     act_location_enter       (EggAction *action, KzWindow *kz);
static void     act_search               (EggAction *action, KzWindow *kz);
static void     act_left_tab             (EggAction *action, KzWindow *kz);
static void     act_right_tab            (EggAction *action, KzWindow *kz);
static void     act_preference           (EggAction *action, KzWindow *kz);

/* callbacks */
static void     menu_merge_add_widget_cb (EggMenuMerge *merge,
					  GtkWidget *widget,
					  GtkBox *box);
static void     notebook_switch_page_cb  (GtkNotebook *notebook,
					  GtkNotebookPage *page,
					  guint page_num, KzWindow *kz);
static void     find_keyword_cb          (GtkWidget *widget, KzWindow *kz);

/* callbacks for mozilla */
static void     title_changed_cb         (GtkMozEmbed *embed, KzWindow *kz);
static void     location_changed_cb      (GtkMozEmbed *embed, KzWindow *kz);
static void     link_message_cb          (GtkMozEmbed *embed, KzWindow *kz);
#if 0
static void     visibility_cb            (GtkMozEmbed *embed,
					  gboolean visibility,
					  KzWindow *kz);
#endif
static void     load_started_cb          (GtkMozEmbed *embed, KzWindow *kz);
static void     load_finished_cb         (GtkMozEmbed *embed, KzWindow *kz);
static void     moz_new_window_cb        (GtkMozEmbed *embed,
					  GtkMozEmbed **newEmbed,
					  guint chromemask,
					  KzWindow *kz);
static void     close_tab_cb             (GtkObject *obj, KzWindow *kz);
static gint     dom_key_up_cb            (GtkMozEmbed *embed, gpointer event,
					  KzWindow *kz);
static gint     dom_mouse_click_cb       (GtkMozEmbed *embed, gpointer event,
					  KzWindow *kz);
static gint     dom_mouse_dbl_click_cb   (GtkMozEmbed *embed, gpointer event,
					  KzWindow *kz);
static gint     dom_mouse_down_cb        (GtkMozEmbed *embed,
					  gpointer event,
					  KzWindow *kz);

/*
 * mozilla doesn't accept these signals, so we connect these funtions to
 * KzWindow instead of KzMozEmbed.
 */
static gboolean motion_notify_event_cb  (GtkWidget *widget,
					 GdkEventMotion *event,
					 KzMozEmbed *kzembed);
static gboolean button_release_event_cb (GtkWidget *widget,
					 GdkEventButton *event,
					 KzMozEmbed *kzembed);

/* notebook received dropped url */
static void drag_data_recieved_cb (GtkWidget *widget,
				   GdkDragContext *drag_context,
				   gint x, gint y,
				   GtkSelectionData *data,
				   guint info,
				   guint time,
				   KzWindow *kz);

#warning FIXME!!
#if 1
/* gesture sequnce */
KzGestureMotion gesture_go_back[]    = {KZ_GESTURE_MOTION_LEFT,
					KZ_GESTURE_MOTION_TERMINATOR};
KzGestureMotion gesture_go_forward[] = {KZ_GESTURE_MOTION_RIGHT,
					KZ_GESTURE_MOTION_TERMINATOR};
KzGestureMotion gesture_close_tab[]  = {KZ_GESTURE_MOTION_UP,
					KZ_GESTURE_MOTION_TERMINATOR};
KzGestureMotion gesture_reload[]     = {KZ_GESTURE_MOTION_DOWN,
					KZ_GESTURE_MOTION_TERMINATOR};
KzGestureMotion gesture_home[]       = {KZ_GESTURE_MOTION_DOWN,
					KZ_GESTURE_MOTION_UP,
					KZ_GESTURE_MOTION_TERMINATOR};
KzGestureMotion gesture_left_tab[]   = {KZ_GESTURE_MOTION_LEFT,
					KZ_GESTURE_MOTION_RIGHT,
					KZ_GESTURE_MOTION_TERMINATOR};
KzGestureMotion gesture_right_tab[]  = {KZ_GESTURE_MOTION_RIGHT,
					KZ_GESTURE_MOTION_LEFT,
					KZ_GESTURE_MOTION_TERMINATOR};
#endif


#define CTRL "<control>"
#define SHFT "<shift>"
#define SFCT "<shift><control>"
static EggActionGroupEntry action_entries[] = {
  {"StockFileMenu", N_("_File"), NULL, NULL, NULL, NULL, NULL},
  {"StockEditMenu", N_("_Edit"), NULL, NULL, NULL, NULL, NULL},
  {"StockViewMenu", N_("_View"), NULL, NULL, NULL, NULL, NULL},
  {"StockGoMenu",   N_("_Go"),   NULL, NULL, NULL, NULL, NULL},
  {"StockHelpMenu", N_("_Help"), NULL, NULL, NULL, NULL, NULL},
  {"NewWindow",     N_("_New Window"),    GTK_STOCK_NEW,        CTRL"N",  NULL, G_CALLBACK(act_new_window),   NULL},
  {"NewTab",        N_("New _Tab"),       GTK_STOCK_NEW,        CTRL"T",  NULL, G_CALLBACK(act_new_tab),      NULL},
  {"CloseWindow",   N_("_Close Window"),  GTK_STOCK_CLOSE,      CTRL"W",  NULL, G_CALLBACK(act_close_window), NULL},
  {"CloseTab",      N_("Close Ta_b"),     GTK_STOCK_CLOSE,      SFCT"W",  NULL, G_CALLBACK(act_close_tab),    NULL},
  {"Home",          N_("_Home"),          GTK_STOCK_HOME,       NULL,     NULL, G_CALLBACK(act_home),         NULL},
  {"Back",          N_("_Back"),          GTK_STOCK_GO_BACK,    SFCT"B",  NULL, G_CALLBACK(act_back),         NULL},
  {"Forward",       N_("_Forward"),       GTK_STOCK_GO_FORWARD, SFCT"F",  NULL, G_CALLBACK(act_forward),      NULL},
  {"Reload",        N_("_Reload"),        GTK_STOCK_REFRESH,    SFCT"R",  NULL, G_CALLBACK(act_reload),       NULL},
  {"Stop",          N_("_Stop"),          GTK_STOCK_STOP,       "Escape", NULL, G_CALLBACK(act_stop),         NULL},
  {"LocationEntry", N_("Location Entry"), GTK_STOCK_NEW,        NULL,     NULL, G_CALLBACK(act_location_enter), NULL, ENTRY_ACTION},
  {"Search",        N_("Search"),         NULL,                 CTRL"F",  NULL, G_CALLBACK(act_search),       NULL},
  {"LeftTab",       N_("_Left Tab"),      GTK_STOCK_GO_BACK,    CTRL"L",  NULL, G_CALLBACK(act_left_tab),     NULL},
  {"RightTab",      N_("_Right Tab"),     GTK_STOCK_GO_FORWARD, CTRL"R",  NULL, G_CALLBACK(act_right_tab),    NULL},
  {"Preference",    N_("_Preference"),    GTK_STOCK_PREFERENCES,NULL,     NULL, G_CALLBACK(act_preference),   NULL},
};
#undef CTRL
#undef SHFT
#undef SFCT


static GtkWindowClass *parent_class = NULL;
static GList *window_list = NULL;

KZ_OBJECT_GET_TYPE(kz_window, "KzWindow", KzWindow,
		   kz_window_class_init, kz_window_init,
		   GTK_TYPE_WINDOW)

static void
kz_window_class_init (KzWindowClass *class)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (class);

	gobject_class = (GObjectClass *)   class;
	object_class  = (GtkObjectClass *) class;
	widget_class  = (GtkWidgetClass *) class;

	/* GObjectClass */
	gobject_class->finalize = kz_window_finalize;

	/* GtkObject signals */
	object_class->destroy = kz_window_destroy;

	/* GtkWidget signals */
	widget_class->delete_event = kz_window_delete_event;
}


static void
kz_window_init (KzWindow *kz)
{
	GtkWidget *menu_box;
	gint i;
	GtkSettings *setting;
	GError *err;

	g_return_if_fail(KZ_IS_WINDOW(kz));
	
	g_object_set(G_OBJECT(kz), "allow-shrink", TRUE, NULL);

	kz->top_vbox           = gtk_vbox_new(FALSE, 0);

        kz->url_entry          = NULL;
	kz->progress_area_hbox = NULL;
        kz->progressbar        = NULL;
	kz->status_align       = gtk_alignment_new(0, 0, 1, 1);
	kz->statusbar          = gtk_statusbar_new();
	kz->notebook           = gtk_notebook_new();
	kz->find_area_hbox     = gtk_hbox_new(FALSE, 0);
	kz->find_area          = gtk_entry_new();
	kz->kztoolbar          = GTK_WIDGET(kz_bookmark_bar_new(kz));

	kz->accel_group        = gtk_accel_group_new();
	kz->action_group       = egg_action_group_new("KzWindow");;
	kz->menu_merge         = egg_menu_merge_new();
	kz->gesture_items      = NULL;

	kz->priv               = g_new0(KzWindowPriv, 1);
	kz->priv->kzhome       = kz_home_new(kz);
	kz->priv->status_msg   = NULL;
	kz->priv->temp_msg     = NULL;

	kz->priv->did_find     = FALSE; /* for find keyword */

	/* actions and gesture definition */
	egg_action_group_set_accel_group(kz->action_group, kz->accel_group);
	for (i = 0; i < G_N_ELEMENTS(action_entries); i++)
		action_entries[i].user_data = kz;
	egg_action_group_add_actions (kz->action_group,
				      action_entries,
				      G_N_ELEMENTS(action_entries));
	for (i = 0; i < G_N_ELEMENTS(action_entries); i++)
		action_entries[i].user_data = NULL;

	kz_window_gesture_update(kz);

	/* top level vbox */
	gtk_container_add(GTK_CONTAINER(kz),
			  kz->top_vbox);
	gtk_widget_show(kz->top_vbox);

	/* menu & toolbar */
	menu_box = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(kz->top_vbox), menu_box,
			   FALSE, FALSE, 0);   
	gtk_widget_show(menu_box);

	egg_menu_merge_set_accel_group(kz->menu_merge, kz->accel_group);
	egg_menu_merge_insert_action_group(kz->menu_merge, kz->action_group, 0);
	g_signal_connect(kz->menu_merge, "add_widget",
			 G_CALLBACK(menu_merge_add_widget_cb), menu_box);
	egg_menu_merge_add_ui_from_file(kz->menu_merge,
					KZ_DATA_DIR"/kz-ui-menu.xml",
					&err);

	/* FIXME */
	kz_window_menu_merge_set_widget (kz);

	setting = gtk_settings_get_default();
	if (setting)
	{
		/* FIXME */
		gtk_settings_set_long_property (setting,
						"gtk-toolbar-style",
						GTK_TOOLBAR_ICONS,
						"");
	}

	gtk_window_add_accel_group(GTK_WINDOW(kz), kz->accel_group);

	/* KzWindow tool bar */
	gtk_box_pack_start(GTK_BOX(kz->top_vbox), 
			   kz->kztoolbar, FALSE, FALSE, 0);    
	gtk_widget_show(kz->kztoolbar);

	/* create notebook widget */
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(kz->notebook), TRUE);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(kz->notebook), TRUE);
	g_signal_connect(G_OBJECT(kz->notebook), "switch_page", 
			 G_CALLBACK(notebook_switch_page_cb), kz);
	
	/* drop url */
	gtk_drag_dest_set(GTK_WIDGET(kz->notebook),
			  GTK_DEST_DEFAULT_ALL,
                          url_drag_types,n_url_drag_types,
			  GDK_ACTION_MOVE);
	g_signal_connect(G_OBJECT(kz->notebook), "drag-data-received",
			 G_CALLBACK(drag_data_recieved_cb), kz);

	gtk_container_add(GTK_CONTAINER(kz->top_vbox), kz->notebook);
	gtk_widget_show(kz->notebook);

	/* mozembed tab */
	kz_window_open_new_tab(kz, NULL);

	/* search area */
	gtk_box_pack_start(GTK_BOX(kz->top_vbox), kz->find_area_hbox,
			   FALSE, FALSE, 0);   
	gtk_widget_show(kz->find_area_hbox);

	/* find area */
	gtk_box_pack_start(GTK_BOX(kz->find_area_hbox), kz->find_area,
			   FALSE, FALSE, 0); 
	g_signal_connect(G_OBJECT(GTK_ENTRY(kz->find_area)), "activate",
			 G_CALLBACK(find_keyword_cb), kz);
	gtk_widget_show(kz->find_area);

	/* create our status area and the alignment object that will keep it */
	/* from expanding */
	gtk_widget_set_size_request(kz->status_align, 1, -1);
	gtk_widget_show(kz->status_align);
	/* create the status bar */
	gtk_container_add(GTK_CONTAINER(kz->status_align), kz->statusbar);
	gtk_box_pack_start(GTK_BOX(kz->find_area_hbox), kz->status_align,
			   TRUE, TRUE, 0);
	gtk_widget_show(kz->statusbar);

	KZ_WINDOW_SET_SENSITIVE(kz, "Back",    FALSE);
	KZ_WINDOW_SET_SENSITIVE(kz, "Stop",    FALSE);
	KZ_WINDOW_SET_SENSITIVE(kz, "Forward", FALSE);
	KZ_WINDOW_SET_SENSITIVE(kz, "Reload",  FALSE);
	KZ_WINDOW_SET_VISIBLE  (kz, "Stop",    FALSE);
	KZ_WINDOW_SET_VISIBLE  (kz, "Reload",  TRUE);

	/* FIXME! set last window size! */ 
	gtk_window_set_default_size(GTK_WINDOW(kz), 
				    prefs.width, prefs.height);

	gtk_widget_show(GTK_WIDGET(kz));

	window_list = g_list_append(window_list, kz);
}

GList *
kz_window_get_window_list(void)
{
	return window_list;
}

GtkWidget *
kz_window_new(const gchar *url)
{
	KzWindow *kz = KZ_WINDOW(g_object_new(KZ_TYPE_WINDOW,
					      "type", GTK_WINDOW_TOPLEVEL,
					      NULL));
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_load_url(GTK_MOZ_EMBED(widget), url);

	return GTK_WIDGET (kz);
}

GtkWidget *
kz_window_get_from_tab (GtkWidget *tab_widget)
{
	GList *node;

	for (node = window_list; node; node = g_list_next (node))
	{
		KzWindow *kz = node->data;
		GtkWidget *label;

		if (!KZ_IS_WINDOW(kz)) continue;

		label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(kz->notebook),
						   tab_widget);
		if (label)
			return GTK_WIDGET(kz);
	}

	return NULL;
}

static void
cb_gesture_stack_motion(KzGesture *gesture, KzGestureMotion motion, KzWindow *kz)
{
	const gchar *label;
	gchar buf1[256], buf2[256];

	g_return_if_fail(KZ_IS_GESTURE(gesture));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kz_gesture_create_gesture_string(gesture, buf1, G_N_ELEMENTS(buf1));
	label = kz_gesture_get_matched_label(gesture);
	if (label)
		g_snprintf(buf2, G_N_ELEMENTS(buf2),
			   "Gesture: %s(Action: %s)", buf1, label);
	else
		g_snprintf(buf2, G_N_ELEMENTS(buf2),
			   "Gesture: %s", buf1);

	gtk_statusbar_push(GTK_STATUSBAR(kz->statusbar),
			   STATUS_GESTURE, buf2);
}

GtkWidget *
kz_window_open_new_tab (KzWindow *kz, const gchar *url)
{
	KzMozEmbed *kzembed;
	KzTabLabel *kztab;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);

	kzembed = KZ_MOZ_EMBED(kz_moz_embed_new(url));
	kztab = KZ_TAB_LABEL(kz_tab_label_new(kz, kzembed));

	kz_gesture_set_items(kzembed->gesture, kz->gesture_items);
	g_signal_connect(G_OBJECT(kzembed->gesture), "stack_motion",
			 G_CALLBACK(cb_gesture_stack_motion), kz);

	kz_window_set_moz_callbacks(kz, kzembed);

	gtk_widget_show(GTK_WIDGET(kzembed));
	gtk_widget_show(GTK_WIDGET(kztab));

	gtk_notebook_append_page(GTK_NOTEBOOK(kz->notebook),
				 GTK_WIDGET(kzembed),
				 GTK_WIDGET (kztab));

	return GTK_WIDGET(kzembed);
}

void
kz_window_close_tab (KzWindow *kz, GtkWidget *widget)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(GTK_IS_WIDGET(widget));

	if (KZ_IS_MOZ_EMBED(widget))
	{
		kz_window_unset_moz_callbacks(kz, KZ_MOZ_EMBED(widget));
	}

	gtk_widget_destroy(widget);
}

typedef struct _CloseTab
{
	KzWindow  *kz;
	GtkWidget *widget;
} CloseTab;

static gboolean
idle_close_tab (gpointer data)
{
	CloseTab *close_tab = data;
	kz_window_close_tab(close_tab->kz, close_tab->widget);
	g_free(close_tab);

	return FALSE;
}

void
kz_window_close_tab_idle (KzWindow *kz, GtkWidget *widget)
{
	CloseTab *close_tab;

	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(GTK_IS_WIDGET(widget));

	close_tab = g_new(CloseTab, 1);
	close_tab->kz = kz;
	close_tab->widget = widget;

	gtk_idle_add(idle_close_tab, close_tab);
}

void
kz_window_move_tab (KzWindow *kz, GtkWidget *widget)
{
	GtkNotebook *src_notebook;
	GtkWidget *label;
	KzWindow *src_kz;
	KzMozEmbed *kzembed;
	KzTabLabel *new_kztab;

	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_MOZ_EMBED(widget));

	kzembed = KZ_MOZ_EMBED(widget);

	src_kz = KZ_WINDOW(kz_window_get_from_tab(GTK_WIDGET(kzembed)));
	if (!src_kz) return;

	label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(kz->notebook),
					   GTK_WIDGET(kzembed));

	/* the kzembed is already the kz's child */
	if (label) return;

	src_notebook = GTK_NOTEBOOK(src_kz->notebook);
	label = gtk_notebook_get_tab_label(src_notebook,
					   GTK_WIDGET(kzembed));
	g_return_if_fail(label);

	/* create new tab label */
	new_kztab = KZ_TAB_LABEL(kz_tab_label_new(kz, kzembed));

	/* move the page to this window */
	kz_window_unset_moz_callbacks(src_kz, kzembed);
	gtk_widget_reparent(GTK_WIDGET(kzembed), kz->notebook);
	gtk_notebook_set_tab_label(GTK_NOTEBOOK(kz->notebook),
				   GTK_WIDGET(kzembed),
				   GTK_WIDGET(new_kztab));
	kz_window_set_moz_callbacks(kz, kzembed);
	gtk_widget_destroy(GTK_WIDGET(label));
}

static gboolean
kz_window_delete_event (GtkWidget *widget, GdkEventAny *event)
{
	KzWindow *kz;
	gint n_pages, i;

	g_return_val_if_fail(KZ_IS_WINDOW(widget), FALSE);

	kz = KZ_WINDOW(widget);

	n_pages = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	for (i = n_pages - 1; i >= 0; i--)
	{
		GtkWidget *widget = GTK_WIDGET(KZ_WINDOW_NTH_PAGE(kz, i));
		gtk_widget_destroy(widget);
	}

	return FALSE;
}

static void
kz_window_destroy (GtkObject *object)
{
	KzWindow *kz = KZ_WINDOW(object);

	g_return_if_fail (KZ_IS_WINDOW(kz));

	window_list = g_list_remove(window_list, kz);

	if (kz->priv->kzhome)
		g_object_unref(kz->priv->kzhome);
	kz->priv->kzhome = NULL;

	g_free(kz->priv->temp_msg);
        kz->priv->temp_msg = NULL;

	if (kz->menu_merge)
		g_object_unref(kz->menu_merge);
	kz->menu_merge = NULL;

	if (kz->action_group)
		g_object_unref(kz->action_group);
	kz->action_group = NULL;

	if (kz->gesture_items)
		kz_gesture_items_unref(kz->gesture_items);
	kz->gesture_items = NULL;

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy(object);

	if (!kz_window_get_window_list())
		gtk_main_quit();
}

static void
kz_window_finalize (GObject *object)
{
	KzWindow *kz = KZ_WINDOW(object);

	if (kz->priv) {
		g_free(kz->priv);
		kz->priv = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->finalize)
		G_OBJECT_CLASS (parent_class)->finalize(object);
}

static void
kz_window_gesture_update (KzWindow *kz)
{
#warning FIXME
	if (!kz->gesture_items)
	{
		EggAction *action;

		kz->gesture_items = kz_gesture_items_new();

		action = egg_action_group_get_action(kz->action_group, "Back");
		kz_gesture_items_add_action(kz->gesture_items, action,
					    0, gesture_go_back);

		action = egg_action_group_get_action(kz->action_group, "Forward");
		kz_gesture_items_add_action(kz->gesture_items, action,
					    0, gesture_go_forward);

		action = egg_action_group_get_action(kz->action_group, "CloseTab");
		kz_gesture_items_add_action(kz->gesture_items, action,
					    0, gesture_close_tab);

		action = egg_action_group_get_action(kz->action_group, "Reload");
		kz_gesture_items_add_action(kz->gesture_items, action,
					    0, gesture_reload);

		action = egg_action_group_get_action(kz->action_group, "LeftTab");
		kz_gesture_items_add_action(kz->gesture_items, action,
					    0, gesture_left_tab);

		action = egg_action_group_get_action(kz->action_group, "RightTab");
		kz_gesture_items_add_action(kz->gesture_items, action,
					    0, gesture_right_tab);
	}
	else
	{
	}
}

/* FIXME */
static void
kz_window_menu_merge_set_widget (KzWindow *kz)
{
	GtkWidget *toolbar;
	EggAction *action;
	GSList *node;

	/*
	 *  set Google bar
	 */
	toolbar = egg_menu_merge_get_widget(kz->menu_merge, "/MainToolBar");
	if (EGG_IS_TOOLBAR(toolbar))
	{
		EggToolItem *toolitem = egg_tool_item_new();
		GtkWidget *button = google_new(kz);

		gtk_container_add(GTK_CONTAINER(toolitem), button);
		gtk_widget_show(button);
		gtk_widget_show(GTK_WIDGET(toolitem));
		egg_toolbar_append_tool_item (EGG_TOOLBAR(toolbar),
					      toolitem);
	}

	/*
	 *  set Home action
	 */
	action = egg_action_group_get_action(kz->action_group, "Home");
	for (node = action->proxies; action && node; node = g_slist_next (node))
	{
		GtkWidget *widget = node->data;

		kz_home_set_widget (kz->priv->kzhome, widget);
	}
}

static void
kz_window_set_moz_callbacks(KzWindow *kz, KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	kz_gesture_set_items(kzembed->gesture, kz->gesture_items);
	g_signal_connect(G_OBJECT(kzembed->gesture), "stack_motion",
			 G_CALLBACK(cb_gesture_stack_motion), kz);

	/*
	 * key event signal
	 */
	g_signal_connect(G_OBJECT(kzembed),
			 "dom_key_up",
			 G_CALLBACK(dom_key_up_cb), kz);

	/*
	 * mouse event signal
	 */
	g_signal_connect(G_OBJECT(kzembed),
			 "dom_mouse_down",
			 G_CALLBACK(dom_mouse_down_cb), kz);
	g_signal_connect(G_OBJECT(kz),
			 "motion_notify_event",
			 G_CALLBACK(motion_notify_event_cb), kzembed);
	g_signal_connect(G_OBJECT(kz),
			 "button_release_event",
			 G_CALLBACK(button_release_event_cb), kzembed);

	/*
	 * other callbacks
	 */
	g_signal_connect(G_OBJECT(kzembed), "title",
			 G_CALLBACK(title_changed_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "location",
			 G_CALLBACK(location_changed_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "link_message",
			 G_CALLBACK(link_message_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "net_start",
			 G_CALLBACK(load_started_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "net_stop",
			 G_CALLBACK(load_finished_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "new_window",
			 G_CALLBACK(moz_new_window_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "dom_mouse_click",
			 G_CALLBACK(dom_mouse_click_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "dom_mouse_dbl_click",
			 G_CALLBACK(dom_mouse_dbl_click_cb), kz);
	g_signal_connect(G_OBJECT(kzembed), "destroy",
			 G_CALLBACK(close_tab_cb), kz);
}

static void
kz_window_unset_moz_callbacks(KzWindow *kz, KzMozEmbed *kzembed)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	g_return_if_fail(KZ_IS_MOZ_EMBED(kzembed));

	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed->gesture),
					     G_CALLBACK(cb_gesture_stack_motion),
					     kz);

	/*
	 * key event signal
	 */
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(dom_key_up_cb),
					     kz);

	/*
	 * mouse event signal
	 */
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(dom_mouse_down_cb),
					     kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kz),
					     G_CALLBACK(motion_notify_event_cb),
					     kzembed);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kz),
					     G_CALLBACK(button_release_event_cb),
					     kzembed);

	/*
	 * other callbacks
	 */
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(title_changed_cb), kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(location_changed_cb),
					     kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(link_message_cb),
					     kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(load_started_cb),
					     kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(load_finished_cb),
					     kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(moz_new_window_cb),
					     kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(dom_mouse_click_cb),
					     kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(dom_mouse_dbl_click_cb),
					     kz);
	g_signal_handlers_disconnect_by_func(G_OBJECT(kzembed),
					     G_CALLBACK(close_tab_cb),
					     kz);
}

static void
kz_window_update_nav_buttons (KzWindow *kz, KzMozEmbed *kzembed)
{
	GtkMozEmbed *embed;
	gboolean can_go_back;
	gboolean can_go_forward;
	gboolean can_stop;

	g_return_if_fail (KZ_IS_WINDOW(kz));
	g_return_if_fail (KZ_IS_MOZ_EMBED (kzembed));

	embed = GTK_MOZ_EMBED(kzembed);

	can_go_back = gtk_moz_embed_can_go_back(embed);
	can_go_forward = gtk_moz_embed_can_go_forward(embed);
	can_stop = kz_moz_embed_is_loading(KZ_MOZ_EMBED(embed));

	KZ_WINDOW_SET_SENSITIVE(kz, "Back",    can_go_back);
	KZ_WINDOW_SET_SENSITIVE(kz, "Forward", can_go_forward);
	KZ_WINDOW_SET_VISIBLE  (kz, "Stop",    can_stop);
	KZ_WINDOW_SET_VISIBLE  (kz, "Reload",  !can_stop);
	KZ_WINDOW_SET_SENSITIVE(kz, "Stop",    can_stop);
	KZ_WINDOW_SET_SENSITIVE(kz, "Reload",  !can_stop);
}

void
kz_window_set_visibility (KzWindow *kz, gboolean visibility)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (!visibility)
		gtk_widget_hide(GTK_WIDGET (kz));
	else
		gtk_widget_show(GTK_WIDGET(kz));
}

void
kz_window_load_url (KzWindow *kz, const gchar *url)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_load_url(GTK_MOZ_EMBED(widget), url);
}

gchar *
kz_window_get_title (KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);

	if (GTK_IS_MOZ_EMBED(widget))
		return mozilla_get_title(GTK_MOZ_EMBED(widget));
	else
		return NULL;
}

gchar *
kz_window_get_uri (KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);

	if (GTK_IS_MOZ_EMBED(widget))
		return gtk_moz_embed_get_location(GTK_MOZ_EMBED(widget));
	else
		return NULL;
}

static void
act_close_window (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	gtk_widget_destroy(GTK_WIDGET(kz));
}

static void
act_close_tab (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);
	if (widget)
		kz_window_close_tab_idle(kz, widget);
}

static void
act_new_window (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	kz_window_new(NULL);
}

static void
act_new_tab (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	kz_window_open_new_tab(kz, NULL);
}

static void
act_home (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	kz_home_load_first_home(kz->priv->kzhome);
}

static void
act_back (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_go_back(GTK_MOZ_EMBED(widget));
}

static void 
act_stop (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_stop_load(GTK_MOZ_EMBED(widget));
}

static void
act_forward (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget = KZ_WINDOW_CURRENT_PAGE(kz);

	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_go_forward(GTK_MOZ_EMBED(widget));
}

static void
act_reload  (EggAction *action, KzWindow *kz)
{
	GdkModifierType state = (GdkModifierType)0;
	gint x, y;
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW(kz));

	gdk_window_get_pointer(NULL, &x, &y, &state);

	widget = KZ_WINDOW_CURRENT_PAGE(kz);

	if (GTK_IS_MOZ_EMBED(widget))
		gtk_moz_embed_reload(GTK_MOZ_EMBED(widget),
				     (state & GDK_SHIFT_MASK) ?
				     GTK_MOZ_EMBED_FLAG_RELOADBYPASSCACHE : 
				     GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}

static void
act_location_enter (EggAction *action, KzWindow *kz)
{
	const gchar *uri;

	g_return_if_fail(EGG_IS_ENTRY_ACTION(action));

	uri = egg_entry_action_get_text(EGG_ENTRY_ACTION(action));
	gtk_moz_embed_load_url(GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)), uri);
}

static void
act_search (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	gtk_widget_grab_focus(kz->find_area);
}

static void
act_left_tab (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	gtk_notebook_prev_page(GTK_NOTEBOOK(kz->notebook));
}

static void
act_right_tab (EggAction *action, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	gtk_notebook_next_page(GTK_NOTEBOOK(kz->notebook));
}


static void
act_preference (EggAction *action, KzWindow *kz)
{
	GtkWidget *widget;

	g_return_if_fail(KZ_IS_WINDOW (kz));

	widget = kz_prefs_win_new ();
	gtk_window_set_transient_for (GTK_WINDOW(widget), GTK_WINDOW(kz));
	gtk_widget_show (widget);
}

static void
menu_merge_add_widget_cb (EggMenuMerge *merge, GtkWidget *widget, GtkBox *box)
{
	gtk_box_pack_start (box, widget, FALSE, FALSE, 0);
	gtk_widget_show (widget);
}

static void
notebook_switch_page_cb (GtkNotebook *notebook, GtkNotebookPage *page,
			 guint page_num, KzWindow *kz)
{
	KzMozEmbed *kzembed = KZ_MOZ_EMBED(KZ_WINDOW_NTH_PAGE(kz, page_num));
	const gchar *newTitle, *newLocation;
	EggAction *action;

	newTitle = kz_moz_embed_get_title(kzembed);
	newLocation = kz_moz_embed_get_location(kzembed);
	if (newTitle)
	{
		gtk_window_set_title(GTK_WINDOW(kz), newTitle);
	}
	else
	{
		gtk_window_set_title(GTK_WINDOW(kz), _("Kazehakase"));
	}
	
	action = egg_action_group_get_action(kz->action_group, "LocationEntry");

	if (EGG_IS_ENTRY_ACTION(action))
	{
		if (newLocation)
			egg_entry_action_set_text(EGG_ENTRY_ACTION(action),
						  newLocation);
		else
			egg_entry_action_set_text(EGG_ENTRY_ACTION(action), "");
	}
	kz_window_update_nav_buttons(kz, kzembed);
}

static void
find_keyword_cb(GtkWidget *widget, KzWindow *kz)
{
	GtkMozEmbed *embed = GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz));
	const gchar *text = gtk_entry_get_text(GTK_ENTRY(widget));

	mozilla_find(embed, text, &kz->priv->did_find);
}

static void
title_changed_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	if (embed == GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz))) {
		const gchar *title = kz_moz_embed_get_title(KZ_MOZ_EMBED(embed));
		gtk_window_set_title(GTK_WINDOW(kz), (gchar *) title);
	}
}

static void
location_changed_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	const char *newLocation;

	g_return_if_fail (KZ_IS_MOZ_EMBED(embed));
	g_return_if_fail (KZ_IS_WINDOW(kz));

	newLocation = kz_moz_embed_get_location(KZ_MOZ_EMBED(embed));
	if (newLocation)
	{
		EggAction *action
			= egg_action_group_get_action(kz->action_group,
						      "LocationEntry");
		if (embed == GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz))
		    && EGG_IS_ENTRY_ACTION(action))
		{
			egg_entry_action_set_text(EGG_ENTRY_ACTION(action),
						  newLocation);
		}
	}
	else
	{
		g_print("failed to get location!\n");
	}

	kz_window_update_nav_buttons(kz, KZ_MOZ_EMBED(embed));
}

static void
link_message_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	gchar *newMessage;

	g_return_if_fail (KZ_IS_MOZ_EMBED(embed));
	g_return_if_fail (KZ_WINDOW(kz));

	newMessage = mozilla_get_link_message(embed);
	if(kz->statusbar)
		gtk_statusbar_push(GTK_STATUSBAR(kz->statusbar),
				   STATUS_LINK_MESSAGE, newMessage);

#if 0
	if (KZ_MOZ_EMBED(embed)->popup_window)
	{
		/* destroy popup window*/
		gtk_widget_destroy(kzembed->popup_window);
		kzembed->popup_window = NULL;
	}
	else if (newMessage != NULL)
	{
		/* fetch image */ 
		/* create popup window */
		kzembed->popup_window = gtk_window_new(GTK_WINDOW_POPUP);
		gtk_window_set_transient_for(GTK_WINDOW(kzembed->popup_window), 
					     GTK_WINDOW(kz->topLevelWindow));
		gtk_widget_show(kzembed->popup_window);
	}
#endif

	g_free(newMessage);
}

#if 0
static void
visibility_cb (GtkMozEmbed *embed, gboolean visibility, KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));
	kz_window_set_visibility(kz, visibility);
}
#endif

static void
load_started_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	if (embed == GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)))
		kz_window_update_nav_buttons(kz, KZ_MOZ_EMBED(embed));
}

static void
load_finished_cb (GtkMozEmbed *embed, KzWindow *kz)
{
	if (embed == GTK_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kz)))
		kz_window_update_nav_buttons(kz, KZ_MOZ_EMBED(embed));
}

static void 
destroy_browser_cb (GtkMozEmbed *embed, GtkWidget *transient_window)
{
	gtk_widget_destroy(GTK_WIDGET(transient_window));
}

static void
moz_new_window_cb (GtkMozEmbed *embed,
		   GtkMozEmbed **newEmbed, guint chromemask,
		   KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	if ((chromemask & GTK_MOZ_EMBED_FLAG_OPENASCHROME) != 0)
	{
		/* FIXME! this is ad hoc. */
		GtkWidget *newWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);

		gtk_window_set_transient_for(GTK_WINDOW(newWindow),
					     GTK_WINDOW(kz));
		*newEmbed = GTK_MOZ_EMBED(kz_moz_embed_new(NULL));
		g_signal_connect(G_OBJECT(*newEmbed),"destroy_browser",
				 G_CALLBACK(destroy_browser_cb),newWindow);
		gtk_container_add(GTK_CONTAINER(newWindow),GTK_WIDGET(*newEmbed));
	}
	else
	{
		*newEmbed = GTK_MOZ_EMBED(kz_window_open_new_tab(kz, NULL));
		gtk_widget_show (GTK_WIDGET(*newEmbed));
	}
}

static void
close_tab_cb (GtkObject *obj, KzWindow *kz)
{
#if 0
	KzMozEmbed *kzembed;
	gint n, n_pages;

	g_return_if_fail(KZ_IS_MOZ_EMBED(obj));
	g_return_if_fail(KZ_IS_WINDOW(kz));

	kzembed = KZ_MOZ_EMBED(obj);
	n = gtk_notebook_page_num(GTK_NOTEBOOK(kz->notebook),
				  GTK_WIDGET(kzembed));
	n_pages = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));

	if (n_pages <= 0)
	{
		gtk_moz_embed_pop_startup();
		/* gtk_window_set_title(GTK_WINDOW(kz), _("Kazehakase")); */
	}
	else if (n_pages >= n)
	{
		gtk_notebook_set_current_page(GTK_NOTEBOOK(kz->notebook), n);
	}
#endif
}

static gint
dom_key_up_cb (GtkMozEmbed *embed, gpointer event, KzWindow *kz)
{

	g_return_val_if_fail(KZ_IS_MOZ_EMBED(embed), FALSE);
	g_return_val_if_fail(KZ_IS_WINDOW(kz), FALSE);

	if (!prefs.typeahead)
	{
		KzMozEmbed *kzembed;
		gulong key_code;
		glong type;
		KeyEventInfo info;

		kzembed = KZ_MOZ_EMBED(embed);
		
		mozilla_get_key_code(event, &key_code);
		
		type = mozilla_get_key_event_info(embed, event,&info);

		if ((key_code == 191) && !(type & CONTEXT_INPUT)) /* '/' */
		{
			gtk_widget_grab_focus(kz->find_area);
		}
	}

	return FALSE;
}

#warning FIXME
static gint
dom_mouse_click_cb (GtkMozEmbed *embed, gpointer event, KzWindow *kz)
{
	gint button;
	guint32 time;
     	glong type;
     	MouseEventInfo info;

	g_return_val_if_fail(GTK_IS_MOZ_EMBED(embed), FALSE);
	g_return_val_if_fail(KZ_IS_WINDOW(kz), FALSE);

	time = GDK_CURRENT_TIME;
	
     	type = mozilla_get_mouse_event_info(embed, event, &info);
	button = mozilla_get_button(event);

	switch (button) {
	 case 0:
		 if ((info.modifier & SHIFT_KEY) &&
		     (type & CONTEXT_LINK) &&
		     info.cinfo.link)
		 {
			 kz_window_open_new_tab(kz, info.cinfo.link);
			 return TRUE;
		 }
		 break;
	 case 1:
		 if ((type & CONTEXT_LINK) && info.cinfo.link)
		 {
			 kz_window_open_new_tab(kz, info.cinfo.link);
		 }
		 break;
	}

	return FALSE;
}

static gint
dom_mouse_dbl_click_cb (GtkMozEmbed *embed, gpointer event, KzWindow *kz)
{
#if 0
	gint button;

	g_return_val_if_fail(GTK_IS_MOZ_EMBED(embed), FALSE);
	g_return_val_if_fail(KZ_WINDOW(kz), FALSE);

	button = mozilla_get_button(event);
	switch (button) {
	 case 0:
		 gtk_notebook_prev_page(GTK_NOTEBOOK(kz->notebook));
		 break;
	 case 2:	
		 gtk_notebook_next_page(GTK_NOTEBOOK(kz->notebook));
		 break;
	}
#endif

	return FALSE;
}

static gint
dom_mouse_down_cb (GtkMozEmbed *embed, gpointer event, KzWindow *kz)
{
	KzMozEmbed *kzembed;
	gint button;
     	glong type;
     	MouseEventInfo info;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), FALSE);
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(embed), FALSE);

	kzembed = KZ_MOZ_EMBED(embed);

     	type = mozilla_get_mouse_event_info(embed, event, &info);	
	button = mozilla_get_button(event);
	
	if (button == 2) 
	{
		static GdkCursor *cursor = NULL;
		glong x_pos, y_pos;
		gint x, y, win_x, win_y, win_x_pos, win_y_pos;

 		mozilla_get_mouse_position(event, &x_pos, &y_pos);
		gdk_window_get_root_origin (GTK_WIDGET(kzembed)->window,
					    &win_x, &win_y);
		gdk_window_get_position (GTK_WIDGET(kzembed)->window,
					 &win_x_pos, &win_y_pos);

		gtk_widget_get_pointer (GTK_WIDGET(kzembed),
					&x, &y);
		kz_gesture_start(kzembed->gesture, 0, x, y);

		if ((type & CONTEXT_LINK) && info.cinfo.link)
			kzembed->gesture_location = g_strdup(info.cinfo.link);

		if (!cursor) cursor = gdk_cursor_new (GDK_HAND1);
		gdk_pointer_grab (GTK_WIDGET(kz)->window,
				  FALSE,
				  GDK_POINTER_MOTION_MASK |
				  GDK_BUTTON_RELEASE_MASK |
				  GDK_BUTTON_PRESS_MASK,
				  NULL, cursor, gtk_get_current_event_time ());
		gtk_statusbar_push(GTK_STATUSBAR(kz->statusbar),
				   STATUS_GESTURE, "Gesture:");
 	}
	
	return FALSE;
}

static gboolean
motion_notify_event_cb (GtkWidget *widget,
			GdkEventMotion *event,
			KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WINDOW(widget), FALSE);
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);

	if (kz_gesture_is_started(kzembed->gesture)) {
		gint win_x, win_y, win_x_pos, win_y_pos, x, y;

		gdk_window_get_root_origin (GTK_WIDGET(kzembed)->window,
					    &win_x, &win_y);
		gdk_window_get_position (GTK_WIDGET(kzembed)->window,
					 &win_x_pos, &win_y_pos);
		gtk_widget_get_pointer (GTK_WIDGET(kzembed),
					&x, &y);
		kz_gesture_update_position(kzembed->gesture, x, y);
	}

	return FALSE;
}

static gboolean
button_release_event_cb (GtkWidget *widget,
			 GdkEventButton *event,
			 KzMozEmbed *kzembed)
{
	g_return_val_if_fail(KZ_IS_WINDOW(widget), FALSE);
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), FALSE);
	if (kz_gesture_is_started(kzembed->gesture))
	{
		if (event->button == 3)
			kz_gesture_perform(kzembed->gesture);
		else
			kz_gesture_cancel(kzembed->gesture);
		gtk_statusbar_push(GTK_STATUSBAR(KZ_WINDOW(widget)->statusbar),
				   STATUS_GESTURE, "");
	}

        if (gdk_pointer_is_grabbed ())
        {
                gdk_pointer_ungrab (gtk_get_current_event_time ());
        }

	g_free(kzembed->gesture_location);
	kzembed->gesture_location = NULL;

	return FALSE;
}

static void
drag_data_recieved_cb (GtkWidget *widget,
		       GdkDragContext *drag_context,
		       gint x, gint y,
		       GtkSelectionData *data,
		       guint info,
		       guint time,
		       KzWindow *kz)
{
	g_return_if_fail(KZ_IS_WINDOW(kz));

	switch (info)
	{
	case TARGET_KAZEHAKASE_TAB:
	{
		GtkWidget *src_widget = gtk_drag_get_source_widget(drag_context);
		KzWindow *src_kz;
		KzTabLabel *src_kztab;

		if (!KZ_IS_TAB_LABEL(src_widget)) return;
		src_kztab = KZ_TAB_LABEL(src_widget);

		src_kz = KZ_WINDOW(kz_window_get_from_tab(GTK_WIDGET(src_kztab->kzembed)));
		if (kz == src_kz)
		{
			gint n;
			n = gtk_notebook_get_n_pages(GTK_NOTEBOOK(kz->notebook));
			gtk_notebook_reorder_child(GTK_NOTEBOOK(kz->notebook), 
						   GTK_WIDGET(src_kztab->kzembed),
						   n - 1);
		}
		else
		{
			kz_window_move_tab(kz, GTK_WIDGET(src_kztab->kzembed));
		}

		break;
	}

	case TARGET_NETSCAPE_URL:
	case TARGET_TEXT_URI_LIST:
	case TARGET_TEXT_PLAIN:
	case TARGET_STRING:
	{
		gchar *string, **strings;

		if (data->length < 0) return;
		string = g_alloca (data->length + 1);

		memcpy (string, data->data, data->length);
		string[data->length] = '\0';

		strings = g_strsplit(string, "\n", 2);
		kz_window_open_new_tab(kz, strings[0]);
		g_strfreev(strings);
		break;

	}
	default:
		break;
	}
}
