/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-gesture.h"

#include <stdlib.h>

#include "gobject-utils.h"

enum {
   START_SIGNAL,
   STACK_MOTION_SIGNAL,
   PERFORM_SIGNAL,
   LAST_SIGNAL
};

static void kz_gesture_class_init   (KzGestureClass *class);
static void kz_gesture_init         (KzGesture *gesture);
static void kz_gesture_dispose      (GObject *object);

/* KzGesture class signals */
static void kz_gesture_stack_motion (KzGesture *gesture,
				     KzGestureMotion motion);

static void kz_gesture_reset        (KzGesture *gesture);
static KzGestureItem *kz_gesture_search_matched_item (KzGesture *gesture);

static GObjectClass *parent_class = NULL;
static gint kz_gesture_signals[LAST_SIGNAL] = {0};

KZ_OBJECT_GET_TYPE(kz_gesture, "KzGesture", KzGesture,
		   kz_gesture_class_init, kz_gesture_init,
		   G_TYPE_OBJECT)

static void
kz_gesture_class_init (KzGestureClass *class)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (class);
	object_class = (GObjectClass *) class;

	kz_gesture_signals[STACK_MOTION_SIGNAL]
		= g_signal_new ("stack-motion",
				G_TYPE_FROM_CLASS (class),
				G_SIGNAL_RUN_FIRST,
				G_STRUCT_OFFSET (KzGestureClass, stack_motion),
				NULL, NULL,
				g_cclosure_marshal_VOID__ENUM,
				G_TYPE_NONE, 1, G_TYPE_INT);

	object_class->dispose = kz_gesture_dispose;

	class->start        = NULL;
	class->stack_motion = kz_gesture_stack_motion;
	class->perform      = NULL;
}

static void
kz_gesture_init (KzGesture *gesture)
{
	gesture->max_sequence_len
		= G_N_ELEMENTS(gesture->sequence) - 1;
	gesture->threshold = 16;

	kz_gesture_reset(gesture); /* init other members */

	gesture->items = NULL;
}

static void
kz_gesture_dispose (GObject *object)
{
	KzGesture *gesture;

	g_return_if_fail(KZ_IS_GESTURE(object));

	gesture = KZ_GESTURE(object);

	if (gesture->items)
		kz_gesture_items_unref(gesture->items);
	gesture->items = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}

KzGesture *
kz_gesture_new (void)
{
	KzGesture *gesture = KZ_GESTURE(g_object_new(KZ_TYPE_GESTURE, NULL));
	return gesture;
}

void
kz_gesture_set_items (KzGesture *gesture, KzGestureItems *items)
{
	g_return_if_fail(KZ_IS_GESTURE(gesture));

	if (gesture->items)
		kz_gesture_items_unref(gesture->items);
	if (items)
		gesture->items = kz_gesture_items_ref(items);
	else
		gesture->items = NULL;
}

static void
kz_gesture_stack_motion (KzGesture *gesture, KzGestureMotion motion)
{
	gint len, max_len;
	KzGestureMotion *seq;

	g_return_if_fail(KZ_IS_GESTURE(gesture));
	g_return_if_fail(motion > KZ_GESTURE_MOTION_TERMINATOR &&
			 motion < KZ_GESTURE_N_MOTIONS);

	len = gesture->sequence_len;
	max_len = gesture->max_sequence_len;
	seq = gesture->sequence;

	g_return_if_fail(len >= 0 && len < max_len);

	seq[len] = motion;
	len = ++gesture->sequence_len;
	seq[len] = KZ_GESTURE_MOTION_TERMINATOR;
}

gboolean
kz_gesture_start (KzGesture *gesture, gint mode, gint x, gint y)
{
	gesture->current_mode = mode;
	gesture->prev_x = gesture->x = x;
	gesture->prev_y = gesture->y = y;
	gesture->started = TRUE;

	return TRUE;
}

gboolean
kz_gesture_perform (KzGesture *gesture)
{
	KzGestureItem *item;

	item = kz_gesture_search_matched_item(gesture);
	if (item)
		egg_action_activate(item->action);

	kz_gesture_reset(gesture);

	return FALSE;
}

void
kz_gesture_cancel (KzGesture *gesture)
{
	kz_gesture_reset (gesture);
}

static void
kz_gesture_reset (KzGesture *gesture)
{
	gesture->sequence[0]       = KZ_GESTURE_MOTION_TERMINATOR;
	gesture->sequence_len      = 0;
	gesture->current_mode      = 0;
	gesture->prev_x            = -1;
	gesture->prev_y            = -1;
	gesture->x                 = -1;
	gesture->y                 = -1;
	gesture->started           = FALSE;
}

gboolean
kz_gesture_is_started (KzGesture *gesture)
{
	g_return_val_if_fail(KZ_IS_GESTURE(gesture), FALSE);
	return gesture->started;
}

void
kz_gesture_create_gesture_string(KzGesture *gesture, gchar buf[], gint len)
{
	gint i, j = 0;

	g_return_if_fail(KZ_IS_GESTURE(gesture));
	g_return_if_fail(buf);

	buf[0] = '\0';

	for (i = 0; i < gesture->sequence_len && j < len - 2; i++)
	{
		switch (gesture->sequence[i])
		{
		case KZ_GESTURE_MOTION_UP:
			buf[j++] = 'U';
			break;
		case KZ_GESTURE_MOTION_DOWN:
			buf[j++] = 'D';
			break;
		case KZ_GESTURE_MOTION_LEFT:
			buf[j++] = 'L';
			break;
		case KZ_GESTURE_MOTION_RIGHT:
			buf[j++] = 'R';
			break;
		default:
			buf[j++] = '?';
			break;
		}
		if (gesture->sequence[i] != KZ_GESTURE_MOTION_TERMINATOR)
			buf[j++] = ' ';
		buf[j] = '\0';
	}
}

void
kz_gesture_update_position (KzGesture *gesture, gint x, gint y)
{
	KzGestureMotion motion;
	gint mx, my;

	g_return_if_fail(KZ_IS_GESTURE(gesture));
	g_return_if_fail(kz_gesture_is_started(gesture));

	mx = x - gesture->prev_x; 
	my = y - gesture->prev_y;

	if (abs(mx) > gesture->threshold || abs(my) > gesture->threshold)
	{
		gint len = gesture->sequence_len;
		gint max_len = gesture->max_sequence_len;
		KzGestureMotion *seq = gesture->sequence;

		if (abs(mx) > abs(my)) {
			if (mx < 0)
				motion = KZ_GESTURE_MOTION_LEFT;
			else
				motion = KZ_GESTURE_MOTION_RIGHT;
		}
		else
		{
			if (my < 0)
				motion = KZ_GESTURE_MOTION_UP;
			else
				motion = KZ_GESTURE_MOTION_DOWN;
		}

		gesture->prev_x = x;
		gesture->x = x;
		gesture->prev_y = y;
		gesture->y = y;

		if (len == 0 || (len > 0 && len < max_len && seq[len - 1] != motion))
		{
			g_signal_emit (G_OBJECT (gesture),
				       kz_gesture_signals[STACK_MOTION_SIGNAL],
				       0, motion);
		}
	}
}

static KzGestureItem *
kz_gesture_search_matched_item (KzGesture *gesture)
{
	GSList *node;
	gint j;

	g_return_val_if_fail(KZ_IS_GESTURE(gesture), NULL);
	if (!kz_gesture_is_started(gesture)) return NULL;
	if (gesture->sequence[0] == 0) return NULL;
	if (!gesture->items) return NULL;

	for (node = gesture->items->list; node; node = g_slist_next(node))
	{
		KzGestureItem *item = node->data;

		if (!item) continue;
		if (item->sequence[0] == 0) continue;

		for (j = 0; gesture->sequence[j] == item->sequence[j]; j++)
		{
			if (gesture->sequence[j + 1] == 0 &&
			    item->sequence[j + 1] == 0)
			{
				return item;
			}
			else if (gesture->sequence[j + 1] == 0 ||
				 item->sequence[j + 1] == 0)
			{
				break;
			}
		}
	}

	return NULL;
}

const gchar *
kz_gesture_get_matched_label (KzGesture *gesture)
{
	KzGestureItem *item;

	g_return_val_if_fail(KZ_IS_GESTURE(gesture), NULL);

	item = kz_gesture_search_matched_item (gesture);
	if (!item) return NULL;
	g_return_val_if_fail(item->action, NULL);
	return item->action->name;
}


/*****************************************************************************
 *
 *   KzGestureItemTable
 *
 *****************************************************************************/
KzGestureItems *
kz_gesture_items_new ()
{
	KzGestureItems *items = g_new0(KzGestureItems, 1);
	items->list = NULL;
	items->ref_count = 1;
	return items;
}

KzGestureItems *
kz_gesture_items_ref (KzGestureItems *items)
{
	g_return_val_if_fail(items, NULL);
	items->ref_count++;
	return items;
}

void
kz_gesture_items_unref (KzGestureItems *items)
{
	g_return_if_fail(items);
	items->ref_count--;

	if (items->ref_count == 0) {
		GSList *node = items->list;

		for(; node; node = g_slist_next(node)) {
			KzGestureItem *item = node->data;

			if (!item) continue;

			g_object_unref(item->action);
			item->action = NULL;
			g_free(item->sequence);
			item->sequence = NULL;
			g_free(item);
		}

		g_slist_free(items->list);
		items->list = NULL;
		g_free(items);
	}
}

void
kz_gesture_items_add_action (KzGestureItems *items,
			     EggAction *action, gint mode,
			     const KzGestureMotion *sequence)
{
	KzGestureItem *item;
	gint len;

	g_return_if_fail(items);
	g_return_if_fail(EGG_IS_ACTION(action));
	g_return_if_fail(sequence && *sequence != KZ_GESTURE_MOTION_TERMINATOR);

	for (len = 0; sequence[len] != KZ_GESTURE_MOTION_TERMINATOR; len++);
	item = g_new0(KzGestureItem, 1);
	item->action   = g_object_ref(action);
	item->mode     = mode;
	item->sequence = g_memdup(sequence, sizeof(KzGestureMotion) * (len + 1));

	items->list = g_slist_append(items->list, item);
}

