/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-zoom-action.h"

#include <glib/gi18n.h>
#include "kazehakase.h"
#include "gobject-utils.h"

struct _KzZoomActionPrivate 
{
	KzWindow *kz;
	GtkWidget *spin;
};

enum {
	PROP_0,
	PROP_KZ_WINDOW
};


static void kz_zoom_action_class_init       (KzZoomActionClass *class);
static void kz_zoom_action_init             (KzZoomAction      *action);
static void kz_zoom_action_dispose          (GObject *obj);
static void kz_zoom_action_finalize         (GObject *obj);
static void kz_zoom_action_set_property     (GObject           *object,
					     guint              prop_id,
					     const GValue      *value,
					     GParamSpec        *pspec);
static void kz_zoom_action_get_property     (GObject           *object,
					     guint              prop_id,
					     GValue            *value,
					     GParamSpec        *pspec);

static GtkWidget *kz_zoom_action_create_tool_item (GtkAction  *action);

static void cb_spin_value_changed (GtkSpinButton *spin, KzZoomAction *action);


static GtkActionClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(kz_zoom_action, "KzZoomAction", KzZoomAction,
		   kz_zoom_action_class_init, kz_zoom_action_init,
		   GTK_TYPE_ACTION)

KZ_OBJECT_FINALIZE(kz_zoom_action, KzZoomAction)

static void
kz_zoom_action_class_init (KzZoomActionClass *klass)
{
	GObjectClass *object_class;
	GtkActionClass *action_class;

	parent_class = g_type_class_peek_parent(klass);
	object_class = G_OBJECT_CLASS(klass);
	action_class = GTK_ACTION_CLASS(klass);

	object_class->set_property     = kz_zoom_action_set_property;
	object_class->get_property     = kz_zoom_action_get_property;
	object_class->dispose          = kz_zoom_action_dispose;
	object_class->finalize         = kz_zoom_action_finalize;

	action_class->create_tool_item = kz_zoom_action_create_tool_item;

	g_object_class_install_property
		(object_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
			 _("KzWindow"),
			 _("The KzWindow to add a zoom button"),
			 KZ_TYPE_WINDOW,
			 G_PARAM_READWRITE |
			 G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_zoom_action_init (KzZoomAction *action)
{
	action->priv = g_new0(KzZoomActionPrivate, 1);
	action->priv->kz   = NULL;
	action->priv->spin = NULL;
}


static void
kz_zoom_action_dispose (GObject *obj)
{
	KzZoomAction *action = KZ_ZOOM_ACTION(obj);

	if (action->priv->kz)
		g_object_unref(action->priv->kz);
	action->priv->kz = NULL;
	action->priv->spin = NULL;

	if (G_OBJECT_CLASS(parent_class)->dispose)
		G_OBJECT_CLASS(parent_class)->dispose(obj);
}


static void
kz_zoom_action_set_property (GObject         *object,
	 		     guint            prop_id,
			     const GValue    *value,
			     GParamSpec      *pspec)
{
	KzZoomAction *action = KZ_ZOOM_ACTION(object);

	switch (prop_id)
	{
		case PROP_KZ_WINDOW:
			action->priv->kz = g_object_ref(g_value_get_object(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}


static void
kz_zoom_action_get_property (GObject         *object,
			     guint            prop_id,
			     GValue          *value,
			     GParamSpec      *pspec)
{
	KzZoomAction *action = KZ_ZOOM_ACTION(object);

	switch (prop_id)
	{
		case PROP_KZ_WINDOW:
			g_value_set_object(value, action->priv->kz);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}


KzZoomAction *
kz_zoom_action_new(KzWindow *kz)
{
	KzZoomAction *action;

	action = g_object_new(KZ_TYPE_ZOOM_ACTION,
			      "name",     "Zoom",
			      "label",    _("_Zoom"),
			      "tooltip",  _("Change the text size of the current page"),
			      "stock_id",  GTK_STOCK_ZOOM_100,
			      "kz-window", kz,
			      NULL);

	return action;
}

static void
cb_spin_button_destroy (GtkWidget *widget, GtkAction *action)
{
	g_return_if_fail(KZ_IS_ZOOM_ACTION(action));
	
	KZ_ZOOM_ACTION(action)->priv->spin = NULL;
}


static GtkWidget * 
kz_zoom_action_create_tool_item (GtkAction  *action)
{
	GtkWidget *widget, *spin;

	widget = GTK_WIDGET(g_object_new(GTK_TYPE_TOOL_ITEM, NULL));

	spin = gtk_spin_button_new_with_range(10, 990, 10);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), 100);
	gtk_container_add(GTK_CONTAINER(widget), spin);
	gtk_widget_show(spin);

	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_spin_value_changed), action);
	g_signal_connect(spin, "destroy",
			 G_CALLBACK(cb_spin_button_destroy), action);
	KZ_ZOOM_ACTION(action)->priv->spin = spin;

	return widget;
}


static void
cb_spin_value_changed (GtkSpinButton *spin, KzZoomAction *action)
{
	gint ratio;
	GtkWidget *embed;

	if (!action->priv->spin || !GTK_WIDGET_VISIBLE(action->priv->spin))
		return;

	ratio = gtk_spin_button_get_value_as_int (spin);
	
	embed = KZ_WINDOW_CURRENT_PAGE(KZ_ZOOM_ACTION(action)->priv->kz);
	
	if (KZ_EMBED(embed))
		kz_embed_zoom_set(KZ_EMBED(embed), ratio, TRUE);
}


void
kz_zoom_action_set_ratio (KzZoomAction *action,
			  gint ratio) 
{
	GtkWidget *spin;
	g_return_if_fail(KZ_IS_ZOOM_ACTION(action));

	if (!action->priv->spin || !GTK_WIDGET_VISIBLE(action->priv->spin))
		return;
	
	spin = action->priv->spin;
	g_signal_handlers_block_by_func(spin,
			G_CALLBACK(cb_spin_value_changed), action);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), ratio);
	g_signal_handlers_unblock_by_func(spin,
			G_CALLBACK(cb_spin_value_changed), action);
}

gint
kz_zoom_action_get_ratio (KzZoomAction *action)
{
	gint ratio;
	GtkWidget *spin;
	
	g_return_val_if_fail(KZ_IS_ZOOM_ACTION(action), 100);
	
	if (!action->priv->spin || !GTK_WIDGET_VISIBLE(action->priv->spin))
		return 100;
	spin = action->priv->spin;
	ratio = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(spin));
    
	return ratio;
}

