unit untHttp;

interface

uses
  Classes,
  IdThread, IdHttp, IdComponent,
  untStreamTool;

type

  TAsyncHttp = class(TIdThread)
  private
    FURL           : string;
    FHttp          : TIdHttp;
    FWriteEvent    : TMemoryStreamEx;
    FBufferReader  : TStreamReader;
    FReadPosition  : Integer;
    FOnReceived    : TNotifyEvent;
    FReceivedLines : TStringList;
    FStartRange    : Integer;
    FLastModified  : string;
    FOnStatus      : TIdStatusEvent;
    FStatus        : TIdStatus;
    FStatusText    : String;
    FOnComplete: TNotifyEvent;
    FContentLength: Integer;
    procedure HttpReceived(const Buff; Count : int64);
    procedure HttpStatus(ASender: TObject; const AStatus: TIdStatus; const AStatusText: string);
    procedure SetOnReceived(const Value: TNotifyEvent);
    procedure SetStartRange(const Value: Integer);
    procedure SetLastModified(const Value: string);
    procedure RaiseReceivedEvent;
    procedure RaiseStatusEvent;
    procedure SetOnStatus(const Value: TIdStatusEvent);
    function  GetReplayCode : Integer;
    procedure SetOnComplete(const Value: TNotifyEvent);
    procedure SetContentLength(const Value: Integer);
  public
    property    OnReceived : TNotifyEvent read FOnReceived write SetOnReceived;
    property    OnStatus: TIdStatusEvent read FOnStatus write SetOnStatus;
    property    OnComplete : TNotifyEvent read FOnComplete write SetOnComplete;
    property    ReceivedLines : TStringList read FReceivedLines;
    property    StartRange : Integer read FStartRange write SetStartRange;
    property    LastModified : string read FLastModified write SetLastModified;
    property    ContentLength : Integer read FContentLength write SetContentLength;
    property    ResponseCode : Integer read GetReplayCode;
    constructor Create;
    destructor  Destroy; override;
    procedure   BeforeRun; override;
    procedure   Run; override;
    procedure   AfterRun; override;
    procedure   Get(URL : string);
  end;

implementation

{ TAsyncHttp }

procedure TAsyncHttp.SetOnReceived(const Value: TNotifyEvent);
begin
  FOnReceived := Value;
end;

constructor TAsyncHttp.Create;
begin
  inherited Create();

  FReceivedLines := TStringList.Create;
end;

destructor TAsyncHttp.Destroy;
begin
  FReceivedLines.Free;

  inherited;
end;

procedure TAsyncHttp.Get(URL: string);
begin

  FURL := URL;
  self.Start;

end;

procedure TAsyncHttp.BeforeRun;
begin
  inherited;

  FHttp := TIdHttp.Create(nil);
  FHTtp.OnStatus := HttpStatus;
  FWriteEvent := TMemoryStreamEx.Create;
  FWriteEvent.OnWrite := HttpReceived;
  FBufferReader       := TStreamReader.Create(FWriteEvent); 
  FReadPosition       := 0;

  {*debug Config.SetReadProxy(FHttp)
  FHttp.Request.UserAgent   := gUserAgent;
  FHttp.Request.RawHeaders.Add('X-2ch-UA: ' + APP_2chUA);
  }

  FHttp.Request.ContentRangeStart := FStartRange;

  FHttp.Request.Connection := 'close';


end;

procedure TAsyncHttp.Run;
begin
  inherited;

  FHttp.Get(FURL, FWriteEvent);

  FHttp.Disconnect;

  Stop;
end;

procedure TAsyncHttp.HttpReceived(const Buff; Count: int64);
var
  line : string;
begin

  line := '';
  FWriteEvent.Seek(FReadPosition, soFromBeginning);

  // ͊̕Oɏo\
  while FBufferReader.ReadLine(line) do
  begin
    FReadPosition := FWriteEvent.Position;
    FReceivedLines.Add(line);
  end;

  Synchronize(RaiseReceivedEvent);

end;

procedure TAsyncHttp.AfterRun;
begin
  inherited;

  FLastModified  := FHttp.Response.RawHeaders.Values['Last-Modified'];
  FContentLength := FHttp.Response.ContentLength;

  if Assigned(FOnComplete) then FOnComplete(self);

  FHttp.Free;
  FWriteEvent.Free;
  FBufferReader.Free;

end;

procedure TAsyncHttp.SetStartRange(const Value: Integer);
begin
  FStartRange := Value;
end;

procedure TAsyncHttp.SetLastModified(const Value: string);
begin
  FLastModified := Value;
end;

procedure TAsyncHttp.RaiseReceivedEvent;
begin
  if Assigned(FOnReceived) then
    FOnReceived(self)
end;

procedure TAsyncHttp.SetOnStatus(const Value: TIdStatusEvent);
begin
  FOnStatus := Value;
end;

procedure TAsyncHttp.HttpStatus(ASender: TObject; const AStatus: TIdStatus;
  const AStatusText: string);
begin

  FStatus     := AStatus;
  FStatusText := AStatusText;

  Synchronize(RaiseStatusEvent);

end;

procedure TAsyncHttp.RaiseStatusEvent;
begin

  if Assigned(FOnStatus) then
    FOnStatus(self, FStatus, FStatusText)

end;

function TAsyncHttp.GetReplayCode: Integer;
begin

  result := FHttp.ResponseCode;

end;

procedure TAsyncHttp.SetOnComplete(const Value: TNotifyEvent);
begin
  FOnComplete := Value;
end;

procedure TAsyncHttp.SetContentLength(const Value: Integer);
begin
  FContentLength := Value;
end;

end.
