require 'PageDataSource.rb'
require 'PropertyFile.rb'

module Wiki

class FilePageDataSource < PageDataSource
	
	def exist?(page_name, lang, version = nil)
		source_file_path = make_source_file_path(page_name, lang, version)
		property_file_path = make_property_file_path(page_name, lang, version)

		File.file?(source_file_path) and 
		File.readable?(source_file_path) and
		File.file?(property_file_path) and 
		File.readable?(property_file_path) 
	end

	def exist_page?(page_name)
		result = false
		page_dir_name = escape_page_name(page_name)
		page_dir_path = "#{self.data_dir}/#{page_dir_name}"
		if File.directory?(page_dir_path ) then
			result = true
		end
		result
	end

	def load_page_source(page_name, lang, version = nil)
		page_source = nil
		source_file_path = make_source_file_path(page_name, lang, version)
		File.open(source_file_path, "r+") do |source_file|
			until source_file.flock(File::LOCK_SH|File::LOCK_NB) do
				sleep 0.01
			end
			page_source = source_file.readlines.to_s
		end
		page_source
	end

	def save_page_source(page_name, lang, version, source)
		source_file_path = make_source_file_path(page_name, lang, version)
		create_page(page_name, lang, version)

		File.open(source_file_path, "w+"){ |source_file|
			until source_file.flock(File::LOCK_EX|File::LOCK_NB) do
				sleep 0.01
			end
			source_file.write(source)
		}

		super(page_name, lang, version, source)
	end

	def delete_page(page_name, lang, version)
		cache_file_path = make_cache_file_path(page_name, lang, version)
		source_file_path = make_source_file_path(page_name, lang, version)
		property_file_path = make_property_file_path(page_name, lang, version)
		[cache_file_path, source_file_path, property_file_path].each(){ |file_path|
			if File.writable?(file_path) then
				File.unlink(file_path)
			end
		}

		lang_dir_path = File.dirname(make_file_path(page_name, lang, version))
		page_dir_path = File.dirname(lang_dir_path)
		if Dir.entries(lang_dir_path).size <= 2 then
			Dir.rmdir(lang_dir_path)
			if Dir.entries(page_dir_path).size <= 2 then
				Dir.rmdir(page_dir_path)
			end
		end
		
	end

	def page_property(page_name, lang, version = nil, &block)
		property = nil
		property_file_path = make_property_file_path(page_name, lang, version)
		property = PropertyFile.property(property_file_path, default_page_property, &block);
		return  property;
	end

	def site_property(&block)
		property = nil
		property_file_path = "#{self.data_dir}/site.properties"
		property = PropertyFile.property(property_file_path, default_site_property, &block);
		return  property;
	end

	def page_list()
		source_file_path_list = Dir.glob("#{data_dir}/*/*/*.txt")
		pages = source_file_path_list.collect(){ |path|
			path.untaint()
			if path =~ %r!#{data_dir}/(.*)/(.*)/(.*).txt$! then
				page_name = URI.unescape($1)
				lang = $2
				version_str = $3
				version_str = nil if version_str == 'recent' 
				if exist?(page_name, lang, version_str) then
					page_id = make_page_id(page_name, lang, version_str)
					property = page_property(page_name, lang, version_str)
					[page_id, property]
				else
					nil
				end
			end
		}
		pages.reject!(){ |page| page == nil }
		pages.sort!()
		return pages;
	end

protected

	def make_source_file_path(page_name, lang, version = nil)
		source_file_path = make_file_path(page_name, lang, version) + ".txt"
		return source_file_path;
	end

	def make_cache_file_path(page_name, lang, version = nil)
		cache_file_path = make_file_path(page_name, lang, version) + ".dat"
		return cache_file_path;
	end

	def make_property_file_path(page_name, lang, version = nil)
		property_file_path = make_file_path(page_name, lang, version) + ".properties"
		return property_file_path;
	end

	def make_file_path(page_name, lang, version = nil)
		page_dir_name = escape_page_name(page_name)
		if version then
			version_str = sprintf("%08d", version.to_s.to_i)
			file_path = "#{self.data_dir}/#{page_dir_name}/#{lang}/#{version_str}"
		else
			file_path = "#{self.data_dir}/#{page_dir_name}/#{lang}/recent"
		end
		return file_path;
	end

	def escape_page_name(original_name)
		result = URI.escape(original_name)
		result.gsub!(/\./, '%2E')
		result.gsub!(/\//, '%2F')
		result
	end

	def create_page(page_name, lang, version)
		source_file_path = make_source_file_path(page_name, lang, version)
		property_file_path = make_property_file_path(page_name, lang, version)

		page_dir = File.dirname(source_file_path) 
		lang_dir = File.dirname(page_dir) 
		Dir.mkdir(lang_dir) unless File.directory?(lang_dir) 
		Dir.mkdir(page_dir) unless File.directory?(page_dir) 

		unless File.file?(source_file_path) then
			File.open(source_file_path, "w+"){ |source_file|
				until source_file.flock(File::LOCK_EX|File::LOCK_NB) do
					sleep 0.01
				end
				source_file.write("\n")
			}
		end
		
		unless File.file?(property_file_path) then
			property = default_page_property
			property['page_name'] = page_name
			property['page_title'] = page_name
			property['language'] = lang
			File.open(property_file_path, "w+"){ |prop_file|
				until prop_file.flock(File::LOCK_EX|File::LOCK_NB) do
					sleep 0.01
				end
				PropertyFile.write_property(prop_file, property)
			}
		end
	end

	def load_cached_document(page_name, lang, version)
		cache_file_path = make_cache_file_path(page_name, lang, version)
		document = nil
		if File.file?(cache_file_path) and 
				File.readable?(cache_file_path) then
			begin
				File.open(cache_file_path, "r+") do |cache_file|
					until cache_file.flock(File::LOCK_EX|File::LOCK_NB) do
						sleep 0.01
					end
					document =  Marshal.restore(cache_file)
				end
			rescue NameError, TypeError, EOFError
				document = nil
			end
		end
		return document;
	end

	def save_cached_document(page_name, lang, version, document)
		cache_file_path = make_cache_file_path(page_name, lang, version)
		File.open(cache_file_path, "w+"){ |cache_file|
			until cache_file.flock(File::LOCK_EX|File::LOCK_NB) do
				sleep 0.01
			end
			Marshal.dump(document, cache_file);
		}

		page_property(page_name, lang, version){ |property|
			property['cache_time'] = Time.now
		}
	end
end

end # module Wiki

