# :Author: SAKIMA Kimihisa
# :Contact: sakima at hooktail.org
# :Date: $Date: 2007/03/10 20:37:33 $
# :Revision: $Revision: 1.3 $

$KCODE = 'UTF8'

# ===================================
# ドキュメントクラス・スタイルの整形
# ===================================
def documentclass(buf, style)
  if style == "report"
    pkg_name = "kagipdf_report"
    class_name = "jsarticle"
    class_opt = "11pt,a4paper"
  elsif style == "thesis"
    pkg_name = "kagipdf_thesis"
    class_name = "jsbook"
    class_opt = "12pt,a4paper,oneside,openany"
  else
    pkg_name = "kagipdf_hooktail"
    class_name = "jsarticle"
    class_opt = "11pt,a4paper"
  end
  
  # ドキュメントクラスを置換，\usepakage{pkg_name} を追加
  buf.sub!(/^\\documentclass.*$/,
  "\\documentclass\[#{class_opt}\]\{#{class_name}\}\n\\usepackage\{#{pkg_name}\}\n\\usepackage\{mydefinition\}")

  return buf
end

# ====================
# プリアンブルの整形
# ====================
def preamble(buf, style)
  # color, graphicx パッケージに対して dvipdfm オプションを追加
  buf.sub!(/\\usepackage\{color\}/, "\\usepackage\[dvipdfm\]\{color\}")
  buf.sub!(/\\usepackage\{graphicx\}/, "\\usepackage\[dvipdfm\]\{graphicx\}")

  # あると困るパッケージ命令を削除
  buf.sub!(/\\usepackage\[DIV12\]\{typearea\}\n/, "")
  buf.sub!(/\\usepackage\{ae\}\n/, "")
  buf.sub!(/\\usepackage\[.*\]\{inputenc\}\n/, "")
  buf.sub!(/\\usepackage\{babel\}\n/, "")
  buf.sub!(/\\usepackage\{aeguill\}\n/, "")

  # amssymb, bm パッケージを追加
  buf.sub!(/\\usepackage\{amsmath\}/, "\\usepackage\{amsmath,amssymb,bm\}")

  # hyperref パッケージのオプションを変更
  buf.sub!(/\\usepackage\[colorlinks=(\w+),linkcolor=(\w+),urlcolor=(\w+)\]\{hyperref\}/){
    "\\usepackage\[dvipdfm,bookmarks=true,bookmarksnumbered=true,pdfstartview=\{FitH -32768\}," +
    "colorlinks=#{$1},linkcolor=#{$2},urlcolor=#{$3},citecolor=#{$3}" +
    "\]\{hyperref\}" +
    "\\ifnum 42146=\\euc\"A4A2 \\AtBeginDvi\{\\special\{pdf:tounicode EUC-UCS2\}\}\\else" +
    "\\AtBeginDvi\{\\special\{pdf:tounicode 90ms-RKSJ-UCS2\}\}\\fi  %\""
  }

  return buf
end

# ====================
# latexタイトルの整形
# ====================
def textitle(buf, style)
  if style == "hooktail"
    # authorに「＠物理のかぎプロジェクト」を加える
    buf.gsub!(/\\author\{(.*?)\}/){"\\author\{" + $1 + "＠物理のかぎプロジェクト\}"}
    buf.gsub!(/pdfauthor=\{(.*?)\}/){"pdfauthor=\{" + $1 + "＠物理のかぎプロジェクト\}"}
  end
  
  # :Date: today で \date{\today}
  buf.gsub!(/\\date\{today\}/, "\\date\{\\today\}")

  return buf
end

# ================
# latex本文の整形
# ================
def texbody(buf, style)
  # 画像スケールを縮小，いつでも中央に配置
  unless style == "thesis"
    buf.gsub!(/\\includegraphics/, "\\includegraphics\[scale=0\.55\]")
    buf.gsub!(/\\includegraphics(.*\}$)/){
      "\\begin\{center\}\\includegraphics" + $1 + "\\end\{center\}"
    }
  end

  # \end{quote} ，\end{itemize} の後の改行を取り除く
  buf.gsub!(/(^\\begin\{quote\}).*?\n/){ $1 }
  buf.gsub!(/(^\\end\{quote\}).*?\n/){ $1 }
  buf.gsub!(/(^\\end\{itemize\}).*?\n/){ $1 }
  buf.gsub!(/(^\\end\{list\}).*?\n/){ $1 }
  
  # 余分な空白を除く
  buf.gsub!(/ ?([、。，．]) ?/){ $1 }
  buf.gsub!(/ ?(\\ref\{.*?\}) ?/){ $1 }
  buf.gsub!(/ ?(\$\\ref\{.*?\}\$) ?/){ $1 }
  
=begin
  # 章番号をつけない
  DO_NOT_NUMBERING = [
  '謝辞',
  'acknowledgment'
  ]
  buf.gsub!(/\\chapter\{(#{DO_NOT_NUMBERING[0]})\}/){
    "\chapter*{#{$1}}\n\addcontentsline{toc}{chapter}{#{$1}}"
  }
=end
  
  # \href{...}中の % をエスケープ
  buf = buf.gsub(/\\href\{(.*?)\}/){
    "\\href\{" + $1.gsub(/%/, "\\%") + "\}"
  }

  unless style == "hooktail"
    buf.gsub!(/\\section\*\{/, "\\section\{")
    buf.gsub!(/\\subsection\*\{/, "\\subsection\{")
  end
  
  if style == "thesis"
    buf.sub!(/\\begin\{center\}.*?\\begin\{tabularx\}\{\\docinfowidth\}\{lX\}.*?\\end\{tabularx\}.*?\\end\{center\}.*?\\setlength\{\\locallinewidth\}\{\\linewidth\}.*?\\tableofcontents.*?\\bigskip/m, "\\frontmatter\n\\tableofcontents\n\\mainmatter")
    buf.gsub!(/\\section\{謝辞\}/, '\chapter*{謝辞}\addcontentsline{toc}{chapter}{謝辞}')
    buf.gsub!(/\\section/, '\chapter')
    buf.gsub!(/\\subsection/, '\section')
    buf.gsub!(/\\subsubsection/, '\subsection')
    buf.gsub!(/ ?\$(\\cite\{.*?\})\$ ?/){ "~#{$1}" }
  end
  
  return buf
end

# ================
# latex本文の整形
# ================

# 挿入画像ファイルを適切に変換
def figure(buf)
  # image もしくは figure ディレクティブのパターン
  re_conv_eps = /\.\.\s+(figure|image)\s*::\s+(.*)\.(png|gif|jpg)/
  re_make_bb = /\.\.\s+(figure|image)\s*::\s+(.*)\.(pdf)/

  # 格納用配列
  images = Array.new

  # 画像ファイル名を取得
  images << buf.scan(re_conv_eps)
  images << buf.scan(re_make_bb)
  
  # png, gifの場合はディレクティブ中の拡張子をepsにする
  buf.gsub!(re_conv_eps) { ".. " + $1 + ":: " + $2 + ".eps" }
  
  # 画像ファイル名情報を格納してある配列をフラットにする
  images.flatten!
  
  # eps画像もしくはbbファイルを生成
  # （使われている画像ファイルの数だけ繰り返す）
  n = images.length
  i = 0
  while i < n
    fBody = images[i+1]  # 拡張子を除いたファイル名
    fTail = images[i+2]  # 拡張子名
    
    orgFile = fBody + "." + fTail  # 元の画像ファイル名
    pngFile = fBody + ".png" # epsファイル名
    epsFile = fBody + ".eps" # epsファイル名

    # 元ファイルの種類により最適化
    if (fTail == "png")
      # bmepsがあれば優先
      if open("| which bmeps 2>/dev/null"){|f| f.gets}
        `bmeps -c -p3 -am255,255,255s #{orgFile} #{epsFile}`
      else
        `convert -compress lzw #{orgFile} eps2:#{epsFile}`
      end
    elsif (fTail == "gif")
      `convert -compress lzw #{orgFile} eps3:#{epsFile}`
    elsif (fTail == "jpg")
      `convert -compress lzw #{orgFile} eps3:#{epsFile}`
    elsif (fTail == "pdf")
      `ebb #{orgFile}`  # bb（BoundingBox）ファイルをつくる
    end
    
    # 配列3つで画像ファイル一つ分なので「+3」すると次の画像
    i += 3
  end

  return buf
end

# =====================
# 物理のかぎしっぽ特有
# =====================

# @@命令の処理
def atat(buf)
  # @@命令の各種パターン
  re_author = /@@author:\s*(.*?)\s*@@/
  re_date = /@@accept:\s*?(.+?)\s*?@@/
  re_reference = /@@reference:\s*(.*?)\s*@@/
  re_category = /@@category:\s*(.*?)\s*@@/
  re_information = /@@information:\s*(.*?)\s*@@/
  re_id = /@@id:\s*(.*?)\s*@@/

  # パターンにマッチする部分を取得
  author = buf.scan(re_author).join
  date = buf.scan(re_date).join
  date = "today" if date == ""

  # 「@@author:...@@」，「@@accept:...@@」を，
  #  それぞれタイトル行直下の「:Author: ...」,「:Date: ...」に変換
  if buf =~ re_author
    buf.sub!(/\n(.*)(====|----|````|::::|\"\"\"\"|~~~~|\^\^\^\^|____|\*\*\*\*|\+\+\+\+|####|<<<<|>>>>)\n/){
      "\n" + $1 + $2 +  "\n:Author: " + author + "\n" + ":Date: " + date  + "\n\n"
    }
  end

  # @@命令そのものは取り除く
  buf.gsub!(re_author, "")
  buf.gsub!(re_date, "")
  buf.gsub!(re_reference, "")
  buf.gsub!(re_category, "")
  buf.gsub!(re_information, "")
  buf.gsub!(re_id, "")
  
  return buf
end

# その他
def etc(buf, style)
  # footnoteは数字にする（[*] → [#]）
  buf.gsub!(/ \[\*\]\_ ?/, " \[#\]\_ ")
  buf.gsub!(/^\.\.\s+\[\*\]\s+/, "\.\. \[#\] ")

  # 行頭の全角スペースを取り除く
  buf.gsub!(/^　/, "")

  # 「、」を「，」に，「。」を「．」に置換
  buf.gsub!(/、/, "，")
  buf.gsub!(/。/, "．") if style == "hooktail"

  return buf
end
