// $Id: PLCMemory.cpp,v 1.5 2003/03/22 16:41:44 fukasawa Exp $

//=============================================================================
/**
 *  @file    PLCMemory.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2002 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "PLCDeviceManager.h"
#include "PLCMemory.h"
#include "ace/Log_Msg.h"

#ifndef BOUNDARY
#define BOUNDARY     0
#define NO_BOUNDARY  1
#endif

static bool _clear_memory = false;
#ifdef WIN32
static char _rootPath[256] = "..\\shared";
#else
static char _rootPath[256] = "../shared";
#endif

void smem_dump(u_short * _startp, u_short * _endp,
               unsigned long base, int mode);

//------------------------------------------------------------------------------
//
// PLC Memory
//
//------------------------------------------------------------------------------
bool PLCMemory::SetClearMemory(bool flag)
{
    bool old = _clear_memory;
    _clear_memory = flag;
    return old;
}

//------------------------------------------------------------------------------
// Initial path name
//------------------------------------------------------------------------------
void PLCMemory::pathName(string& name)
{
    ACE_OS::strncpy(_rootPath, name.c_str(), sizeof(_rootPath) - 1);
    _rootPath[sizeof(_rootPath) - 1] = '\0';
}

//------------------------------------------------------------------------------
// Open data from virtual memory
//------------------------------------------------------------------------------
int PLCMemory::open(long bus, long stnum, long address, long offset)
{
    m_path = bus;
    m_stnum = stnum;

#if defined(PLC_SHARED_MEMORY)
    size_t size = this->isWord() ? m_areaSize : (m_areaSize + 7) / 8;
    int result = mapOpen(m_mmap, m_devname, size, address, offset);
    if (result < 0)
    {
        return result;
    }

#endif
    return 0;
}

//------------------------------------------------------------------------------
// Open map memory for simulate.
//------------------------------------------------------------------------------
#if defined(PLC_SHARED_MEMORY)
int PLCMemory::mapOpen(ACE_Mem_Map& mmptr, string& fname, size_t size,
                       long address, long offset)
{
    int result;
    string path = _rootPath;
#ifdef WIN32
    path += "\\";
#else
    path += "/";
#endif
    path += fname;
    path += ".mel";
    if (address == 0)
    {
        result = mmptr.map(path.c_str(), size * sizeof(short), O_RDWR | O_CREAT,
                            ACE_DEFAULT_FILE_PERMS, PROT_RDWR, ACE_MAP_SHARED);
        if (result == -1)
        {
            ACE_ERROR_RETURN((LM_ERROR, ACE_TEXT("%n: %p \"%s\"\n"),
                              ACE_TEXT("mmap"), path.c_str()), -1);
        }
    }
    else
    {
        result = mmptr.map(path.c_str(), size * sizeof(short), O_RDWR | O_CREAT,
                            ACE_DEFAULT_FILE_PERMS, PROT_RDWR, ACE_MAP_SHARED,
                            (void *)address, offset);
        if (result == -1)
        {
            ACE_ERROR_RETURN ((LM_ERROR, ACE_TEXT("%n: %p\n"), ACE_TEXT("mmap")), -1);
        }
    }

    if (_clear_memory)
    {
        unsigned char * top = (unsigned char *)mmptr.addr();
        for (size_t i = 0; i < size * sizeof(short); i++) 
        {   // Clear
            *(top + i) = 0;
        }
        mmptr.sync();
    }
    return result;
}
#endif

//------------------------------------------------------------------------------
// Close data map memory
//------------------------------------------------------------------------------
int PLCMemory::close() 
{
#if defined(PLC_SHARED_MEMORY)
    m_mmap.sync();
    m_mmap.close();
#endif
    return 0;
}

//------------------------------------------------------------------------------
// Set data to virtual memory
//------------------------------------------------------------------------------
int PLCMemory::setvm(long , size_t , u_short * )
{
    throw runtime_error("PLCMemory::setvm : not suport method");
    return 0;
}

//------------------------------------------------------------------------------
// Read data from plc memory
//------------------------------------------------------------------------------
int PLCMemory::read(long , size_t , u_short * ) const
{
    throw runtime_error("PLCMemory::read : not suport method");
    return -1;
}

//------------------------------------------------------------------------------
// Write data to plc memory
//------------------------------------------------------------------------------
int PLCMemory::write(long , size_t , u_short * )
{
    throw runtime_error("PLCMemory::write : not suport method");
    return -1;
}

//------------------------------------------------------------------------------
// Get a datum from plc memory
//------------------------------------------------------------------------------
u_short PLCMemory::get(long ) const
{
    throw runtime_error("PLCMemory::get : not suport method");
    return 0xFFFF;
}

//------------------------------------------------------------------------------
// Put a datum to plc memory
//------------------------------------------------------------------------------
int PLCMemory::put(long , u_short )
{
    throw runtime_error("PLCMemory::put : not suport method");
    return -1;
}

//------------------------------------------------------------------------------
// Get a datum from virtual memory
//------------------------------------------------------------------------------
u_short PLCMemory::vmget(long ) const
{
    throw runtime_error("PLCMemory::vmget : not suport method");
    return 0xFFFF;
}

//------------------------------------------------------------------------------
// Read data from virtual memory
//------------------------------------------------------------------------------
int PLCMemory::vmread(long , size_t , u_short * ) const
{
    throw runtime_error("PLCMemory::vmread : not suport method");
    return -1;
}


//------------------------------------------------------------------------------
// Dump logical memory
//------------------------------------------------------------------------------
void PLCMemory::dump(long , size_t ) const
{
    throw runtime_error("PLCMemory::dump : not suport method");
}

//------------------------------------------------------------------------------
//
// Word Memory
//
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// Set data to virtual memory
//------------------------------------------------------------------------------
int PLCWordMemory::setvm(long address, size_t size, u_short * data)
{
    if (m_vmaddr == NULL)
    {
        throw runtime_error("PLCWordMemory::setvm : virtual memory is null");
        return -1;
    }
    size_t setsize = (size == (size_t)ACTION_END) ? m_areaSize : size;
    if ((address + setsize) > m_areaSize)
    {
        setsize = m_areaSize - address;
    }
    memmove(m_vmaddr + address, data, setsize * 2);
    return 0;
}

//------------------------------------------------------------------------------
// Get a datum from virtual memory
//------------------------------------------------------------------------------
u_short PLCWordMemory::vmget(long address) const
{
    if (m_vmaddr == NULL)
    {
        throw runtime_error("PLCWordMemory::vmread : virtual memory is null");
        return 0xFFFF;
    }
    if ((u_long)address >= m_areaSize)
    {
        throw runtime_error("PLCWordMemory::get : bad device address");
        return 0xFFFF;
    }
    return (*(m_vmaddr + address) & 0xFFFF);
}

//------------------------------------------------------------------------------
// Read data from virtual memory
//------------------------------------------------------------------------------
int PLCWordMemory::vmread(long address, size_t size, u_short * data) const
{
    if (m_vmaddr == NULL)
    {
        throw runtime_error("PLCWordMemory::vmread : virtual memory is null");
        return -1;
    }
    size_t getsize = (size == (size_t)ACTION_END) ? m_areaSize : size ;
    if ((address + getsize) > m_areaSize)
    {
        getsize = m_areaSize - address;
    }
    memmove(data, m_vmaddr + address, getsize * 2);
    return 0;
}


//------------------------------------------------------------------------------
// Dump logical memory
//------------------------------------------------------------------------------
void PLCWordMemory::dump(long addr, size_t size) const
{
    size_t wordsize;
    size_t dumpsize = (size == (size_t)ACTION_END) ? m_areaSize : size;
    if (((u_long)addr + dumpsize) >= m_areaSize)
    {
        dumpsize = m_areaSize - addr;
    }

    wordsize = dumpsize;
    u_short * buf = (u_short *)malloc(wordsize * sizeof(u_short));
    if (buf == NULL)
    {
        throw runtime_error("PLCWordMemory::dump : lack heap memory");
    }
    this->read(addr, size, buf);
    smem_dump(buf, buf + wordsize, 0, NO_BOUNDARY);
    free(buf);
}


//------------------------------------------------------------------------------
//
// Bit Memory
//
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// Read data from virtual memory
//------------------------------------------------------------------------------
unsigned int PLCBitMemory::BitPattern[32] =
{
    0x00000001, 0x00000002, 0x00000004, 0x00000008,
    0x00000010, 0x00000020, 0x00000040, 0x00000080,
    0x00000100, 0x00000200, 0x00000400, 0x00000800,
    0x00001000, 0x00002000, 0x00004000, 0x00008000,
    0x00010000, 0x00020000, 0x00040000, 0x00080000,
    0x00100000, 0x00200000, 0x00400000, 0x00800000,
    0x01000000, 0x02000000, 0x04000000, 0x08000000,
    0x10000000, 0x20000000, 0x40000000, 0x80000000,
};

//------------------------------------------------------------------------------
// Set data to virtual memory
//------------------------------------------------------------------------------
int PLCBitMemory::setvm(long address, size_t size, u_short * data)
{
    if (m_vmaddr == NULL)
    {
        throw runtime_error("PLCBitMemory::setvm : virtual memory is null.");
        return -1;
    }
    if (((address % 16) != 0) || ((size % 16) != 0))
    {
        ACE_ERROR((LM_ERROR,
            ACE_TEXT("PLCBitMemory::setvm : Not boundary at word.\n")));
        return -1;
    }
    memmove((m_vmaddr + (address / 16)), data, size / 8);
    return 0;
}

//------------------------------------------------------------------------------
// Get a datum from virtual memory
//------------------------------------------------------------------------------
u_short PLCBitMemory::vmget(long address) const
{
    if (m_vmaddr == NULL)
    {
        throw runtime_error("PLCBitMemory::vmread : virtual memory is null");
        return 0;
    }
    if ((u_long)address >= m_areaSize)
    {
        throw runtime_error("PLCBitMemory::get : bad device address");
        return 0;
    }
    u_short word = *(m_vmaddr + ((address != 0) ? address / 16 : 0));
    return ((word & BitPattern[address & 15]) == 0) ? 0 : 1;
}

//------------------------------------------------------------------------------
// Read data from virtual memory
//------------------------------------------------------------------------------
int PLCBitMemory::vmread(long address, size_t size, u_short * data) const
{
    if (m_vmaddr == NULL)
    {
        throw runtime_error("PLCBitMemory::vmread : virtual memory is null");
        return -1;
    }
    size_t getsize = (size == (size_t)ACTION_END) ? m_areaSize : size ;
    if ((address + getsize) > m_areaSize)
    {
        getsize = m_areaSize - address;
    }

    if ((address & 15) == 0)
    {   // move words
        u_long rel = (address == 0) ? address : address / 16;
        memmove(data, m_vmaddr + rel, ((getsize + 15) / 16) * sizeof(short));
    }
    else
    {   // bit shifting
        u_char * top = (u_char *)m_vmaddr;
        bool     is_bound = true;
        size_t   offset = 0;
        u_short  bits = 0;
        for (u_int pos = 0; pos < getsize; pos++)
        {
            if ((pos != 0) && (pos % BITS_OF_SHORT == 0))
            {
                *(data + offset) = bits;
                offset++;
                bits = 0;
                is_bound = true;
            }
            else
            {
                is_bound = false;
            }
            u_long reladr = ((address + pos) > 0) ? (address + pos) / 8 : 0;
            u_long bnum = ((address + pos) > 0) ? (address + pos) % 8 : 0;
            u_char bt = *(top + reladr);
            bool bit = ((bt & PLCBitMemory::BitPattern[bnum]) != 0);
            int bitnum = pos & MASK_OF_SHORT;
            if (bit)
            {
                bits |= PLCBitMemory::BitPattern[bitnum];
            }
        }
        if (! is_bound)
        {   // set data to memory
            *(data + offset) = bits;
        }
    }
    return getsize;
}

//------------------------------------------------------------------------------
// Dump logical memory
//------------------------------------------------------------------------------
void PLCBitMemory::dump(long addr, size_t size) const
{
    size_t wordsize;
    size_t dumpsize = (size == (size_t)ACTION_END) ? m_areaSize : size;
    if (((u_long)addr + dumpsize) >= m_areaSize)
    {
        dumpsize = m_areaSize - addr;
    }

    wordsize = ((dumpsize + BITS_OF_SHORT) / BITS_OF_SHORT);
    u_short * buf = (u_short *)malloc(wordsize * sizeof(u_short));
    if (buf == NULL)
    {
        throw runtime_error("MELBitMemory::dump : lack heap memory");
    }
    this->read(addr, size, buf);
    smem_dump(buf, buf + wordsize, 0, NO_BOUNDARY);
    free(buf);
}


/*---------------------------------------------------------------------------*/
/*                                                                           */
/* hex dump                                                                  */
/*                                                                           */
/*---------------------------------------------------------------------------*/
#ifndef BOUNDARY
#define BOUNDARY     0
#define NO_BOUNDARY  1
#endif

static char lower_code[] = "0123456789abcdef";
//static char upper_code[] = "0123456789ABCDEF";

#define HEX_POINT      0
#define SEPARATE_POINT 44
#define CHAR_POINT     47
static char linebuf[] =
    "1234 1234  1234 1234  1234 1234  1234 1234  : 1234 1234 1234 1234\n";

/*---------------------------------------------------------------------------*/
/* word format                                                               */
/*---------------------------------------------------------------------------*/
void smem_dump(u_short * _startp, u_short * _endp,
               unsigned long base, int mode)
{
    char * startp = (char *)_startp;
    char * endp = (char *)_endp;
    char * curptr;
    unsigned int dumpnt;
    char   c;
    char * hexp;
    char * charp;
    char * hexcodep;

    if (mode == BOUNDARY)
    {   /*  */
        curptr = (char *)((unsigned long)startp & ~0x0F);
        base &= ~0x0F;
    }
    else
        curptr = startp;

    hexcodep = (char *)lower_code;     /*  */

/*---------------------------------------------------------------------------*/
/* dump hex data                                                             */
/*---------------------------------------------------------------------------*/
    for ( ; curptr < endp; curptr += 16)
    {
        hexp = (char *)(linebuf + HEX_POINT);
        charp = (char *)(linebuf + CHAR_POINT);

        for (dumpnt = 0; *(hexp + dumpnt) != '\n'; dumpnt++)
            *(hexp + dumpnt) = ' ';
        *(hexp + SEPARATE_POINT) = ':';

        for (dumpnt = 0; dumpnt < 16; dumpnt++)
        {
            c = *(curptr + dumpnt);
            if (dumpnt != 0 && (dumpnt & 0x01) == 0)
                hexp++;
            if (dumpnt != 0 && (dumpnt & 0x03) == 0)
            {
                hexp++;
                charp++;
            }
/*---------------------------------------------------------------------------*/
/* dump character                                                            */
/*---------------------------------------------------------------------------*/
            if ((curptr + dumpnt) < startp || (curptr + dumpnt) >= endp)
            {
                hexp += 2;
                charp++;
            }
            else
            {
                *hexp++ = hexcodep[(c >> 4) & 0x0F];
                *hexp++ = hexcodep[c & 0x0F];
                if (c < ' ' || c > '~')
                    *charp++ = '.';
                else
                    *charp++ = c;
            }
        }
        printf("%8lx    %s", base, linebuf);
        base += 16;
    }
}

