// $Id: b_matrix.h,v 1.2 2003/02/02 15:53:51 fukasawa Exp $

//=============================================================================
/**
 *  @file    b_matrix.h
 *
 *  @author Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 1998-2003 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef B_MATRIX_H
#define B_MATRIX_H

#include "b_ace.h"

typedef pair<string, string > KeyPair;

struct pair_less
{
    bool operator() (KeyPair x, KeyPair y) const {
        if (x.first == y.first) {
            return (x.second < y.second);
        } else {
            return (x.first < y.first);
        }
    }
};


//----------------------------------------------------------------------------
//
// Matrix Element
//
//----------------------------------------------------------------------------
template<class T, class ContT>
class MatrixElement
{
public:
    typedef pair<string, string >  KeyPair;
    typedef MatrixElement<T, ContT> reference;

    MatrixElement(ContT& cont, string state, string event)
        : m_container(cont), m_iter(m_container.find(KeyPair(state, event))),
          m_state(state), m_event(event)  {}

    T transition()
    {
        if (m_iter == m_container.end())
            return (T)0;
        else
            return (*m_iter).second;
    }

    operator T ()
    {
       return transition();
    }

    reference operator=(const T& x)
    {
        if (x != NULL)
        {
            if (m_iter == m_container.end())
            {
                if (! (m_container.size() < m_container.max_size()))
                    throw out_of_range("illegal table size.");
                KeyPair key(m_state, m_event);
                typename ContT::value_type val(key, x);
                m_iter = (m_container.insert(val)).first;
            }
            else
                (*m_iter).second = x;
       }
       else
       {
          if (m_iter != m_container.end())
          {
              m_container.erase(m_iter);
              m_iter = m_container.end();
          }
       }
       return *this;
    }

    reference operator=(const reference& rhs)
    {
       if (this != &rhs)
       {
           return operator=(rhs.transition());
       }
       return *this;
    }

//
private:
    ContT&  m_container;
    typename ContT::iterator m_iter;
    string m_state;
    string m_event;

};

//----------------------------------------------------------------------------
//
// State Matrix
//
//----------------------------------------------------------------------------
template<class T>
class b_matrix
{
public:
    typedef map<KeyPair, T, pair_less > ContT;
    typedef MatrixElement<T, ContT>     MatrixElement;
//
public:
    b_matrix() {}

   // usual container type definitions
   typedef typename ContT::iterator iterator;
   typedef typename ContT::const_iterator const_iterator;

   //
   size_t size() const          { return m_container.size(); }
   size_t max_size() const      { return m_container.max_size(); }
   iterator begin()             { return m_container.begin(); }
   iterator end()               { return m_container.end(); }
   const_iterator begin() const { return m_container.begin(); }
   const_iterator end() const   { return m_container.end(); }

   void clear() { m_container.clear(); }

   //
   // for key operation
   //
   class KeyOpe
   {
   public:
       KeyOpe(string state, ContT& cont) : m_state(state), m_container(cont) {}

       MatrixElement& operator[](string event)
       {
           MatrixElement * result =
                   new MatrixElement(m_container, m_state, event);
           return *result;
       }
    //
    private:
        string m_state;
        ContT& m_container;
    };

    //
    KeyOpe operator[](string state)
    {
        return KeyOpe(state, m_container);
    }

    string first(iterator& iter) const
    {
        return (*iter).first.first;
    }

    string second(iterator& iter) const
    {
        return (*iter).first.second;
    }

    T value(iterator& iter) const
    {
        return (*iter).second;
    }
    T value(const_iterator& iter) const
    {
        return (*iter).second;
    }

    void dump();

//
private:
    ContT m_container;

};

#endif
