// $Id: JGMemoryMappedFile.cpp,v 1.3 2003/03/22 16:41:43 fukasawa Exp $

//=============================================================================
/**
 *  @file    JGMemoryMappedFile.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2003 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "JGMemoryMappedFile.h"
#include "JGFile.h"


//-----------------------------------------------------------------------------
// Copy
//-----------------------------------------------------------------------------
void JGMemoryMappedFile::copy(const JGMemoryMappedFile& rhs)
{
    m_dir = rhs.m_dir;                // directory that make mmap file.
    m_fname = rhs.m_fname;            // filename that make mmap file.
    m_ext = rhs.m_ext;
    m_path = rhs.m_path;
    m_max = rhs.m_max;                // max byte size
    m_mmap.map(rhs.m_mmap.handle(), -1, 3, 2);
}

//-----------------------------------------------------------------------------
// Initial physical program area
//-----------------------------------------------------------------------------
int JGMemoryMappedFile::init(const string& dir, const string& fname,
                             const string& ext, int maxq, bool clear)
{
    m_dir = dir;              // directory that make mmap file.
    m_fname = fname;
    m_ext = ext;
    m_max = maxq;             // max byte size
    return map(clear);
}

//-----------------------------------------------------------------------------
int JGMemoryMappedFile::map(bool clear)
{
    //
    // Open map file
    //
    JGFile::makeFileName(m_dir, m_fname, m_ext, m_path);
    ACE_HANDLE hl = ACE_OS::open(m_path.c_str(), O_RDONLY);
    if (hl < 0)
    {   // mmap file is nothing
        ACE_DEBUG((LM_DEBUG, _TX("mmap is nothing(%s: %d)\n"), m_path.c_str(), m_max));
        clear = true;
    }
    else
    {
        ACE_DEBUG((LM_DEBUG, _TX("mmap exist(%s: %d)\n"), m_path.c_str(), m_max));
        ACE_OS::close(hl);
    }
    
    int result = m_mmap.map(m_path.c_str(), m_max, O_RDWR | O_CREAT,
                            ACE_DEFAULT_FILE_PERMS, PROT_RDWR,
                            ACE_MAP_SHARED);
    if (result < 0)
    {
        return -1;
    }

    // Clear all memory
    if (clear)
    {
        if (m_max > (size_t)0)
        {
            memset(m_mmap.addr(), 0, m_max);
        }
        m_mmap.sync();
    }
    m_current = (BYTE *)m_mmap.addr();
    return 0;
}

//-----------------------------------------------------------------------------
// Read binary data from mmap
//-----------------------------------------------------------------------------
int JGMemoryMappedFile::read(BYTE * addr, BYTE * buf, size_t bytes)
{
    if ((addr + bytes) > m_current)
    {
        return -1;
    }
    memmove(buf, addr, bytes);
    return 0;
}

//-----------------------------------------------------------------------------
// Write new binary data
//-----------------------------------------------------------------------------
int JGMemoryMappedFile::write(BYTE * buf, size_t bytes)
{
    memmove(m_current, buf, bytes);
    m_mmap.sync (m_current, bytes);
    m_current += bytes;
    return 0;
}


//-----------------------------------------------------------------------------
// Dump
//-----------------------------------------------------------------------------
void JGMemoryMappedFile::dump() const
{
    ACE_DEBUG((LM_DEBUG, ACE_TEXT("*** %s ***\n"), m_path.c_str()));
    size_t size = (BYTE *)m_mmap.addr() - m_current;
    ACE_HEX_DUMP((LM_DEBUG, (char *)m_mmap.addr(), size));
}

