// $Id: JGErrorList.h,v 1.5 2003/03/16 14:51:00 fukasawa Exp $

//=============================================================================
/**
 *  @file    JGErrorList.h
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2003 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#ifndef JGERRORLIST_H
#define JGERRORLIST_H

#include "jyugem.h"

#ifndef DEF_ERRCODE
#define DEF_ERRCODE
// Error code
enum
{
    ERRCODE_NORMAL = 0,                     // No error
    ERRCODE_UNKNOWN_OBJECT = 1,             // Unknown object in Object Specifier
    ERRCODE_UNKNOWN_TARGET = 2,             // Unknown target object type
    ERRCODE_UNKNOWN_OBJECT_INSTANCE = 3,    // Unknown object instance
    ERRCODE_UNKNOWN_ATTRIBUTE = 4,          // Unknown attribute name
    ERRCODE_READONLY_ATTRIBUTE = 5,         // Read-only attribute - access denied
    ERRCODE_UNKNOWN_OBJTYPE = 6,            // Unknown object type
    ERRCODE_INVALID_VALUE = 7,              // Invalide attribute value
    ERRCODE_SYNTAX_ERROR = 8,               // Syntax error
    ERRCODE_VERIFICATION_ERROR = 9,         // Verification error
    ERRCODE_VALIDATION_ERROR = 10,          // Validation error
    ERRCODE_IDENTIFIER_USED = 11,           // Object identifier in use
    ERRCODE_PARM_IMPROPERLY = 12,           // Parameters improperly specified
    ERRCODE_INSUFFICIENT_PARM = 13,         // Insufficient parameters specified
    ERRCODE_UNSUPPORTED_OPTION = 14,        // Unsupported option requested
    ERRCODE_BUSY = 15,                      // Busy
    ERRCODE_NOT_AVAILABLE = 16,             // Not available for processing
    ERRCODE_NOT_COMMAND = 17,               // Command not valid for current state
    ERRCODE_NO_MATERIAL = 18,               // No material altered
    ERRCODE_MATERIAL_PARTIALLY = 19,        // Material partially processed
    ERRCODE_MATERIAL_PROCESSED = 20,        // All material processed
    ERRCODE_ILLEGAL_RECIPE = 21,            // Recipe specification related error
    ERRCODE_DURING_PROCESSING = 22,         // Failed during processing
    ERRCODE_NOT_PROCESSING = 23,            // Failed while not processing
    ERRCODE_LACK_OF_MATERIAL = 24,          // Failed due to lack of material
    ERRCODE_JOB_ABORTED = 25,               // Job aborted
    ERRCODE_JOB_STOPPED = 26,               // Job stopped
    ERRCODE_JOB_CANCELED = 27,              // Job cancelled
    ERRCODE_CANNOT_CHANGE_RECIPE = 28,      // Cannot change selected recipe
    ERRCODE_UNKNOWN_EVENT = 29,             // Unknown event
    ERRCODE_DUPLICATE_RPTID = 30,           // Duplicate report ID
    ERRCODE_UNKNOWN_RPTDATA = 31,           // Unknown Data report
    ERRCODE_NOT_LINKED_REPORT = 32,         // Data report not linked
    ERRCODE_UNKNOWN_TRACE = 33,             // Unknown trace report
    ERRCODE_DUPLICATE_TRID = 34,            // Duplicate trace ID
    ERRCODE_TOO_MANY_REPORTS = 35,          // Too many data reports
    ERRCODE_SAMPLE_PERIOD_OVER = 36,        // Sample period out of range
    ERRCODE_GROUP_SIZE_OVER = 37,           // Group size to large
    ERRCODE_RECOVERY_CURRENT_INVALID = 38,  // Recovery action currently invalid
    ERRCODE_BUSY_ANOTHER_RECOVERY = 39,     // Busy with another recovery currently unable to perform the recovery
    ERRCODE_NO_ACTIVE_RECOVERY = 40,        // No active recovery action
    ERRCODE_RECOVERY_FAILED = 41,           // Exception recovery failed
    ERRCODE_RECOVERY_ABORTED = 42,          // Exception recovery aborted
    ERRCODE_INVALID_TABLE_ELEMENT = 43,     // Invalid table element
    ERRCODE_UNKNOWN_TABLE_ELEMENT = 44,     // Unknown table element
    ERRCODE_CANNOT_DELETE = 45,             // Cannot delete predefined
    ERRCODE_INVALID_TOKEN = 46,             // Invalid token
    ERRCODE_INVALID_PARAMETER = 47,         // Invalid parameter
};

const int MAX_ERRTEXT = 80;
const int MAX_ERRCODE = 47;

#endif /* DEF_ERRCODE */

class BS2Item;
class JGErrorList;

//
// Error information
//
class BEE_Export JGError
{
public:
    JGError() : m_code(0), m_text(_TX("")) { }
    JGError(u_long code, const string& text)
            : m_code(code), m_text(text) { }
    JGError(u_long code, const BCHAR * text = NULL)
            : m_code(code), m_text((text == NULL) ? _TX("") : text) { }
    JGError(const JGError& rhs) {
            m_code = rhs.m_code;
            m_text = rhs.m_text;
        }
    ~JGError() {}

    JGError& operator=(const JGError& rhs) {
            if (this == &rhs)
                return *this;
            m_code = rhs.m_code;
            m_text = rhs.m_text;
            return *this;
        }

    void  text(const string& txt) { m_text = txt; }
    const BCHAR * text() const    { return m_text.c_str(); }
    void   code(u_long cd) { m_code = cd; }
    u_long code() const    { return m_code; }

    // Make error item for SECS message
    BS2Item * makeItem(int cpackType = ATOM_UINT4) const; // ERRCODE + ERRTEXT
    BS2Item * cpackItem(int cpackType) const;             // CPNAME + CPACK

protected:
    u_long m_code;
    string m_text;

};

typedef vector<JGError>  JGErrors;


//
// Container of Error data.
//
class BEE_Export JGErrorList
{
public:
    JGErrorList() : m_name(_TX("")), m_format(ATOM_UINT1),
                    m_errFormat(ATOM_UINT4), m_ack(0) {}
    JGErrorList(const BCHAR * name, int form = ATOM_UINT1)
            : m_name(name), m_format(form), m_errFormat(ATOM_UINT4), m_ack(0) {}
    JGErrorList(const BCHAR * name, int form, int cpackType)
            : m_name(name), m_format(form),  m_errFormat(cpackType), m_ack(0) {}
    JGErrorList(const JGErrorList& rhs);
    ~JGErrorList() {}

    JGErrorList& operator=(const JGErrorList& rhs);

    void set(const string& name, int form = ATOM_UINT1) {
            m_name = name;  m_format = form;
        }
    void set(const string& name, int form, int cpackType) {
            m_name = name;  m_format = form;  m_errFormat = cpackType;
        }
    void   ack(int ack) { m_ack = ack & 0xFF; }
    int    ack() const  { return m_ack; }
    size_t count()      { return m_errors.size(); }
    JGErrors& getList() { return m_errors; }

    static const BCHAR * getErrText(u_long cd);

    // Append error data
    void append(JGError& errdata) {
            m_errors.push_back(errdata); m_ack = 1;
        }
    void append(u_long errcode) {
            BS2Assert(errcode <= (u_long)MAX_ERRCODE);
            JGError err(errcode, JGErrorList::getErrText(errcode));
            m_errors.push_back(err);
            m_ack = 1;
        }
    void append(u_long errcode, const string& detail) {
            BEEAssert(errcode <= (u_long)MAX_ERRCODE);
            string errstr(JGErrorList::getErrText(errcode));
            errstr += _TX(" [");
            errstr += detail;
            errstr += _TX("]");
            JGError err(errcode, errstr);
            m_errors.push_back(err);
            if (m_ack == 0) {
                m_ack = 1;
            }
        }

    // Make error item for SECS message
    // if iname is NULL, ACK item does not include in list
    BS2Item * makeItem() const;

protected:
    void copy(const JGErrorList& rhs);

protected:
    string m_name;
    int m_format;
    int m_errFormat;
    int m_ack;
    JGErrors m_errors;

};

#endif  /* JGERRORLIST_H */
