/* $Id: UInteger.java,v 1.5 2004/06/27 08:01:56 fukasawa Exp $ */

/**
 *  @file    UInteger.java
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

package com.jyugem.secs;

import java.*;

public final class UInteger extends Number implements Comparable {

    public static final long MIN_VALUE = 0L;
    public static final long MAX_VALUE = 0xFFFFFFFFL;

    private long value;    // *** value ***

    // Constructor
    public UInteger(UInteger value) {
        this.value = value.value;
    }

    public UInteger(int value) {
        this.value = value;
    }

    public UInteger(String s) throws NumberFormatException {
        this.value = parseUInteger(s);
    }

    public static String toString(int x) {
        return Integer.toString(x);
    }

    public static int parseUInteger(String s) throws NumberFormatException {
        return parseUInteger(s, 16);
    }

    public static int parseUInteger(String s, int radix)
                          throws NumberFormatException {
        long l = Long.parseLong(s, radix);
        if (l < MIN_VALUE || l > MAX_VALUE)
            throw new NumberFormatException();
        return (int)l;
    }

    public static UInteger valueOf(String s, int radix)
                           throws NumberFormatException {
        return new UInteger(parseUInteger(s, radix));
    }

    public static UInteger valueOf(String s) throws NumberFormatException {
        return valueOf(s, 10);
    }

    // Decodes a String into a UInteger.  Accepts decimal, hexadecimal, and octal
    // numbers, in the following formats:
    //    1-9   decimal constant
    //    0x    hex constant
    //    #     hex constant
    //    0     octal constant
    public static UInteger decode(String numStr) throws NumberFormatException {
        int radix = 10;
        int index = 0;
        boolean negative = false;
        UInteger result;

        if (numStr.startsWith("-")) {
            negative = true;
            index++;
        }

        if (numStr.startsWith("0x", index) ||
            numStr.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (numStr.startsWith("#", index)) {
            index++;
            radix = 16;
        } else if (numStr.startsWith("0", index) &&
                   numStr.length() > (index + 1)) {
            index++;
            radix = 8;
        }

        if (numStr.startsWith("-", index)) {
            throw new NumberFormatException("Negative sign in wrong position");
        }

        try {
            result = UInteger.valueOf(numStr.substring(index), radix);
            result = negative ? new UInteger(-result.intValue()) : result;
        } catch (NumberFormatException e) {
            String constant = "";
            if (negative) {
                constant = "-";
            }
            constant += numStr.substring(index);
            result = UInteger.valueOf(constant, radix);
        }
        return result;
    }

    public byte byteValue() {
        return (byte)value;
    }

    public short shortValue() {
        return (short)value;
    }

    public int intValue() {
        return (int)value;
    }

    public long longValue() {
        return (long)value;
    }

    public float floatValue() {
        return (float)value;
    }

    public double doubleValue() {
        return (double)value;
    }

    public String toString() {
        return Long.toString(value, 10);
    }

    public boolean equals(Object obj) {
        if (obj instanceof UInteger) {
            return value == ((UInteger)obj).intValue();
        }
        return false;
    }

    public int compareTo(UInteger another) {
        return (value < another.value ? -1 : (value == another.value ? 0 : 1));
    }

    public int compareTo(Object obj) {
        return compareTo((UInteger)obj);
    }

}
