/* $Id: SECSTestAssert.java,v 1.4 2004/06/20 15:23:40 fukasawa Exp $ */

/**
 *  @file    SECSTestAssert.java
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package com.jyugem.jsunit.tfw;

import junit.framework.*;
import org.apache.oro.text.regex.*;
import com.jyugem.secs.*;

//
//
//
public class SECSTestAssert extends TestCase {
    public static int PASS  = 0;
    public static int XPASS = 1;
    public static int FAIL  = 2;
    public static int XFAIL = 3;
    public static int UNRESOLVED = 4;
    public static int UNSUPPORTED = 5;
    public static int UNTESTED = 6;
    public static int ERROR = -1;
    public static int WARNING = -2;

    protected Perl5Compiler m_compiler;
    protected Perl5Matcher  m_matcher;
    protected Pattern m_pattern;

    public SECSTestAssert(String name) {
        super(name);
        m_compiler = new Perl5Compiler();
        m_matcher  = new Perl5Matcher();
        m_matcher.setMultiline(true);
    }

    /**
     * Fails a test with the given message.
     */
    public static void pass(String description) {
        System.out.println("PASS: " + description);
    }

    public static void fail(String expected, String message) {
        throwError("FAIL: ", null, expected, message);
    }
    public static void fail(String guide, String expected, String message) {
        throwError("FAIL: ", guide, expected, message);
    }

    public static void xfail(String expected, String message) {
        throwError("XFAIL: ", null, expected, message);
    }
    public static void xfail(String guide, String expected, String message) {
        throwError("XFAIL: ", guide, expected, message);
    }

    public static void unresolved(String expected, String message) {
        throwError("UNRESOLVED: ", null, expected, message);
    }
    public static void unresolved(String guide, String expected,
                                  String message) {
        throwError("UNRESOLVED: ", guide, expected, message);
    }

    public static void unsupported(String expected, String message) {
        throwError("UNSUPPORTED: ", null, expected, message);
    }
    public static void unsupported(String guide, String expected,
                                   String message) {
        throwError("UNSUPPORTED: ", guide, expected, message);
    }

    public static void warning(String expected, String message) {
        throwError("WARNING: ", null, expected, message);
    }
    public static void warning(String guide, String expected, String message) {
        throwError("WARNING: ", guide, expected, message);
    }

    public static void error(String expected, String message) {
        throwError("ERROR: ", null, expected, message);
    }
    public static void error(String guide, String expected, String message) {
        throwError("ERROR: ", guide, expected, message);
    }

    private static void throwError(String result, String guide, String expected,
                                   String message) {
        String formatted = "\n### [" + result + "] ";
        if (guide != null) {
            formatted += guide;
        }
        formatted += "\n### expected ###\n" + expected +
                     "\n### but was  ###\n" + message + "\n";

        throw new AssertionFailedError(formatted);
    }

    /**
     * Asserts that message match expect pattern. If they are not
     * an AssertionFailedError is thrown.
     */
    public void assertMatch(String expected, SECSMessage msg) {
        assertMatch(null, expected, msg, TestPair.MATCH);
    }
    public void assertMatch(String description, String expected,
                            SECSMessage msg) {
        assertMatch(description, expected, msg, TestPair.MATCH);
    }

    public void assertMatch(String expected, SECSMessage msg, int match) {
        assertMatch(null, expected, msg, match);
    }

    public void assertMatch(String description, String expected,
                            SECSMessage msg, int match) {
        m_compiler = new Perl5Compiler();
        m_matcher  = new Perl5Matcher();

        if (expected == null && msg == null)
            return;

        String xmlStr = msg.toXMLString();
        assertTrue((xmlStr != null));

        try {
            m_pattern = (Perl5Pattern)m_compiler.compile(expected,
                                                Perl5Compiler.SINGLELINE_MASK);
        } catch (MalformedPatternException e) {
            error(description, expected, "Perl5Pattern compile.");
        }

        PatternMatcherInput input = new PatternMatcherInput(xmlStr);

        if (match == TestPair.NOT_CONTAIN) {
            if (! m_matcher.contains(input, m_pattern)) {
                pass(description);
                return;
            }
            xfail(description, expected, xmlStr);
        } else if (match == TestPair.CONTAIN) {
            if (m_matcher.contains(input, m_pattern)) {
                pass(description);
                return;
            }
        } else {  // MATCH
            if (m_matcher.matches(input, m_pattern)) {
                pass(description);
                return;
            }
        }
        fail(description, expected, xmlStr);
    }

    /**
     * Asserts that message match expect pattern. If they are not,
     * an AssertionFailedError is thrown.
     */
    public SECSMessage assertMatch(TestPair pairs, SECSMessage msg) {
        return assertMatch(null, pairs, msg);
    }

    public SECSMessage assertMatch(String description, TestPair pairs,
                                   SECSMessage msg) {
        m_compiler = new Perl5Compiler();
        m_matcher  = new Perl5Matcher();

        if (pairs == null && msg == null)
            return null;

        String xmlStr = msg.toXMLString();
        assertTrue((xmlStr != null));

        PatternMatcherInput input = new PatternMatcherInput(xmlStr);

        int checkNum = pairs.size();
        int i;
        for (i = 0; i < checkNum; i++) {
            String expected = pairs.getExpect(i);
            try {
                m_pattern = (Perl5Pattern)m_compiler.compile(expected,
                                                   Perl5Compiler.SINGLELINE_MASK);
            } catch (MalformedPatternException e) {
                error(description, expected, "Perl5Pattern compile.");
            }
            if (pairs.getMatch() == TestPair.NOT_CONTAIN) {
                if (! m_matcher.contains(input, m_pattern)) {
                    pass(description);
                    return pairs.getMessage(i);
                }
                xfail(description, pairs.getExpect(), xmlStr);
            } else if (pairs.getMatch() == TestPair.CONTAIN) {
                if (m_matcher.contains(input, m_pattern)) {
                    pass(description);
                    return pairs.getMessage(i);
                }
            } else if (pairs.getMatch() == TestPair.NOT_MATCH) {
                if (! m_matcher.contains(input, m_pattern)) {
                    pass(description);
                    return pairs.getMessage(i);
                }
            } else {  // MATCH
                if (m_matcher.matches(input, m_pattern)) {
                    pass(description);
                    return pairs.getMessage(i);
                }
            }
        }
        fail(description, pairs.getExpect(), xmlStr);
        return null;
    }

    /**
     * Asserts that message match expect pattern. If they are not,
     * null is returned.
     */
    public SECSMessage compareMatch(TestPair pairs, SECSMessage msg) {
        return assertMatch(null, pairs, msg);
    }

    public SECSMessage compareMatch(String description, TestPair pairs,
                                   SECSMessage msg) {
        m_compiler = new Perl5Compiler();
        m_matcher  = new Perl5Matcher();

        if (pairs == null && msg == null)
            return null;

        String xmlStr = msg.toXMLString();
        assertTrue((xmlStr != null));

        PatternMatcherInput input = new PatternMatcherInput(xmlStr);

        int checkNum = pairs.size();
        int i;
        for (i = 0; i < checkNum; i++) {
            String expected = pairs.getExpect(i);
            try {
                m_pattern = (Perl5Pattern)m_compiler.compile(expected,
                                                   Perl5Compiler.SINGLELINE_MASK);
            } catch (MalformedPatternException e) {
                error(description, expected, "Perl5Pattern compile.");
            }
            if (pairs.getMatch() == TestPair.NOT_CONTAIN) {
                if (! m_matcher.contains(input, m_pattern)) {
                    pass(description);
                    return pairs.getMessage(i);
                }
                return null;
            } else if (pairs.getMatch() == TestPair.CONTAIN) {
                if (m_matcher.contains(input, m_pattern)) {
                    pass(description);
                    return pairs.getMessage(i);
                }
            } else if (pairs.getMatch() == TestPair.NOT_MATCH) {
                if (! m_matcher.contains(input, m_pattern)) {
                    pass(description);
                    return pairs.getMessage(i);
                }
            } else {  // MATCH
                if (m_matcher.matches(input, m_pattern)) {
                    pass(description);
                    return pairs.getMessage(i);
                }
            }
        }
        return null;
    }

    /**
     * Asserts that the message is same name and item. If they are not
     * an AssertionFailedError is thrown.
     */
    public void assertAck(String sfname, String itemName,
                          SECSMessage message) {
        assertAckCode(null, sfname, itemName, message, 0);
    }
    public void assertAck(String description, String sfname, String itemName,
                          SECSMessage message) {
        assertAckCode(description, sfname, itemName, message, 0);
    }

    public void assertNack(String sfname, String itemName,
                           SECSMessage message) {
        assertAckCode(null, sfname, itemName, message, 1);
    }
    public void assertNack(String description, String sfname, String itemName,
                           SECSMessage message) {
        assertAckCode(description, sfname, itemName, message, 1);
    }

    public void assertAckCode(String description, String sfname, String itemName,
                              SECSMessage message, int expcode) {
        if (! sfname.equals(message.name())) {
            fail(description, sfname, message.name() );
        }

        String codeStr = (expcode == 0) ? " = 0" : " != 0";

        ItemGetter icart = new ItemGetter(itemName);
        message.traverse(icart);
        Object ack = icart.getValue();
        if (ack != null) {
            if (ack instanceof Number) {
                int result = ((Number)ack).intValue();
                if (expcode == 0 && result == 0) {
                    return;
                } else if (expcode != 0 && result != 0) {
                    return;
                }
            }
            fail(description, sfname + "." + itemName + codeStr ,
                 message.name() + "." + itemName + " = " + ack.toString() + "\n");
        }
        fail(description, sfname + "." + itemName + codeStr + "\n",
             message.name() + "\n");
    }


    //
    // Get value of the item.
    //
    protected class ItemGetter implements SECSParser {
        String      m_iname;
        Object      m_result = null;
        int         m_type = -1;
        SECSMessage m_message;

        // Constractor
        protected ItemGetter(String iname) { m_iname = iname;  }
        protected ItemGetter(String iname, int type) { m_iname = iname; m_type = type; }

        protected Object getValue() { return m_result; }

        // Interface :
        public void set(SECSMessage msg) { m_message = msg; }

        public int callWithItem(int type, String itemName, Object value) {
            if (m_iname.equals(itemName) && (m_type < 0 || m_type == type)) {
                m_result = value;
                return -1;
            }
            return 0;
        }

        // Callback to parse list
        public int beginList(int nest, int size) { return 0; }
        public int endList(int nest) { return 0; }
    }
}

