// $Id: AnalyDevice.cpp,v 1.1 2004/11/27 06:51:33 fukasawa Exp $

//=============================================================================
/**
 *  @file    AnalyDevice.cpp
 *
 *  @author Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 1998-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "Analyzer.h"
#include "AnalyDevice.h"
#include "BS2DeclAtoms.h"
#include "BS2AnalyzeReceiver.h"
#include "BS2Sender.h"
#include "BS2Socket.h"
#include "BS2Driver.h"
#include "BS2TransactionManager.h"
#include "BS2MessageInfo.h"
#include "BS2ErrorMessage.h"

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
AnalyDevice::AnalyDevice() : BS2Gateway(), m_viewType(VIEW_JYUGEM)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "AnalyDevice::AnalyDevice");
}

//-----------------------------------------------------------------------------
AnalyDevice::~AnalyDevice()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "AnalyDevice::~AnalyDevice");
}

//-----------------------------------------------------------------------------
// View Formatted SECS Message (Base)
//-----------------------------------------------------------------------------
class Formatter : public BS2Traverser
{
    friend class JGInfoManager;
public:
    string m_msgtext;
    string m_baseind;
    BS2Item * m_listitem;

    Formatter() : BS2Traverser(), m_msgtext(_TX("")), m_listitem(NULL) {}
    virtual ~Formatter() {}

    virtual int begin(BS2Message * msg) = 0;
    virtual int end() = 0;
};

//-----------------------------------------------------------------------------
// View Formatted SECS Message (JYUGEM)
//-----------------------------------------------------------------------------
class JYUGEMFormatter : public Formatter
{
public:

    JYUGEMFormatter() : Formatter()
    {
        m_baseind = _TX("  ");
    }
    virtual ~JYUGEMFormatter() {}

    virtual int begin(BS2Message * msg)
    {
        m_msgtext = "<message sf=\"";
        m_msgtext += msg->name();
        if (msg->reply())
        {
            m_msgtext += "\" wait=\"true\" transaction=\"";
        }
        else
        {
            m_msgtext += "\" transaction=\"";
        }
        m_msgtext += "\">\n";
        m_msgtext += "  <body>\n";
        return 0;
    }

    virtual int end()
    {
        m_msgtext += "  </body>\n";
        m_msgtext += "</message>\n";
        m_msgtext += "\n";
        return 0;
    }

    // Append value string to message string
    int itemString(int indent, BS2value& val, const string& itemName,
                   string& msgbuf)
    {
        TRACE_FUNCTION(TRL_LOW, "Formatter::itemString");

        int format = val.format();
        string indbuf(indent * 2, ' ');
        indbuf += m_baseind;
        const char * formatStr = b_value::formatStr(format & 0x3F);

        msgbuf += indbuf + "<item name=\"";
        msgbuf += itemName + "\" type=\"";
        msgbuf += formatStr;
        msgbuf += "\">";

        if (format & ATOM_ARRAY)
        {
            msgbuf += "\n";
            msgbuf += val.toString() + "\n";
            msgbuf += indbuf;
        }
        else if (format == ATOM_BINARY)
        {
            if (val.size() > 8)
            {
                msgbuf += "\n";
                msgbuf += val.toString() + "\n";
                msgbuf += indbuf;
            }
            else
            {
                msgbuf += val.toString();
            }
        }
        else
        {
            if (format == ATOM_ASCII || format == ATOM_JIS)
            {
                msgbuf += val.toString();
            }
            else
            {
                msgbuf += val.toString();
            }
        }
        msgbuf += "</item>\n";
        return 0;
    }

    // <item>
    virtual int parseItem(BS2Item * item)
    {
        TRACE_FUNCTION(TRL_LOW, "Formatter::parseItem");

        BS2Atom * atom = item->atom();
        BS2value val;
        atom->get(val);
        itemString(m_nest + 1, val, item->name(), m_msgtext);

        return 0;
    }

    // <list>
    int beginList(BS2ListItem * listitem)
    {
        TRACE_FUNCTION(TRL_LOW, "Formatter::beginList");

        this->BS2Traverser::beginList(listitem);

        string indbuf(m_nest * 2, ' ');
        indbuf += m_baseind;
        m_msgtext += indbuf + "<list>\n";

        return 0;
    }

    // </list>
    int endList(BS2ListItem * listitem)
    {
        TRACE_FUNCTION(TRL_LOW, "Formatter::endList");

        string indbuf(m_nest * 2, ' ');
        indbuf += m_baseind;
        m_msgtext += indbuf + "</list>\n";
        this->BS2Traverser::endList(listitem);

        return 0;
    }

    // <list>
    int beginValueList(BS2Item * item)
    {
        TRACE_FUNCTION(TRL_LOW, "Formatter::beginValueList");
        this->BS2Traverser::beginValueList(item);

        m_listitem = item;
        BS2List * listatom = (BS2List *)item->atom();
        string indbuf(m_nest * 2, ' ');
        indbuf += m_baseind;
        m_msgtext += indbuf + "<list>\n";

        BS2Atoms& atoms = listatom->getList();
        if (atoms.size() > 0)
        {
            this->parseListAtom(m_nest + 1, atoms, m_msgtext);
        }
        return 0;
    }

    // </list>
    int endValueList(BS2Item * item)
    {
        TRACE_FUNCTION(TRL_LOW, "Formatter::endValueList");

        string indbuf(m_nest * 2, ' ');
        indbuf += m_baseind;
        m_msgtext += indbuf;
        m_msgtext += "</list>\n";

        this->BS2Traverser::endValueList(item);
        return 0;
    }

    // <list>
    int parseListAtom(int indent, BS2Atoms& atoms, string& msgbuf)
    {
        TRACE_FUNCTION(TRL_LOW, "Formatter::parseListAtom");

        int    memberNum = atoms.size();
        string indbuf(indent * 2, ' ');
        indbuf += m_baseind;
        for (int i = 0; i < memberNum; i++)
        {
            BS2Atom * mbr_atom = atoms[i];
            if (mbr_atom->isList())
            {
                BS2Atoms& vlist = ((BS2List *)mbr_atom)->getList();
                if (vlist.size() > 0)
                {
                    msgbuf += indbuf + "<list>\n";
                    this->parseListAtom(indent + 1, vlist, msgbuf);
                    msgbuf += indbuf + "</list>\n";
                }
                else
                {
                    msgbuf += indbuf + "<list/>\n";
                }
            }
            else
            {
                this->parseValueAtom(indent, mbr_atom, msgbuf);
            }
        }
        return 0;
    }

    // <item>
    int parseValueAtom(int indent, BS2Atom * atom, string& msgbuf)
    {
        TRACE_FUNCTION(TRL_LOW, "Formatter::parseValueAtom");

        BS2value val;
        atom->get(val);
        this->itemString(indent, val,
                         (m_listitem != NULL) ? m_listitem->name() : _TX(""),
                         msgbuf);

        return 0;
    }
};

//-----------------------------------------------------------------------------
// View Formatted SECS Message (SML)
//-----------------------------------------------------------------------------
class SMLFormatter : public Formatter
{
public:
    SMLFormatter() : Formatter()
    {
        m_baseind = _TX("");
    }
    virtual ~SMLFormatter() {}

    virtual int begin(BS2Message * msg)
    {
        m_msgtext = "<";
        m_msgtext += msg->name();
        if (msg->reply())
        {
            m_msgtext += " WAIT";
        }
        m_msgtext += "\n";
        return 0;
    }

    virtual int end()
    {
        m_msgtext += ">\n";
        return 0;
    }

    // Append value string to message string
    int itemString(int indent, BS2value& val, string& msgbuf)
    {
        TRACE_FUNCTION(TRL_LOW, "SMLFormatter::itemString");

        const char * formatStr;
        int format = val.format();
        string indbuf(indent * 2, ' ');
        indbuf += m_baseind;

        if (format & ATOM_ARRAY)
        {
            BCHAR sizebuf[64];
            sprintf(sizebuf, _TX("%d"), val.m_q);

            formatStr = b_value::smlStr(format & 0x3F);
            if (formatStr == NULL)
            {
                TRACE_ERROR((_TX("Invalid format: format = %o\n"),
                        format));
                return -1;
            }
            msgbuf += indbuf + "<";
            msgbuf += formatStr;
            msgbuf += "[";
            msgbuf += sizebuf;
            msgbuf += "]\n";
            msgbuf += indbuf;
            msgbuf += val.toString() + "\n";
            msgbuf += indbuf + ">\n";
        }
        else if (format == ATOM_BINARY)
        {
            BCHAR sizebuf[64];
            sprintf(sizebuf, _TX("%d"), val.size());

            msgbuf += indbuf + "<B[";
            msgbuf += sizebuf;
            msgbuf += "] ";
            if (val.size() > 8)
            {
                msgbuf += "\n";
                msgbuf += val.toString() + "\n";
                msgbuf += indbuf + ">\n";
            }
            else
            {
                msgbuf += val.toString();
                msgbuf += ">\n";
            }
        }
        else
        {
            formatStr = b_value::smlStr(format);
            if (formatStr == NULL)
            {
                TRACE_ERROR((_TX("Invalid format: format = %o\n"),
                        format));
            }
            msgbuf += indbuf + "<";
            msgbuf += formatStr;
            msgbuf += " ";

            if (format == ATOM_ASCII || format == ATOM_JIS)
            {
                msgbuf += "\"";
                msgbuf += val.toString();
                msgbuf += "\"";
            }
            else
            {
                msgbuf += val.toString();
            }
            msgbuf += ">\n";
        }
        return 0;
    }

    // <item>
    virtual int parseItem(BS2Item * item)
    {
        TRACE_FUNCTION(TRL_LOW, "SMLFormatter::parseItem");

        BS2Atom * atom = item->atom();
        BS2value val;
        atom->get(val);
        itemString(m_nest + 1, val, m_msgtext);

        return 0;
    }

    // <list>
    int beginList(BS2ListItem * listitem)
    {
        TRACE_FUNCTION(TRL_LOW, "SMLFormatter::beginList");

        this->BS2Traverser::beginList(listitem);

        string indbuf(m_nest * 2, ' ');
        indbuf += m_baseind;
        BCHAR sizebuf[64];
        sprintf(sizebuf, _TX("%d"),  listitem->memberq());

        m_msgtext += indbuf + "<L, ";
        m_msgtext += sizebuf;
        m_msgtext += "\n";

        return 0;
    }

    // </list>
    int endList(BS2ListItem * listitem)
    {
        TRACE_FUNCTION(TRL_LOW, "SMLFormatter::endList");

        string indbuf(m_nest * 2, ' ');
        indbuf += m_baseind;
        m_msgtext += indbuf + ">\n";
        this->BS2Traverser::endList(listitem);

        return 0;
    }

    // <list>
    int beginValueList(BS2Item * item)
    {
        TRACE_FUNCTION(TRL_LOW, "SMLFormatter::beginValueList");
        this->BS2Traverser::beginValueList(item);

        m_listitem = item;
        BS2List * listatom = (BS2List *)item->atom();
        string indbuf(m_nest * 2, ' ');
        indbuf += m_baseind;
        BCHAR sizebuf[64];
        sprintf(sizebuf, _TX("%d"),  listatom->getList().size());

        m_msgtext += indbuf + "<L, ";
        m_msgtext += sizebuf;
        m_msgtext += "\n";

        BS2Atoms& atoms = listatom->getList();
        int memberNum = atoms.size();
        if (atoms.size() > 0)
        {
            this->parseListAtom(m_nest + 1, atoms, m_msgtext);
        }
        return 0;
    }

    // </list>
    int endValueList(BS2Item * item)
    {
        TRACE_FUNCTION(TRL_LOW, "SMLFormatter::endValueList");

        string indbuf(m_nest * 2, ' ');
        indbuf += m_baseind;
        m_msgtext += indbuf + ">\n";

        this->BS2Traverser::endValueList(item);
        return 0;
    }

    // <list>
    int parseListAtom(int indent, BS2Atoms& atoms, string& msgbuf)
    {
        TRACE_FUNCTION(TRL_LOW, "SMLFormatter::parseListAtom");

        int memberNum = atoms.size();
        string indbuf(indent * 2, ' ');
        indbuf += m_baseind;
        for (int i = 0; i < memberNum; i++)
        {
            BS2Atom * mbr_atom = atoms[i];
            if (mbr_atom->isList())
            {
                BS2List * mbrlist = (BS2List *)mbr_atom;
                BS2Atoms& vlist = mbrlist->getList();
                BCHAR sizebuf[64];
                _stprintf(sizebuf, _TX("%d"), vlist.size());

                msgbuf += indbuf + "<L, ";
                msgbuf += sizebuf;
                msgbuf += "\n";
                if (vlist.size() > 0)
                {
                    this->parseListAtom(indent + 1, vlist, msgbuf);
                }
                msgbuf += indbuf + ">\n";
            }
            else
            {
                this->parseValueAtom(indent, mbr_atom, msgbuf);
            }
        }
        return 0;
    }

    // <item>
    int parseValueAtom(int indent, BS2Atom * atom, string& msgbuf)
    {
        TRACE_FUNCTION(TRL_LOW, "SMLFormatter::parseValueAtom");

        BS2value val;
        atom->get(val);
        this->itemString(indent, val, msgbuf);

        return 0;
    }
};


//-----------------------------------------------------------------------------
// Receive message thread.
//-----------------------------------------------------------------------------
int AnalyDevice::svc(void)
{
    TRACE_FUNCTION(TRL_LOW, "AnalyDevice::svc");

    int result = 0;
    BS2Message * msg = NULL;
    BS2MessageInfo msginfo;
    ACE_Time_Value mb_tv(0, 250 * 1000);
    ACE_Time_Value msg_tv(1);
    Formatter * formatter;
    JYUGEMFormatter  jyugem_form;
    SMLFormatter     sml_form;

    for (;;)
    {
        ACE_Message_Block *mb;

        // delay(0.25 sec) for task switching
        result = this->getq(mb, &mb_tv);
        if (result == -1)
        {   // time out
            // if ((result = this->receiveWithLimit(msginfo, &msg_tv)) >= 0)
            if ((result = this->receive(msginfo)) >= 0)
            {
                BS2Message * msg = msginfo.message();
                if (msginfo.getResult() == BS2RET_NORMAL && msg != NULL)
                {
                    // msg->dump();    // for DEBUG
                    if (m_viewType == VIEW_JYUGEM)
                    {
                        formatter = &jyugem_form;
                    }
                    else
                    {
                        formatter = &sml_form;
                    }
                    formatter->begin(msg);
                    msg->traverse(formatter);
                    formatter->end();
                    ACE_ERROR((LM_INFO, _TX("\n%s"), formatter->m_msgtext.c_str()));

                    delete msg;
                }
                else
                {   // Line Error
                    TRACE_ERROR((_TX("Timeout ! Is communication disconnect(%d) ?\n"),
                              msginfo.getResult()));
                }
            }
            else
            {
                ACE_ERROR((LM_ERROR, "Received error.\n"));
            }
        }
        else
        {
            int length = mb->length();
            if (length > 0)
            {
                BCHAR * top = mb->rd_ptr();
                TRACE_ERROR((_TX("AnalyDevice: Received unexpected message.\n")));
            }
            mb->release ();

            if (length == 0)             // shutdown
                break;
        }
    }
    TRACE_DEBUG((_TX("(%t) Analyzer::svc \"device %d\" dispatcher is deleted.\n"),
                m_devid));
    return 0;
}

