// $Id: BS2JavaDevice.cpp,v 1.6 2004/06/20 15:23:40 fukasawa Exp $

//=============================================================================
/**
 *  @file    BS2JavaDevice.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "BS2JavaDevice.h"
#include "BS2Device.h"
#include "BS2Driver.h"
#include "BS2Message.h"
#include "BS2MessageInfo.h"
#include "jni.h"
#include "JSMessage.h"
#include "JSListItem.h"
#include "JSItem.h"
#include "JSAtom.h"

static BS2JavaDevice * _instance = NULL;

//-----------------------------------------------------------------------------
// timeout of Delay Time Handler
//-----------------------------------------------------------------------------
void JavaDelayHandler::handle_time_out(const ACE_Time_Value &tv,
                                       const void * arg)
{
    TRACE_FUNCTION(TRL_LOW, "JavaDelayHandler::handle_time_out");
    ACE_UNUSED_ARG(tv);

    BS2JavaDevice * mngr = (BS2JavaDevice *)arg;
    //
    // notify xxxx to sevice
    //
    mngr->timerId(INIT_TIMERID);     // Clear timer id
}

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
BS2JavaDevice::BS2JavaDevice(BCHAR * name) : m_name(name), m_device(NULL)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "BS2JavaDevice::BS2JavaDevice");
    m_tv = ACE_Time_Value::zero;
    m_logmask = ACE_Log_Msg::instance()->priority_mask();
    m_jenv = NULL;
}

//-----------------------------------------------------------------------------
// Instance
//-----------------------------------------------------------------------------
BS2JavaDevice * BS2JavaDevice::instance()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "BS2JavaDevice::instance");
    if (_instance == NULL)
    {
        _instance = new BS2JavaDevice(_TX("JYUSECS"));
    }
    return _instance;
}

//-----------------------------------------------------------------------------
// Initialize java enviroument
//-----------------------------------------------------------------------------
int BS2JavaDevice::init(JNIEnv * jenv)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "BS2JavaDevice::init");
//    if (m_jenv == jenv)
//        return 0;

    // Update java enviroument
    m_jenv = jenv;
    try
    {
        JSMessage::instance()->init(jenv);
        JSListItem::instance()->init(jenv);
        JSItem::instance()->init(jenv);
        JSAtom::init(jenv, JSItem::instance()->getClass());
    }
    catch (runtime_error e)
    {
        TRACE_ERROR((_TX("Error class setting(%s).\n"), e.what()));
        return BEE_ERROR;
    }
    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Open and import database.
//-----------------------------------------------------------------------------
int BS2JavaDevice::open(void * parm)
{
    TRACE_FUNCTION(TRL_LOW, "BS2JavaDevice::open");

    DeviceParameter * dev_parm = (DeviceParameter *)parm;
    m_device = new BS2Device;
    m_device->setName(m_name.c_str());
    if (m_device->initialize(dev_parm) < 0)
    {
        TRACE_ERROR((_TX("Jyusecs manager does not initialize device.\n")));
        delete m_device;
        m_device = NULL;
        return BEE_ERROR;
    }

    m_tv.set(1, 0);      // set timeout value of message queue (1 sec)

    if (this->activate(THR_NEW_LWP | THR_DETACHED) == -1)
        ACE_ERROR_RETURN((LM_ERROR, ACE_TEXT("%p\n"), ACE_TEXT("spawn")),
                          BEE_ERROR);

    m_device->getDriver()->hexDump(true);

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Close .
//-----------------------------------------------------------------------------
int BS2JavaDevice::close(int flag)
{
    TRACE_FUNCTION(TRL_LOW, "BS2JavaDevice::close");
    ACE_UNUSED_ARG(flag);

    m_device->close();      // start to receive message
    delete m_device;
    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Start Delay Timer for Manager.
//-----------------------------------------------------------------------------
int BS2JavaDevice::startTimer(long sec)
{
    TRACE_FUNCTION(TRL_LOW, "BS2JavaDevice::startTimer");
    ACE_Time_Value tv(sec);
    m_tmid = ACE_Proactor::instance()->schedule_timer(m_timer, this, tv);
    if (m_tmid == ERROR_TIMERID)
    {
        ACE_ERROR_RETURN((LM_ERROR, ACE_TEXT("%p\n"), ACE_TEXT("schedule_timer")),
                          BEE_ERROR);
    }
    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Start Delay Timer for Manager.
//-----------------------------------------------------------------------------
int BS2JavaDevice::cancelTimer()
{
    TRACE_FUNCTION(TRL_LOW, "BS2JavaDevice::cancelTimer");

    if (m_tmid != INIT_TIMERID)
    {
        ACE_Proactor::instance()->cancel_timer(m_tmid);
        m_tmid = INIT_TIMERID;
    }
    else
    {
        ACE_ERROR_RETURN((LM_ERROR, ACE_TEXT("timer id is not found.\n")),
                          BEE_ERROR);
    }
    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Stop loop in thread.
//-----------------------------------------------------------------------------
int BS2JavaDevice::stopLoop()
{
    TRACE_FUNCTION(TRL_LOW, "BS2JavaDevice::stopLoop");

    ACE_Message_Block * mb;
    ACE_NEW_RETURN(mb, ACE_Message_Block(BUFSIZ + 1), BEE_ERROR);

    // Send a shutdown message to the other thread and exit.
    mb->length(0);
    if (this->put(mb) == -1)
    {
        TRACE_ERROR((_TX("(%t) %p\n"), _TX("put")));
    }
    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Message handling.
//-----------------------------------------------------------------------------
int BS2JavaDevice::queue(BS2Message * msg)
{
    ACE_Guard< ACE_Thread_Mutex > recv_mon(m_lock);
    while (1)
    {
        if (recv_mon.locked () != 0)
            break;
        ACE_OS::sleep(1);
    }

    m_queue.push_back(msg);

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
BS2Message * BS2JavaDevice::dequeue()
{
    ACE_Guard< ACE_Thread_Mutex > recv_mon(m_lock);
    while (1)
    {
        if (recv_mon.locked () != 0)
        {
            break;
        }
        ACE_OS::sleep(1);
    }

    if (m_queue.empty())
    {   // Receive error occured.
        return NULL;
    }
    BS2Message * msg = m_queue.front();
    if (msg != NULL)
    {
        m_queue.pop_front();
    }
    return msg;
}

//-----------------------------------------------------------------------------
// Send message.
//-----------------------------------------------------------------------------
int BS2JavaDevice::send(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "BS2JavaDevice::send");
#if 1
    int t_id = m_device->send(msg);
    return t_id;
#else
    this->queue(msg);
    return 1;
#endif
}

//-----------------------------------------------------------------------------
// Receive message
//-----------------------------------------------------------------------------
BS2Message * BS2JavaDevice::receive()
{
    TRACE_FUNCTION(TRL_LOW, "BS2JavaDevice::receive");
    BS2Message * msg = dequeue();
    while (msg == NULL)
    {
        if (m_received.wait() == -1)
        {
            TRACE_ERROR((_TX("JYUSECS Driver receiving error.\n")));
            return NULL;
        }
        msg = dequeue();
    }
    return msg;
}

//-----------------------------------------------------------------------------
// Receive message thread.
//-----------------------------------------------------------------------------
int BS2JavaDevice::svc(void)
{
    TRACE_FUNCTION(TRL_LOW, "BS2JavaDevice::svc");

    int result = 0;
    BS2Message * msg = NULL;
    BS2MessageInfo msginfo;
    ACE_Time_Value mb_tv(1);

    for (;;)
    {
        ACE_Message_Block * mb;
        ACE_Log_Msg::instance()->priority_mask(this->logmask());

        result = this->getq(mb, &mb_tv);
        if (result == -1)
        {   // time out
            if ((result = m_device->receive(msginfo)) >= 0)
            {
                msg = msginfo.message();
                if ((msginfo.getResult() == BS2RET_NORMAL) && (msg != NULL))
                {
                    // msg->dump();    // for DEBUG

                    this->queue(msg);
                    // Notify receipt message to Java
                    this->m_received.signal();
                }
                else
                {
                    TRACE_ERROR((_TX("Received time out message.\n")));
                    this->m_received.signal();
                }
            }
            else
            {
                TRACE_ERROR((_TX("Received error.\n")));
                this->m_received.signal();
            }
        }
        else
        {
            int length = mb->length();
            if (length > 0)
            {
                // char * top = mb->rd_ptr();

                TRACE_ERROR((_TX("Received unexpected message.\n")));
            }
            mb->release ();

            if (length == 0)             // shutdown
                break;
        }
    }

    ACE_DEBUG((LM_DEBUG, ACE_TEXT("BS2JavaDevice is deleted.\n")));
    return 0;
}

