// $Id$

//=============================================================================
/**
 *  @file    HSAlarmManager.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSAlarmManager.h"
#include "HSMainManager.h"
#include "HSServer.h"
#include "HSEventManager.h"
#include "HSInfoManager.h"
#include "HSSECSDevice.h"
#include "HSSECSDevice.h"
#include "JGTaskTrigger.h"
#include "JGLogTime.h"
#include "BS2ACKMessage.h"
#include "BS2ErrorMessage.h"
#include "BS2ListItem.h"
#include "BS2DeclAtoms.h"

#include "JGVariable.h"

static HSAlarmManager * _manager = NULL;

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
HSAlarmManager::HSAlarmManager() : JGAlarmManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSAlarmManager::HSAlarmManager");
}

//-----------------------------------------------------------------------------
HSAlarmManager::~HSAlarmManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSAlarmManager::~HSAlarmManager");

}

//-----------------------------------------------------------------------------
// Return own.
//-----------------------------------------------------------------------------
HSAlarmManager * HSAlarmManager::instance()
{
    TRACE_FUNCTION(TRL_LOW, "HSAlarmManager::instance");
    if (_manager == NULL)
    {
        _manager = new HSAlarmManager;
    }
    return _manager;
}


//-----------------------------------------------------------------------------
// Change the enable flag
//-----------------------------------------------------------------------------
BS2Message * HSAlarmManager::enabled(bool tf, JGid& id)
{
    TRACE_FUNCTION(TRL_LOW, "HSAlarmManager::enabled");

    this->changeEnabled(id, tf);

    BS2Message * msg = BS2Message::factory(SFCODE(5,3));
    BS2ListItem * bodylist = new BS2ListItem();
    BS2Item * item;
    item = BS2Item::factory(_TX("ALED"), JGvalue(tf));
    bodylist->add(item);
    item = BS2Item::factory(_TX("ALID"), id);
    bodylist->add(item);

    msg->add(bodylist);
    return msg;
}

//-----------------------------------------------------------------------------
// Change all enable flags
//-----------------------------------------------------------------------------
BS2Message * HSAlarmManager::allEnabled(bool tf)
{
    TRACE_FUNCTION(TRL_LOW, "HSAlarmManager::allEnabled");

    this->changeEnabledAll(tf);

    BS2Message * msg = BS2Message::factory(SFCODE(5,3));
    BS2ListItem * bodylist = new BS2ListItem();
    BS2Item * item;
    item = BS2Item::factory(_TX("ALED"), JGvalue(tf));
    bodylist->add(item);
    JGid empty_id;
    item = BS2Item::factory(_TX("ALID"), empty_id);
    bodylist->add(item);

    msg->add(bodylist);
    return msg;
}


//-----------------------------------------------------------------------------
// List Enabled Alarm message (S5F1 -> S5F2)
//-----------------------------------------------------------------------------
extern "C" char * vfei_make_report(const char * cmd, const char * rpt_body);
extern "C" int    vfei_make_report_1(const char * cmd, const char * rpt_body,
                                     char * rbuf, int size);

class _ParseAlarm : public BS2Traverser
{
    friend class HSAlarmManager;
public:
    _ParseAlarm() : m_alcd(false), m_alid(0), m_alidstr(""), m_altx(NULL) {}
    virtual ~_ParseAlarm() {}

    int    m_alcd;
    int    m_alid;
    string m_alidstr;
    const char * m_altx;

    virtual int parseItem(BS2Item * item) {
        TRACE_FUNCTION(TRL_LOW, "HSAlarmManager::_ParseAlarm::parseItem");
        BS2Atom * atom = item->atom();
        if (item->name() == _TX("ALCD"))
        {
             if (atom->isBinary())
             {
                 BYTE * bin = ((BS2Binary *)atom)->value();
                 m_alcd = ((*bin & 0x80) != 0) ? 1 : 0;
             }
             else
             {
                 TRACE_DEBUG((_TX("ALCD format is not binary(0x%x).\n"),
                              atom->format()));
                 return -1;
             }
        }
        else if (item->name() == _TX("ALID"))
        {
            JGid alid;
            atom->get(alid);
            m_alid = alid.getInt();

            int itemFormat = alid.format();
            const BCHAR * formatStr = HSMainManager::formatToSml(itemFormat);
            if (formatStr == NULL)
            {
                TRACE_ERROR((_TX("Warning: ALID invalid format = %o\n"),
                            itemFormat));
                formatStr = _TX("I4");
            }
            m_alidstr = formatStr;
            m_alidstr += "=";
            m_alidstr += alid.toString();
        }
        else if (item->name() == _TX("ALTX"))
        {
            if (! atom->isAscii())
            {
                TRACE_ERROR((_TX("Illegal alarm text format(0x%x) \n"),
                            atom->format()));
                return -1;
            }
            m_altx = ((BS2Ascii *)atom)->value().c_str();
        }
        return 0;
    }
};

//-----------------------------------------------------------------------------
BS2Message * HSAlarmManager::alarmReport(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "HSAlarmManager::alarmReport");

    BS2Message * replymsg;
    BYTE ack = ACK_OK;
    _ParseAlarm alarmInfo;
    int result = msg->traverse(&alarmInfo);    // parse S5F1
    if (result < 0)
    {   // reply normal acknowlede, always.
        //replymsg = BS2Message::response(msg, (BYTE)ACK_NG, _TX("ACKC5"));
        replymsg = BS2Message::response(msg, (BYTE)ACK_OK, _TX("ACKC5"));
        return replymsg;
    }

    //
    HSMainManager * vfeisrv = HSMainManager::instance();
    char vfei_buf[256];
    sprintf(vfei_buf, "ALARM_ID/%s ALARM_STATE/U1=%d ALARM_TEXT/A=\"",
            alarmInfo.m_alidstr.c_str(),
            alarmInfo.m_alcd);
    string vfei_msg = vfei_buf;
    vfei_msg += alarmInfo.m_altx;
    vfei_msg += "\"";

    // "CMD/A="ALARM_REPORT" MID/A="vfeixxxxx" MTY/A="E" ECD/U4=0 ETX/A=""
    ACE_Message_Block * mb;
    int buf_size = vfei_msg.size() + 128;
    mb = new ACE_Message_Block(buf_size);
    if (mb == NULL)
    {
        TRACE_ERROR((_TX("Lack message block buffer(%d)\n"), buf_size));
    }
    else
    {
        int total = vfeisrv->makeReport(this->name(), "ALARM_REPORT", vfei_msg,
                                        mb->wr_ptr(), buf_size);
        if (total > 0)
        {
            mb->length(total + 1);
            vfeisrv->put(mb);
        }
        else
        {
            mb->release();
        }
    }
    replymsg = NULL;

    return replymsg;
}

//-----------------------------------------------------------------------------
// Thread of received message event.
//-----------------------------------------------------------------------------
BS2Message * HSAlarmManager::msg_svc(JGMessageTrigger * trigger,
                                     BS2Message * msg)
{
    ACE_UNUSED_ARG(trigger);
    BS2Message * replymsg = NULL;

    if (msg->sf() == SFCODE(5,1))
    {
        replymsg = this->alarmReport(msg);
    }
    else
    {   // Unexpected message
        replymsg = this->unrecognized(msg);
    }

    return replymsg;
}

