// $Id$

//=============================================================================
/**
 *  @file    HSTraceManager.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2003 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSTraceManager.h"
#include "HSServer.h"
#include "HSInfoManager.h"
#include "HSSECSDevice.h"
#include "jyugem/gem/JGTaskTrigger.h"
#include "jyugem/gem/JGTrace.h"
#include "BS2ACKMessage.h"
#include "BS2ErrorMessage.h"
#include "BS2ListItem.h"
#include "BS2DeclAtoms.h"

static HSTraceManager * _manager = NULL;

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
HSTraceManager::HSTraceManager() : JGTraceManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSTraceManager::HSTraceManager");
}

//-----------------------------------------------------------------------------
HSTraceManager::~HSTraceManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSTraceManager::~HSTraceManager");
}

//-----------------------------------------------------------------------------
// Return own.
//-----------------------------------------------------------------------------
HSTraceManager * HSTraceManager::instance()
{
    TRACE_FUNCTION(TRL_LOW, "HSTraceManager::instance");
    if (_manager == NULL)
    {
        _manager = new HSTraceManager;
    }
    return _manager;
}

//-----------------------------------------------------------------------------
// Setup trace parameters.
//-----------------------------------------------------------------------------
int HSTraceManager::setup(int tid, bool enable, int trace_id, size_t frequency,
                          size_t samples, size_t report_size,
                          const std::vector<std::string>& vids,
                          const std::string& dir)
{
    TRACE_FUNCTION(TRL_LOW, "HSTraceManager::setup");

    int result = 0;
    BS2Message * msg = NULL;
    JGid trid(trace_id);
    JGTrace * trcptr = this->find(trid);
    if (enable && (trcptr != NULL))
    {
        TRACE_ERROR((_TX("It has been tracing (%d).\n"), trace_id));
        if (trcptr->active())
        {
            trcptr->totsmp(0);
            msg = BS2Message::factory(SFCODE(2,23));
            BS2Item * item = trcptr->makeInitItem();
            msg->add(item);
        }
    }
    else if (enable)
    {
        HSInfoManager * info_mngr = (HSInfoManager *)this->equipment()->getInfoManager();
        int seconds = frequency;
        if (frequency >= 60*60*100)
        {
            seconds = 60*60*100 - 1;
        }
        BCHAR buf[32];
        _stprintf(buf, _TX("%02d%02d%02d"), seconds / (60 * 60), (seconds / 60) %60,
                                            seconds % 60);
        JGTrace trace(trid, samples, report_size, buf);
        trace.dir(dir);
        trace.manager(this);
        for (size_t i = 0; i < vids.size(); i++)
        {
            JGid vid;
            vid.set(vids[i], info_mngr->vidFormat());
            JGVariable * var = info_mngr->variable(vid);
            if (var == NULL)
            {
                TRACE_ERROR((_TX("Variable is not found (%s).\n"),
                            vids[i].c_str()));
                continue ;
            }
            trace.insert(var);
        }
        this->entry(&trace);

        msg = BS2Message::factory(SFCODE(2,23));
        BS2Item * item = trace.makeInitItem();
        msg->add(item);

        trcptr = this->find(trid);
    }
    else if (trcptr != NULL)
    {   // Disable trace
        if (trcptr->active())
        {
            trcptr->totsmp(0);
            msg = BS2Message::factory(SFCODE(2,23));
            BS2Item * item = trcptr->makeInitItem();
            msg->add(item);
        }
    }
    if (msg != NULL)
    {
        result = this->send(msg, tid);
        m_recentTrace = trcptr;
    }
    return result;
}

//-----------------------------------------------------------------------------
// Responce of Initialize Trace : S2F24
//-----------------------------------------------------------------------------
int HSTraceManager::traceStart(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "JGTraceManager::traceStart");

    BS2Atom * tiaack = msg->getAtom(_TX("TIAACK"));
    if (tiaack == NULL)
    {
        TRACE_ERROR((_TX("Can't parse message: unknown item(TIAACK).\n")));
        return -1;
    }

    int ack = tiaack->getInt();
    if (ack == ACK_OK)
    {
        if (m_recentTrace->totsmp() > 0)
        {   // enable trace
            m_recentTrace->active(true);
        }
        else
        {   // disable trace
        }   // ignore trace ack
    }
    else
    {
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Trace Record : S6F1 --> S6F2
//-----------------------------------------------------------------------------
BS2Message * HSTraceManager::traceRecord(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "JGTraceManager::traceRecord");

    BYTE ack = ACK_OK;
    BS2Message * replymsg;

    //
    // (1) Save trace data
    // (2) Check trace end
    //

    // Create reply message.
    replymsg = BS2Message::response(msg, ack, _TX("ACKC6"));
    return replymsg;
}

//-----------------------------------------------------------------------------
// Thread of received trigger event.
//-----------------------------------------------------------------------------
BS2Message * HSTraceManager::msg_svc(JGMessageTrigger * trigger,
                                     BS2Message * msg)
{
    ACE_UNUSED_ARG(trigger);
    BS2Message * replymsg = NULL;

    if (msg->sf() == SFCODE(2,24))
    {
        int result = this->traceStart(msg);
    }
    else if (msg->sf() == SFCODE(6,1))
    {   // Ignore
        replymsg = this->traceRecord(msg);
    }
    else
    {   // Unexpected message
        replymsg = this->unrecognized(msg);
    }
    return replymsg;
}
