// $Id$

//=============================================================================
/**
 *  @file    HSServer.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSServer.h"
#include "HSAlarmManager.h"
#include "HSEventManager.h"
#include "HSDevice.h"
#include "HSSECSDevice.h"
#include "HSMainManager.h"
#include "HSInfoManager.h"
#include "HSTraceManager.h"
#include "HSCmdManager.h"
#include "HSTerminalManager.h"
#include "HSControlManager.h"
#include "HSProcProgManager.h"
#include "HSInitializeCmd.h"
#include "jyugem/gem/JGTimeModule.h"
#include "jyugem/gem/JGObjectManager.h"
#include "jyugem/gem/JGRCmdManager.h"
#include "jyugem/gem/JGTaskTrigger.h"
#include "jyugem/cell/host_message.h"
#include "c_wrapper.h"

using namespace jyugem::cell;

//
// send report or reply to host
//
static int _report_to_host(host_message * message)
{
    TRACE_FUNCTION(TRL_LOW, "_report_to_host@HSServer");
    std::string buf;
    HSMainManager * mainsrv = HSMainManager::instance();
    int buf_size = message->toXml(buf) + 128;
    ACE_Message_Block * mb = new ACE_Message_Block(buf_size);
    if (mb == NULL)
    {
        TRACE_ERROR((_TX("Lack message block buffer(%d)\n"), buf_size));
        return -1;
    }
    else
    {
        _tcscpy(mb->wr_ptr(), buf.c_str());
        mb->length(buf_size + 1);
        mainsrv->put(mb);
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Initialize
//-----------------------------------------------------------------------------
int HSServer::init(void * parm)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::init");

    int retval;

    //
    // Create gem managers
    //
    m_obj_mngr = ObjManager::instance();
    m_obj_mngr->equipment(this);
    m_managers.push_back(m_obj_mngr);

    m_info_mngr = HSInfoManager::instance();
    m_info_mngr->equipment(this);
    m_managers.push_back(m_info_mngr);

    m_systime = JGTimeModule::instance();
    m_systime->equipment(this);
    m_managers.push_back(m_systime);

    m_alarm_mngr = HSAlarmManager::instance();
    m_alarm_mngr->equipment(this);
    m_managers.push_back(m_alarm_mngr);

    m_event_mngr = HSEventManager::instance();
    m_event_mngr->equipment(this);
    m_managers.push_back(m_event_mngr);

    m_control_mngr = HSControlManager::instance();
    m_control_mngr->equipment(this);
    m_managers.push_back(m_control_mngr);

    m_trace_mngr = HSTraceManager::instance();
    m_trace_mngr->equipment(this);
    m_managers.push_back(m_trace_mngr);

    m_rcmd_mngr = JGRCmdManager::instance();
    m_rcmd_mngr->equipment(this);
    m_managers.push_back(m_rcmd_mngr);

    m_pp_mngr = HSProcProgManager::instance();
    m_pp_mngr->equipment(this);
    m_managers.push_back(m_pp_mngr);

    // Load data from DB
    retval = this->load();
    if (retval < 0)
    {
        return retval;
    }

    // create device
    m_devptr = this->initDevice();
    if (m_devptr < 0)
    {
        ACE_ERROR((LM_ERROR, ACE_TEXT("JGEquipment Loaded Error SECS Driver.\n")));
        return BEE_ERROR;
    }

    this->act_managers(m_devptr);

    ACE_DEBUG((LM_DEBUG, ACE_TEXT("Start Managers.\n")));

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Initialize equipment/host device
//-----------------------------------------------------------------------------
HSSECSDevice * HSServer::initDevice()
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::initDevice");

    HSSECSDevice * device = new HSSECSDevice;
    // read config table
    DeviceParameter * dparm = m_info_mngr->getDeviceParameter();

    const string& equipName = this->getConf(CONF_EQUIP_NAME);
    BS2Assert(equipName.size() > 0);    // check parameter value
    device->setName(equipName.c_str());

    if (device->initialize(dparm) >= 0)
    {
        device->setEquipment(this);
    }
    else
    {
        TRACE_ERROR((_TX("%s manager does not initialize device.\n"),
                     this->charName()));
        delete device;
        device = NULL;
    }
    delete dparm;
    m_device = device;
    return device;
}

//-----------------------------------------------------------------------------
// Entry Initial Command Module
//-----------------------------------------------------------------------------
int HSServer::entryInitializeCmdModue(void * param)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::entryInitializeCmdModue");

    HSInitializeCmd::InitConfig * conf = (HSInitializeCmd::InitConfig *)param;
    int result = 0;
    if (m_initcmd == NULL)
    {
        m_initcmd = HSInitializeCmd::instance();
        m_initcmd->equipment(this);
        m_managers.push_back(m_initcmd);

        if ((result = m_initcmd->init(param)) < 0)   // link variables
        {
            TRACE_ERROR((_TX("InitializeCmd init error.\n")));
            return BEE_ERROR;
        }
        m_initcmd->device(m_devptr);
    }
    m_initcmd->tid(conf->m_tid);
    m_initcmd->open();
    ACE_OS::sleep(1);
    m_initcmd->notify(_TX("start"));
    return result;
}

//-----------------------------------------------------------------------------
// Online
//-----------------------------------------------------------------------------
int HSServer::reqOnline()
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::reqOnline");

    int result = 0;


    return result;
}


//-----------------------------------------------------------------------------
// Abort command
//-----------------------------------------------------------------------------
int HSServer::abort(int tid, const std::string& port, int level)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::abort");

    HSCmdManager * cmd_mngr = (HSCmdManager *)this->findManager(CATEGORY_RCMD);
    int result = cmd_mngr->abort(tid, port, level);

    return result;
}

//-----------------------------------------------------------------------------
// Alarm Report
//-----------------------------------------------------------------------------
int HSServer::alarm_report(int alid, BYTE alstate, const std::string& altx,
                           BYTE alcd, int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::alarm_report");

    HSMainManager * mainsrv = HSMainManager::instance();
    bool state = (alstate) ? true : false;
    alarm_report_message alarm(mainsrv, this->name(), alid, state, altx,
                               alcd, errcode, errtext);
    int result = _report_to_host(&alarm);
    return result;
}

//-----------------------------------------------------------------------------
// Alarm Setup command
//-----------------------------------------------------------------------------
int HSServer::alarm_setup(int tid, bool tf,
                          const std::vector<std::string>& alids)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::setupAlarm");

    int result = 0;
    BYTE aled = (tf) ? 0x80 : 0;
    HSAlarmManager * alarm_mngr = (HSAlarmManager *)m_alarm_mngr;
    if (alids.size() == 0)
    {
        result = alarm_mngr->setupAll(tid, aled);
    }
    else
    {
        HSInfoManager * info_mngr = (HSInfoManager *)m_info_mngr;
        for (size_t i = 0; i  < alids.size(); i++)
        {
            JGid alid;
            alid.set(alids[i], info_mngr->alidFormat());
            result = alarm_mngr->setup(tid, aled, alid);
            if (result < 0)
            {
                break;
            }
        }
    }
    return result;
}

//-----------------------------------------------------------------------------
// Command Ack
//-----------------------------------------------------------------------------
int HSServer::command_ack(int tid, int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::command_ack");

    HSMainManager * mainsrv = HSMainManager::instance();
    cmd_ack_message ack(mainsrv, this->name(), tid, errcode, errtext);
    int result = _report_to_host(&ack);
    return result;
}


//-----------------------------------------------------------------------------
// Display command
//-----------------------------------------------------------------------------
int HSServer::display(int tid, BYTE terminal, const std::string& text)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::display");

    HSTerminalManager * term_mngr;
    term_mngr = (HSTerminalManager *)this->findManager(CATEGORY_TERMINAL);
    int result = term_mngr->display(tid, terminal, text);

    return result;
}

//-----------------------------------------------------------------------------
// Event Report
//-----------------------------------------------------------------------------
int HSServer::event_report(const std::string& ceid, const std::string& rpt,
                           const std::string& file,
                           int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::event_report");

    HSMainManager * mainsrv = HSMainManager::instance();
    event_report_message event(mainsrv, this->name(), ceid, rpt, file,
                               errcode, errtext);
    int result = _report_to_host(&event);
    return result;
}

//-----------------------------------------------------------------------------
// Event Setup command
//-----------------------------------------------------------------------------
int HSServer::event_setup(int tid, bool ceed,
                          const std::vector<std::string>& ceids,
                          const std::vector<std::string>& vids,
                          const std::string& dir)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::event_setup");

    HSEventManager * event_mngr = (HSEventManager *)m_event_mngr;
    int result = event_mngr->setup(tid, ceed, ceids, vids, dir);
    return result;
}

//-----------------------------------------------------------------------------
// Initialize command
//-----------------------------------------------------------------------------
int HSServer::initialize(int tid)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::initialize");

    HSInitializeCmd::InitConfig conf;
    conf.m_tid = tid;
    int result = this->entryInitializeCmdModue(&conf);
    return result;
}

//-----------------------------------------------------------------------------
// Machine command
//-----------------------------------------------------------------------------
int HSServer::machine_command(int tid, const std::string& cmdtype,
                              const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::machine_command");

    int result = 0;
    HSControlManager * control_mngr =
            (HSControlManager *)this->findManager(CATEGORY_CONTROL);
    if (cmdtype == _TX("LOOPBACK"))
    {
        result = control_mngr->loopback(tid, params);
    }
    else if (cmdtype == _TX("HELLO"))
    {
        result = control_mngr->heartbeat(tid, params);
    }
    else
    {
        TRACE_ERROR((_TX("Unsuport command type (%s).\n"),
                    cmdtype.c_str()));
        result = -1;
    }

    return result;
}

//-----------------------------------------------------------------------------
// Move command
//-----------------------------------------------------------------------------
int HSServer::move_command(int tid, const std::string& tgtype,
                           const std::string& tgid, const std::string& src_loc,
                           const std::string& dest_loc,
                           const std::string& src_port,
                           const std::string& dest_port)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::move_command");

    int result = 0;

    return result;
}

//-----------------------------------------------------------------------------
// Pause command
//-----------------------------------------------------------------------------
int HSServer::pause(int tid, const std::string& port)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::pause");

    HSCmdManager * cmd_mngr = (HSCmdManager *)this->findManager(CATEGORY_RCMD);
    int result = cmd_mngr->pause(tid, port);

    return result;
}

//-----------------------------------------------------------------------------
// Resouce Activate command
//-----------------------------------------------------------------------------
int HSServer::resource_activate(int tid, const std::string& res_type,
                                const std::vector<std::string>& resids,
                                const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::resource_activate");

    int result = 0;
    if (res_type == _TX("RECIPE"))
    {   // To the recipe in the executable space
        if (resids.size() != 3)
        {
            return -1;
        }
        std::string port = resids[0];
        std::string lotid = resids[1];
        std::string ppid = resids[2];
        HSCmdManager * cmd_mngr = (HSCmdManager *)this->findManager(CATEGORY_RCMD);
        result = cmd_mngr->pp_select(tid, port, lotid, ppid, params);
    }
    else if (res_type == _TX("TOOL"))
    {   //
    }
    else if (res_type == _TX("COMPONENT"))
    {
    }
    else if (res_type == _TX("CONSUMABLE"))
    {
    }
    else if (res_type == _TX("CASSETTE_MAP"))
    {
    }
    else if (res_type == _TX("WAFER_MAP"))
    {
    }
    else
    {
        TRACE_ERROR((_TX("Unsuport command type (%s).\n"), res_type.c_str()));
        result = -1;
    }

    return result;
}

//-----------------------------------------------------------------------------
// Resouce Deactivate command
//-----------------------------------------------------------------------------
int HSServer::resource_deactivate(int tid, const std::string& res_type,
                                  const std::vector<std::string>& resids)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::resource_deactivate");

    int result = 0;
    if (res_type == _TX("RECIPE"))
    {
    }
    else if (res_type == _TX("TOOL"))
    {   //
    }
    else if (res_type == _TX("COMPONENT"))
    {
    }
    else if (res_type == _TX("CONSUMABLE"))
    {
    }
    else if (res_type == _TX("CASSETTE_MAP"))
    {
    }
    else if (res_type == _TX("WAFER_MAP"))
    {
    }
    else
    {
        TRACE_ERROR((_TX("Unsuport command type (%s).\n"), res_type.c_str()));
        result = -1;
    }

    return result;
}

//-----------------------------------------------------------------------------
// Resouce Delete command
//-----------------------------------------------------------------------------
int HSServer::resource_delete(int tid, const std::string& type,
                              const std::vector<std::string>& resids)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::resource_delete");

    int result = 0;

    return result;
}

//-----------------------------------------------------------------------------
// Resouce List command
//-----------------------------------------------------------------------------
int HSServer::resource_list(int tid, const std::string& type,
                            const std::vector<std::string>& resids,
                            const std::string& file,
                            int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::resource_list");

    HSMainManager * mainsrv = HSMainManager::instance();
    res_list_message reslist(mainsrv, this->name(), tid, type, resids,
                             file, errcode, errtext);
    int result = _report_to_host(&reslist);
    return result;
}

//-----------------------------------------------------------------------------
// Resouce Query command
//-----------------------------------------------------------------------------
int HSServer::resource_query(int tid, const std::string& res_type,
                             const std::vector<std::string>& resids,
                             const std::string& file)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::resource_query");

    int result = 0;
    if (res_type == _TX("RECIPE"))
    {
        HSProcProgManager * pp_mngr = (HSProcProgManager *)m_recipe_mngr;
        pp_mngr->pp_query(tid, file);
    }
    else if (res_type == _TX("TOOL"))
    {   //
    }
    else if (res_type == _TX("COMPONENT"))
    {
    }
    else if (res_type == _TX("CONSUMABLE"))
    {
    }
    else if (res_type == _TX("CASSETTE_MAP"))
    {
    }
    else if (res_type == _TX("WAFER_MAP"))
    {
    }
    else
    {
        TRACE_ERROR((_TX("Unsuport command type (%s).\n"), res_type.c_str()));
        result = -1;
    }

    return result;
}

//-----------------------------------------------------------------------------
// Resouce Request command
//-----------------------------------------------------------------------------
int HSServer::resource_request(int tid, const std::string& res_type,
                               const std::vector<std::string>& resids,
                               const std::string& file)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::resource_request");

    int result = 0;
    if (res_type == _TX("RECIPE"))
    {
        HSProcProgManager * pp_mngr = (HSProcProgManager *)m_recipe_mngr;
        pp_mngr->pp_upload(tid, resids[0], file);
    }
    else if (res_type == _TX("TOOL"))
    {   //
    }
    else if (res_type == _TX("COMPONENT"))
    {
    }
    else if (res_type == _TX("CONSUMABLE"))
    {
    }
    else if (res_type == _TX("CASSETTE_MAP"))
    {
    }
    else if (res_type == _TX("WAFER_MAP"))
    {
    }
    else
    {
        TRACE_ERROR((_TX("Unsuport command type (%s).\n"), res_type.c_str()));
        result = -1;
    }

    return result;
}

//-----------------------------------------------------------------------------
// Resouce Transfer command
//-----------------------------------------------------------------------------
int HSServer::resource_transfer(int tid, const std::string& res_type,
                                const std::string& resid, size_t length,
                                const char * body)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::resource_transfer");

    int result = 0;
    if (res_type == _TX("RECIPE"))
    {
        HSProcProgManager * pp_mngr = (HSProcProgManager *)m_recipe_mngr;
        pp_mngr->pp_download(tid, resid, length, (BYTE *)body);
    }
    else if (res_type == _TX("TOOL"))
    {   //
    }
    else if (res_type == _TX("COMPONENT"))
    {
    }
    else if (res_type == _TX("CONSUMABLE"))
    {
    }
    else if (res_type == _TX("CASSETTE_MAP"))
    {
    }
    else if (res_type == _TX("WAFER_MAP"))
    {
    }
    else
    {
        TRACE_ERROR((_TX("Unsuport command type (%s).\n"), res_type.c_str()));
        result = -1;
    }

    return result;
}

//-----------------------------------------------------------------------------
// Restart command
//-----------------------------------------------------------------------------
int HSServer::restart(int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::restart");

    HSMainManager * mainsrv = HSMainManager::instance();
    restart_message restart(mainsrv, this->name(), errcode, errtext);
    int result = _report_to_host(&restart);
    return result;
}

//-----------------------------------------------------------------------------
// Resume command
//-----------------------------------------------------------------------------
int HSServer::resume(int tid, const std::string& port,
                     const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::resume");

    HSCmdManager * cmd_mngr = (HSCmdManager *)this->findManager(CATEGORY_RCMD);
    int result = cmd_mngr->resume(tid, port, params);

    return result;
}

//-----------------------------------------------------------------------------
// Set Parameter command
//-----------------------------------------------------------------------------
int HSServer::set_param(int tid, const std::string& res_type,
                        const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::set_param");

    int result = 0;
    if (res_type == _TX("RECIPE"))
    {   // To the recipe in the executable space
    }
    else if (res_type == _TX("TOOL"))
    {   //
    }
    else if (res_type == _TX("COMPONENT"))
    {
    }
    else if (res_type == _TX("CONSUMABLE"))
    {
    }
    else if (res_type == _TX("MACHINE"))
    {   // Sets equipment constant
        HSInfoManager * info_mngr = (HSInfoManager *)this->getInfoManager();
        result = info_mngr->setParam(tid, params);
    }
    else if (res_type == _TX("VFEI"))
    {   // Sets vfei driver
    }
    else
    {
        TRACE_ERROR((_TX("Unsuport resource type (%s).\n"), res_type.c_str()));
        result = -1;
    }
    return result;
}

//-----------------------------------------------------------------------------
// Start command
//-----------------------------------------------------------------------------
int HSServer::start(int tid, const std::string& port, const std::string& lotid,
                    int mbc_size, const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::start");

    HSCmdManager * cmd_mngr = (HSCmdManager *)this->findManager(CATEGORY_RCMD);
    int result = cmd_mngr->start(tid, port, lotid, mbc_size, params);

    return result;
}

//-----------------------------------------------------------------------------
// Status List command
//-----------------------------------------------------------------------------
int HSServer::status_list(int tid, const std::string& status,
                          const std::string& file,
                          int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::status_list");

    HSMainManager * mainsrv = HSMainManager::instance();
    status_list_message svlist(mainsrv, this->name(), tid, status, file,
                               errcode, errtext);
    int result = _report_to_host(&svlist);
    return result;
}

//-----------------------------------------------------------------------------
// Status Query command
//-----------------------------------------------------------------------------
int HSServer::status_query(int tid, const std::vector<std::string>& vids,
                           const std::string& file)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::status_query");

    HSInfoManager * info_mngr = (HSInfoManager *)this->getInfoManager();
    int result = info_mngr->inqStatus(tid, vids, file);
    return result;
}

//-----------------------------------------------------------------------------
// Trace Report command
//-----------------------------------------------------------------------------
int HSServer::trace_report(int tid, int trace_id, size_t report_length,
                           const std::string& rpt, const std::string& file,
                           int errcode, const std::string& errtext)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::trace_report");

    HSMainManager * mainsrv = HSMainManager::instance();
    trace_report_message trace(mainsrv, this->name(), trace_id, report_length,
                               rpt, file, errcode, errtext);
    int result = _report_to_host(&trace);
    return result;
}


//-----------------------------------------------------------------------------
// Trace Setup command
//-----------------------------------------------------------------------------
int HSServer::trace_setup(int tid, bool enable, int trace_id, size_t frequency,
                          size_t samples, size_t report_size,
                          const std::vector<std::string>& vids,
                          const std::string& dir)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::trace_setup");

    HSTraceManager * trace_mngr =
            (HSTraceManager *)this->findManager(CATEGORY_TRACE);
    int result = trace_mngr->setup(tid, enable, trace_id, frequency, samples,
                                   report_size, vids, dir);

    return result;
}

//------------------------------------------------------------------------------
// Request host command to equipment.
//------------------------------------------------------------------------------
int HSServer::request(const host_cmd& hcmd)
{
    TRACE_FUNCTION(TRL_LOW, "HSServer::request");

    HSCmdManager * cmd_mngr = (HSCmdManager *)this->findManager(CATEGORY_RCMD);
    int result;
    if (hcmd.name() == _TX("ABORT"))
    {
        result = cmd_mngr->request(hcmd);
    }
    else if (hcmd.name() == _TX("ALARM_SETUP"))
    {
        alarm_setup_message& alarm_setup_cmd = (alarm_setup_message&)hcmd;
        result = alarm_setup(alarm_setup_cmd.m_tid, alarm_setup_cmd.m_enable,
                             alarm_setup_cmd.m_alids);
    }
    else if (hcmd.name() == _TX("DISPLAY_MSG"))
    {
        display_message& display_cmd = (display_message&)hcmd;
        result = display(display_cmd.m_tid, display_cmd.m_terminal, display_cmd.m_text);
    }
    else if (hcmd.name() == _TX("EVENT_SETUP"))
    {
        event_setup_message& event_setup_cmd = (event_setup_message&)hcmd;
        result = event_setup(event_setup_cmd.m_tid, event_setup_cmd.m_enable,
                             event_setup_cmd.m_ceids, event_setup_cmd.m_vars,
                             event_setup_cmd.m_dir);
    }
    else if (hcmd.name() == _TX("INITIALIZE"))
    {
        result = initialize(hcmd.m_tid);
    }
    else if (hcmd.name() == _TX("MACH_CMD"))
    {
        mach_cmd_message& mach_cmd = (mach_cmd_message&)hcmd;
        result = machine_command(mach_cmd.m_tid, mach_cmd.m_command_type,
                                 mach_cmd.m_items);
    }
    else if (hcmd.name() == _TX("MOVE"))
    {
        result = cmd_mngr->request(hcmd);
    }
    else if (hcmd.name() == _TX("PAUSE"))
    {
        result = cmd_mngr->request(hcmd);
    }
    else if (hcmd.name() == _TX("RES_ACTIVATE"))
    {
        res_activate_message& res_act_cmd = (res_activate_message&)hcmd;
        result = resource_activate(res_act_cmd.m_tid, res_act_cmd.m_type,
                                   res_act_cmd.m_resid, res_act_cmd.m_items);
    }
    else if (hcmd.name() == _TX("RES_DEACTIVATE"))
    {
        res_deactivate_message& res_deact_cmd = (res_deactivate_message&)hcmd;
        result = resource_deactivate(res_deact_cmd.m_tid, res_deact_cmd.m_type,
                                     res_deact_cmd.m_resid);
    }
    else if (hcmd.name() == _TX("RES_DELETE"))
    {
        res_delete_message& res_del_cmd = (res_delete_message&)hcmd;
        result = resource_delete(res_del_cmd.m_tid, res_del_cmd.m_type,
                                 res_del_cmd.m_resid);
    }
    else if (hcmd.name() == _TX("RES_QUERY"))
    {
        res_query_message& res_query_cmd = (res_query_message&)hcmd;
        result = resource_query(res_query_cmd.m_tid, res_query_cmd.m_type,
                                res_query_cmd.m_resid, res_query_cmd.m_file);
    }
    else if (hcmd.name() == _TX("RES_REQUEST"))
    {
        res_request_message& res_req_cmd = (res_request_message&)hcmd;
        result = resource_request(res_req_cmd.m_tid, res_req_cmd.m_type,
                                  res_req_cmd.m_resid, res_req_cmd.m_file);
    }
    else if (hcmd.name() == _TX("RES_TRANSFER"))
    {
        res_transfer_message& res_trans_cmd = (res_transfer_message&)hcmd;
        result = resource_transfer(res_trans_cmd.m_tid, res_trans_cmd.m_type,
                                   res_trans_cmd.m_resid, res_trans_cmd.m_length,
                                   res_trans_cmd.m_body);
    }
    else if (hcmd.name() == _TX("RESTART"))
    {
        result = cmd_mngr->request(hcmd);
    }
    else if (hcmd.name() == _TX("RESUME"))
    {
        result = cmd_mngr->request(hcmd);
    }
    else if (hcmd.name() == _TX("SET_PARAM"))
    {
        set_param_message& set_param_cmd = (set_param_message&)hcmd;
        result = set_param(set_param_cmd.m_tid, set_param_cmd.m_type,
                           set_param_cmd.m_items);
    }
    else if (hcmd.name() == _TX("START"))
    {
        result = cmd_mngr->request(hcmd);
    }
    else if (hcmd.name() == _TX("STATUS_QUERY"))
    {
        status_query_message& status_query_cmd = (status_query_message&)hcmd;
        result = status_query(status_query_cmd.m_tid, status_query_cmd.m_vids,
                              status_query_cmd.m_file);
    }
    else if (hcmd.name() == _TX("TRACE_SETUP"))
    {
        trace_setup_message& trace_setup_cmd = (trace_setup_message&)hcmd;
        result = trace_setup(trace_setup_cmd.m_tid, trace_setup_cmd.m_enable,
                             trace_setup_cmd.m_traceid,
                             trace_setup_cmd.m_frequency,
                             trace_setup_cmd.m_sample,
                             trace_setup_cmd.m_rptsize,
                             trace_setup_cmd.m_vids, trace_setup_cmd.m_dir);
    }
    else if (hcmd.name() == _TX("ALARM_REPORT"))
    {
    }
    else if (hcmd.name() == _TX("CMD_ACK"))
    {
    }
    else if (hcmd.name() == _TX("EVENT_REPORT"))
    {
    }
    else if (hcmd.name() == _TX("RES_LIST"))
    {
    }
    else if (hcmd.name() == _TX("STATUS_LIST"))
    {
    }
    else if (hcmd.name() == _TX("TRACE_REPORT"))
    {
    }
    else
    {
    }
    return result;
}

//------------------------------------------------------------------------------
// Start communication in service
//------------------------------------------------------------------------------
int HSServer::svc()
{
    TRACE_FUNCTION(TRL_MIDDLE, "HSServer::svc");

    // Check ready state in driver
    // m_comm_mngr->start();           // Start communication

    ACE_Reactor::instance()->handle_events();

    TRACE_DEBUG((_TX("Return ......\n")));

    return BEE_SUCCESS;
}

