// $Id$

//=============================================================================
/**
 *  @file    HSProcProgManager.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2002 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSProcProgManager.h"
#include "HSServer.h"
#include "jyugem/gem/JGVariable.h"
#include "jyugem/gem/JGTrigger.h"
#include "jyugem/gem/JGProcessManager.h"
#include "BS2ErrorMessage.h"
#include "BS2Message.h"
#include "BS2ListItem.h"
#include "BS2DeclAtoms.h"


static HSProcProgManager * _manager = NULL;

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
HSProcProgManager::HSProcProgManager() : JGManager(CATEGORY_PROCPROG)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSProcProgManager::HSProcProgManager");
}

//-----------------------------------------------------------------------------
HSProcProgManager::~HSProcProgManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSProcProgManager::~HSProcProgManager");
}

//-----------------------------------------------------------------------------
// Return own.
//-----------------------------------------------------------------------------
HSProcProgManager * HSProcProgManager::instance()
{
    TRACE_FUNCTION(TRL_LOW, "HSProcProgManager::instance");
    if (_manager == NULL)
    {
        _manager = new HSProcProgManager;
    }
    return _manager;
}

//-----------------------------------------------------------------------------
// Initialize
//-----------------------------------------------------------------------------
int HSProcProgManager::init(void * parm)
{
    TRACE_FUNCTION(TRL_LOW, "HSProcProgManager::init");
    ACE_UNUSED_ARG(parm);

    return BEE_SUCCESS;
}

//-----------------------------------------------------------------------------
// Current Process Program Directory Request
//-----------------------------------------------------------------------------
int HSProcProgManager::pp_query(int tid, const std::string& file)
{
    TRACE_FUNCTION(TRL_LOW, "HSProcProgManager::pp_query");

    m_eppd = file;
    BS2Message * msg = BS2Message::factory(SFCODE(7,19));
    int result = this->send(msg, tid);
    if (result < 0)
    {
         TRACE_ERROR((_TX("SECS Message send error.\n")));
    }
    return result;
}

//-----------------------------------------------------------------------------
// Request Process Program
//-----------------------------------------------------------------------------
int HSProcProgManager::pp_upload(int tid, const std::string& ppid,
                               const std::string& file)
{
    TRACE_FUNCTION(TRL_LOW, "HSProcProgManager::pp_upload");

    m_ppfile = file;
    BS2Message * msg = BS2Message::factory(SFCODE(7,5));
    BS2Item * item = BS2Item::factory(_TX("PPID"), new BS2Ascii(ppid));
    msg->add(item);
    int result = this->send(msg, tid);
    if (result < 0)
    {
         TRACE_ERROR((_TX("SECS Message send error.\n")));
    }
    return result;
}

//-----------------------------------------------------------------------------
// Send Process Program
//-----------------------------------------------------------------------------
int HSProcProgManager::pp_download(int tid, const std::string& ppid, size_t size,
                                 BYTE * body)
{
    TRACE_FUNCTION(TRL_LOW, "HSProcProgManager::pp_download");

    m_ppid = ppid;
    m_ppbody = new BS2Binary(body, size);  // save ppbody
    BS2Message * msg = BS2Message::factory(SFCODE(7,1));
    BS2ListItem * bodylist = new BS2ListItem();
    msg->add(bodylist);
    BS2Item * item = BS2Item::factory(_TX("PPID"), new BS2Ascii(ppid));
    bodylist->add(item);
    item = BS2Item::factory(_TX("LENGTH"), new BS2UInt4(size));
    bodylist->add(item);
    // send inquiry
    int result = this->send(msg, tid);
    if (result < 0)
    {
         TRACE_ERROR((_TX("SECS Message send error.\n")));
    }
    return result;
}

//-----------------------------------------------------------------------------
// Process Program Load Grant (S7F2) and Load (S7F3)
//-----------------------------------------------------------------------------
int HSProcProgManager::pp_grant(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "HSProcProgManager::pp_grant");

    BS2Atom * atom = msg->getAtom(_TX("PPGNT"));
    if (atom == NULL)
    {
        TRACE_ERROR((_TX("PPGNT item is nothing.\n")));
        return -1;
    }
    if (! atom->isBinary())
    {
        TRACE_ERROR((_TX("Invalid PPGNT item format.\n")));
        return -1;
    }
    BYTE * bin = ((BS2Binary *)atom)->value();
    if (*bin != 0)
    {
        static BCHAR * _ppgnt[] =
        {
            "OK", "Already have", "No space", "Invalid PPID", "Busy, try later",
            "Will not accept", "Other errors" , "Unknown error"
        };
        TRACE_ERROR((_TX("Can't load process program (%s).\n"), _ppgnt[*bin & 0x7]));
        return -1;
    }

    msg = BS2Message::factory(SFCODE(7,3));
    BS2ListItem * bodylist = new BS2ListItem();
    msg->add(bodylist);
    BS2Item * item = BS2Item::factory(_TX("PPID"), new BS2Ascii(m_ppid));
    bodylist->add(item);
    item = BS2Item::factory(_TX("PPBODY"), m_ppbody);
    bodylist->add(item);

    int result = this->send(msg, m_tid);
    if (result < 0)
    {
         TRACE_ERROR((_TX("SECS Message send error.\n")));
    }
    return result;
}

//-----------------------------------------------------------------------------
// PP Load Grant (S7F6)
//-----------------------------------------------------------------------------
int HSProcProgManager::pp_body(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "HSProcProgManager::pp_body");

    BS2Atom * atom = msg->getAtom(_TX("PPBODY"));
    if (atom == NULL)
    {
        TRACE_ERROR((_TX("PPBODY item is nothing.\n")));
        return -1;
    }
    size_t size;
    BYTE * body;
    if (atom->isBinary())
    {
        body = ((BS2Binary *)atom)->value();
        size = ((BS2Binary *)atom)->size();
    }
    else if (atom->isAscii())
    {
        const std::string bodystr = ((BS2Ascii *)atom)->value();
        body = (BYTE *)bodystr.c_str();
        size = bodystr.size();
    }
    else
    {
        TRACE_ERROR((_TX("Invalid PPBODY item format.\n")));
        return -1;
    }
    // save ppbody
    if (m_ppbody->size() == 0)
    {
        //
    }
    else
    {
        // store recipe file
    }
    return 0;
}

//-----------------------------------------------------------------------------
// Process Program Load (S7F1)
//-----------------------------------------------------------------------------
int HSProcProgManager::pp_inquire(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "HSProcProgManager::pp_grant");

    BS2Atom * atom = msg->getAtom(_TX("PPID"));
    if (atom == NULL)
    {
        TRACE_ERROR((_TX("PPID item is nothing.\n")));
        return -1;
    }
    if (! atom->isAscii())
    {
        TRACE_ERROR((_TX("Invalid PPBODY item format.\n")));
        return -1;
    }
    m_inqPpid = ((BS2Ascii *)atom)->value();

    atom = msg->getAtom(_TX("LENGTH"));
    if (atom == NULL)
    {
        TRACE_ERROR((_TX("LENGTH item is nothing.\n")));
        return -1;
    }
    m_inqLength = atom->getInt();

    return 0;
}

//-----------------------------------------------------------------------------
// Thread of received trigger event.
//-----------------------------------------------------------------------------
BS2Message * HSProcProgManager::msg_svc(JGMessageTrigger * trigger,
                                        BS2Message * msg)
{
    ACE_UNUSED_ARG(trigger);
    BS2Message * replymsg = NULL;

    if (msg->sf() == SFCODE(7,20))
    {
    }
    else if (msg->sf() == SFCODE(7,1))
    {
        this->pp_inquire(msg);
    }
    else if (msg->sf() == SFCODE(7,2))
    {
        this->pp_grant(msg);
    }
    else if (msg->sf() == SFCODE(7,4))
    {
        // ACK;
    }
    else if (msg->sf() == SFCODE(7,6))
    {
        this->pp_body(msg);
    }
    else
    {   // Unexpected message
        replymsg = this->unrecognized(msg);
    }

    return replymsg;
}


