// $Id$

//=============================================================================
/**
 *  @file    HSControlManager.cpp
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
//=============================================================================

#define BEE_BUILD_DLL

#include "HSControlManager.h"
#include "HSMainManager.h"
#include "HSServer.h"
#include "HSEventManager.h"
#include "HSInfoManager.h"
#include "HSSECSDevice.h"
#include "jyugem/gem/JGTaskTrigger.h"
#include "jyugem/gem/JGLogTime.h"
#include "BS2ACKMessage.h"
#include "BS2ErrorMessage.h"
#include "BS2ListItem.h"
#include "BS2DeclAtoms.h"

#include "jyugem/gem/JGVariable.h"

static HSControlManager * _manager = NULL;
static BYTE _test_pattern[] =
{
    0xFF, 0xAA, 0x55, 0xA5, 0x5A, 0xEE, 0x77, 0x66, 0x88, 0xFF
};

//-----------------------------------------------------------------------------
// Heart Beat Timer Handler
//-----------------------------------------------------------------------------
static BCHAR * _offline_alarm = _TX("OffLine");
const int VFEI_TIMEOUT = 10028;

class HeartbeatTimerHandler : public TaskTimerHandler
{
public:
    HeartbeatTimerHandler() : TaskTimerHandler() {}

    virtual void handle_time_out(const ACE_Time_Value &tv,
                                 const void * arg)
    {
        TRACE_FUNCTION(TRL_LOW, "HeartbeatTimerHandler::handle_time_out");
        HSControlManager * control = (HSControlManager *)arg;
        // Lock
        if (control->timer_id() == m_tmid)
        {
            HSServer * eqsvr = (HSServer *)control->equipment();
            eqsvr->alarm_report(VFEI_TIMEOUT, 0x80, _offline_alarm, 0);
            control->timer_id(0);
            eqsvr->online(false);
        }
        delete this;
    }
};

//-----------------------------------------------------------------------------
// Constructor/Destructor
//-----------------------------------------------------------------------------
HSControlManager::HSControlManager() : JGControlManager(), m_heartbeat(NULL)
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSControlManager::HSControlManager");
}

//-----------------------------------------------------------------------------
HSControlManager::~HSControlManager()
{
    TRACE_FUNCTION(TRL_CONSTRUCT, "HSControlManager::~HSControlManager");

}

//-----------------------------------------------------------------------------
// Return own.
//-----------------------------------------------------------------------------
HSControlManager * HSControlManager::instance()
{
    TRACE_FUNCTION(TRL_LOW, "HSControlManager::instance");
    if (_manager == NULL)
    {
        _manager = new HSControlManager;
    }
    return _manager;
}


//-----------------------------------------------------------------------------
// Send LoopBack
//-----------------------------------------------------------------------------
int HSControlManager::loopback(int tid, const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSControlManager::loopback");

    BS2Message * s2f25 = BS2Message::factory(SFCODE(2,25));
    BS2Item * item = BS2Item::factory(_TX("ABS"),
                         new BS2Binary(_test_pattern, sizeof(_test_pattern)));
    s2f25->add(item);
    int result = this->send(s2f25, tid);
    if (result < 0)
    {
         TRACE_ERROR((_TX("SECS Message send error.\n")));
    }
    return result;
}

//-----------------------------------------------------------------------------
// Are You There
//-----------------------------------------------------------------------------
int HSControlManager::heartbeat(int tid, const std::vector<any_item>& params)
{
    TRACE_FUNCTION(TRL_LOW, "HSControlManager::heartbeat");

    BS2Message * s1f1 = BS2Message::factory(SFCODE(1,1));
    int result = this->send(s1f1, tid);
    if (result < 0)
    {
         TRACE_ERROR((_TX("SECS Message send error.\n")));
    }
    if (m_heartbeat == NULL)
    {
        m_heartbeat = new HeartbeatTimerHandler();
        JGVariable * heartbeatTime = this->equipment()->variable(_TX("Heartbeat"));
        std::string timeStr = heartbeatTime->getv().toString();
        m_expire = _tcstol(timeStr.c_str(), NULL, 10);
    }
    else
    {
        if (m_timerid.value() != 0)
        {
            // duplicate;
        }
    }
    if (m_expire != 0)
    {
        ACE_Time_Value tv(m_expire);
        m_timerid = this->addTimer(tv, m_heartbeat);
    }
    return result;
}

//-----------------------------------------------------------------------------
// List Enabled Alarm message (S1F2)
//-----------------------------------------------------------------------------
BS2Message * HSControlManager::onLineData(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "HSControlManager::onLineData");

    if (m_timerid.value() != 0)
    {
        this->cancelTimer(m_timerid.value());
        m_timerid = 0;

        ACE_Time_Value tv(m_expire);
        m_timerid = this->addTimer(tv, m_heartbeat);
    }
    else
    {
        // gost reply message;
    }
    //
    return NULL;
}

//-----------------------------------------------------------------------------
// Are You There from Equipment (S1F1 -> S1F2)
//-----------------------------------------------------------------------------
BS2Message * HSControlManager::areYouThere(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "HSControlManager::areYouThere");

    BS2Message * replymsg = BS2Message::response(msg);
    BS2ListItem * bodylist = new BS2ListItem();
    replymsg->add(bodylist);

    HSServer * eqsvr = (HSServer *)this->equipment();
    eqsvr->restart();
    eqsvr->online(true);

    //
    return replymsg;
}

//-----------------------------------------------------------------------------
// Received Loopback (S2F26)
//-----------------------------------------------------------------------------
int HSControlManager::recvLoopback(BS2Message * msg)
{
    TRACE_FUNCTION(TRL_LOW, "HSControlManager::recvLoopback");

    int errcode = 0;
    BCHAR * errtext = _TX("");
    BS2Atom * atom = msg->getAtom(_TX("ABS"));
    if (atom == NULL)
    {
        TRACE_ERROR((_TX("ABS item is nothing.\n")));
        errcode = -1;
    }
    if (! atom->isBinary())
    {
        TRACE_ERROR((_TX("Invalid ABS item format.\n")));
        errcode = -1;
    }
    if (atom->size() != sizeof(_test_pattern))
    {
        TRACE_ERROR((_TX("Unexpect loopback size (%d:%d).\n"), atom->size(),
                    sizeof(_test_pattern)));
        errcode = -1;
    }
    BYTE * bin = ((BS2Binary *)atom)->value();
    if (memcmp(bin, _test_pattern, sizeof(_test_pattern)) != 0)
    {
        TRACE_ERROR((_TX("Unexpect loopback data(%s).\n"),
                    atom->toString().c_str()));
        errcode = -1;
    }
    HSServer * eqsvr = (HSServer *)this->equipment();
    UINT tid = this->eraseTransaction(msg->transNum());
    eqsvr->command_ack(tid, errcode, errtext);
    return 0;
}

//-----------------------------------------------------------------------------
// Thread of received message event.
//-----------------------------------------------------------------------------
BS2Message * HSControlManager::msg_svc(JGMessageTrigger * trigger,
                                       BS2Message * msg)
{
    ACE_UNUSED_ARG(trigger);
    BS2Message * replymsg = NULL;

    if (msg->sf() == SFCODE(1,1))
    {
        this->areYouThere(msg);
    }
    else if (msg->sf() == SFCODE(1,2))
    {
        this->onLineData(msg);
    }
    else if (msg->sf() == SFCODE(2,26))
    {
        this->recvLoopback(msg);
    }
    else
    {   // Unexpected message
        replymsg = this->unrecognized(msg);
    }

    return replymsg;
}

