/* $Id: ULong.java,v 1.6 2004/06/27 08:01:56 fukasawa Exp $ */

/**
 *  @file    ULong.java
 *
 *  @author  Fukasawa Mitsuo
 *
 *
 *    Copyright (C) 2001-2004 BEE Co.,Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

package com.jyugem.secs;

import java.*;
import java.lang.Integer;
import java.lang.Long;
import java.math.BigInteger;

public final class ULong extends Number implements Comparable {

    public final static BigInteger MIN_VALUE = new BigInteger("0");
    public final static BigInteger MAX_VALUE = new BigInteger("18446744073709551615");

    private BigInteger value;

    // Constructor
    public ULong(BigInteger v) {
        this.value = v;
    }

    public ULong(ULong v) {
        this.value = v.value;
    }

    public ULong(long v) {
        this.value = BigInteger.valueOf(v);
    }

    public ULong(int v) {
        this.value = BigInteger.valueOf((long)v);
    }

    public ULong(String s) throws NumberFormatException {
        this.value = new BigInteger(s);
        if ((this.value.compareTo(MIN_VALUE) < 0) ||
            (this.value.compareTo(MAX_VALUE) > 0)) {
           throw new NumberFormatException();
        }
    }

    public static String toString(int x) {
        BigInteger bi = BigInteger.valueOf((long)x);
        return bi.toString();
    }

    public static BigInteger parseULong(String s) throws NumberFormatException {
        return parseULong(s, 16);
    }

    public static BigInteger parseULong(String s, int radix)
                          throws NumberFormatException {
        BigInteger bi = new BigInteger(s, radix);
        if ((bi.compareTo(MIN_VALUE) < 0) ||
            (bi.compareTo(MAX_VALUE) > 0)) {
           throw new NumberFormatException();
        }
        return bi;
    }

    public static ULong valueOf(String s, int radix)
                           throws NumberFormatException {
        return new ULong(parseULong(s, radix));
    }

    public static ULong valueOf(String s) throws NumberFormatException {
        return valueOf(s, 10);
    }

    // Decodes a String into a ULong.  Accepts decimal, hexadecimal, and octal
    // numbers, in the following formats:
    //    1-9   decimal constant
    //    0x    hex constant
    //    #     hex constant
    //    0     octal constant
    public static ULong decode(String numStr) throws NumberFormatException {
        int radix = 10;
        int index = 0;
        boolean negative = false;
        ULong result;

        if (numStr.startsWith("-")) {
            throw new NumberFormatException("Can't sign.");
        }

        if (numStr.startsWith("0x", index) ||
            numStr.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (numStr.startsWith("#", index)) {
            index++;
            radix = 16;
        } else if (numStr.startsWith("0", index) &&
                   numStr.length() > (index + 1)) {
            index++;
            radix = 8;
        }

        if (numStr.startsWith("-", index)) {
            throw new NumberFormatException("Negative sign in wrong position");
        }

        try {
            result = ULong.valueOf(numStr.substring(index), radix);
            result = negative ? new ULong(-result.intValue()) : result;
        } catch (NumberFormatException e) {
            String constant = "";
            if (negative) {
                constant = "-";
            }
            constant += numStr.substring(index);
            result = ULong.valueOf(constant, radix);
        }
        return result;
    }

    //public byte byteValue() {
    //   return value.byteValue();
    //}

    public short shortValue() {
        return value.shortValue();
    }

    public int intValue() {
        return value.intValue();
    }

    public long longValue() {
        return value.longValue();
    }

    public float floatValue() {
        return value.floatValue();
    }

    public double doubleValue() {
        return value.doubleValue();
    }

    public String toString() {
        return value.toString();
    }

    public boolean equals(Object obj) {
        try {
            ULong u = (ULong)obj;
            return u.value.equals(this.value);
        } catch (ClassCastException ce) {
            // This was not an ULong, so equals should fail.
            return false;
        }
    }

    public int compareTo(ULong another) {
        return value.compareTo(another);
    }

    public int compareTo(Object obj) {
        return value.compareTo(((ULong)obj).value);
    }

}
