/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/preferences/ExtensionEntryDialog.java,v 1.2 2007/09/03 08:59:39 kohnosuke Exp $
 * $Revision: 1.2 $
 * $Date: 2007/09/03 08:59:39 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.preferences;

import java.util.Iterator;
import java.util.Map;

import jp.sourceforge.jeextension.common.plugin.DefaultDialog;
import jp.sourceforge.jeextension.styleeditor.core.StyleEditorTools;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.TableColumn;

/**
 * PartitionEntryDialog
 * @since 1.0.0
 */
class ExtensionEntryDialog extends DefaultDialog {
    
    /** Add button. */
    private Button fButtonAdd;
    
    /** Remove button. */
    private Button fButtonDel;
    
    /** Modify button. */
    private Button fButtonMod;
    
    /** partition entry table. */
    private Table fTable;
    
    /**
     * Initiate this dialog page.
     * @param shell parent shell.
     */
    ExtensionEntryDialog(Shell shell) {
        super(shell);
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
     */
    public void handleEvent(Event event) {
        if (event.widget == fButtonAdd) {
            fireAddEntry();
        }
        if (event.widget == fButtonMod) {
            fireModifyEntry();
        }
        if (event.widget == fButtonDel) {
            fireDeleteEntry();
        }
    }
        
    /**
     * Add extension into target Table.
     * @param   ext   extension name.
     */
    private void addTableItem(String ext) {
        /* add table item. */
        TableItem item = new TableItem(fTable, SWT.NONE);
        item.setText(0, ext);
    }
    
    /**
     * Create Button.
     * @param composite parent composite.
     * @param str button's string.
     * @return created Button.
     */
    private Button createButton(Composite composite, String str) {
        final Button button = new Button(composite, SWT.NONE);
        button.setLayoutData(new GridData(
          GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_BEGINNING));
        button.setText(str);
        button.addListener(SWT.Selection, this);
        return button;
    }
    
    /**
     * Create TableColumn.
     * @param table parent composite (Table).
     * @param str column's string.
     * @return created TableColumn.
     */
    private TableColumn createTableColumn(Table table, String str) {
        final TableColumn tableColumn = new TableColumn(table, SWT.NONE);
        tableColumn.setWidth(100);
        tableColumn.setText(str);
        return tableColumn;
    }
    
    /**
     * fire Add button action.
     */
    private void fireAddEntry() {
        String[] container = new String[1];
        AddExtensionDialog d = new AddExtensionDialog(getShell(), container);
        if (d.open() == IDialogConstants.OK_ID) {
            // add table item.
            if (container[0].equals("") == false) {
                addTableItem(container[0]);
            }
        }
    }
    
    /**
     * fire Delete button action.
     */
    private void fireDeleteEntry() {
        int sel = fTable.getSelectionIndex();
        if (sel < 0) {
            return;
        }
        TableItem item = fTable.getItem(sel);
        String ext = item.getText();
        
        // remove from table. and remove from conf list.
        fTable.remove(sel);
        StyleEditorTools.getPartitionEntryMap().remove(ext);
    }
    
    /**
     * fire Modify button action.
     */
    private void fireModifyEntry() {
        int sel = fTable.getSelectionIndex();
        if (sel < 0) {
            return;
        }
        TableItem item = fTable.getItem(sel);
        String ext = item.getText();
        String[] container = new String[1];
        container[0] = ext;
        AddExtensionDialog d = new AddExtensionDialog(getShell(), container);
        if (d.open() == IDialogConstants.OK_ID) {
            item.setText(0, container[0]);
        }
    }
    
    /**
     * Reflesh table entries.
     */
    private void refleshTableEntry() {
        // clear all table content.
        fTable.removeAll();
        
        // add entry extensions to table.
        Map map = StyleEditorTools.getPartitionEntryMap();
        Iterator iterator = map.keySet().iterator();
        while (iterator.hasNext()) {
            addTableItem((String) iterator.next());
        }
    }

    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
     */
    protected Control createDialogArea(Composite parent) {
        
        final Composite composite = (Composite) super.createDialogArea(parent);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        final GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        composite.setLayout(gridLayout);
        
        fTable = new Table(composite, SWT.FULL_SELECTION | SWT.BORDER);
        final GridData gridData = new GridData(GridData.FILL_BOTH);
        gridData.verticalSpan = 5;
        fTable.setLayoutData(gridData);
        fTable.setLinesVisible(true);
        fTable.setHeaderVisible(true);
        
        createTableColumn(fTable, getMessage("pref.com.colmn.ext"));
        
        fButtonAdd = createButton(composite, getMessage("pref.com.btn.add"));
        fButtonMod = createButton(composite, getMessage("pref.com.btn.mod"));
        fButtonDel = createButton(composite, getMessage("pref.com.btn.del"));
        
        // entry data.
        refleshTableEntry();
        
        return composite;
    }

    /**
     * Adds buttons to this dialog's button bar.
     * <p>
     * The <code>Dialog</code> implementation of this framework method adds 
     * standard ok and cancel buttons using the <code>createButton</code>
     * framework method. These standard buttons will be accessible from 
     * <code>getCancelButton</code>, and <code>getOKButton</code>.
     * Subclasses may override.
     * </p>
     *
     * @param parent the button bar composite
     */
    protected void createButtonsForButtonBar(Composite parent) {
        // create OK button only
        createButton(
            parent,
            IDialogConstants.OK_ID,
            IDialogConstants.OK_LABEL,
            true);
    }
}
