/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/core/EditorTools.java,v 1.4 2005/07/01 09:27:18 kohnosuke Exp $
 * $Revision: 1.4 $
 * $Date: 2005/07/01 09:27:18 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sourceforge.jeextension.common.xml.XMLNode;
import jp.sourceforge.jeextension.styleeditor.StyleEditorPlugin;
import jp.sourceforge.jeextension.styleeditor.entry.DefinedStyleEntry;
import jp.sourceforge.jeextension.styleeditor.entry.DependsEntry;
import jp.sourceforge.jeextension.styleeditor.entry.DependsEntryList;
import jp.sourceforge.jeextension.styleeditor.entry.FontEntry;
import jp.sourceforge.jeextension.styleeditor.entry.IDependsEntry;
import jp.sourceforge.jeextension.styleeditor.entry.IEntry;
import jp.sourceforge.jeextension.styleeditor.entry.IPartitionEntry;
import jp.sourceforge.jeextension.styleeditor.entry.PartitionEntry;
import jp.sourceforge.jeextension.styleeditor.entry.PartitionEntryList;
import jp.sourceforge.jeextension.styleeditor.text.ExtEndOfLineRule;
import jp.sourceforge.jeextension.styleeditor.text.ExtMultiLineRule;
import jp.sourceforge.jeextension.styleeditor.text.ExtSingleLineRule;
import jp.sourceforge.jeextension.styleeditor.text.ExtTextAttribute;
import jp.sourceforge.jeextension.styleeditor.text.ExtWordRule;
import jp.sourceforge.jeextension.styleeditor.text.IColorActivator;
import jp.sourceforge.jeextension.styleeditor.text.IExtRule;

import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.TextAttribute;
import org.eclipse.jface.text.rules.IPredicateRule;
import org.eclipse.jface.text.rules.IRule;
import org.eclipse.jface.text.rules.IToken;
import org.eclipse.jface.text.rules.RuleBasedPartitionScanner;
import org.eclipse.jface.text.rules.RuleBasedScanner;
import org.eclipse.jface.text.rules.Token;
import org.eclipse.jface.text.rules.WordRule;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.RGB;

/**
 * ScannerCreator
 */
public class EditorTools {
    
    /**
     * Change IDependsEntry with XMLNode.
     * @param entry IDependsEntry
     * @param node  XMLNode
     * @param ext   extension name. (ex. java, means *.java)
     */
    public static void changeDependsEntry(IDependsEntry entry, XMLNode node,
            String ext) {
        DependsEntry newEntry = new DependsEntry(node, ext);
        entry.setDepends(newEntry.getDepends());
        entry.setRGB(newEntry.getRGB());
        entry.setRule(newEntry.getRule());
        entry.setWords(newEntry.getWords());
    }
    
    /**
     * Change IPartitionEntry with XMLNode.
     * @param entry IPartitionEntry
     * @param node  XMLNode
     * @param ext   extension name. (ex. java, means *.java)
     */
    public static void changePartitionEntry(IPartitionEntry entry,
            XMLNode node, String ext) {
        IPartitionEntry newEntry = new PartitionEntry(node, ext);
        entry.setRGB(newEntry.getRGB());
        entry.setRule(newEntry.getRule());
        entry.setStart(newEntry.getStart());
        entry.setEnd(newEntry.getEnd());
        entry.setName(newEntry.getName());
    }
    
    /**
     * Create IExtRule form XMLNode.
     * @param node XMLNode.
     * @param ext   extension name. (ex. java, means *.java)
     * @return IExtRule
     */
    public static IExtRule createExtRule(XMLNode node, String ext) {
        IExtRule rule = null;
        String nodeName = node.getNodeName();
        RGB rgb = new RGB(0, 0, 0);
        rgb.red   = Integer.parseInt(node.getAttribute("red"));
        rgb.green = Integer.parseInt(node.getAttribute("green"));
        rgb.blue  = Integer.parseInt(node.getAttribute("blue"));
        
        if (nodeName.equals("partition")) {
            String start = node.getAttribute("start");
            String end   = node.getAttribute("end");
            IToken token = new Token(node.getAttribute("name"));
            rule = new ExtMultiLineRule(start, end, token);
        }
        if (nodeName.equals("multiLine")) {
            String start = node.getAttribute("start");
            String end   = node.getAttribute("end");
            IToken token = new Token(new ExtTextAttribute(rgb));
            rule = new ExtMultiLineRule(start, end, token);
        }
        if (nodeName.equals("singleLine")) {
            IToken token = new Token(new ExtTextAttribute(rgb));
            String start = node.getAttribute("start");
            String end   = node.getAttribute("end");
            rule = new ExtSingleLineRule(start, end, token);
        }
        if (nodeName.equals("endOfLine")) {
            IToken token = new Token(new ExtTextAttribute(rgb));
            rule = new ExtEndOfLineRule(node.getAttribute("start"), token);
        }
        if (nodeName.equals("keyword")) {
            
            // get depends partition entry.
            String partName = node.getParentNode().getAttribute("name");
            IPartitionEntry depends = getPartitionEntry(partName, ext, false);
            
            // create default token depends parent partition entry.
            IToken defaultToken = new Token(
                new ExtTextAttribute(depends.getRGB())
            );
            
            // create WordRule.
            IToken token = new Token(new ExtTextAttribute(rgb));
            ExtWordRule wRule =
                    new ExtWordRule(new WordDetector(), defaultToken);
            String word = node.getAttribute("word");
            String csensitive = node.getAttribute("case");
            if (csensitive == null) {
                csensitive = "true";
            }
            wRule.addWord(word, token, csensitive);
            rule = wRule;
        }
        return rule;
    }
    
    /**
     * Shortcut method for getting PartitionEntryMap.
     * @return PartitionEntryMap.
     */
    public static Map getPartitionEntryMap() {
        DefinedStyleEntry styleEntry =
                StyleEditorPlugin.getDefinedStyleEntry();
        return styleEntry.getPartitionEntryMap();
    }
    
    /**
     * Shortcut method for getting FontEntryMap.
     * @return  FontEntryMap
     */
    public static Map getFontEntryMap() {
        DefinedStyleEntry styleEntry =
                StyleEditorPlugin.getDefinedStyleEntry();
        return styleEntry.getFontEntryMap();
    }
    
    /**
     * Return Editor Font.
     * @param ext   file's extension.
     * @return  editor font.
     */
    public static Font getFont(String ext) {
        Map fontEntryMap = getFontEntryMap();
        FontEntry fontEntry = (FontEntry) fontEntryMap.get(ext);
        if (fontEntry != null) {
            return fontEntry.getFont();
        }
        else {
            fontEntry = (FontEntry) fontEntryMap.get("*");
            if (fontEntry != null) {
                return fontEntry.getFont();
            }
            else {
                return JFaceResources.getTextFont();
            }
        }
    }
    
    /**
     * Shortcut method for getting IPartitionEntry.
     * @param partName depends partition name.
     * @param ext   extension name. (ex. java, means *.java)
     * @param create if not exist, create one.
     * @return IPartitionEntry.
     */
    public static IPartitionEntry getPartitionEntry(String partName,
            String ext, boolean create) {
        DefinedStyleEntry styleEntry =
                StyleEditorPlugin.getDefinedStyleEntry();
        return styleEntry.getPartitionEntry(partName, ext, create);
    }
    
    /**
     * Shortcut method for getting PartitionEntryList.
     * @param   ext extension name. (ex. java, means *.java)
     * @param create if not exist, create one.
     * @return PartitionEntryList.
     */
    public static PartitionEntryList getPartitionEntryList(String ext,
            boolean create) {
        DefinedStyleEntry styleEntry =
                StyleEditorPlugin.getDefinedStyleEntry();
        return styleEntry.getPartitionEntryList(ext, create);
    }
    
    /** Color manager. */
    private ColorManager fColorManger;
    
    /** Default token. */
    private IToken fDefaultToken;
    
    /**
     * Initiate Tools using current editor's Configuration.
     * @param conf current editor's Configuration.
     */
    public EditorTools(EditorConfiguration conf) {
        super();
        this.fColorManger = conf.getColorManager();
        TextAttribute attr =
                new TextAttribute(fColorManger.getColor(new RGB(0, 0, 0)));
        this.fDefaultToken = new Token(attr);
    }
    
    /**
     * Create Partition Scanner.
     * @param ext       extension name. (ex. java, means *.java)
     * @return RuleBasedPartitionScanner.
     */
    public RuleBasedPartitionScanner createPartitionScanner(String ext) {
        RuleBasedPartitionScanner scanner = new RuleBasedPartitionScanner();
        scanner.setPredicateRules(createPredicateRule(ext));
        return scanner;
    }
    
    /**
     * Create BasicScanner for default content type.
     * @param partitionName target's partition name.
     * @param ext   extension name. (ex. java, means *.java)
     * @return RuleBasedScanner.
     */
    public RuleBasedScanner createScanner(String partitionName, String ext) {
        RuleBasedScanner scanner = new RuleBasedScanner();
        scanner.setRules(createRules(partitionName, ext));
        return scanner;
    }
    
    /**
     * Create ContentTypes includes IDocument.DEFAULT_CONTENT_TYPE.
     * @param ext       extension name. (ex. java, means *.java)
     * @return String[] contains ContentTypes.
     */
    public String[] getContentTypes(String ext) {
        List typeList = new ArrayList();
        typeList.add(IDocument.DEFAULT_CONTENT_TYPE);
        return createContentTypes(typeList, ext);
    }
    
    /**
     * Create ContentTypes exclude IDocument.DEFAULT_CONTENT_TYPE.
     * @param ext       extension name. (ex. java, means *.java)
     * @return String[] contains ContentTypes.
     */
    public String[] getContentTypesExcludeDefault(String ext) {
        return createContentTypes(new ArrayList(), ext);
    }
    
    /**
     * Return default token.
     * @return default token.
     */
    public IToken getDefaultToken() {
        return this.fDefaultToken;
    }
    
    /**
     * Return default token using depends partition entry.
     * @param entry partition entry.
     * @return default token.
     */
    public IToken getDefaultToken(IPartitionEntry entry) {
        TextAttribute attr =
                new TextAttribute(fColorManger.getColor(entry.getRGB()));
        IToken token = new Token(attr);
        return token;
    }
    
    /**
     * Activate color divice in rule for current editor.
     * @param ruleList List contains IExtRules.
     */
    private void activateColor(List ruleList) {
        for (int i = 0; i < ruleList.size(); i++) {
            IColorActivator rule = (IColorActivator) ruleList.get(i);
            rule.activateColor(fColorManger);
        }
    }
    
    /**
     * Append to ruleList.
     * @param entry     IEntry that defined entry.
     * @param ruleList  append's target.
     */
    private void appendRuleList(IEntry entry, List ruleList) {
        IRule rule = null;
        if (entry.getName().equals("keyword")) {
            // search WordRule
            for (int i = 0; i < ruleList.size(); i++) {
                Object obj = ruleList.get(i);
                if (obj instanceof WordRule) {
                    rule = (WordRule) obj;
                }
            }
            if (rule != null) {
                ExtWordRule wRule = (ExtWordRule) entry.getRule();
                ((ExtWordRule) rule).addWordRuleMap(wRule.getWordRuleMap());
                return;
            }
            else {
                rule = entry.getRule();
                
                // once clear all.
                ExtWordRule wRule = (ExtWordRule) entry.getRule();
                Map currentMap = new HashMap(wRule.getWordRuleMap());
                wRule.getWordRuleMap().clear();
                
                // append current entry.
                wRule.getWordRuleMap().putAll(currentMap);
            }
        }
        if (entry.getName().equals("multiLine")) {
            rule = entry.getRule();
        }
        if (entry.getName().equals("endOfLine")) {
            rule = entry.getRule();
        }
        if (entry.getName().equals("singleLine")) {
            rule = entry.getRule();
        }
        if (entry instanceof IPartitionEntry) {
            if (entry.getName().equals(IDocument.DEFAULT_CONTENT_TYPE)) {
                return;
            }
            rule = entry.getRule();
        }
        ruleList.add(rule);
    }
    
    /**
     * Create ContentTypes using List.
     * @param typeList  List contains ContentTypes.
     * @param ext       extension name. (ex. java, means *.java)
     * @return String[] contains ContentTypes.
     */
    private String[] createContentTypes(List typeList, String ext) {
        PartitionEntryList entryList = getPartitionEntryList(ext, false);
        for (int i = 0; i < entryList.size(); i++) {
            IPartitionEntry entry = entryList.getPartitionEntry(i);
            typeList.add(entry.getName());
        }
        String[] types = new String[typeList.size()];
        return (String[]) typeList.toArray(types);
    }
    
    /**
     * Create IPredicateRule for Parition Scanner.
     * @param ext       extension name. (ex. java, means *.java)
     * @return created IPredicateRules.
     */
    private IPredicateRule[] createPredicateRule(String ext) {
        PartitionEntryList entryList = getPartitionEntryList(ext, false);
        
        List ruleList = new ArrayList();
        for (int i = 0; i < entryList.size(); i++) {
            IPartitionEntry entry = entryList.getPartitionEntry(i);
            appendRuleList(entry, ruleList);
        }
        activateColor(ruleList);
        
        IPredicateRule[] rules = new IPredicateRule[ruleList.size()];
        return (IPredicateRule[]) ruleList.toArray(rules);
    }
    
    /**
     * Create IRule for Basic Scanner.
     * @param partName target partition name.
     * @param ext   extension name. (ex. java, means *.java)
     * @return created IRules.
     */
    private IRule[] createRules(String partName, String ext) {
        IPartitionEntry partList = getPartitionEntry(partName, ext, false);
        DependsEntryList entryList = partList.getDependsEntryList();
        
        List ruleList = new ArrayList();
        for (int i = 0; i < entryList.size(); i++) {
            IDependsEntry entry = entryList.getDependsEntry(i);
            String partition = entry.getDepends().getName();
            if (partition.equals(partName)) {
                appendRuleList(entry, ruleList);
            }
        }
        activateColor(ruleList);
        
        IRule[] rules = new IRule[ruleList.size()];
        return (IRule[]) ruleList.toArray(rules);
    }
}
