package jp.jsuite.snmp;

import org.apache.log4j.Logger;
import org.snmp4j.CommunityTarget;
import org.snmp4j.PDU;
import org.snmp4j.PDUv1;
import org.snmp4j.Snmp;
import org.snmp4j.mp.SnmpConstants;
import org.snmp4j.smi.Address;
import org.snmp4j.smi.GenericAddress;
import org.snmp4j.smi.Integer32;
import org.snmp4j.smi.IpAddress;
import org.snmp4j.smi.OID;
import org.snmp4j.smi.OctetString;
import org.snmp4j.smi.VariableBinding;
import org.snmp4j.transport.DefaultUdpTransportMapping;

import com.google.inject.Inject;

/**
 * SNMP トラップ送信.
 * <pre>
 * コンストラクタで送信先のSNMP宛先 アドレスを指定する。
 * 特別なアノテーションにより、SNMP宛先アドレスの、Google guice インジェクトを
 * 次のように約束されている。
 *     @Inject public SnmpTrapSender(@SnmpIPaddress String sendIP)
 * これにより、SnmpTrapSenderFactory で SnmpTrapSenderインスタンスを生成するか、
 * new 演算子を使うかである。
 *
 * </pre>
 */
public final class SnmpTrapSender{
   String sendIP;
   String targetip;
   int trapNo;
   int trapType;
   /**
    * コンストラクタ
    * @param sendIP SNMP宛先アドレス
    */
   @Inject
   public SnmpTrapSender(@SnmpIPaddress String sendIP){
      this.sendIP = sendIP;
   }

   /**
    * トラップ送信.
    * void send(targetIP,trapno,type,1); の呼び出しと同じ効果があります。
    * @param targetIP ターゲットＩＰ
    * @param trapno 任意トラップ番号
    * @param type org.snmp4j.PDUv1.LINKDOWN か、org.snmp4j.PDUv1.LINKUP を指定します。
    */
   public void send(String targetIP,int trapno,int type){
      this.send(targetIP,trapno,type,1);
   }

   /**
    * トラップ送信.
    * @param targetIP ターゲットＩＰ
    * @param trapno 任意トラップ番号
    * @param type org.snmp4j.PDUv1.LINKDOWN か、org.snmp4j.PDUv1.LINKUP を指定します。
    * @param retry UDP通信の為、この回数分送信します。
    */
   public void send(String targetIP,int trapno,int type,int retry){
      this.targetip = targetIP;
      this.trapNo = trapno;
      this.trapType = type;     // PDUv1.LINKDOWN or PDUv1.LINKUP
      int loops = retry > 1 ? retry : 1;
      for(int i=0;i < loops;i++){
         Thread th = new Thread(new _SendThread());
         th.start();
         try{
         th.join();
         }catch(InterruptedException e){
         }
      }
   }

   class _SendThread implements Runnable{
      private Logger logger;
      _SendThread(){
         this.logger = Logger.getLogger(this.getClass());
      }
      /*
       * @see java.lang.Runnable#run()
       */
      @Override
      public void run(){
         Address targetAddress = GenericAddress.parse("udp:"+SnmpTrapSender.this.sendIP+"/162");
         CommunityTarget target = new CommunityTarget();
         target.setCommunity(new OctetString("public"));
         target.setAddress(targetAddress);
         target.setVersion(SnmpConstants.version1);

         PDUv1 pdu = new PDUv1();
         pdu.setType(PDU.V1TRAP);

         VariableBinding vb = new VariableBinding();
         vb.setOid(new OID("1.3.6.1.2.1.2.2.1.1"));
         vb.setVariable(new Integer32(SnmpTrapSender.this.trapNo));
         pdu.add(vb);
         pdu.setEnterprise(new OID("1.3.6.1.2.1.11"));

         pdu.setAgentAddress(new IpAddress(SnmpTrapSender.this.targetip));
         pdu.setGenericTrap(SnmpTrapSender.this.trapType);

         try{
         DefaultUdpTransportMapping udpTransportMap = new DefaultUdpTransportMapping();
         Snmp snmp = new Snmp(udpTransportMap);
         snmp.send(pdu,target);
         snmp.close();
         this.logger.debug("## Trap send target = "+target.toString());
         this.logger.debug("## Trap send PDUv1  = "+pdu.toString());
         }catch(Exception e){
            this.logger.error("## SnmpTrapSender._SendThread Exception : "+e.getMessage(),e);
         }
      }
   }
}
