package jp.jsuite.snmp;

import java.io.IOException;

import org.apache.log4j.Logger;
import org.snmp4j.CommandResponder;
import org.snmp4j.MessageDispatcherImpl;
import org.snmp4j.Snmp;
import org.snmp4j.mp.MPv1;
import org.snmp4j.mp.MPv2c;
import org.snmp4j.mp.MPv3;
import org.snmp4j.security.SecurityModels;
import org.snmp4j.security.SecurityProtocols;
import org.snmp4j.security.USM;
import org.snmp4j.smi.GenericAddress;
import org.snmp4j.smi.OctetString;
import org.snmp4j.smi.UdpAddress;
import org.snmp4j.transport.DefaultUdpTransportMapping;
import org.snmp4j.util.MultiThreadedMessageDispatcher;
import org.snmp4j.util.ThreadPool;

import com.google.inject.Inject;

/**
 * SNMP トラップ受信.
 * <pre>
 * コンストラクタでSNMP受信するＩＰ，受信スレッド数、CommandResponderインスタンスを指定する。
 * 特別なアノテーションにより、SNMP受信するＩＰ，受信スレッド数の、
 * Google guice インジェクトを次のように約束されている。
 *   @Inject public SnmpTrapReceiver(@SnmpIPaddress       String ip
 *                                  ,@SnmpRecvThreadnum   int threadNum
 *                                  ,CommandResponder     commandResponder)
 * よって、任意のModule で、annotatedWith で、＠SnmpIPaddress , ＠SnmpRecvThreadnum に対して
 * インジェクトするものを作成するか、または、SnmpTrapReceiveModule を使用するかである。
 *
 * 生成したSnmpTrapReceiver は、受信を終了させるまでインスタンスが有効になるようにすること。
 *
 * </pre>
 */
public final class SnmpTrapReceiver{
   private String recvIP;
   private int threadNum = 2;
   private CommandResponder commandResponder;
   private Snmp snmp = null;
   private ThreadPool threadPool;
   private Logger logger;

   /**
    * コンストラクタ
    * @param ip トラップ受信するＩＰアドレス
    * @param threadNum 受信スレッドの数
    * @param commandResponder CommandResponder実装インスタンス
    */
   @Inject
   public SnmpTrapReceiver(@SnmpIPaddress String ip,@SnmpRecvThreadnum int threadNum,CommandResponder commandResponder){
      this.recvIP = ip;
      this.threadNum = threadNum;
      this.commandResponder = commandResponder;
      this.logger = Logger.getLogger(this.getClass());
   }
   /**
    * 受信開始.
    * CommandResponder実装インスタンスがスレッドとして待機
    * @throws IOException
    */
   public void start() throws IOException{
      // port 162 をバインド
      this.threadPool = ThreadPool.create("SnmpTrapReceiver",this.threadNum);
      MultiThreadedMessageDispatcher dispatcher = new MultiThreadedMessageDispatcher(this.threadPool,new MessageDispatcherImpl());
      org.snmp4j.smi.Address listenAddress = GenericAddress.parse(System.getProperty("snmp4j.listenAddress","udp:"+this.recvIP+"/162"));
      this.snmp = new Snmp(dispatcher,new DefaultUdpTransportMapping((UdpAddress)listenAddress));
      this.snmp.getMessageDispatcher().addMessageProcessingModel(new MPv1());
      this.snmp.getMessageDispatcher().addMessageProcessingModel(new MPv2c());
      this.snmp.getMessageDispatcher().addMessageProcessingModel(new MPv3());
      USM usm = new USM(SecurityProtocols.getInstance(),
      new OctetString(MPv3.createLocalEngineID()),0);
      SecurityModels.getInstance().addSecurityModel(usm);
      this.snmp.addCommandResponder(this.commandResponder);
      this.snmp.listen();
      this.logger.debug("## SnmpTrapReceiver  "+this.recvIP+"/162  listen");
   }
   /**
    * 受信終了
    * 受信スレッド終了、ポートを閉じる。
    */
   public void end(){
      this.threadPool.cancel();
      try{
      this.snmp.close();
      }catch(IOException e){
         this.logger.error("## IOException : "+e.getMessage(),e);
      }finally{
         this.logger.debug("## SnmpTrapReceiver  close");
      }
   }

}
