/*
  SvgEditor.js

  iSlideMaker
    http://sourceforge.jp/projects/islidemaker/simple/

  Copyright(c) 2010, Isao Hara, isao-hara@users.sourceforge.jp

  All rights reserved. This program is made available under the terms of the
  Eclipse Public License v1.0 which accompanies this distribution, and is
  available at http://www.eclipse.org/legal/epl-v10.html

  Contributors:  Isao Hara.
 */
var systemDB;
var currentMode;
var editarea_w=80;
var editarea_h=10;
var MainTitle="Simple SVG Editor";

var preview=null;
var targetItem=null;
var selectedItems=new Array();
var sx=0;
var sy=0;
var ex=0;
var ey=0;
var modeSVG=null;
var iSlideMgr='iSlideManager.php';
var MgrPath="";
var nextId=1;

var lineEdit=null;

////// SVG
var svg_ns = 'http://www.w3.org/2000/svg';
var svg_x=10;
var svg_y=10;
var svg_width = 100;
var svg_height = 50;
var svg_rx = 50;
var svg_ry = 30;
var svg_lw = 1;
var svg_wo = null;
var svg_ho = null;
var svg_ro = null;
var svg_rxo = null;
var svg_ryo = null;
var svg_fsize = null;
var svg_scale_dir = null;

var svg_top = null;
var svg_select = null;
var svg_defs = null;

var svg_font_size = 24;
var svg_font_family = 'Helvetica';
var svg_color = '#000000';
var svg_fill_color = '#ffffff';
var svg_line_width = 1;

var editingTextObj = null;
var svg_rotate_locked = false;
var svg_scale_locked = false;

var dupItems;
var dupX;
var dupY;
var is_newPath;

var firstTouch = new Date();

var lineW = new Array(0.25,0.5,0.75,1,1.5,2.25,3,4.5,6);
var StrokeDash = new Array("", "2", "6 2", "6 2 1 2", "8 2", "8 2 1 2", "8 2 1 2 1 2" );

///// Color
var colors=new Array('none', '#ffffff',
  '#000000', '#3f3f3f', '#7f7f7f', '#bfbfbf',
  '#ff0000', '#ff7f00', '#ffff00', '#7fff00',
  '#00ff00', '#00ff7f', '#00ffff', '#007fff',
  '#0000ff', '#7f00ff', '#ff00ff', '#ff007f',
  '#7f0000', '#7f3f00', '#7f7f00', '#3f7f00',
  '#007f00', '#007f3f', '#007f7f', '#003f7f',
  '#00007f', '#3f007f', '#7f007f', '#7f003f',
  '#7f0000', '#7f3f00', '#7f7f00', '#3f7f00',
  '#ffaaaa', '#ffd4aa', '#ffffaa', '#d4ffaa',
  '#aaffaa', '#aaffd4', '#aaffff', '#aad4ff',
  '#aaaaff', '#d4aaff', '#ffaaff', '#ffaad4'
);

//// Initialize
function initEditor(name, dispname, size){
  systemDB = initDB(name, dispname, size);
  fileSelector();
  preview=document.getElementById('preview');
  if(preview){
    preview.addEventListener("touchstart", onTouchStart,false);
    preview.addEventListener("touchmove", onTouchMove, false);
    preview.addEventListener("touchend", onTouchEnd, false);
    preview.addEventListener("gesturestart", onGestureStart,false);
    preview.addEventListener("gesturechange", onGestureChange, false);
    preview.addEventListener("gestureend", onGestureEnd, false);
  }
  restoreValues();
}

//// localStorage
function restoreValues(){
  if(typeof(localStorage) == 'undefined'){ alert('local storage not suported'); }
  MgrPath = localStorage.getItem('MgrPath') ? localStorage.getItem('MgrPath')  : "";
  window.onbeforeuload=function(){ return saveChanges(); } 
}

function clearAll(){
  localsStorage.clear();
  restoreValudes();
}

function saveChanges(){
  localStorage.setItem('MgrPath', MgrPath);
}

// initialize a database
function initDB(name, dispname, size) {
  try {
    if (!window.openDatabase) {
        alert('not supported');
    } else {
        var version = '1.0';
        var myDB = openDatabase(name, version, dispname, size);
    }
  }catch(e){
    if (e == INVALID_STATE_ERR){ alert("Invalid database version."); }else{ alert("Unknown error "+e+"."); }
    return null;
  }
  createTables(myDB);
  return myDB;
}

function createTables(db) {
  db.transaction(
    function (transaction) {
      transaction.executeSql('CREATE TABLE IF NOT EXISTS files(id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT, name TEXT NOT NULL, filedata_id INTEGER NOT NULL, deleted INTEGER NOT NULL DEFAULT 0);', [], nullDataHandler, killTransaction);
      transaction.executeSql('CREATE TABLE IF NOT EXISTS filedata(id INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT, datablob BLOB NOT NULL DEFAULT "");', [], nullDataHandler, errorHandler);
    }
  );
}

function dropTables(db) {
  db.transaction(
    function (transaction) {
      transaction.executeSql('DROP TABLE files;');
      transaction.executeSql('DROP TABLE filedata;');
    }
  );
}

/// Create New File
function reallyCreateNewFileAction(name, db){
  var myDB = db;
  if(!db) myDB = systemDB;

  myDB.transaction(
    function (transaction) {
      var myfunc = new Function("transaction","results","transaction.executeSql('INSERT INTO files (name, filedata_id) VALUES (?, ?);', [ '"+name+"', results.insertId], nullDataHandler, killTransaction);");

       transaction.executeSql('INSERT INTO filedata (datablob) VALUES ("");', [], myfunc, errorHandler);
    }
  );
  fileSelector();
}

function createNewFileAction(db){
  var myDB = db;
  if(!db) myDB = systemDB;

  var name = document.getElementById('createFilename').value
  if(name == "") {
    if (confirm('Filename is required, try agein?')) { createNewFile(); }else{ fileSelector(); }
    return;
  }

  myDB.transaction(
    new Function("transaction", "transaction.executeSql('SELECT id,name from files where name=?;',['"+name+"'],"+
        "function (transaction, results) {"+
           "if(results.rows.length == 0){"+
               "reallyCreateNewFileAction('"+name+"');"+
           "}else{"+
             "if (confirm(results.rows.item(0)['name']+' already exist, try agein?')) {"+
               "createNewFile();"+
             "}else{"+
               "fileSelector();"+
             "}"+
           "}"+
        "}, errorHandler);")
  );

}

//// Delete File
function deleteUpdateResults(transaction, results){
  if (results.rowsAffected) {
    fileSelector();
  }
}

function reallyDelete(id, db){
  var myDB = db;
  if(!db) myDB = systemDB;

  myDB.transaction(
    new Function("transaction",
      "transaction.executeSql('DELETE FROM  files where id=?;', [ "+id+" ], null, errorHandler);"
      +"transaction.executeSql('DELETE FROM filedata where id=?;', [ "+id+" ], deleteUpdateResults, errorHandler);")
  );
}

function deleteFile(name, db){
  var myDB = db;
  if(!db) myDB = systemDB;

  myDB.transaction(
    new Function("transaction", "transaction.executeSql('SELECT id,name from files where name=?;',['"+name+"'],"+
        "function (transaction, results) {"+
           "if (confirm('Really delete '+results.rows.item(0)['name']+'?')) {"+
               "reallyDelete(results.rows.item(0)['id']);"+
           "}"+
        "}, errorHandler);")
  );
}

function reallyDeleteAll(db){
  var myDB = db;
  if(!db) myDB = systemDB;

  myDB.transaction(
    new Function("transaction",
      "transaction.executeSql('DELETE FROM  files;',[],null, errorHandler);"
      +"transaction.executeSql('DELETE FROM filedata;',[],deleteUpdateResults, errorHandler);")
  );
}

function deleteAllFile(db){
  if (confirm('Really delete all file?')) {
     reallyDeleteAll(db);
  }
}

///// Rename
function reallyRenameFileAction(id, newname, db){
  var myDB = db;
  if(!db) myDB = systemDB;

  myDB.transaction(
    function (transaction) {
      transaction.executeSql('UPDATE files set name=? where id=?;', [newname,id ], null, errorHandler);
    }
  );
  fileSelector();
}

function renameFileAction(db){
  var myDB = db;
  if(!db) myDB = systemDB;

  var new_name = document.getElementById('newFilename').value
  var id = document.getElementById('fileId').value

  if(new_name == "") {
    alert('Filename is required.');
    fileSelector();
    return;
  }

  myDB.transaction(
    new Function("transaction",
      "transaction.executeSql('SELECT id,name from files where name=?;',['"+new_name+"'],"+
        "function (transaction, results) {"+
           "if(results.rows.length == 0){"+
               "reallyRenameFileAction('"+id+"','"+new_name+"');"+
           "}else{"+
             "alert(results.rows.item(0)['name']+' already exist');"+
             "fileSelector();"+
           "}"+
        "}, errorHandler);")
  );

}

/// Save File
function saveFile(db){
  var myDB = db;
  if(!db) myDB = systemDB;

  var editarea = document.getElementById('editarea');
  var contents = "";
  contents = editarea.value;

  myDB.transaction(
    function (transaction) {
      var datadiv = document.getElementById('tempdata');
      var filedata_id = datadiv.getAttribute('lfdataid');

      transaction.executeSql("UPDATE filedata set datablob=? where id=?;",
        [ contents, filedata_id ], nullDataHandler, errorHandler);

      alert('Saved.');
    }
  );
}

function saveData(db){
  var myDB = db;
  if(!db) myDB = systemDB;

  var contents = "";
  contents = getSVGContent();

  myDB.transaction(
    function (transaction) {
      var datadiv = document.getElementById('tempdata');
      var filedata_id = datadiv.getAttribute('lfdataid');

      transaction.executeSql("UPDATE filedata set datablob=? where id=?;",
        [ contents, filedata_id ], nullDataHandler, errorHandler);

      alert('Saved.');
    }
  );
}

function saveContent(contents, db){
  var myDB = db;
  if(!db) myDB = systemDB;

  if(!contents) {
    alert("Invalid content");
    return;
  }

  myDB.transaction(
    function (transaction) {
      var datadiv = document.getElementById('tempdata');
      var filedata_id = datadiv.getAttribute('lfdataid');

      transaction.executeSql("UPDATE filedata set datablob=? where id=?;",
        [ contents, filedata_id ], nullDataHandler, errorHandler);

      alert('Saved.');
    }
  );
}

function reallySaveNewFileAction(fname, db){
  var myDB = db;
  if(!db) myDB = systemDB;

  var datadiv = document.getElementById('tempdata');
  content = datadiv.textContent;

  myDB.transaction(
    function (transaction) {
      var myfunc = new Function("transaction", "results", "transaction.executeSql('INSERT INTO files (name, filedata_id) VALUES (?, ?);', [ '"+fname+"', results.insertId], nullDataHandler, killTransaction);");

       transaction.executeSql('INSERT INTO filedata (datablob) VALUES (?);', [content], myfunc, errorHandler);
    }
  );
  datadiv.textContent="";
  alert("download to "+fname);
}

function saveContentFilename(id, db){
  var datadiv = document.getElementById('tempdata');
  datadiv.setAttribute('lfdataid', id);
  saveContent(datadiv.innerHTML, db);
  datadiv.textContent="";
}

function saveContentWithFilename(fname, contents, db){
  var myDB = db;
  if(!db) myDB = systemDB;
  if(!contents) { alert("Invalid content"); return; }

  var datadiv = document.getElementById('tempdata');
  datadiv.setAttribute('lfname', fname);
  datadiv.textContent = contents;

  myDB.transaction(
    new Function("transaction", "transaction.executeSql('SELECT id,name from files where name=?;',['"+fname+"'],"+
        "function (transaction, results) {"+
           "if(results.rows.length == 0){"+
               "reallySaveNewFileAction('"+fname+"');"+
           "}else{"+
             "if (confirm(results.rows.item(0)['name']+' already exist, Overwrite?')) {"+
               "saveContentFilename(results.rows.item(0)['id']);"+
             "}else{"+
               "fileSelector();"+
             "}"+
           "}"+
        "}, errorHandler);")
  );

}

function showFileData(transaction, results){
  var data = results.rows.item(0);
  var filename = data['name'];
  var filedata = data['datablob'];

  var datadiv = document.getElementById('tempdata');
  datadiv.setAttribute('lfdataid', parseInt(data['filedata_id']));
  document.title=filename;

  previewFile(filedata);

  updateShowMenu();
  setMode('Preview');
}

function showFile(name){ getFile(name, 'showFileData'); }

function getFile(name, func, db){
  var myDB = db;
  if(!db) myDB = systemDB;

  var datadiv = document.getElementById('tempdata');
  if(datadiv) datadiv.setAttribute('lfname', name);
 
  myDB.transaction(
    function (transaction) {
      transaction.executeSql('SELECT * from files, filedata where files.name=? and files.filedata_id = filedata.id;', [name], eval(func), errorHandler);
    }
  );
}
 
//  Error Handlers 
function killTransaction(transaction, error){ return true; }
 
function errorHandler(transaction, error){
  alert('Oops.  Error was '+error.message+' (Code '+error.code+')');
 
  var we_think_this_error_is_fatal = true;
  if (we_think_this_error_is_fatal) return true;
  return false;
}
 
function nullDataHandler(transaction, results){ }
 
/////////
function fileSelector(db) {
  var myDB = db;
  if(!db) myDB = systemDB;
 
  myDB.transaction(
    function (transaction) {
      transaction.executeSql("SELECT * from files where deleted=0;", [ ], 
         function (transaction, results) {
            var filelist = '';
            var menuDiv = document.getElementById('menuDiv');
            if(menuDiv){
              for(var i=0; i<results.rows.length; i++) {
                var row = results.rows.item(i);
                filelist = filelist + fileEntry(row);
              }
              if (filelist == "") {
                filelist = "No files.<br />\n";
              } else {
                filelist = "<center><table class='filetable'>"+filelist+"</table></center>";
              }
              menuDiv.innerHTML="<H1 class='title'>"+MainTitle+"</H1>"+createMenuBar()+filelist;
            }
          }, errorHandler);
    }
  );
  setMode('List');
}

function fileSelectorOnServer(val) {
  var filelist = '';
  var menuDiv = document.getElementById('menuDiv');
  var results = val.split(',');
  
  if(menuDiv){
    for(var i=0; i<results.length; i++) {
      var row = results[i];
      filelist = filelist + fileEntryOnServer(row);
    }
    if (filelist == "") {
     filelist = "No files.<br />\n";
    } else {
     filelist = "<center><table class='filetable'>"+filelist+"</table></center>";
    }
    var Menu = "<button onClick=\"fileSelector();\">Local Storage</button>";
    Menu += "<button onClick=\"getRemoteFileList();\">List on Server</button>";
     menuDiv.innerHTML="<H1 class='title'>"+MainTitle+"</H1>"+Menu+filelist;
   }
  setMode('List');
}

function fileEntryOnServer(name){
  name = name.split('.')[0];
  var res =  "<tr class='filerow'>";
  res += "<td class='filenamecell'  onClick=\"getRemoteFile('"+name+"');\">"+name+"</td>";
  res += "<td class='filelinkcell'>";
  res += "<button class='green' onClick=\"showRemoteFile('"+name+"');\">View</button>";
  res += "<button onClick=\"downloadFile('"+name+"');\">Download to local storage</button>";
  res += "<button onClick=\"downloadToFile('"+name+"');\">Download as SVG File</button>";
  res += "</td></tr>\n";

  return res;
}

function fileEntry(row){
  var name = row['name'];
  var files_id = row['id'];

  var res =  "<tr class='filerow'>";
  res += "<td class='filenamecell'  onClick=\"showFile('"+name+"');\">"+name+"</td>";
  res += "<td class='filelinkcell'>";
  res += "<button class='blue' onClick=\"editFile('"+name+"');\">&nbsp;Edit&nbsp;</button> &nbsp;";
  res += "<button class='green' onClick=\"renameFile('"+name+"',"+files_id+");\">&nbsp;Rename&nbsp;</button> &nbsp;";
  res += "<button  onClick=\"uploadFile('"+name+"');\">Upload</button>";
  res += "<button class='red' onClick=\"deleteFile('"+name+"');\">Delete</button>";
  res += "</td></tr>\n";

  return res;
}

function createMenuBar(){
  var menu = "<ul id='menubar'>";
  menu += "<li><button onClick='createNewFile()'>Create New File</button></li>";
  menu += "<li><button onClick='deleteAllFile()'>Delete All</button></li>";
  menu += "<li><button onClick='getRemoteFileList()'>File List on a Server</button></li>";
  menu += "<li><button onClick='configServer()'>Server Configuration</button></li>";
  menu += "</ul><p class='cls'>";
  return menu;
}

//// Config
function setConfig(){
  var ele = document.getElementById('ServerURL');
  MgrPath=ele.value;
  saveChanges();
  alert("Done");
}

function configServer(){
  var popupDiv = document.getElementById('popup');
  var string = "";

  string += "<H1 class='title'>Server Configration</H1>\n";
  string += "<div class=\"input_form\">\n";
  string += "Filename:<input id='ServerURL' name='url' value=\""+MgrPath+"\" size=\"80\"/><br>\n";
  string += "<button onClick=\"setConfig();hideItemById('popup'); \">Done</button>\n";
  string += "<button onClick=\"hideItemById('popup'); \">Cancel</button>\n";
  string += "</div>\n";
 
  popupDiv.innerHTML=string;
  popupDiv.style.display='block';
}

////
function createNewFile(){
  var popupDiv = document.getElementById('popup');
  var string = "";

  string += "<H1 class='title'>Create New File</H1>\n";
  string += "<div class=\"input_form\">\n";
  string += "Filename:<input id='createFilename' name='name' value=\"\" />\n";
  string += "<button onClick=\"createNewFileAction();hideItemById('popup'); \">Create</button>\n";
  string += "<button onClick=\"hideItemById('popup'); \">Cancel</button>\n";
  string += "</div>\n";
 
  popupDiv.innerHTML=string;
  popupDiv.style.display='block';
}

/////
function renameFile(name, id){
  var popupDiv = document.getElementById('popup');
  var string = "";

  string += "<H1 class='title'>Rename File</H1>\n";
  string += "<div class='input_form'>\n";
  string += "Old Filename: "+name+"<br>";
  string += "New Filename:<input id='newFilename' name='newname' value=\"\" />\n";
  string += "<input type='hidden' id='fileId' value=\""+id+"\" /><br>\n";
  string += "<button onClick=\"renameFileAction();hideItemById('popup');\">Rename</button>\n";
  string += "<button onClick=\"hideItemById('popup'); \">Cancel</button>\n";
  string += "</div>\n";
 
  popupDiv.innerHTML=string;
  popupDiv.style.display='block';
}

/// EditMenu
function insertStr(str, len){
  var editarea = document.getElementById('editarea');
  if(len < 0) len = str.length;

  if (editarea){
    if(str=='DQ') str='"';
    var strs = editarea.value;
    var cPos = editarea.selectionStart;
    var tmp = strs.substr(0,cPos);
    editarea.value = tmp +str+strs.substr(cPos, strs.length);
    cPos += len;
    editarea.setSelectionRange(cPos, cPos);
    editarea.focus();
  }
}

function insertTag(tag){
  var str;
  var len = -1;
  if (tag == "p"){
    str = "<p>  </p>";
    len = 4;
  }else if (tag == "ul"){
    str = "<ul class=\"  \"> \n\n</ul>";
    len = 17;
  }else if (tag == "li"){
    str = "<li>   </li>";
    len = 5;
  }else if (tag == "href"){
    str = "xlink:href=\"\"";
    len = 12;
  }else if (tag == "EQ"){
    str = "=\"\"";
    len = 2;
  }else if (tag == "svg:text"){
    str = "<svg:text x=\"100\" y=\"100\" fill=\"black\" font-size=\"12\"> </svg:text>";
    len = 58;
  }else if (tag == "svg:rect"){
    str = "<svg:rect x=\"10\" y=\"10\" width=\"100\" height=\"100\" fill=\"white\" stroke=\"black\" />";
    len = 13;
  }else if (tag == "svg:circle"){
    str = "<svg:circle cx=\"10\" cy=\"100\" r=\"100\" fill=\"white\" stroke=\"black\" strokc-width=\"1\"/>";
    len = 17;
  }
  insertStr(str, len);
}

function editMenuBar()
{
    var str = "";
    str += "<button onClick=\"insertStr('/',1);\">/</button>\n";
    str += "<button onClick=\"insertStr('DQ',1);\">\"</button>\n";
    str += "<button onClick=\"insertTag('EQ',1);\">=\"\"</button>\n";
    str += "<button onClick=\"insertTag('p');\">p</button>\n";
    str += "<button onClick=\"insertTag('href');\">href</button>\n";

    str += "<button onClick=\"insertTag('svg:text');\">TEXT</button>\n";
    str += "<button onClick=\"insertTag('svg:rect');\">rect</button>\n";
    str += "<button onClick=\"insertTag('svg:circle');\">circle</button>\n";

    str += "<button onClick=\"chEditareaHeight();\">...</button>\n";

    return str;
}

function updateEditMenu(){
  var menuDiv = document.getElementById('menuDiv');
  menuDiv.innerHTML="<h1 class='title'>" +document.title+"</h1>\n";
  menuDiv.innerHTML+= "<button onClick=\"saveFile();fileSelector();\"> Save </button>\n";
  menuDiv.innerHTML+= "<button onClick=\"previewData();\"> Preview </button>\n";
  menuDiv.innerHTML+= "<button onClick=\"fileSelector();\"> File List </button>\n";

  menuDiv.style.display='block';
}

//// Editor
function editFileData(transaction, results){
  var editDiv = document.getElementById('editDiv');
  var datadiv = document.getElementById('tempdata');
 
  if( results.rows.length == 0) return;
  var data = results.rows.item(0);
  var filename = data['name'];
  var filedata = data['datablob'];
  datadiv.setAttribute('lfdataid', parseInt(data['filedata_id']));

  var editcontent="<textarea id=\"editarea\" rows=\""+editarea_h+"\" cols=\""+editarea_w+"\">"+filedata+"</textarea>\n";
 
  document.title="EditFile: "+filename;
  updateEditMenu();
  editDiv.innerHTML = editMenuBar() +"<br>"+ editcontent;

  setMode('Edit');
}

function editFile(name){
   if(currentMode == 'Preview'){
     var data = getSVGContent();
     var editcontent="<textarea id=\"editarea\" rows=\""+editarea_h+"\" cols=\""+editarea_w+"\">"+data+"</textarea>\n";
     updateEditMenu();
     editDiv.innerHTML = editMenuBar() +"<br>"+ editcontent;

     setMode('Edit');
   }else{
     getFile(name, 'editFileData');
   }
}

function editCurrentFile(){
  var datadiv = document.getElementById('tempdata');
  var name = datadiv.getAttribute('lfname');

  editFile(name);
}


//  Show File (GUI Editor)
function updateShowMenu(){
  var menuDiv = document.getElementById('menuDiv');
  menuDiv.innerHTML="<h1 class='title'>" +document.title+"</h1>\n";
  var menu_str = "<img src=\"images/menu.png\" usemap=\"#topmenu\" />\n";
  menu_str+= "<map name=\"topmenu\">";
  menu_str+= "<area shape=\"rect\" coords=\"0,0,30,25\"  onClick=\"fileSelector();\">";
  menu_str+= "<area shape=\"rect\" coords=\"30,0,60,25\" onClick=\"saveData();\">";
  menu_str+= "<area shape=\"rect\" coords=\"60,0,90,25\" onClick=\"showSVGSource();\">";
  menu_str+= "</map>";

  menu_str+= updateSVGObjMenu();
  menuDiv.innerHTML+= menu_str;
  updateStrokeWidth();
  updatePathTypeMenu();
  updateArrowMenu();
  menuDiv.style.display='block';
  updateToolBar();
}

function showColorPalette(val){
  var palette = document.getElementById('color-palette');
  var ele;
  var current = palette.getAttribute("targetType");

  if(palette.style.display=='block' && current == val){
    palette.style.display='none';
    return;
  }

  if(val == 'fill'){
   ele = document.getElementById('toolFill');
  }else{
   ele = document.getElementById('toolStroke');
  }
  
  var pos= ele.offsetTop + 110;
  palette.style.top = pos +"px";
  palette.style.display='block';
  palette.setAttribute("targetType", val);

}

function selectToolBar(idx){
  clearSelectedItems();
  var ele=document.getElementById('tool_select');
  var pos = idx *25;
  ele.style.top= pos+'px';
}


function updateToolBar(){
  var toolbar = document.getElementById('toolBar');
  var str = "";
  if(!toolbar.innerHTML){
    str += "<li><img src=\"images/tools.png\" usemap=\"#toolbar\" />\n";
    str += "<img id=\"tool_select\" src=\"images/select.png\" style=\"position:absolute;top:0;left:10px;\" />\n";
    str += "<map name=\"toolbar\">";
    str += "<area shape=\"rect\" coords=\"0,0,30,25\"  onClick=\"setSVGMode('selector');\">";
    str += "<area shape=\"rect\" coords=\"0,25,30,50\" onClick=\"setSVGMode('newPath');\">";
    str += "<area shape=\"rect\" coords=\"0,50,30,75\" onClick=\"setSVGMode('newLine');\">";
    str += "<area shape=\"rect\" coords=\"0,75,30,100\" onClick=\"setSVGMode('text');\">";
    str += "<area shape=\"rect\" coords=\"0,100,30,125\" onClick=\"setSVGMode('rect');\">";
    str += "<area shape=\"rect\" coords=\"0,125,30,150\" onClick=\"setSVGMode('circle');\">";
    str += "<area shape=\"rect\" coords=\"0,150,30,175\" onClick=\"setSVGMode('ellipse');\">";
    str += "<area shape=\"rect\" coords=\"0,175,30,200\" onClick=\"setSVGMode('image');\">";
    str += "<area shape=\"rect\" coords=\"0,210,30,235\" onClick=\"showColorPalette('fill');\">";
    str += "<area shape=\"rect\" coords=\"0,240,30,265\" onClick=\"showColorPalette('stroke');\">";
    str += "<area shape=\"rect\" coords=\"0,280,30,305\" onClick=\"topItem();\">";
    str += "<area shape=\"rect\" coords=\"0,305,30,330\" onClick=\"upItem();\">";
    str += "<area shape=\"rect\" coords=\"0,330,30,355\" onClick=\"downItem();\">";
    str += "<area shape=\"rect\" coords=\"0,355,30,380\" onClick=\"bottomItem();\">";
    str += "<area shape=\"rect\" coords=\"0,390,30,415\" onClick=\"dupObject();\">";
    str += "<area shape=\"rect\" coords=\"0,415,30,440\" onClick=\"delSVGObj();\">";
    str += "<area shape=\"rect\" coords=\"0,450,30,470\" onClick=\"setRotLock();\">";
    str += "<area shape=\"rect\" coords=\"0,470,30,490\" onClick=\"setScaleLock();\">";
    str += "</map>";
    str += "<img src=\"images/lock.png\" onClick=\"toggleRotateScaleLock();\" id=\"lock\" style=\"display:none;z-index:100;position:absolute;\"/>\n";
    str += "<div id=\"toolFill\"></div>\n";
    str += "<div id=\"toolStroke\"></div>\n";
    str += "</li>\n";
    toolbar.innerHTML= "<ul>"+str+"</ul>";
  }
  toolbar.style.display='block';
}

function setMode(m){
  currentMode=m;
  switch(m){
    case 'List':
      hideItemById('editDiv');
      hideItemById('preview');
      hideItemById('popup');
      hideItemById('toolBar');
      hideItemById('color-palette');
      showItemById('menuDiv');
      break;
    case 'Edit':
      hideItemById('preview');
      hideItemById('popup');
      hideItemById('toolBar');
      hideItemById('color-palette');
      showItemById('editDiv');
      showItemById('menuDiv');
      break;
    case 'Preview':
      showItemById('preview');
      hideItemById('editDiv');
      hideItemById('popup');
      hideItemById('color-palette');
      showItemById('menuDiv');
      showItemById('toolBar');
      break;
    default:
      break;
  }
}

function hideItemById(id){
  var itm = document.getElementById(id);
  if(itm) itm.style.display='none';
}

function showItemById(id){
  var itm = document.getElementById(id);
  if(itm) itm.style.display='block';
}

function removeChildNodes(id){
  var itm = document.getElementById(id);
  if(!itm) return;

  var child = itm.firstChild;

  while(child){
    itm.removeChild(child);
    child = itm.firstChild;
  }
}

function chEditareaHeight(){
  var itm = document.getElementById('editarea');
  if(!itm) return;
  var cv = itm.getAttribute('rows');
  if(parseInt(cv) > editarea_h){
    itm.setAttribute('rows', editarea_h);
  }else{
    itm.setAttribute('rows', editarea_h * 2);
  }
}

function format_file(str){
  return "<pre>"+str+"</pre>";
}

function previewData(data){
  if(!data) data = document.getElementById('editarea').value;

  previewFile(data);

  updateShowMenu();
  setMode('Preview');
}
  
function previewFile(data){
  removeChildNodes('preview');

  preview.style.display='block';
  preview.style.position='absolute';
  preview.style.top='180px';
  preview.style.bottom='50px';
  preview.style.left='0px';
  preview.style.right='10px';
  preview.style.width='800px';
  preview.style.height='525px';
  mkColorPalette();

  var ele = toSVGElement(data, '100%','100%');

  preview.appendChild(ele);

  svg_top = document.getElementById('svg_top');
  initSVGElementId(svg_top);
  svg_select = createSVGObj('rect', 'x=1,y=1,width=1,height=1,visibility=hidden,stroke-dasharray=9 5', 'none', 'blue', 2);
  svg_select.setAttribute("id","svg_select");
  appendSVGObj(svg_select);
}

function mkColorPalette(){
  var palette = document.getElementById('color-palette');
  if(!palette) return;
  if(palette.innerHTML) return;

  palette.innerHTML="";
  palette.addEventListener("touchstart", onTouchStartColor, false);

  for(var i=0; i<colors.length ;i++){
    if(colors[i] == 'none')
      palette.innerHTML +="<div class=\"item\" style=\"background-color:"+colors[i]+";\" color-val=\""+colors[i]+"\">X</div>";
    else
      palette.innerHTML +="<div class=\"item\" style=\"background-color:"+colors[i]+";\" color-val=\""+colors[i]+"\"> </div>";
  }
  palette.style.width='120px';
}

////// for SVG object
////////////////
function downItem(){
  if(selectedItems.length != 1) return;
  var itm = selectedItems[0];
  var nodes = svg_top.childNodes;
  for(var i=0; i< nodes.length; i++){ if(nodes[i] == itm) break; }
  if (i > 0) svg_top.insertBefore(itm, nodes[i-1]); 
}

function upItem(){
  if(selectedItems.length != 1) return;
  var itm = selectedItems[0];
  var nodes = svg_top.childNodes;
  for(var i=0; i< nodes.length; i++){ if(nodes[i] == itm) break; }

  if (i == nodes.length-2){
    svg_top.appendChild(itm);
  }else if (i < nodes.length-1){
    svg_top.insertBefore(itm, nodes[i+2]); 
  }
}

function bottomItem(){
  if(selectedItems.length != 1) return;
  var itm = selectedItems[0];
  svg_top.insertBefore(itm, svg_top.firstChild); 
}

function topItem(){
  if(selectedItems.length != 1) return;
  var itm = selectedItems[0];
  svg_top.removeChild(itm);
  svg_top.appendChild(itm);
}

function updateSVGObjMenu(){
  var res = "";
  var tag = modeSVG;
  if(selectedItems.length > 1){
     res += '<button onClick="groupSVGObjects(selectedItems);">Group</button>';
     return res;
  }
  if(selectedItems.length == 1){
      tag = getElementTag(selectedItems[0]);
      res += setSVGObjectProp(selectedItems[0]);
  }else{
    switch(tag){
      case 'text':
        res += propSVGText("", svg_font_size, svg_color, 0);
        break;
      case 'rect':
      case 'circle':
      case 'ellipse':
      case 'newPath':
      case 'newLine':
      case 'path':
      case 'line':
        res += propSVGObj(tag+":", svg_line_width, svg_color, svg_fill_color, "",  0);
        break;
      case 'image':
        res += propSVGImage("", 100, 100, 0);
        break;
      default:
        break;
    }
  }
  return res;
}

function propSVGText(str, size, color, rot){
  if(!size) size = 24;
  if(!color) color = '#000000';
  if(!rot) rot = 0;

  var res = "Text:";
  res += "<input type=\"hidden\" id=\"svg_text\" value=\""+str+"\"/>";
  res += "<input type=\"hidden\" id=\"svg_color\" value=\""+color+"\" size=\"8\"/>";
  res += "Rot:<input type=\"text\" onChange=\"updateSVGObj();\" id=\"svg_rotate\" value=\""+rot+"\" size=\"4\"/>";
  
  res += "Size:<input type=\"text\" onChange=\"updateSVGObj();\" id=\"svg_size\" value=\""+size+"\" size=\"4\"/>";

  document.getElementById('toolFill').style.backgroundColor=color;
  return res;
}

function propSVGObj(type, stroke, color, fill, style, rot){
  var res = type;
  if(!stroke) stroke = 1;
  if(!color) color = '#000000';
  if(!fill) fill = '#ffffff';
  if(!rot) rot = 0;

  res += "<input type=\"hidden\" id=\"svg_fill\" value=\""+fill+"\" size=\"8\"/>";
  res += "<input type=\"hidden\" id=\"svg_color\" value=\""+color+"\" size=\"8\"/>";
  res += "<input type=\"hidden\" id=\"svg_stroke\" onChange=\"updateSVGObj();\" value=\""+stroke+"\" size=\"2\"/>";
  res += "Rot:<input type=\"text\" id=\"svg_rotate\" onChange=\"updateSVGObj();\" value=\""+rot+"\" size=\"4\"/>";

  if(style==null) style="";
  res += "<input type=\"hidden\" id=\"svg_stroke_type\" onChange=\"updateSVGObj();\" value=\""+style+"\" size=\"8\"/>";

  res += "<img id=\"strokeW\" src=\"images/strokeW.png\" usemap=\"#strokemenu\" />\n";
  res += "<map name=\"strokemenu\">";
  res += "<area shape=\"rect\" coords=\"0,0,60,25\" onClick=\"selectStrokeW(0);\">";
  res += "<area shape=\"rect\" coords=\"0,25,60,50\" onClick=\"selectStrokeW(1);\">";
  res += "<area shape=\"rect\" coords=\"0,50,60,75\" onClick=\"selectStrokeW(2);\">";
  res += "<area shape=\"rect\" coords=\"0,75,60,100\" onClick=\"selectStrokeW(3);\">";
  res += "<area shape=\"rect\" coords=\"0,100,60,125\" onClick=\"selectStrokeW(4);\">";
  res += "<area shape=\"rect\" coords=\"0,125,60,150\" onClick=\"selectStrokeW(5);\">";
  res += "<area shape=\"rect\" coords=\"0,150,60,175\" onClick=\"selectStrokeW(6);\">";
  res += "<area shape=\"rect\" coords=\"0,175,60,200\" onClick=\"selectStrokeW(7);\">";
  res += "<area shape=\"rect\" coords=\"0,200,60,225\" onClick=\"selectStrokeW(8);\">";
  res += "</map>";

  res += "<img id=\"strokeDash\" src=\"images/dash_type.png\" usemap=\"#dashtype\" />\n";
  res += "<map name=\"dashtype\">";
  res += "<area shape=\"rect\" coords=\"0,0,50,25\" onClick=\"selectStrokeDash(0);\">";
  res += "<area shape=\"rect\" coords=\"0,25,50,50\" onClick=\"selectStrokeDash(1);\">";
  res += "<area shape=\"rect\" coords=\"0,50,50,75\" onClick=\"selectStrokeDash(2);\">";
  res += "<area shape=\"rect\" coords=\"0,75,50,100\" onClick=\"selectStrokeDash(3);\">";
  res += "<area shape=\"rect\" coords=\"0,100,50,125\" onClick=\"selectStrokeDash(4);\">";
  res += "<area shape=\"rect\" coords=\"0,125,50,150\" onClick=\"selectStrokeDash(5);\">";
  res += "<area shape=\"rect\" coords=\"0,150,50,175\" onClick=\"selectStrokeDash(6);\">";
  res += "</map>";

  document.getElementById('toolFill').style.backgroundColor=fill;
  document.getElementById('toolStroke').style.backgroundColor=color;
  return res;
}

function propSVGLine(type, stroke, color, fill, style, rot){
  var res =  propSVGObj(type, stroke, color, fill, style, rot);
  if(!rot) rot = 0;

  res += "<img id=\"arrow_l\" src=\"images/arrow_l.png\" usemap=\"#arrow_l\" />\n";
  res += "<map name=\"arrow_l\">";
  res += "<area shape=\"rect\" coords=\"0,0,30,25\"  onClick=\"setLeftArrow();\">";
  res += "<area shape=\"rect\" coords=\"0,25,30,50\"  onClick=\"removeLeftArrow();\">";
  res += "</map>";

  res += "<img id=\"arrow_r\" src=\"images/arrow_r.png\" usemap=\"#arrow_r\" />\n";
  res += "<map name=\"arrow_r\">";
  res += "<area shape=\"rect\" coords=\"0,0,30,25\"  onClick=\"setRightArrow();\">";
  res += "<area shape=\"rect\" coords=\"0,25,30,50\"  onClick=\"removeRightArrow();\">";
  res += "</map>";


  res += "<img id=\"path_type\" src=\"images/path_type.png\" usemap=\"#path_type\" />\n";
  res += "<map name=\"path_type\">";
  res += "<area shape=\"rect\" coords=\"0,0,30,100\"  onClick=\"togglePathType();\">";
  res += "</map>";
  document.getElementById('toolFill').style.backgroundColor=fill;
  document.getElementById('toolStroke').style.backgroundColor=color;

  return res;
}

function getStrokeWIdx(lw){
  var res=0;
  for(var i=0;i<lineW.length;i++){
    if(lw < lineW[i]){ return res; }
    res = i; 
  }
  return lineW.length-1;
}

function getStrokeW(idx){ return lineW[idx]; }

function updateStrokeWidth(){
  if(selectedItems.length == 1){
    var obj = selectedItems[0];
    if(!obj){ return; }
    var val = obj.getAttribute('stroke-width'); 
    if(!val) return;
    var lw = parseFloat(val);
    updateStrokeWImg(getStrokeWIdx(lw));
    updateStrokeDash();
  }
}

function updateStrokeWImg(idx){
  var ele = document.getElementById('strokeW');
  if(!ele) return;
  var v1 = idx*25;
  var v2 = v1+25;
  var v3 = 81 - v1;
  ele.style.clip = "rect("+v1+"px,70px,"+v2+"px,0px)";
  ele.style.top = v3+"px";
}

function selectStrokeW(idx){
  var ele = document.getElementById('strokeW');
  if(ele.style.clip != "rect(auto auto auto auto)"){
    ele.style.clip = "rect(auto auto auto auto)";
    ele.style.top = "81px";
    return;
  }
  if(selectedItems.length == 1){
    var obj = selectedItems[0];
    var lw = getStrokeW(idx);
    var plw = parseFloat(obj.getAttribute('stroke-width'));
    obj.setAttribute('stroke-width', lw);
    updateDashArray(obj, lw, plw, obj.getAttribute('stroke-dasharray'));
    updateStrokeWImg(idx);
  }else{
    updateStrokeWImg(1);
  }
}

function updateStrokeDash(){
  if(selectedItems.length == 1){
    var obj = selectedItems[0];
    if(!obj){ return; }
    var val = obj.getAttribute('stroke-width'); 
    if(!val) return;
    var lw = parseFloat(val);
    var darr = obj.getAttribute('stroke-dasharray'); 
    if(!darr) return;
    updateStrokeDashImg( getDashArrayIndex(lw, darr) );
  }
}
function updateStrokeDashImg(idx){
  var ele = document.getElementById('strokeDash');
  if(!ele) return;
  var v1 = idx*25;
  var v2 = v1+25;
  var v3 = 81 - v1;
  ele.style.clip = "rect("+v1+"px,70px,"+v2+"px,0px)";
  ele.style.top = v3+"px";
}

function selectStrokeDash(idx){
  var ele = document.getElementById('strokeDash');
  if(ele.style.clip != "rect(auto auto auto auto)"){
    ele.style.clip = "rect(auto auto auto auto)";
    ele.style.top = "81px";
    return;
  }

  if(selectedItems.length == 1){
    var obj = selectedItems[0];
    var lw = obj.getAttribute('stroke-width');
    if(!lw ) lw=1;
    else lw=parseFloat(lw);

    var darr = StrokeDash[idx];
    if(darr=="") try{ obj.removeAttribute('stroke-dasharray'); }catch(e){}
    else updateDashArray(obj, lw, 1, darr);

    updateStrokeDashImg(idx);
  }else{
    updateStrokeDashImg(1);
  }
}

function getDashArrayIndex(plw, darr){
  if(!darr || darr=="") return 0;
  var d_arr = darr.split(' ');
  var top = Math.round(parseFloat(d_arr[0])/plw);
  if(top == 6){
    if(d_arr.length == 2) return 2;
    if(d_arr.length == 4) return 3;
  }else if(top == 8){
    if(d_arr.length == 2) return 4;
    if(d_arr.length == 4) return 5;
    if(d_arr.length == 6) return 6;
  }else if(top == 2){
    return 1;
  }
  return 0;
}

function updateDashArray(obj, lw, plw, darr){
  if(darr && darr != ""){
     var d_arr = darr.split(' ');
     darr = "";
     for(var i=0; i<d_arr.length; i++){
        var val = Math.round(parseFloat(d_arr[i])/plw);
        darr += val * lw  + ' ';
     } 
     obj.setAttribute('stroke-dasharray', trim(darr));
     return true;
  }
  return false;
}

function togglePathType(){
  if(selectedItems.length != 1) return;
  var itm = selectedItems[0];
  var path = itm.getAttribute("d").split(' ');
  var newpath = "";

  for(var i=0;i<path.length ;i++){
    if(path[i] == "L"){ path[i] = "Q"; }
    else if(path[i] == "Q"){ path[i] = "S"; }
    else if(path[i] == "S"){ path[i] = "T"; }
    else if(path[i] == "T"){ path[i] = "L"; }
    newpath += path[i] + ' ';
  }
  var mm = path.length % 4;
  if(mm != 0) {
    newpath += path[i-2] + ' ';
    newpath += path[i-1] + ' ';
  }
  
  itm.setAttribute("d",trim(newpath));
  updatePathTypeMenu();
}

function  updatePathTypeMenu(){
  if(selectedItems.length != 1) return;
  var itm = selectedItems[0];
  if(!itm) return;
  var path = itm.getAttribute("d");
  var img = document.getElementById('path_type');
  if(!path){
     if(img) img.style.display='none';
     return;
  }

  if(path.indexOf("Q") > 0){
     selectMenuImage(img, 1);
  }else if(path.indexOf("T") > 0){
     selectMenuImage(img, 2);
  }else if(path.indexOf("S") > 0){
     selectMenuImage(img, 3);
  }else{
     selectMenuImage(img, 0);
  }
  img.style.display='block';
}

function  updateArrowMenu(){
  try{
    var imgl = document.getElementById('arrow_l');
    if(hasArrow('start')){ selectMenuImage(imgl, 1); }else{ selectMenuImage(imgl, 0); }
  }catch(e){}

  try{
    var imgr = document.getElementById('arrow_r');
    if(hasArrow('end')){ selectMenuImage(imgr, 1); }else{ selectMenuImage(imgr, 0); }
  }catch(e){}
}

function selectMenuImage(img, pos){
  if(!img) return;
   var clip = new Array("rect(0px,30px,25px,0px)","rect(25px,30px,50px,0px)",
        "rect(50px,30px,75px,0px)", "rect(75px,30px,100px,0px)");
   var top = new Array( "81px", "56px", "31px", "6px");

  img.style.clip = clip[pos];
  img.style.top = top[pos];
}

function propSVGImage(str, w, h, rot){
  var res = "image:";
  if(!rot) rot = 0;
  res += "<input type=\"text\" id=\"svg_text\" value=\""+str+"\" />";
  res += "Width:<input type=\"text\" onChange=\"updateSVGObj();\" id=\"svg_w\" value=\""+w+"\" size=\"4\"/>";
  res += "Height:<input type=\"text\" onChange=\"updateSVGObj();\" id=\"svg_h\" value=\""+h+"\" size=\"4\"/>";
  res += "Rot:<input type=\"text\" onChange=\"updateSVGObj();\" id=\"svg_rotate\" value=\""+rot+"\" size=\"4\"/>";
  return res;
}


function setSVGMode(m){
  modeSVG = m;
  switch(m){
    case 'selector':
      selectToolBar(0);
      break;
    case 'newPath':
      selectToolBar(1);
      break;
    case 'newLine':
      selectToolBar(2);
      break;
    case 'text':
      selectToolBar(3);
      break;
    case 'rect':
      selectToolBar(4);
      break;
    case 'circle':
      selectToolBar(5);
      break;
    case 'ellipse':
      selectToolBar(6);
      break;
    case 'image':
      selectToolBar(7);
      break;
    default:
      break;
  }
  updateShowMenu();
}


function toSVGElement(str, w, h){
  var xmlsvg = "xmlns:svg=\"http://www.w3.org/2000/svg\" xmlns:xlink=\"http://www.w3.org/1999/xlink\"";
  var parser = new DOMParser();
  var header = "<svg:svg width=\""+w+"\" height=\""+h+"\" "+xmlsvg+" id=\"svg_top\">";
  var footer = "</svg:svg>";
  var xmlDoc = parser.parseFromString(header+str+footer, "text/xml");
  var xmlRoot = xmlDoc.documentElement;
  var ele =  document.importNode(xmlRoot,true);

  return ele;
}

function newID(){
  var id = 'svg_'+nextId;
  nextId++;
  return id;
}

function createSVGElement(tag, id){
  if(!id || id=='new') id = newID(); 
  var ele= document.createElementNS(svg_ns, tag);
  ele.setAttribute("id", id);
  return ele;
}

function defSVGElement(node){
  if(!svg_defs) svg_defs = document.getElementById('svg_defs');
  if(!svg_defs){
    svg_defs = createSVGElement('defs', 'svg_defs');
    svg_top.insertBefore(svg_defs, svg_top.firstChild);
  }
  if(node) svg_defs.appendChild(node);
}

function initSVGElementId(svg_top){
  nextId = 1;
  checkSVGElementId(svg_top);
  appendSVGElementId(svg_top);
}

function checkSVGElementId(top){
  var objs = top.childNodes;

  for(var i=0; i<objs.length ;i++){
    if(objs[i].tagName){
      var id = objs[i].getAttribute("id");

      if(id && id.match(/svg_[0-9]+/i)){
         var n = parseInt(RegExp.lastMatch.substr(4));
         if(n >= nextId){
           nextId = n+1;
         }
       
      }
      checkSVGElementId(objs[i]);
    }
  }
}

function appendSVGElementId(top){
  var objs = top.childNodes;

  for(var i=0; i<objs.length ;i++){
    if(objs[i].tagName){
      var id = objs[i].getAttribute("id");
      if(!id){
         objs[i].setAttribute("id", "svg_"+nextId);
         nextId++;
      }
    }
    appendSVGElementId(objs[i]);
  }
}

function setAttributes(obj, attrs){
  var attr_array = attrs.split(',');

  for (var i=0; i<attr_array.length;i++){
    var x = attr_array[i].split('=');
    if(x.length == 2){
      obj.setAttribute(x[0], x[1]);
    }
  }
}

function createSVGObj(tag, attrs, fill, color, lw){
  var ele = createSVGElement(tag, 'new');
  setAttributes(ele, attrs);
  if (fill) ele.setAttribute('fill', fill);
  if (color) ele.setAttribute('stroke', color);
  if(lw) ele.setAttribute('stroke-width', lw);

  return ele;
}

function createSVGText(txt, x, y, size, color){
  var ele = createSVGElement('text', 'new');

  ele.setAttribute('x', x);
  ele.setAttribute('y', y);
  ele.setAttribute('font-size', size);
  ele.setAttribute('fill', color);
  ele.textContent=txt;

  return ele;
}

function createSVGImage(fname, width, height, attrs){
  var ele = createSVGElement('image', 'new');
  ele.setAttributeNS('http://www.w3.org/1999/xlink', 'xlink:href', fname);
  ele.setAttribute('width', width);
  ele.setAttribute('height', height);
  setAttributes(ele, attrs);

  return ele;
}

function groupSVGObjects(){
  try{ 
    var objs = selectedItems;
    if(objs.length == 0) return null;
    var ele = createSVGElement('g', 'new');
    for(var i=0;i<objs.length;i++){
      var obj = objs[i];
       obj.parentNode.removeChild(obj);
       ele.appendChild(obj);
    }
    appendSVGObj(ele);
    selectedItems = new Array(ele);
    updateShowMenu();
    return ele;
  }catch(e){ return null;}
}

function ungroupSVGObjects(){
  try{ 
    var obj = selectedItems[0];
    if(getElementTag(obj) != 'g') return;
    var parent = obj.parentNode;
    selectedItems = new Array();
    while(obj.firstChild != null){
      selectedItems.push(obj.firstChild);
      parent.appendChild(obj.firstChild);
    }
    parent.removeChild(obj);
    updateShowMenu();
  }catch(e){ return;}
}

function createSVGMarker(pid, id, child){
  var parent = document.getElementById(pid);
  if(!parent) return;
  var ele = createSVGElement('marker', pid+'_'+id);
  ele.appendChild(child);
  return ele;
}

function setLeftArrow(){
 if( selectedItems.length == 1 ){
    setArrow(selectedItems[0], 'start', '');
    updateShowMenu();
 }
}

function setRightArrow(){
 if( selectedItems.length == 1 ){
    setArrow(selectedItems[0], 'end', '');
    updateShowMenu();
 }
}

function removeLeftArrow(){
 if( selectedItems.length == 1 ){
   if(!svg_defs) svg_defs = document.getElementById('svg_defs');
   try{
     var marker = getArrowMarker(selectedItems[0],'start');
     svg_defs.removeChild(marker);
     selectedItems[0].removeAttribute('marker-start');
   }catch(e){ }
   updateShowMenu();
 }
}

function removeRightArrow(){
 if( selectedItems.length == 1 ){
   if(!svg_defs) svg_defs = document.getElementById('svg_defs');
   try{
     var marker = getArrowMarker(selectedItems[0],'end');
     svg_defs.removeChild(marker);
     selectedItems[0].removeAttribute('marker-end');
   }catch(e){ }
   updateShowMenu();
 }
}

function removeAllMarker(obj){
  var smarker = getArrowMarker(obj,'start');
  if(smarker) svg_defs.removeChild(smarker);
  var emarker = getArrowMarker(obj,'end');
  if(emarker) svg_defs.removeChild(emarker);
  return true;
}

function hasArrow(pos){
  var pobj = selectedItems[0];
  if(!pobj) return false;
  if(getArrowMarker(pobj,pos)) return true;
  return false;
}

function getArrowMarker(obj,pos){
  try{
    return document.getElementById(obj.getAttribute('id')+'_'+pos+'arrow');
  }catch(e){ return null; }
 
}

function setArrow(pobj, pos, type){
  if(!pobj) return;
  var marker = getArrowMarker(pobj,pos);
  if(marker){ return; }

  var obj = createSVGElement('path', 'new');

  var refX = 10;
  var refY = 10; 

  switch(pos){
    case 'start':
      setAttributes(obj,'d=M 20 0 L 0 10 20 20 Z');
      break;
    case 'end':
      setAttributes(obj,'d=M 0 0 L 20 10 0 20 Z');
      break;
    default:
      return; 
  }
  
  var pid = pobj.getAttribute("id");
  marker = createSVGMarker(pid, pos+'arrow' , obj);

  setAttributes(marker,'markerWidth=10,markerHeight=10,orient=auto,viewBox=0 0 20 20,markerUnits=strokeWidth,refX='+refX+',refY='+refY);

  marker.setAttribute("fill",pobj.getAttribute("stroke")); 

  defSVGElement(marker);
  if(!document.getElementById(pid+'_'+pos+'arrow')){
     alert("Error in setArrow");
     return;
  }
  var mid = "url(#"+marker.getAttribute("id")+")";
  var mattr = "marker-"+pos;
  pobj.setAttribute(mattr, mid);
  updateShowMenu();
}

function scalePath(itm, scale){
  if(!scale) return;
  var path = itm.getAttribute("d").split(' ');
  var bbox = itm.getBBox();
  var newpath = "";
  var sX=scale[0];
  var sY=scale[1];
  var isX=true;
  var dx = bbox.x - sX*bbox.x;
  var dy = bbox.y - sY*bbox.y;

  for(var i=0;i<path.length ;i++){
    if(path[i].match(/[0-9]+/)){
      var val = parseInt(path[i]);
      if(isX){
        path[i] =  Math.round(val*sX) + dx ;
        isX = false;
      }else{
        path[i] =  Math.round(val*sY) + dy ;
        isX = true;
      }  
    }
    newpath += path[i] + ' ';
  }
  
  itm.setAttribute("d",trim(newpath));
}

function scaleLine(itm, scale){
  if(!scale) return;
  var sX=scale[0];
  var sY=scale[1];
  var x1 = parseInt(itm.getAttribute("x1"));
  var y1 = parseInt(itm.getAttribute("y1"));
  var x2 = parseInt(itm.getAttribute("x2"));
  var y2 = parseInt(itm.getAttribute("y2"));
  var bbox = itm.getBBox();
  var dx = bbox.x - sX*bbox.x;
  var dy = bbox.y - sY*bbox.y;

  itm.setAttribute("x1", Math.round(x1*sX)+dx);
  itm.setAttribute("y1", Math.round(y1*sY)+dy);
  itm.setAttribute("x2", Math.round(x2*sX)+dx);
  itm.setAttribute("y2", Math.round(y2*sY)+dy);
}

function appendSVGObj(obj){
  var svg_top = document.getElementById('svg_top');
  if(!svg_top) return;

  svg_top.appendChild(obj);
}

function removeSVGObj(obj){
  var svg_top = document.getElementById('svg_top');
  if(!svg_top) return;

  svg_top.removeChild(obj);
}


function isChildById(element, id) {
  if (element == null || element.parentNode == null || element.parentNode.nodeName=='BODY') return false;
  else if (element.parentNode.id == id) return true;
  else return isChildById(element.parentNode, id);
}

/////////////////////// Formatting SVG DOM
function escapeHTML(text) {
  return text.replace( /[<>"&]/g,
    function (m) { return { '<': '&lt;', '>': '&gt;', '"': '&quot;', '&': '&amp;' }[m]; }
  );
};

function formatTag(ele){
  var str="";
  if(ele.nodeType == 1){
    var tag_a = ele.tagName.split(':');
    var tag;
    if(tag_a.length == 1){ tag = "svg:"+tag_a[0]; }else{ tag = ele.tagName; }

    str += "<"+tag;
    var attrs = ele.attributes;
    for(var i=0; i<attrs.length; i++){
      str += " "+attrs[i].nodeName+"=\""+attrs[i].nodeValue+"\"";
    }
    var cn = ele.childNodes;
    if(cn.length > 0){
      str +=">\n";
      for(var i=0; i<cn.length; i++){
        var tmp = trim(formatTag(cn[i]));
        if(tmp) str += " "+tmp+"\n";
      }
      str += "</"+tag+">";
    }else{
      str +=" />";
    }
    return str;
  }else if(ele.nodeType==3){
    return ele.textContent;
  }
}

function getSVGContent(){
  if(!svg_top) return "";

  var str = "";
  var elements = svg_top.childNodes;
  for(var i=0; i<elements.length; i++){
    if(elements[i] != svg_select){
      var tmp = trim(formatTag(elements[i]));
      if(tmp) str += tmp + '\n';
    }
  }
  return str;
}

function trim(str){
  return str.replace(/(^\s+)|(\s+$)/g, "");
}

///////  Access Server
function newXMLRequest(){
  if(this.XMLHttpRequest){
    return new XMLHttpRequest();
  }else {
    return new ActiveXObject("Microsoft.XMLHTTP");
  }
}

function createRequestData(data){
  var str="filetype=svg";
  for (var i in data){
    str = str +"&"+ i +"="+encodeURIComponent(data[i]);
  }
  return str;
}

function postRequest(url, data, func){
  var postData=createRequestData(data);
  var obj=newXMLRequest();

  obj.onreadystatechange = function(){
    if (obj.readyState == 4 && obj.status == 200){
      func(obj.responseText);
    }
  }
  obj.open("POST", url, true);
  obj.setRequestHeader("Content-Type","application/x-www-form-urlencoded");
  obj.send(postData);
}

function commnadFinishAlert(s){
  alert(s);
}

function uploadFileData(transaction, results){
  var data = results.rows.item(0);
  var updata = new Array(0);
  updata['name']= data['name'];
  updata['datalob'] = data['datablob'];
  updata['cmd'] = 'upload'

  postRequest(MgrPath+iSlideMgr, updata, commnadFinishAlert);
}

function uploadFile(name){
   getFile(name, 'uploadFileData');
}

function getRemoteFileList(){
  var data=new Array();
  data['name'] = "SVG";
  data['cmd'] = "list";
  postRequest(MgrPath+iSlideMgr, data, fileSelectorOnServer);
}

function previewRemoteFile(content){
  previewFile(content);
  setMode('Preview');
}

function getRemoteFile(name){
  var data=new Array();
  data['name'] = name;
  data['cmd'] = "get";
  postRequest(MgrPath+iSlideMgr, data, previewRemoteFile);
}

function saveRemoteFile(content){
  var datadiv = document.getElementById('tempdata');
  var fname = datadiv.getAttribute('lfname');
  saveContentWithFilename(fname, content);
}

function downloadFile(name){
  var data=new Array();
  data['name'] = name;
  data['cmd'] = "get";
  var datadiv = document.getElementById('tempdata');
  datadiv.setAttribute('lfname', name);
  postRequest(MgrPath+iSlideMgr, data, saveRemoteFile);
}

function showRemoteFile(name){
  getRemoteFile(name);
}

function downloadToFile(name){
  var downloadForm = "";
  downloadForm += "<form action=\"iSlideManager.php\" method=\"post\">";
  downloadForm += "<input type=\"hidden\" name=\"cmd\" value=\"download\">";
  downloadForm += "<input type=\"hidden\" name=\"name\" value=\""+name+"\">";
  downloadForm += "<input type=\"hidden\" name=\"filetype\" value=\"svg\">";
  downloadForm += "<input type=\"submit\">";
  downloadForm += "</form>";

  var cmdForm = document.getElementById('cmdForm');
  cmdForm.innerHTML = downloadForm;
  cmdForm.firstChild.submit();
}

//////// Event Handler
function selectedRectangle(x, y, ex, ey){
  if(!svg_select) return null;

  svg_select.setAttribute("x",x);
  svg_select.setAttribute("y",y);
  svg_select.setAttribute("width", ex-x);
  svg_select.setAttribute("height", ey-y);
  svg_select.setAttribute("visibility", "visible");
  return svg_select;
}

function hideSelectedRectangle(){
  svg_select.setAttribute("x",0);
  svg_select.setAttribute("y",0);
  svg_select.setAttribute("width",0);
  svg_select.setAttribute("height",0);
  svg_select.setAttribute("visibility","hidden");
}

function setInnerHTML(id, val){
  var itm=document.getElementById(id);
  if(itm) itm.innerHTML=val;
}

function toggleRotateScaleLock(){
  var lock = document.getElementById('lock');
  if(svg_rotate_locked){
    svg_rotate_locked=false;
    if(lock.style.top == '450px'){
      lock.style.display='none';
    }else{
      svg_scale_locked=true;
      lock.style.top = '470px';
    }
    return;
  }
  if(svg_scale_locked){
    svg_scale_locked=false;
    if(lock.style.top == '470px'){
      lock.style.display='none';
    }else{
      svg_rotate_locked=true;
      lock.style.top = '450px';
    }
    return;
  }

}

function setRotLock(){
  var lock = document.getElementById('lock');
  if(!svg_rotate_locked){
    svg_rotate_locked = true;
    svg_scale_locked = false;
    lock.style.top = '450px';
    lock.style.display='block';
  }else alert("setRotLock");
}

function setScaleLock(){
  var lock = document.getElementById('lock');
  if(!svg_scale_locked){
    svg_scale_locked = true;
    svg_rotate_locked = false;
    lock.style.top = '470px';
    lock.style.display='block';
  }else alert("setScaleLock");
}

function popupInfo(val){
  var str="<button onClick=\"hideItemById('popup');\">Close</button><hr> ";
  str += val;
  setInnerHTML('popup', str);
  showItemById('popup');
}

function addAttributeVal(obj, itm, dv){
  if(obj.hasAttribute(itm)){
    var x = parseInt(obj.getAttribute(itm));
    x = x + dv;
    obj.setAttribute(itm, x)
  }else alert("No such attribute:"+itm);
}

function showSVGSource(){
  var str = getSVGContent();
  var escstr="<button onClick=\"hideItemById('popup');\">Close</button>";
  escstr+="<button onClick=\"saveContent(document.getElementById('ContentView').value); previewData(document.getElementById('ContentView').value);\">Save</button><hr> ";
  escstr += "<textarea cols=\"100\" rows=\"30\" id=\"ContentView\">"+str+"</textarea>";
  setInnerHTML('popup', escstr);
  showItemById('popup');
}


function getElementTag(obj){
  if (!obj) return null;
  if(!obj.tagName) return "NoName";
  var tag = obj.tagName.split(':');
  if(tag.length == 2 && tag[0]=='svg') return tag[1];
  return tag[0];
}

function setSVGObjectProp(obj){
  var tag = getElementTag(obj);
  var res ="";
  switch(tag){
    case 'text':
      res = propSVGText(obj.textContent, obj.getAttribute("font-size"), obj.getAttribute("fill"),getRotateAngle(obj));
      break;
    case 'rect':
    case 'circle':
    case 'ellipse':
     res =  propSVGObj(tag+":", obj.getAttribute("stroke-width"), obj.getAttribute("stroke"),
         obj.getAttribute("fill"),obj.getAttribute("stroke-dasharray"),getRotateAngle(obj));
     break;
    case 'path':
    case 'line':
     res =  propSVGLine(tag+":",obj.getAttribute("stroke-width"),
       obj.getAttribute("stroke"),obj.getAttribute("fill"), obj.getAttribute("stroke-dasharray"),getRotateAngle(obj));
     break;
    case 'image':
     res =  propSVGImage(obj.getAttribute("xlink:href"),obj.getAttribute("width"),obj.getAttribute("height"),getRotateAngle(obj));
     break;
    case 'g':
      res += '<button onClick="ungroupSVGObjects();">Ungroup</button>';
      break;
    default:
     break;
  }

  return res
}

function updateSVGObj(){
  if(selectedItems.length != 1) return;
  var obj = selectedItems[0];
  var tag = getElementTag(obj);
  var deg=document.getElementById('svg_rotate');
  var res ="";
  switch(tag){
    case 'text':
      var txt=document.getElementById('svg_text');
      var color=document.getElementById('svg_color');
      var size=document.getElementById('svg_size');
      obj.textContent = txt.value;
      obj.setAttribute("fill", color.value);
      obj.setAttribute("font-size", size.value);
      setRotate(obj,deg.value);
      break;
    case 'rect':
    case 'circle':
    case 'ellipse':
      var fill=document.getElementById('svg_fill');
      var color=document.getElementById('svg_color');
      var L=document.getElementById('svg_stroke');
      var dash=document.getElementById('svg_stroke_type');
      obj.setAttribute("fill", fill.value);
      obj.setAttribute("stroke", color.value);
      obj.setAttribute("stroke-width", L.value);
      if(dash) obj.setAttribute("stroke-dasharray", dash.value);
      setRotate(obj,deg.value);
     break;
    case 'path':
    case 'line':
      var fill=document.getElementById('svg_fill');
      var color=document.getElementById('svg_color');
      var L=document.getElementById('svg_stroke');
      var dash=document.getElementById('svg_stroke_type');
      obj.setAttribute("fill", fill.value);
      obj.setAttribute("stroke", color.value);
      obj.setAttribute("stroke-width", L.value);
      if(dash) obj.setAttribute("stroke-dasharray", dash.value);
      setRotate(obj,deg.value);

      var id = obj.getAttribute("id");
      var marker = document.getElementById(id+'_startarrow');
      if(marker) marker.setAttribute("fill", color.value);
      var marker = document.getElementById(id+'_endarrow');
      if(marker) marker.setAttribute("fill", color.value);
     break;
    case 'image':
      var fname=document.getElementById('svg_text');
      var w=document.getElementById('svg_w');
      var h=document.getElementById('svg_h');
      obj.setAttribute("xlink:href", fname.value);
      obj.setAttribute("width", w.value);
      obj.setAttribute("height", h.value);
      setRotate(obj,deg.value);
     break;
    default:
     break;
  }

}

function delSVGObj(){
  if(!svg_top) return;
  for(var i=0; i< selectedItems.length; i++){
    var obj = selectedItems[i];
    removeAllMarker(obj);
    svg_top.removeChild(obj);
  }
  hideSelectedRectangle();
}

///// For MobileSafari
function getPreviewX(x){ return x - preview.offsetLeft; }
function getPreviewY(y){ return y - preview.offsetTop; }

function calcDict(x1, y1, x2, y2){
 return (x1-x2)*(x1-x2) +(y1-y2)*(y1-y2);
}

function isIncludeEllipse( x1, y1, cx, cy, rx, ry){
 return ((x1-cx)*(x1-cx)/rx/rx +(y1-cy)*(y1-cy)/ry/ry) < 1;
}

function getAttributeVal(obj, name){
  try{
    return parseInt(obj.getAttribute(name)); 
  }catch(e){ return 0; }
}

function checkIntersection(obj, x, y, ex, ey){
  if(!obj) return false;
  var res = true;
  var bbox = obj.getBBox();
  var ox = bbox.x;
  var oy = bbox.y;
  var oex = bbox.x+bbox.width;
  var oey = bbox.y+bbox.height;
  if( ex < ox ||  oex < x || ey < oy || oey < y) res = false;

  var tag = getElementTag(obj);
  switch(tag){
    case 'circle':
    case 'ellipse':
      var cx = getAttributeVal(obj,"cx"); 
      var cy = getAttributeVal(obj,"cy"); 
      var rx = getAttributeVal(obj,"r"); 
      var ry = rx;
      if(!rx){
        var rx = getAttributeVal(obj,"rx"); 
        var ry = getAttributeVal(obj,"ry"); 
      }

      if(res){
        if(cx <x && cy < y && !isIncludeEllipse(x,y,cx,cy,rx,ry)) res=false;
        else if(cx < x && cy > ey && !isIncludeEllipse(x,ey,cx,cy,rx,ry)) res=false;
        else if(cx > ex && cy > ey && !isIncludeEllipse(ex,ey,cx,cy,rx,ry)) res=false;
        else if(cx > ex && cy < y && !isIncludeEllipse( ex,y,cx,cy,rx,ry)) res=false;
      }
      break;
    case 'path':
      var d = obj.getAttribute("d"); 
      var p = getPoints(d);
      for(var i=0;i<p.length;i++){
        var ox=p[i][0];
        var oy=p[i][1];
        if(x < ox && ox < ex && y < oy && oy < ey) {
           return true;
        }
      } 
      return false;
      break;
    case 'line':
      var x1 = getAttributeVal(obj,"x1"); 
      var y1 = getAttributeVal(obj,"y1"); 
      var x2 = getAttributeVal(obj,"x2"); 
      var y2 = getAttributeVal(obj,"y2"); 
      var d = (y2-y1)/(x2-x1);

      if(res){
        var xx = Math.abs(x2-x1);
        var sign = 1;
        if(x2-x1 < 0){ sign = -1; }
        for(var i=0; i < xx; i++){
          var nx = i*sign + x1;
          var ny = d * i*sign + y1;
          if(x < nx && nx < ex && y < ny && ny < ey) return true;
        }
      }
      return false;

      return res;
      break;
    case 'text':
    case 'rect':
    case 'polygon':
    case 'polyline':
    case 'image':
    case 'g':
      break;
    default:
      res=false;
      break;
  }
  return res;
}

function getBoundingBox(obj){
  var res = new Array(4);
  var bbox = obj.getBBox();
  res[0] = bbox.x-1;
  res[1] = bbox.y-1;
  res[2] = bbox.x+bbox.width+2;
  res[3] = bbox.y+bbox.height+2;
  return res;
}

function setSelectBox(){
  if(!svg_select) return;

  if(selectedItems.length == 0){
    hideSelectedRectangle()
    return;
  }

  var bbox = new Array(1000,1000,0,0);
  
  for(var i=0;i<selectedItems.length;i++){
    var bp = getBoundingBox(selectedItems[i]);
    if(bp[0] < bbox[0]) bbox[0]=bp[0];
    if(bp[1] < bbox[1]) bbox[1]=bp[1];
    if(bp[2] > bbox[2]) bbox[2]=bp[2];
    if(bp[3] > bbox[3]) bbox[3]=bp[3];
  } 
  selectedRectangle(bbox[0], bbox[1], bbox[2], bbox[3]);
}

function getSelectedObjects(x1, y1, x2, y2){
   if(x1 > x2) { var tmp = x1; x1=x2; x2=tmp; } 
   if(y1 > y2) { var tmp = y1; y1=y2; y2=tmp; } 

   var val="";
   if(svg_top){
     var val ="";
     var objs = svg_top.childNodes;
     selectedItems = new Array();
     for(var i=0; i<objs.length;i++){
      if(objs[i].tagName){
       
        if(objs[i] != svg_select && checkIntersection(objs[i], x1, y1, x2, y2)){
          selectedItems.push(objs[i]);
        }
          val += objs[i].tagName+" ";
      }
    }
  }
  setSelectBox();
}

function dupObject(){
  if(selectedItems.length == 0){ return; }
  dupItems = selectedItems;
  dupX = parseInt(svg_select.getAttribute("x"));
  dupY = parseInt(svg_select.getAttribute("y"));
  setSVGMode('Duplicate');
}

function pasteObject(x,y){
  if(selectedItems.length == 0){ return; }
  for(var i=0;i<dupItems.length;i++){
    var itm = dupItems[i].cloneNode(true);
    itm.setAttribute("id", newID());
    replaceTranslate(itm,x-dupX,y-dupY);
    updateTransform(itm);
    appendSVGObj(itm);
  }
}

function onTouchStartCore(){
  if((!modeSVG || modeSVG == 'selector') && selectedItems.length == 0){ // Selector Mode
    var x1=getPreviewX(sx-1);
    var y1=getPreviewY(sy-1);
    var x2=getPreviewX(sx+2);
    var y2=getPreviewY(sy+2);
    getSelectedObjects(x1, y1, x2, y2);
    
    if(selectedItems.length == 0){
      setSVGMode('selector');
    }else if(selectedItems.length == 1){
      targetItem=selectedItems[0];
    }else{
      setSVGMode('Group');
    }
 }else { // CreateMode
   if(selectedItems.length == 0){

     var fill=document.getElementById('svg_fill');
     var color=document.getElementById('svg_color');
     var L=document.getElementById('svg_stroke');

     var x = getPreviewX(sx);
     var y = getPreviewY(sy);

     switch(modeSVG){
       case 'text':
         var txt=document.getElementById('svg_text');
         var size=document.getElementById('svg_size');
         if(txt.value){
           y = y + parseInt(size.value)*0.8;
           targetItem=createSVGText(txt.value, x, y, size.value, color.value);
         }else{
           putInputForm(x, y, txt.value, size.value);
         }
         break;
       case 'rect':
         var attr = 'x='+x+',y='+y+',width='+svg_width+',height='+svg_height;
         targetItem=createSVGObj(modeSVG,attr, fill.value, color.value, L.value);
         break;
       case 'circle':
         var attr = 'cx='+x+',cy='+y+',r='+svg_rx;
         targetItem=createSVGObj(modeSVG,attr, fill.value, color.value, L.value);
         break;
       case 'ellipse':
         var attr = 'cx='+x+',cy='+y+',rx='+svg_rx+',ry='+svg_ry;
         targetItem=createSVGObj(modeSVG,attr, fill.value, color.value, L.value);
         break;
       case 'newPath':
         //var attr = 'd=M '+x+' '+y+' L '+x+' '+y;
         var attr = 'd=M '+x+' '+y+' L';
         targetItem=createSVGObj('path' ,attr, 'none', color.value, L.value);
	 is_newPath=true;
         break;
       case 'newLine':
         var x2=x+1;
         var attr = 'x1='+x+',y1='+y+',x2='+x2+',y2='+y;
         targetItem=createSVGObj('line' ,attr, 'none', color.value, L.value);
         break;
       case 'image':
         var attr = 'x='+x+',y='+y;
         var txt=document.getElementById('svg_text');
         var w=document.getElementById('svg_w');
         var h=document.getElementById('svg_h');
         if(txt.value) targetItem=createSVGImage(txt.value ,w.value, h.value, attr);
       default:
         break;
     }
     if (targetItem){
       appendSVGObj(targetItem);
       selectedItems[0]=targetItem;
     }
   }else{
     var x1=getPreviewX(sx-1);
     var y1=getPreviewY(sy-1);
     var x2=getPreviewX(sx+2);
     var y2=getPreviewY(sy+2);

     if(modeSVG == 'Duplicate'){ pasteObject(x1,y1); }
     if(modeSVG == 'newPath'){
       if(targetItem.tagName == 'path' ){
         var path = targetItem.getAttribute("d");
         path = path + ' '+ getPreviewX(sx) + ' '+ getPreviewY(sy) ;
         targetItem.setAttribute("d",path);
       }
       return;
     }
     if(!checkIntersection(svg_select, x1, y1, x2, y2)){
        setSVGMode('selector');
     }
   }
 }
}


function onDoubleTap(e){
  if(selectedItems.length == 1 ){
    hideSelectedRectangle();

    var obj = selectedItems[0];
    if(!obj) return;
    switch(obj.tagName){
      case 'svg:text':
      case 'text':
        var txt = trim(obj.textContent);
        var size = parseInt(obj.getAttribute("font-size"));
        var x = parseInt(obj.getAttribute("x"));
        var y = parseInt(obj.getAttribute("y"));
        x = x-20;
        y = y-size*0.8 -10;
        putInputForm(x, y, txt, size, obj.id);
        obj.style.display = 'none';
        editingTextObj = obj;
        break;
      case 'path':
        if(modeSVG == 'newPath'){ setSVGMode('selector'); }
        break;
      default:
        break;
    }
   return false;
  }
}

function getPoints(d){
  var p = d.split(' ');
  var res = new Array();
  var isx=true;
  var x, y;
  for(var i=0; i<p.length;i++){
    if(p[i].match('[MLHVCSQTA]','i')){
    }else{
      if(isx){
        x = parseInt(p[i]);
      }else{
        y = parseInt(p[i]);
        res.push(new Array(x, y));
      }
      isx = !isx;
    }
  }
  return res;
}
 
function updatePath(d, x, y){
  var p = d.split(' ');
  var res = "";
  var isx=true;
  var val;
  for(var i=0; i<p.length;i++){
    if(trim(p[i]) == "") continue;
    if(p[i].match('[MLHVCSQTA]','i')){
      res += ' '+p[i]; 
    }else{
      if(isx){
        val = parseInt(p[i])+x;
      }else{
        val = parseInt(p[i])+y;
      }

      res += ' '+ val; 
      isx = !isx;
    }
  }
  return trim(res);
}

function translateObject(obj, dx, dy){
  switch(getElementTag(obj)){
    case 'text':
    case 'rect':
    case 'image':
      addAttributeVal(obj, "x", dx);
      addAttributeVal(obj, "y", dy);
      break;
    case 'circle':
    case 'ellipse':
      addAttributeVal(obj, "cx", dx);
      addAttributeVal(obj, "cy", dy);
      break;

    case 'path':
      var path = obj.getAttribute("d");
      obj.setAttribute("d", updatePath(path, dx, dy));
      break;
    case 'line':
      var x1 = getAttributeVal(obj,"x1"); 
      var y1 = getAttributeVal(obj,"y1"); 
      var x2 = getAttributeVal(obj,"x2"); 
      var y2 = getAttributeVal(obj,"y2"); 
      obj.setAttribute("x1", x1+dx); 
      obj.setAttribute("y1", y1+dy); 
      obj.setAttribute("x2", x2+dx); 
      obj.setAttribute("y2", y2+dy); 

      break;
    case 'g':
      var objs = obj.childNodes;
      for(var i=0;i<objs.length;i++){ translateObject(objs[i], dx, dy); }
      break;
    default:
      break;
  }
}

function updateTransform(obj){
  try{
    var trans = obj.getAttribute("transform");
    if(!trans) return;
    if(trans.match(new RegExp("translate(.+,.+)","i"))){
      var str = RegExp.lastMatch;
      var vals = str.substr(10,str.length-11).split(',') ;
      var dx = parseInt(vals[0]);
      var dy = parseInt(vals[1]);
      translateObject(obj, dx, dy);
      replaceTranslate(obj, 0, 0);
      updateRotate(obj);
    }
  }catch (e){
  }
}

function moveSelectedRectangle(dx, dy){
  if(!svg_select) return;
  svg_select.setAttribute("transform","translate("+dx+","+dy+")");
}

function updateSelectedRectangle(){
   if(!svg_select) return;
   if(selectedItems.length > 0){
     svg_select.setAttribute("visibility","visible");
   }else{
     hideSelectedRectangle();
   }
}

function clearSelectedItems(){
  targetItem=null;
  for(i in selectedItems) delete selectedItems[i];
  selectedItems=new Array();
  updateSelectedRectangle();
}


function replaceTranslate(obj, dx, dy){
  var trans = obj.getAttribute("transform");

  if(trans && trans.indexOf("translate")>=0){
    var strs = trans.split(' ');
    trans = "";
    for(var i=0; i<strs.length; i++){
      if(strs[i].indexOf("translate")>=0){
         if (dx == 0 && dy == 0){ strs[i]=""; }else{ strs[i] = "translate("+dx+","+dy+")"; }
      }
      
      trans += strs[i]+" ";
    }
    if(trim(trans)){ obj.setAttribute("transform", trim(trans));}
    else obj.removeAttribute("transform");
  }else{
    if(trans){
      trans += " translate("+dx+","+dy+")";
      obj.setAttribute("transform", trans);
    }else{
      obj.setAttribute("transform","translate("+dx+","+dy+")");
    }
  }
  updateRotate(obj,dx,dy);
}

function updateRotate(obj,dx,dy){
  var trans = obj.getAttribute("transform");

  if(!trans || trans.indexOf("rotate") < 0) return; 
  var bbox = obj.getBBox();
  var x = bbox.x+bbox.width/2;
  var y = bbox.y+bbox.height/2;
  var strs = trans.split(' ');
  trans = "";

  for(var i=0; i<strs.length ;i++){
    if(strs[i].indexOf("rotate") >= 0){
       var deg = strs[i].substr(7, strs[i].indexOf(",")-7);
       if(dx) x += dx;
       if(dy) y += dy;
       strs[i] = "rotate("+deg+","+x+","+y+")";
    }
    trans += strs[i]+" ";
  }
  obj.setAttribute("transform",trim(trans));
}

function getRotateAngle(obj){
  var trans = obj.getAttribute("transform");
  if(!trans || trans.indexOf("rotate") < 0) return 0; 
  var strs = trans.split(' ');
  for(var i=0; i<strs.length ;i++){
    if(strs[i].indexOf("rotate") >= 0){
       var deg = strs[i].substr(7, strs[i].indexOf(",")-7);
       return parseInt(deg);
    }
  }
  return 0;
}

function setRotate(obj,deg){
  var trans = obj.getAttribute("transform");
  var bbox = obj.getBBox();
  var x = bbox.x+bbox.width/2;
  var y = bbox.y+bbox.height/2;

  if(!trans) trans="";
  if(trans.indexOf("rotate") < 0){
    trans = "rotate("+deg+","+x+","+y+")";
    obj.setAttribute("transform", trans);
  }else{
    var strs = trans.split(' ');
    trans = "";
    for(var i=0; i<strs.length ;i++){
      if(strs[i].indexOf("rotate") >= 0){ strs[i] = "rotate("+deg+","+x+","+y+")"; }
      trans += strs[i]+" ";
    }
    obj.setAttribute("transform",trim(trans));
  }
}

function getScale(obj){
  var trans = obj.getAttribute("transform");
  if(!trans || trans.indexOf("scale") < 0) return null; 
  var strs = trans.split(' ');
  for(var i=0; i<strs.length ;i++){
    if(strs[i].indexOf("scale") >= 0){
       var degs = strs[i].substr(6, strs[i].indexOf(")")-6);
       var degs = degs.split(',');
       return new Array(parseFloat(degs[0]), parseFloat(degs[1]));
    }
  }
  return null;
}

function setScale(obj,scaleX, scaleY){
  var trans = obj.getAttribute("transform");
  var x = parseInt(svg_select.getAttribute("x"));
  var y = parseInt(svg_select.getAttribute("y"));
  var dx = x - scaleX*x;
  var dy = y - scaleY*y;

  if(!trans) trans="";
  if(trans.indexOf("scale") < 0){
    trans = "scale("+scaleX+","+scaleY+")";
    obj.setAttribute("transform", trans);
  }else{
    var strs = trans.split(' ');
    trans = "";
    for(var i=0; i<strs.length ;i++){
      if(strs[i].indexOf("scale") >= 0){ strs[i] = "scale("+scaleX+","+scaleY+")"; }
      if(scaleX == 1 && scaleY == 1) continue;
      trans += strs[i]+" ";
    }
    obj.setAttribute("transform",trim(trans));
  }
  replaceTranslate(obj, dx/scaleX, dy/scaleY);
}

function onTouchMoveCode1(pageX, pageY){
  if(targetItem || selectedItems.length > 0){
    switch(modeSVG){
      case 'newPath':
        if(targetItem.tagName == 'path' ){
          var path = targetItem.getAttribute("d");
          path = path + ' '+ getPreviewX(pageX) + ' '+ getPreviewY(pageY) ;
          targetItem.setAttribute("d",path);
        }
        break;
      case 'newLine':
        if(targetItem.tagName == 'line' ){
          var x2 = getPreviewX(pageX);
          var y2 = getPreviewY(pageY);
          targetItem.setAttribute("x2",x2);
          targetItem.setAttribute("y2",y2);
        }
        break;
      default:
        var dx = pageX - sx;
        var dy = pageY - sy;

        if(selectedItems.length == 1 && getElementTag(selectedItems[0]) == 'line'){
          var lx = getPreviewX(pageX);
          var ly = getPreviewY(pageY);

          if(!lineEdit){
            var x1 = getAttributeVal(selectedItems[0],"x1");
            var y1 = getAttributeVal(selectedItems[0],"y1");
            var x2 = getAttributeVal(selectedItems[0],"x2");
            var y2 = getAttributeVal(selectedItems[0],"y2");
            var xc = (x1+x2)/2;
            var yc = (y1+y2)/2;
            var eS = Math.min(Math.abs(x1-lx), Math.abs(y1-ly));
            var eC = Math.min(Math.abs(xc-lx), Math.abs(yc-ly));
            var eE = Math.min(Math.abs(x2-lx), Math.abs(y2-ly));

            var minVal = Math.min(eS, Math.min(eC, eE));
            if(minVal == eS) lineEdit='start';
            else if(minVal == eE) lineEdit='end';
            else lineEdit='move';
          }
          if(lineEdit=='start'){
            selectedItems[0].setAttribute("x1",lx );
            selectedItems[0].setAttribute("y1",ly );
            setSelectBox();
          }else if(lineEdit == 'end'){
            selectedItems[0].setAttribute("x2",lx );
            selectedItems[0].setAttribute("y2",ly );
            setSelectBox();
          }else{
            replaceTranslate(selectedItems[0], dx, dy);
            moveSelectedRectangle(dx, dy);
          }
        }else{
          for(var i=0; i<selectedItems.length;i++){
            if(selectedItems[i]){
              replaceTranslate(selectedItems[i], dx, dy);
            }
          }
          moveSelectedRectangle(dx, dy);
        }
        break;
    }
    updateShowMenu();
  }else if(modeSVG == 'selector'){
    ex = pageX;
    ey = pageY;
    var x1=sx; 
    var y1=sy; 
    var x2=ex; 
    var y2=ey; 

    if(sx > ex){ x1=ex; x2=sx; }
    if(sy > ey){ y1=ey; y2=sy; }

    selectedRectangle(getPreviewX(x1), getPreviewY(y1), getPreviewX(x2), getPreviewY(y2));
  }
}

function svgInputTextExec(e){
  var inputform = document.getElementById('svg_input');
  var color=document.getElementById('svg_color');
  var size=document.getElementById('svg_size');
  if(!color || !size) return;
  var y = parseInt(inputform.style.top) + parseInt(size.value)*0.8 + 12;
  var x = parseInt(inputform.style.left) + 12;
  if(inputform.value){
    targetItem=createSVGText(inputform.value, x, y, size.value, color.value);

    appendSVGObj(targetItem);
  }
  inputform.setAttribute("type", "hidden");
  setSVGMode('selector');
}

function svgModifyTextExec(e){
  var inputform = document.getElementById('svg_input');
  editingTextObj.textContent = inputform.value;
  inputform.setAttribute("type", "hidden");
  editingTextObj.style.display='block';

  editingTextObj=null;
  setSVGMode('selector');
}

function svgInputFormAdjust(e){
  var inputform = document.getElementById('svg_input');
  inputform.size = jstrlen(inputform.value) + 1;
}

function jstrlen(str) {
   var len = 0;
   str = escape(str);
   for (var i = 0; i < str.length; i++, len++) {
      if (str.charAt(i) == "%") {
         if (str.charAt(++i) == "u") {
            i += 3;
            len++;
         }
         i++;
      }
   }
   return len;
}

function putInputForm(x, y, txt, size, id){
  var inputform = document.getElementById('svg_input');
  if(!inputform){
    inputform = document.createElement('input');
    inputform.setAttribute("id", "svg_input");
    inputform.setAttribute("style", "position:absolute;top:0px;left:0px; border:0px none");
    preview.appendChild(inputform);
  }

  if(document.addEventListner){
    if(id){
      inputform.addEventListener("onChange", svgModifyTextExec,false);
    }else{
      inputform.addEventListener("onChange", svgInputTextExec,false);
    }
    inputform.addEventListener("onkeydown", svgInputFormAdjust,false);
  }else{
    if(id){
      inputform.setAttribute("onChange", "svgModifyTextExec()");
    }else{
      inputform.setAttribute("onChange", "svgInputTextExec()");
    }
    inputform.setAttribute("onkeydown", "svgInputFormAdjust()");
  }

  inputform.setAttribute("type", "text");
  inputform.style.left=x+'px';
  inputform.style.top=y+'px';
  inputform.style.fontSize=size+'px';
  inputform.style.background='none';
  inputform.value = txt;
  inputform.size = jstrlen(txt) + 1;
  inputform.focus();
}

///// EventHandler for iPad

function onTouchStart(e){
  //e.preventDefault();
  sx=e.touches[0].pageX;
  sy=e.touches[0].pageY;
  ex=e.touches[0].pageX;
  ey=e.touches[0].pageY;

  if (e.touches.length == 1){
    var touchtime = new Date();
    var dt = touchtime.getTime() - firstTouch.getTime();
    if(editingTextObj){
       var inputform = document.getElementById('svg_input');
       inputform.setAttribute("type", "hidden");
       editingTextObj.style.display='block';
       editingTextObj=null;
    }
    if(dt < 300 ){ onDoubleTap(); }else{ onTouchStartCore(); }
    firstTouch = touchtime;
  }
    updateSelectedRectangle();
}

function onTouchMove(e){
  if (e.touches.length == 1){
    e.preventDefault();
    onTouchMoveCode1(e.touches[0].pageX, e.touches[0].pageY);
  }else if (e.touches.length == 2){
    if(selectedItems.length == 1){
      e.preventDefault();
      var dx = e.touches[0].pageX-e.touches[1].pageX;
      var dy = e.touches[0].pageY-e.touches[1].pageY;
      var th = Math.abs(Math.atan2(dy , dx)/Math.PI *180);

      if(th > 165 ||  th < 25) svg_scale_dir = 'x';
      else if(th > 65 &&  th < 115) svg_scale_dir = 'y';
      else svg_scale_dir = null;
    }
  }
}

function onTouchEnd(e){
  e.preventDefault();

  for(var i=0; i<selectedItems.length;i++){
     updateTransform(selectedItems[i]);
  }

  if(modeSVG == 'selector'){
    var x1 = getPreviewX(sx);
    var y1 = getPreviewY(sy);
    var x2 = getPreviewX(ex);
    var y2 = getPreviewY(ey);
    getSelectedObjects(x1, y1, x2, y2);
  }

  updateSelectedRectangle();
  updateTransform(svg_select);

  setSelectBox();
  updateShowMenu();
  lineEdit=null;
}


function onGestureStart(e){
  targetItem = selectedItems[0];
  if(targetItem){
   svg_wo = targetItem.getAttribute("width");
   svg_ho = targetItem.getAttribute("height");
   svg_ro = targetItem.getAttribute("r");
   svg_rxo = targetItem.getAttribute("rx");
   svg_ryo = targetItem.getAttribute("ry");
   svg_fsize = targetItem.getAttribute("font-size");
 }else{
   svg_wo = null;
   svg_ho = null;
   svg_ro = null;
   svg_rxo = null;
   svg_ryo = null;
   svg_fsize = null;
 }
}
 
function scaleGesture(obj,scale){
  switch(getElementTag(obj)){
    case 'text':
      if(svg_fsize) obj.setAttribute("font-size", Math.round(svg_fsize*scale) );
      break;
    case 'rect':
    case 'image':
      if(svg_wo && svg_scale_dir != 'y') obj.setAttribute("width",Math.round(svg_wo*scale ));
      if(svg_ho && svg_scale_dir != 'x') obj.setAttribute("height",Math.round(svg_ho*scale ));
      break;
    case 'circle':
     if(svg_ro) obj.setAttribute("r",Math.round(svg_ro*scale ));
     break;
    case 'ellipse':
     if (svg_rxo && svg_scale_dir != 'y') obj.setAttribute("rx",Math.round(svg_rxo*scale) );
     if (svg_ryo && svg_scale_dir != 'x') obj.setAttribute("ry",Math.round(svg_ryo*scale) );
     break;
    case 'path':
    case 'line':
     var scaleX = scale;
     var scaleY = scale;
     if(svg_scale_dir == 'x') scaleY=1;
     else if(svg_scale_dir == 'y') scaleX=1;
     setScale(obj, scaleX, scaleY);

     break;
   defult:
     break;
  }
}

function updatePathAndLine(){
  if(getElementTag(targetItem) == 'path'){
    var scale = getScale(targetItem);
    if(scale){
      setScale(targetItem,1,1);
      scalePath(targetItem, scale);
    }
  }else if(getElementTag(targetItem) == 'line'){
    var scale = getScale(targetItem);
    if(scale){
      setScale(targetItem, 1,1);
      scaleLine(targetItem, scale);
    }
  }
}

function onGestureChange(e){
  var scale = e.scale;
  var rotation = e.rotation;

  if(targetItem){
    e.preventDefault();
    if(!svg_scale_locked) scaleGesture(targetItem,scale);
    if(!svg_rotate_locked) setRotate(targetItem, rotation);

    updateShowMenu();
  }
}


function onGestureEnd(e){
  svg_wo = null;
  svg_ho = null;
  svg_ro = null;
  svg_rxo = null;
  svg_ryo = null;
  svg_fsize = null;

  updatePathAndLine();
}

function setObjectColor(obj, color_sel, color){
  try{
    var tag = getElementTag(obj);
    if(tag == "NoName") return;
    if(tag == "g"){
      var objs=obj.childNodes;
      for(var i=0;i<objs.length;i++){
        setObjectColor(objs[i], color_sel, color);
      }
    }
    obj.setAttribute(color_sel, color);
    if(color_sel == 'fill'){
      if(fill_input) fill_input.value=color;
      document.getElementById('toolFill').style.backgroundColor=color;
    }else{
      color_input.value=color;
      document.getElementById('toolStroke').style.backgroundColor=color;
    }
  }catch(e){}
}

function onTouchStartColor(e){
  if(e.touches.length == 1){
    var ele = e.touches[0].target;
    var color;
    try{ 
      color = ele.getAttribute("color-val");
    }catch(err){  hideItemById('color-palette'); return;}
    var palette = document.getElementById('color-palette');
    var color_sel = palette.getAttribute("targetType");
    var fill_input = document.getElementById('svg_fill');
    var color_input = document.getElementById('svg_color');
    if(color) {
      for(var i=0;i < selectedItems.length; i++){
        setObjectColor(selectedItems[i], color_sel, color);
      }
    }
    palette.style.display='none'; }
}

///// For Safari
var mouse_state=0;

function onMouseDown(e){
  mouse_state=e.which;

  sx=e.pageX;
  sy=e.pageY;
  ex=e.pageX;
  ey=e.pageY;

  if(isChildById(e.target, 'preview') || e.target.id == 'preview') {
    if(editingTextObj){
       var inputform = document.getElementById('svg_input');
       inputform.setAttribute("type", "hidden");
       editingTextObj.style.display='block';
       editingTextObj=null;
    }

    if(e.altKey){
      targetItem = selectedItems[0];
      if(targetItem){
        svg_wo = targetItem.getAttribute("width");
        svg_ho = targetItem.getAttribute("height");
        svg_ro = targetItem.getAttribute("r");
        svg_rxo = targetItem.getAttribute("rx");
        svg_ryo = targetItem.getAttribute("ry");
        svg_fsize = targetItem.getAttribute("font-size");
      }else{
        svg_wo = null;
        svg_ho = null;
        svg_ro = null;
        svg_rxo = null;
        svg_ryo = null;
        svg_fsize = null;
      }
    }else{
      onTouchStartCore();
    }
    updateSelectedRectangle();
    return false;
  }else if(isChildById(e.target, 'color-palette') && e.target.tagName == 'DIV'){
    var ele = e.target;
    var palette = document.getElementById('color-palette');
    var color = ele.getAttribute("color-val");
    var color_sel = palette.getAttribute("targetType");
    var fill_input = document.getElementById('svg_fill');
    var color_input = document.getElementById('svg_color');
    if(color) {
      for(var i=0;i < selectedItems.length; i++){
        setObjectColor(selectedItems[i], color_sel, color);
      }
    palette.style.display='none';
    }
    return false;
  }
 return true;
}

function onMouseMove(e){
 if(!isChildById(e.target, 'preview') && e.target.id != 'preview') { return true; }

 if(mouse_state == 1) {
  if(e.altKey){
    var dx = sx-e.pageX;
    var dy = sy-e.pageY;
    var th = Math.abs(Math.atan2(dy , dx)/Math.PI *180);
    var scale;

    if(th > 165 ||  th < 25){
      svg_scale_dir = 'x';
      scale = -dx/50; 
    }else if(th > 65 &&  th < 115){
      svg_scale_dir = 'y';
      scale = -dy/50; 
    }else{
      svg_scale_dir = null;
      if(dx > 0 && dy > 0) scale = -dx*dy/2500; 
      else if(dx < 0 && dy < 0) scale = dx*dy/2500; 
      else scale = 1;
    }

   if(scale >= 0){
    if(scale < 1) scale=1;
   }else{
    if(scale > -1) scale=1;
    else scale= -1/scale;
   }
    if(targetItem && !svg_scale_locked){
      scaleGesture(targetItem,scale);
      updateShowMenu();
    }
  }else if(e.shiftKey){
    if(targetItem && !svg_rotate_locked){
      var dy = e.pageY -sy;
      var deg = dy % 360;
      setRotate(targetItem, deg);
      var ele = document.getElementById('svg_rotate');
      if(ele) ele.value=deg;
    }
  }else {
   onTouchMoveCode1(e.pageX, e.pageY);
  }
 }

 return false;
}

function onMouseUp(e){
  mouse_state = 0;
 if(!isChildById(e.target, 'preview') && e.target.id != 'preview') { return true; }

  updatePathAndLine() ;
  for(var i=0; i<selectedItems.length;i++){
     updateTransform(selectedItems[i]);
  }

  if(modeSVG == 'selector'){
    var x1 = getPreviewX(sx);
    var y1 = getPreviewY(sy);
    var x2 = getPreviewX(ex);
    var y2 = getPreviewY(ey);
    getSelectedObjects(x1, y1, x2, y2);
  }


  updateSelectedRectangle();
  updateTransform(svg_select);

  setSelectBox();
  updateShowMenu();
  lineEdit=null;

}

if(!navigator.userAgent.match("iPad")){
  document.onmousedown=onMouseDown;
  document.onmousemove=onMouseMove;
  document.onmouseup=onMouseUp;
  document.ondblclick=onDoubleTap;
}

