<?php
/*
 * This file is part of INQMAN
 *
 * Copyright(c) 2008 BULLHEAD,INC. ALL RIGHTS RESERVED.
 *
 * http://www.bullhead.co.jp/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation;
 * either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

require_once MOD_INQMAN_CONTROLLERS_DIR . '/InquiryController.php';

/**
 * 未受付問い合わせ管理画面のコントローラークラス
 *
 * @author 
 * @copyright COPYRIGHT &copy 2008 BULLHEAD,INC. ALL RIGHTS RESERVED.
 * @version 
 * @package 
 * @see InquiryController
 */
class Inqman_UnacceptController extends Inqman_InquiryController 
{
    
    /**
     * 受付待ち（or却下）状態の問い合わせ一覧を表示する。
     * 
     * @access public
     * 
     */
    public function viewlistAction()
    {
        //ロジックインスタンスの生成
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('view_list', 'unaccept');
        
        //リクエストパラメータからサービスIDを取得し、サービス情報を取得
        $service_id = $this->getRequest()->getParam(Inqman_InquiryForm::SERVICE_ID);
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_UNACCEPT, ACL_PRIVILEGE_KEY_READ, $service_id);
        
        //表示情報をVIEWに設定
        $this->view->assign('service', $service);
    }
    
    /**
     * 未処理問い合わせ情報をJSON形式で取得する
     * 
     */
    public function jsonAction()
    {
        // 出力にテンプレートを使わないので、NoRender
        $this->_helper->viewRenderer->setNoRender();
        
        // リクエストのフィルタリング。
        $f = new Zend_Filter_StripTags();
        $qtype      = $f->filter($this->_request->getParam('qtype'));
        $query      = $f->filter($this->_request->getParam('query'));
        $page       = $f->filter($this->_request->getParam('page',1));
        $rp         = $f->filter($this->_request->getParam('rp',10));
        $sortname   = $f->filter($this->_request->getParam('sortname', 'id'));
        $sortorder  = $f->filter($this->_request->getParam('sortorder', 'desc'));
        $service_id = $f->filter($this->_request->getParam('service_id'));
        
        $criteria  = $this->_request->getParam('crtr');
        $this->_logger->debug("service_id:{$service_id}, qtype:{$qtype}, query:{$query}, page:{$page}, rp:{$rp}, sortname:{$sortname}, sortorder:{$sortorder}");
        
        $service_model = new Inqman_ServiceModel();
        $inquiry_model = new Inqman_InquiryModel();
        
        if (!empty($service_id)) {
            $service = $service_model->getOneService($service_id);
            $userInfo = $this->_getUserInfo();
        } else {
            $services = $service_model->getEnableServiceOptions($this->_getUserInfo()->id);
            $service_id = array_keys($services);
        }
        
        $order = "{$sortname}_{$sortorder}";
        
        /**
         * 問い合わせ情報取得処理
         */
        $rows  = $inquiry_model->getUnacceptInquiries($service_id, $order, $page, $rp, $criteria);
        $count = $inquiry_model->getUnacceptInquiriesTotalCount($service_id, $criteria);
        
        // jsonの作成。
        $json = array(
                'page' => (int) $page,
                'total' => (int) $count,
        );

        foreach ($rows as $row) {
            switch ($row['status']) {
                case Inqman_InquiryModel::STATUS_CODE_UNACCEPT:
                case Inqman_InquiryModel::STATUS_CODE_REJECT:
                    $ctrlName = "unaccept";
                    break;
                default:
                    break;
            }
            
            $tmp = array();
            $tmp[] = $row['id'];
            $tmp[] = $this->view->escape($row['service_name']);
            $tmp[] = '<a href="'. $ctrlName . '/view_detail/inquiry_id/' . $row['id'] . '/">' . $row['subject'] . '</a>';
            $tmp[] = empty($row['received_from']) ? "--" : $this->view->escape($row['received_from']);
            $tmp[] = empty($row['content']) ? "--" : $this->view->escape($row['content']);
            $tmp[] = empty($row['create_datetime']) ? "--" : $this->view->escape($row['create_datetime']);
            $tmp[] = empty($row['registrant_name']) ? "--" : $this->view->escape($row['registrant_name']);
            $tmp[] = empty($row['accept_datetime']) ? "--" : $this->view->escape($row['accept_datetime']);
            $tmp[] = empty($row['operator_name'])   ? "--" : $this->view->escape($row['accepter_name']);
            $tmp[] = '<div style="width:100%;padding:0;margin:0;background-color:'. $this->_config->view->bgcolor->status->get($row['status']) .';border:solid 1px #999">' . $this->view->translate("inquiry.status.label." . $row['status']) . '</div>';
            $tmp[] = $this->view->escape($this->view->translate("inquiry.priority.label." . $row['priority']));
            $json['rows'][] = array('cell' => $tmp);
        }
        $this->_helper->json->sendJson($json);
    }
    
    /**
     * 受付待ち（or却下）状態の問い合わせ詳細情報を表示する。
     * 
     * @access public
     * 
     */
    public function viewdetailAction()
    {
        //ロジックインスタンスの生成
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('view_detail', 'unaccept');
        
        //リクエストパラメータから問い合わせIDを取得し、問い合わせ情報を取得
        $inquiry_id = $this->getRequest()->getParam(REQUEST_KEY_INQUIRY_ID);
        $inquiry = $inquiry_model->getOneInquiry($inquiry_id);
        
        //問い合わせ情報からサービスIDを取得し、サービス情報を取得
        $service_id = $inquiry['service_id'];
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_READ, $service_id);
        
        //問い合わせ情報を取得してVIEWに設定
        $this->view->assign(VIEW_KEY_FORM_INPUT, $inquiry);
        
        //フォームオブジェクトの生成
        $form = new Inqman_InquiryForm();
        
        //OPTION情報を取得
        $status_options = $inquiry_model->getStatusOptions();
        $priority_options = $inquiry_model->getPriorityOptions();
        
        //権限情報をVIEWに設定
        $this->_assignAclFlag($service_id, $inquiry_id);
        
        //表示情報をVIEWに設定
        $this->view->assign(VIEW_KEY_FORM, $form);
        $this->view->assign('service', $service);
        $this->view->assign('status_options', $status_options);
        $this->view->assign('priority_options', $priority_options);
    }
    
    
    /**
     * 受付（起票）確認画面を表示する。
     * 
     */
    public function acceptconfirmAction()
    {
        //ロジックインスタンスの生成
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('accept_confirm', 'unaccept');
        
        //リクエストパラメータから問い合わせIDを取得し問い合わせ情報を取得
        $inquiry_id = $this->getRequest()->getParam(REQUEST_KEY_INQUIRY_ID);
        $data = $inquiry_model->getOneInquiry($inquiry_id);
        
        //問い合わせ情報からサービスIDを取得しサービス情報を取得
        $service_id = $data[Inqman_InquiryModel::SERVICE_ID];
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_ACCEPT, $service_id);
        
        //起票者名/IDに、現ログインユーザ名/IDをセット
        $user_info = $this->_getUserInfo();
        $data['accepter_id']   = $user_info->id;
        $data['accepter_name'] = $user_info->name;
        
        //フォーム入力値の検証
        $form = new Inqman_InquiryForm();
        $error_messages = null;
        $tmp_names = null;
        if(!$form->isValid($_POST)) {
            $error_messages = $form->getMessages();
        }
        
        //添付ファイルの検証と保存
        $attach_files = null;
        $attach_error_messages = $this->_validateUploadFiles($_FILES, REQUEST_KEY_ATTACH_FILES);
        if (!empty($attach_error_messages)) {
            $error_messages = array_merge($error_messages, $attach_error_messages);
        } else {
            $attach_files = $this->_saveUploadFiles($_FILES, REQUEST_KEY_ATTACH_FILES);
        }
        
        //エラーメッセージの有無で表示するページを振り分け
        if(!empty($error_messages)) {
            $form_input = array_merge($data, $form->getUnfilteredValues());
            $this->_helper->viewRenderer->setRender('view_detail');
        } else {
            $form_input = array_merge($data, $form->getValues());
            $form_input[REQUEST_KEY_ATTACH_FILES] = $attach_files;
        }
        
        //OPTION情報を取得
        $status_options = $inquiry_model->getStatusOptions();
        $priority_options = $inquiry_model->getPriorityOptions();
        
        //権限情報をVIEWに設定
        $this->_assignAclFlag($service_id, $inquiry_id);
        
        //表示情報をVIEWに設定
        $this->view->assign(VIEW_KEY_FORM_INPUT, $form_input);
        $this->view->assign('service', $service);
        $this->view->assign('status_options', $status_options);
        $this->view->assign('priority_options', $priority_options);
        $this->view->assign(VIEW_KEY_ERROR_MESSAGES, $error_messages);
    }
    
    /**
     * 受付（起票）処理を実行する。
     * 
     * @access public
     * 
     */
    public function acceptcommitAction()
    {
        //ロジックインスタンスの生成
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //リクエストパラメータから問い合わせIDを取得し、問い合わせ情報を取得
        $inquiry_id = $this->getRequest()->getParam(REQUEST_KEY_INQUIRY_ID);
        $data = $inquiry_model->getOneInquiry($inquiry_id);
        
        //問い合わせ情報からサービスIDを取得し、サービス情報を取得
        $service_id = $data[Inqman_InquiryModel::SERVICE_ID];
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_ACCEPT, $service_id);
        
        //入力値の再検証を実施し、正常なら登録処理を行う
        $form = new Inqman_InquiryForm();
        if ($form->isValid($_POST)) {
            $values = $form->getValues();
            
            //添付ファイルを登録する
            $attach_files = !empty($_POST[REQUEST_KEY_ATTACH_FILES]) ? (array) $_POST[REQUEST_KEY_ATTACH_FILES] : array();
            $enced_attach_files = $this->_encodeUploadFiles($attach_files);
            $inquiry_model->registerAttachFiles($inquiry_id, $enced_attach_files);
            
            $userInfo = $this->_getUserInfo();
            $values['accepter_id'] = $userInfo->id;
            
            $inquiry_model->acceptInquiry($inquiry_id, $values);
        
            //承認フローが無効の場合は、同時に承認処理も行う
            if (intval($this->_config->workflow->approve->inquiry->accept) != 1) {
                $inquiry_model->approveInquiry($inquiry_id, $values);
            }
            
            $event_action_id = 4;
            $criteria = array('service_id'=>$service_id, 'inquiry_id'=>$inquiry_id);
            $inquiry_model->sendMail($event_action_id, $criteria);
            
        } else {
            //不正入力があった場合は、確認画面のアクションを経て入力画面に戻す
            $this->acceptconfirmAction();
        }
        
        //表示情報をVIEWに設定する
        $this->view->assign('service_id', $service_id);
    }

}
