/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkldictimpl.c,v 1.8 2004/04/07 01:55:45 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <string.h>
#include <fcntl.h>
#include <glib.h>
#include "skkldictimpl.h"
#include "skkdictitem.h"
#include "skktypes.h"

static SkkQueryFunctionImpl *get_new (SkkQuery * query);
static GList *do_query (SkkQueryFunctionImpl * impl, const gchar * buf,
			const gchar * okuri, gint * found_num);
static void destroy (SkkQueryFunctionImpl * impl);
static gchar *get_info (void);
static gint get_type (void);
static GList *get_arg (void);
static void set_arg (GList * arg);
static GList *completion (SkkQueryFunctionImpl * impl, const gchar * buf);
static void add (SkkQueryFunctionImpl * impl, const gchar * key,
		 const gchar * okuri, const gchar * value);

static SkkQueryFunctionImpl ldictimpl = {
  get_new,
  NULL,
  NULL,
  NULL,
  do_query,
  completion,
  add,
  destroy,
  get_info,
  get_type,
  get_arg,
  set_arg,
};

static gboolean ldict_init (SkkQueryFunctionImpl * impl);

static void
add_word (SkkQueryFunctionImpl * impl, const gchar * key, const gchar * value)
{
  SkkLdictImpl *l = (SkkLdictImpl *) impl;
  DBT db_key, db_data;
  GList *list;
  GList *result_list = NULL;
  int err;
  gchar *result;
  if (!key || !value)
    {
      return;
    }
  if (!l->initialized)
    ldict_init (impl);

  memset (&db_key, 0, sizeof (DBT));
  memset (&db_data, 0, sizeof (DBT));
  db_key.size = strlen (key) + 1;
  db_key.data = (void *) key;
  result_list =
    g_list_append (result_list, skk_dict_item_new_with_string (value, "*;"));
  list = impl->do_query (impl, key, NULL, NULL);
  if (list)
    {
      result_list = skk_dict_item_merge_list (result_list, list);
    }
  result = skk_dict_item_to_string_all (result_list, "/", "*;");
  db_data.size = strlen (result) + 1;
  db_data.data = (void *) result;
  if ((err = l->ldict->put (l->ldict, NULL, &db_key, &db_data, 0) != 0))
    {
      fprintf (stderr, "iiimf-skk: add_word %s\n", db_strerror (err));
    }
  skk_dict_item_destroy_all (result_list);
  g_free (result);
  return;
}

static void
add_word_okuri (SkkQueryFunctionImpl * impl, const gchar * key,
		const gchar * okuri, const gchar * value)
{
  SkkLdictImpl *l = (SkkLdictImpl *) impl;
  DBT db_key, db_data;
  GList *list;
  GList *result_list = NULL;
  gchar *result;
  gchar *real_key;
  int err;
  if (!key || !value)
    return;
  if (!l->initialized)
    {
      ldict_init (impl);
    }
  memset (&db_key, 0, sizeof (DBT));
  memset (&db_data, 0, sizeof (DBT));
  result_list =
    g_list_append (result_list, skk_dict_item_new_with_string (value, "*;"));
  list = impl->do_query (impl, key, okuri, NULL);
  if (list)
    {
      result_list = skk_dict_item_merge_list (result_list, list);
    }
  result = skk_dict_item_to_string_all (result_list, "/", "*;");
  /* TODO store db */
  real_key = g_strdup_printf ("%s%c", key, okuri[0]);
  db_key.size = strlen (real_key) + 1;
  db_key.data = (void *) real_key;
  db_data.size = strlen (result) + 1;
  db_data.data = (void *) result;
  if ((err =
       l->ldict_okuri->put (l->ldict_okuri, NULL, &db_key, &db_data, 0) != 0))
    {
      fprintf (stderr, "iiimf-skk: add_buffer_okuri %s\n", db_strerror (err));
    }
  skk_dict_item_destroy_all (result_list);
  g_free (real_key);
  g_free (result);
  return;
}

static const gchar *
get_word (SkkLdictImpl * impl, const gchar * key)
{
  SkkLdictImpl *l = (SkkLdictImpl *) impl;
  DBT db_key, db_data;
  int err;
  if (!key)
    return NULL;
  memset (&db_key, 0, sizeof (DBT));
  memset (&db_data, 0, sizeof (DBT));
  db_key.size = strlen (key) + 1;
  db_key.data = (void *) key;
  if ((err = l->ldict->get (l->ldict, NULL, &db_key, &db_data, 0)) != 0)
    {
      if (err != DB_NOTFOUND)
	fprintf (stderr, "iiimf-skk: get_word %s\n", db_strerror (err));
      return NULL;
    }
  return (const gchar *) db_data.data;
}

static const gchar *
get_word_okuri (SkkLdictImpl * impl, const gchar * key)
{
  SkkLdictImpl *l = (SkkLdictImpl *) impl;
  DBT db_key, db_data;
  int err;
  if (!key)
    return NULL;
  memset (&db_key, 0, sizeof (DBT));
  memset (&db_data, 0, sizeof (DBT));
  db_key.size = strlen (key) + 1;
  db_key.data = (void *) key;
  if ((err =
       l->ldict_okuri->get (l->ldict_okuri, NULL, &db_key, &db_data, 0)) != 0)
    {
      if (err != DB_NOTFOUND)
	fprintf (stderr, "iiimf-skk: get_word_okuri %s\n", db_strerror (err));
      return NULL;
    }
  return (const gchar *) db_data.data;
}

static GList *
do_query (SkkQueryFunctionImpl * impl, const gchar * buf, const gchar * okuri,
	  gint * found_num)
{
  SkkLdictImpl *l = (SkkLdictImpl *) impl;
  GList *ret = NULL;
  gchar *query_str;
  const gchar *query_result;
  if (!buf)
    return NULL;
  if (!l->initialized)
    {
      ldict_init (impl);
    }
  if (okuri)
    {
      query_str = g_strdup_printf ("%s%c", buf, okuri[0]);
      query_result = get_word_okuri (l, query_str);
    }
  else
    {
      query_str = g_strdup (buf);
      query_result = get_word (l, query_str);
    }
  if (query_result)
    {
      ret = skk_dict_item_new_from_line (query_result, "/", "*;");
    }
  else
    {
      g_free (query_str);
      if (found_num)
	*found_num = 0;
      return NULL;
    }
  if (found_num)
    *found_num = g_list_length (ret);
  g_free (query_str);
  return ret;
}

static GList *
completion (SkkQueryFunctionImpl * impl, const gchar * buf)
{
  SkkLdictImpl *l = (SkkLdictImpl *) impl;
  DBT db_key, db_data;
  DBC *db_cursor;
  int err;
  GList *ret = NULL;
  if (!buf)
    return NULL;
  if (!l->initialized)
    {
      ldict_init (impl);
    }
  memset (&db_key, 0, sizeof (DBT));
  memset (&db_data, 0, sizeof (DBT));
  l->ldict->cursor (l->ldict, NULL, &db_cursor, 0);
  if ((err = db_cursor->c_get (db_cursor, &db_key, &db_data, DB_LAST) != 0))
    {
      if (err != DB_NOTFOUND)
	{
	  fprintf (stderr, "iiimf-skk: get_completion %s\n",
		   db_strerror (err));
	}
      db_cursor->c_close (db_cursor);
      return NULL;
    }
  if (!strncmp (buf, (const char *) db_key.data, strlen (buf)))
    {
      ret = g_list_append (ret, g_strdup (db_key.data));
    }
  while (TRUE)
    {
      if ((err =
	   db_cursor->c_get (db_cursor, &db_key, &db_data, DB_PREV) != 0))
	{
	  if (err != DB_NOTFOUND)
	    fprintf (stderr, "iiimf-skk: get_completion %s\n",
		     db_strerror (err));
	  break;
	}
      if (!strncmp (buf, (const char *) db_key.data, strlen (buf)))
	{
	  ret = g_list_append (ret, g_strdup (db_key.data));
	}
    }
  db_cursor->c_close (db_cursor);
  return ret;
}

static SkkQueryFunctionImpl *
get_new (SkkQuery * query)
{
  SkkLdictImpl *impl;
  impl = g_new0 (SkkLdictImpl, 1);
  impl->impl = ldictimpl;
  impl->file_ldict = skk_query_get_conf_string (query, "skk-jisyo");
  impl->file_ldict_okuri =
    skk_query_get_conf_string (query, "skk-okuri-jisyo");
  impl->ldict = NULL;
  impl->ldict_okuri = NULL;
  impl->initialized = FALSE;
  return (SkkQueryFunctionImpl *) impl;
}

static gint
get_type (void)
{
  return SKKQUERY_LOCAL;
}

static gchar *
get_info (void)
{
  return NULL;
}

static void
destroy (SkkQueryFunctionImpl * impl)
{
  SkkLdictImpl *l = (SkkLdictImpl *) impl;
#ifdef DB_HAVE_TRUNCATE
  u_int32_t countp;
#endif
  /* TODO */
  if (l->initialized)
    {
#ifdef DB_HAVE_TRUNCATE
      if (!l->file_ldict)
	{
	  l->ldict->truncate (l->ldict, NULL, &countp, 0);
	}
#endif
      l->ldict->close (l->ldict, 0);
#ifdef DB_HAVE_TRUNCATE
      if (!l->file_ldict_okuri)
	{
	  l->ldict_okuri->truncate (l->ldict_okuri, NULL, &countp, 0);
	}
#endif
      l->ldict_okuri->close (l->ldict_okuri, 0);
      l->initialized = FALSE;
    }
  if (l->file_ldict)
    g_free (l->file_ldict);
  if (l->file_ldict_okuri)
    g_free (l->file_ldict_okuri);
  g_free (l);
  g_message ("skk ldict destroy");
  return;
}

static GList *
get_arg (void)
{
  return NULL;

}

static void
set_arg (GList * arg)
{
  return;
}

static gboolean
ldict_init (SkkQueryFunctionImpl * impl)
{
  SkkLdictImpl *l = (SkkLdictImpl *) impl;
  int err;

  if ((err = db_create (&(l->ldict), NULL, 0) != 0))
    {
    }
  l->ldict->set_errfile (l->ldict, stderr);
  l->ldict->set_errpfx (l->ldict, "iiimf-skk");
#ifdef DB_NEW_OPEN_API
  if ((err = l->ldict->open (l->ldict, NULL, l->file_ldict, NULL,
			     DB_HASH, DB_CREATE , 0600)))
    {
#else
  if ((err = l->ldict->open (l->ldict, l->file_ldict, NULL,
			     DB_HASH, DB_CREATE , 0600)))
    {
#endif

      printf ("ldict_init %s\n", db_strerror (err));
      if (l->file_ldict)
	{
	  g_free (l->file_ldict);
	  l->file_ldict = NULL;
#ifdef DB_NEW_OPEN_API
	  l->ldict->open (l->ldict, NULL, NULL, NULL, DB_HASH, DB_CREATE,
			  0600);
#else
	  l->ldict->open (l->ldict, NULL, NULL, DB_HASH, DB_CREATE , 0600);
#endif
	}
    }

  if ((err = db_create (&(l->ldict_okuri), NULL, 0) != 0))
    {
    }
  l->ldict_okuri->set_errfile (l->ldict_okuri, stderr);
  l->ldict_okuri->set_errpfx (l->ldict_okuri, "iiimf-skk");
#ifdef DB_NEW_OPEN_API
  if ((err =
       l->ldict_okuri->open (l->ldict_okuri, l->file_ldict_okuri, NULL, NULL,
			     DB_HASH, DB_CREATE , 0644)))
    {
#else
  if ((err =
       l->ldict_okuri->open (l->ldict_okuri, l->file_ldict_okuri, NULL,
			     DB_HASH, DB_CREATE , 0644)))
    {
#endif
      printf ("ldict_init %s\n", db_strerror (err));
      if (l->file_ldict_okuri)
	{
	  g_free (l->file_ldict_okuri);
	  l->file_ldict_okuri = NULL;
#ifdef DB_NEW_OPEN_API
	  l->ldict_okuri->open (l->ldict_okuri, NULL, NULL, NULL, DB_HASH,
				DB_CREATE , 0600);
#else
	  l->ldict_okuri->open (l->ldict_okuri, NULL, NULL, DB_HASH,
				DB_CREATE , 0600);
#endif
	}
    }

  l->initialized = TRUE;
  return TRUE;
}

void
add (SkkQueryFunctionImpl * impl, const gchar * key, const gchar * okuri,
     const gchar * value)
{
  if (!key)
    return;
  if (!value)
    return;
  if (!okuri)
    add_word (impl, key, value);
  else
    add_word_okuri (impl, key, okuri, value);
  return;
}

SkkQueryFunctionImpl *
skk_ldict_new (SkkQuery * query)
{
  return get_new (NULL);
}

SkkQueryFunctionImpl *
impl_init (SkkQuery * query)
{
  return get_new (query);
}

#ifdef SKKLDICTIMPL_DEBUG
#include "skkutils.h"

int
main (void)
{
  SkkQueryFunctionImpl *ldict;
  GList *comp;
  GList *item;
  ldict = skk_ldict_new ();
  ldict->add (ldict, "", NULL, "դۤ");
  ldict->add (ldict, "", NULL, "դդ");
  ldict->add (ldict, "", NULL, "դۤ");
  ldict->add (ldict, "", NULL, "դۤ");
  ldict->add (ldict, "", NULL, "դۤ");
  comp = ldict->completion (ldict, "");
  g_message ("completion %s", skk_utils_list_to_string_all (comp, "/"));
  item = ldict->do_query (ldict, "", NULL, NULL);
  g_message ("do_query %s", skk_dict_item_to_string_all (item, "/", "*;"));
  comp = ldict->completion (ldict, "");
  return 0;
}
#endif
