/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 *
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkquery.c,v 1.2.2.14 2002/10/28 11:15:39 famao Exp $ */

/* vi:set ts=4 sw=4: */

#include <dirent.h>
#include <string.h>
#include "config.h"
#include "skkquery.h"
#include "skkqueryimpl.h"
#include "skkutils.h"

#if 0
static gboolean initialized = FALSE;
static GList *plugin_list = NULL;
#endif

static SkkQueryPlugin *
plugin_new (SkkQuery *query, const gchar *filename)
{
	SkkQueryPlugin *plugin;
	SkkQueryFunctionImpl *impl;
	if (!filename)
		return NULL;
	plugin = g_new0 (SkkQueryPlugin, 1);
	plugin->handle = g_module_open (filename, 0);
	if (!plugin->handle) {
		g_free (plugin);
		return NULL;
	}
	plugin->file = g_strdup (filename);
	if (!g_module_symbol (plugin->handle, "impl_init",
				(gpointer *)&(plugin->impl_init))) {
		g_module_close (plugin->handle);
		g_free (plugin->file);
		g_free (plugin);
		return NULL;
	}
	impl = plugin->impl_init (query);
	plugin->type = impl->get_type ();
	plugin->info = impl->get_info ();
	impl->destroy (impl);
	plugin->num_use = 0;
	return plugin;
}

static void
query_init (SkkQuery *query, const gchar *path)
{
	SkkQueryPlugin *plugin;
	DIR *d;
	struct dirent *dent;
	if (!query)
		return;
	if ((d =  opendir (path ? path : PACKAGE_PLUGIN_DIR)) == NULL) {
		return;
	}
	/* FIXME use g_module_build_path ? */
	while ((dent = readdir (d)) != NULL) {
		if (!strncmp (dent->d_name + strlen (dent->d_name) - 3, ".so", 3)) {
			gchar *file;
			if (path) {
				file = g_strconcat (path, "/", dent->d_name, NULL);
			} else {
				file = g_strconcat (PACKAGE_PLUGIN_DIR, "/", dent->d_name, NULL);
			}
			plugin = plugin_new (query, file);
			if (plugin) {
				query->plugin_list = g_list_append (query->plugin_list, plugin);
			}
			g_free (file);
		}
	}
	closedir (d);
	query->initialized = TRUE;
	return;
}

static void
query_finalize_func  (gpointer free_item, gpointer user_data)
{
	SkkQueryPlugin *plugin = (SkkQueryPlugin *)free_item;
	if (!plugin)
		return;
	if (plugin->file)
		g_free (plugin->file);
	if (plugin->handle)
		g_module_close (plugin->handle);
	g_free (plugin);
}

static void
query_finalize (SkkQuery *query)
{
	if (!query || !query->plugin_list)
		return;
	skk_utils_list_free (query->plugin_list, TRUE, query_finalize_func, NULL);
	query->plugin_list = NULL;
	query->initialized = FALSE;
}

SkkQuery *
skk_query_new (void)
{
	SkkQuery *ret;
	ret = g_new (SkkQuery, 1);
	memset (ret, 0, sizeof (SkkQuery));
	ret->plugin_list = NULL;
	ret->initialized = FALSE;
	ret->query_list = NULL;
	ret->ref_count++;
	ret->cur_item = NULL;
	ret->conf = NULL;
	query_init (ret, NULL);
	return ret;
}

SkkQuery *
skk_query_new_with_path (const gchar *path)
{
	SkkQuery *ret;
	ret = g_new0 (SkkQuery, 1);
	ret->plugin_list = NULL;
	ret->initialized = FALSE;
	ret->query_list = NULL;
	ret->ref_count++;
	ret->cur_item = NULL;
	ret->conf = NULL;
	query_init (ret, path);
	return ret;
}

SkkQueryItem *
skk_query_item_new (SkkQuery *query, SkkQueryType type)
{
	SkkQueryItem *item = NULL;
	GList *l;
	if (!query)
		return NULL;
	if (!query->initialized)
		query_init (query, NULL);
	for (l = query->plugin_list; l ; l = g_list_next (l)) {
		SkkQueryPlugin *p = l->data;
		if (p->type == type) {
			item = g_new0 (SkkQueryItem, 1);
			item->type = type;
			item->impl = p->impl_init (query);
			p->num_use++;
			break;
		}
	}
	return item;
}

static void
query_item_destroy_func  (gpointer free_item, gpointer user_data)
{
	SkkQueryItem *item = (SkkQueryItem *)free_item;
	if (!item)
		return;
	if (item->impl) {
		item->impl->destroy (item->impl);
		item->impl = NULL;
	}
	g_free (item);
	return;
}

void
skk_query_destroy (SkkQuery *query)
{
	if (!query)
		return;
	query->ref_count--;
	if (query->ref_count > 0)
		return;
	if (query->query_list) {
		skk_utils_list_free (query->query_list, TRUE, query_item_destroy_func , NULL);
		query->query_list = NULL;
	}
	if (query->conf) {
		skk_conf_unref (query->conf);
	}
	query_finalize (query);
	g_free (query);
	return;
}

void
skk_query_add_item (SkkQuery *query, SkkQueryItem *item)
{
	if (!query)
		return;
	if (!item)
		return;
	query->query_list = g_list_append (query->query_list, (gpointer)item);
	return;
}

gboolean
skk_query_set_with_type (SkkQuery *query, SkkQueryType type)
{
	GList *l;
	SkkQueryItem *item;
	if (!query)
		return FALSE;
	query->cur_list_num = 0;
	for (l = query->query_list; l ; l = g_list_next (l)) {
		item = l->data;
		if (item && item->type == type) {
			query->cur_item = item;
			return TRUE;
		}
		query->cur_list_num++;
	}
	return FALSE;
}

void
skk_query_remove_item (SkkQuery *query, SkkQueryType type)
{
	GList *l;
	SkkQueryItem *item;
	/* FIXME */
	if (!query)
		return;
	for (l = query->query_list; l ; l = g_list_next (l)) {
		item = l->data;
		if (item && item->type == type) {
			query_item_destroy_func (item, NULL);
			query->query_list = g_list_remove_link (query->query_list, l);
			g_list_free_1 (l);
			break;
		}
	}
	return;
}

gboolean
skk_query_set_next (SkkQuery *query)
{
	if (!query)
		return FALSE;
	query->cur_list_num++;
	query->cur_item = g_list_nth_data (query->query_list, query->cur_list_num);
	if (query->cur_item)
		return TRUE;
	return FALSE;
}

gboolean
skk_query_set_nth (SkkQuery *query, gint nth)
{
	SkkQueryItem *tmp_item;
	if (!query)
		return FALSE;
	tmp_item = query->cur_item;	
	query->cur_item = (SkkQueryItem *)g_list_nth_data (query->query_list, nth);
	if (query->cur_item) {
		query->cur_list_num = nth;
		return TRUE;
	}
	query->cur_item = tmp_item;
	return FALSE;
}	

gboolean
skk_query_set_prev (SkkQuery *query)
{
	if (!query)
		return FALSE;
	if (query->cur_list_num == 0)
		return FALSE;
	query->cur_list_num--;
	query->cur_item = g_list_nth_data (query->query_list, query->cur_list_num);
	if (query->cur_item)
		return TRUE;
	return FALSE;
}

GList *
skk_query_do_completion (SkkQuery *query, const gchar *buf)
{
	if (!query)
		return NULL;
	if (!query->cur_item) {
		query->cur_item = (SkkQueryItem *)g_list_nth_data (query->query_list, 0);
	}
	if (!query->cur_item)
		return NULL;
	if (!query->cur_item->impl || !query->cur_item->impl->completion)
		return NULL;
	return query->cur_item->impl->completion (query->cur_item->impl, buf);
}

GList *
skk_query_do_query (SkkQuery *query, const gchar *buf, const gchar *okuri, gint *found_num)
{
	if (!query)
		return NULL;
	if (!query->cur_item) {
		query->cur_item = (SkkQueryItem *)g_list_nth_data (query->query_list, 0);
	}
	if (!query->cur_item)
		return NULL;
	if (!query->cur_item->impl || !query->cur_item->impl->do_query)
		return NULL;
	return query->cur_item->impl->do_query (query->cur_item->impl, buf, okuri, found_num);
}

void
skk_query_add (SkkQuery *query, const gchar *key, const gchar *okuri, const gchar *value)
{
	if (!query)
		return;
	if (!query->cur_item) {
		query->cur_item = (SkkQueryItem *)g_list_nth_data (query->query_list, 0);
	}
	if (!query->cur_item)
		return;
	if (!query->cur_item->impl || !query->cur_item->impl->add)
		return;
	query->cur_item->impl->add (query->cur_item->impl, key, okuri, value);
	return;
}

void
skk_query_ref (SkkQuery *query)
{
	if (!query)
		return;
	query->ref_count++;
	return;
}

void
skk_query_unref (SkkQuery *query)
{
	if (!query)
		return;
	query->ref_count--;
	if (!query->ref_count) {
		skk_query_destroy (query);
	}
	return;
}

void
skk_query_set_conf (SkkQuery *query, SkkConf *conf)
{
	if (!query)
		return;
	if (query->conf) {
		skk_conf_unref (conf);
	}
	query->conf = conf;
	skk_conf_ref (query->conf);
}

gchar *
skk_query_get_conf_string_by_name (SkkQuery *query, const gchar *name)
{
	if (!query)
		return NULL;
	if (!query->conf)
		return NULL;
	return skk_conf_get_string_by_name (query->conf, name);
}

gint
skk_query_get_conf_num_by_name (SkkQuery *query, const gchar *name)
{
	if (!query)
		return -1;
	if (!query->conf)
		return -1;
	return skk_conf_get_num_by_name (query->conf, name);
}

gboolean
skk_query_get_conf_bool_by_name (SkkQuery *query, const gchar *name)
{
	if (!query)
		return FALSE;
	if (!query->conf)
		return FALSE;
	return skk_conf_get_bool_by_name (query->conf, name);
}
