/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2003 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkconf.c,v 1.3.2.52 2003/02/12 11:36:08 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <glib.h>
#include <string.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "skkutils.h"
#include "skkconf.h"
#include "skkconf_private.h"

static void clear (SkkConf *conf);

/* Parse Item */
static void parse_item_one (SkkConf *conf, xmlNodePtr node);
static void parse_item (SkkConf *conf, xmlDocPtr doc);
/* Parse Rule */
static void parse_rule_one (SkkConf *conf, xmlNodePtr node);
static void parse_rule (SkkConf *conf, xmlDocPtr doc);
static void parse_rule_delete (SkkConf *conf, xmlDocPtr doc);

static SkkConfType find_type (gchar *str);
static SkkConfItemType find_item_type (const gchar *str);
static SkkConfItem *find_item (GList *list, SkkConfItemType type);

/* Add Item to List */
static void add_item_bool (SkkConf *conf, SkkConfItem *item);
static void add_item_string (SkkConf *conf, SkkConfItem *item);
static void add_item_num (SkkConf *conf, SkkConfItem *item);
static void add_defaults (SkkConf* conf);

/* Find Item */
static gboolean find_item_func (gconstpointer p1, gconstpointer p2);
static gint find_item_each_for_delete (gconstpointer p1, gconstpointer p2);
static GList *find_item_for_delete (GList *list, SkkConfItem *item);

/* Load Configuration */
static void load_rc (SkkConf *conf, const gchar *filename);

/* Event */
static void add_rule_emit (SkkConf *conf, const gchar *key, const gchar *hira,
		const gchar *kata, const gchar *append);
static void delete_rule_emit (SkkConf *conf, const gchar *key);

typedef struct _SkkConfAddRuleFunc SkkConfAddRuleFunc;
typedef struct _SkkConfDeleteRuleFunc SkkConfDeleteRuleFunc;

struct _SkkConfAddRuleFunc {
	SkkConfAddRuleListener listener;
	gpointer user_data;
};

struct _SkkConfDeleteRuleFunc {
	SkkConfDeleteRuleListener listener;
	gpointer user_data;
};

static void
add_rule_emit (SkkConf *conf, const gchar *key, const gchar *hira,
		const gchar *kata, const gchar *append)
{
	GSList *tmp_list;
	SkkConfAddRuleFunc *func;
	if (!conf)
		return;
	for (tmp_list = conf->conf_add_rule_listener;
			tmp_list; tmp_list = g_slist_next (tmp_list)) {
		func = (SkkConfAddRuleFunc *)tmp_list->data;
		if (func) {
			func->listener (conf, key, hira, kata, append, func->user_data);
		}
	}
	return;
}

static void
delete_rule_emit (SkkConf *conf, const gchar *key)
{
	GSList *tmp_list;
	SkkConfDeleteRuleFunc *func;
	if (!conf)
		return;
	for (tmp_list = conf->conf_delete_rule_listener;
			tmp_list; tmp_list = g_slist_next (tmp_list)) {
		func = (SkkConfDeleteRuleFunc *)tmp_list->data;
		if (func) {
			func->listener (conf, key, func->user_data);
		}

	}
	return;
}

static gboolean
find_item_func (gconstpointer p1, gconstpointer p2)
{
	SkkConfItem *item;
	if (!p1)
		return TRUE;
	if (!p2)
		return TRUE;
	item = (SkkConfItem *)p1;
	if (GPOINTER_TO_INT (p2) == item->itemtype) {
		return FALSE;
	}
	return TRUE;
	
}

static gint
find_item_each_for_delete (gconstpointer a, gconstpointer b)
{
	if (!a)
		return 1;
	if (!b)
		return 1;
	if (((SkkConfItem *)a)->itemtype == ((SkkConfItem *)b)->itemtype)
		return 0;
	return 1;
}

static GList *
find_item_for_delete (GList *list, SkkConfItem *item)
{
	GList *ret = NULL;
	if (!list)
		return NULL;
	ret = g_list_find_custom (list, item, find_item_each_for_delete);
	if (ret) {
		return ret;
	}
	return NULL;
}

static void
add_defaults (SkkConf* conf)
{
	gint i;
	for (i = 0; i < SKKCONF_DEFAULT_ITEM_SIZE ; i++) {
		skk_conf_add_item (conf, &itemp[i]);
	}
}

static void
add_item_bool (SkkConf *conf, SkkConfItem *item)
{
	GList *found;
	if (item->user_defined) {
		found = find_item_for_delete (conf->conf_bool, item);
		if (found) {
			skk_conf_item_destroy ((SkkConfItem*)found->data);
			conf->conf_bool = g_list_remove_link (conf->conf_bool, found);
			g_list_free_1 (found);
		}
	}
	conf->conf_bool = g_list_prepend (conf->conf_bool, item);
	return;
}

static void
add_item_string (SkkConf *conf, SkkConfItem *item)
{
	GList *found;
	if (item->user_defined) {
		found = find_item_for_delete (conf->conf_string, item);
		if (found) {
			skk_conf_item_destroy ((SkkConfItem *)found->data);
			conf->conf_string = g_list_remove_link (conf->conf_string, found);
			g_list_free_1 (found);
		}
	}
	conf->conf_string = g_list_prepend (conf->conf_string, item);
	return;
}

static void
add_item_num (SkkConf *conf, SkkConfItem *item)
{
	GList *found;
	if (item->user_defined) {
		found = find_item_for_delete (conf->conf_num, item);
		if (found) {
			skk_conf_item_destroy ((SkkConfItem *)found->data);
			conf->conf_num = g_list_remove_link (conf->conf_num, found);
			g_list_free_1 (found);
		}
	}
	conf->conf_num = g_list_prepend (conf->conf_num, item);
	return;
}

static SkkConfType
find_type (gchar *str)
{
	if (!str)
		return SKKCONF_TYPE_INVALID;
	if (!g_strcasecmp (str, "bool")) {
		return SKKCONF_TYPE_BOOL;
	} else if (!g_strcasecmp (str, "string")) {
		return SKKCONF_TYPE_STRING;
	} else if (!g_strcasecmp (str, "num")) {
		return SKKCONF_TYPE_NUM;
	}
	g_assert (0);
	return SKKCONF_TYPE_INVALID;
}

static gint
find_item_type_p (gconstpointer p1, gconstpointer p2)
{
	return g_strcasecmp ((const gchar *)p1, ((SkkConfItem*)p2)->name);
}

static SkkConfItemType
find_item_type (const gchar *str)
{
	SkkConfItem *item;
	item = bsearch (str, itemp, SKKCONF_DEFAULT_ITEM_SIZE,
			sizeof (SkkConfItem), find_item_type_p);
	if (item)
		return item->itemtype;

	return SKKCONF_ITEM_INVALID;
}

static void
parse_item_one (SkkConf *conf, xmlNodePtr node)
{
	xmlNodePtr child;
	xmlChar *name = NULL; /* always ASCII */
	xmlChar *value = NULL;
	xmlChar *info = NULL;
	xmlChar *type = NULL;
	gchar *info_e = NULL;
	gpointer value_e = NULL;
	SkkConfItem *item;
	SkkConfType ctype;
	SkkConfItemType itype;
	if (!node)
		return;
	child = node->xmlChildrenNode;
	if (!child)
		return;
	type = xmlGetProp (node, "type");
	if (!type)
		return;
	if ((ctype = find_type (type)) == SKKCONF_TYPE_INVALID) {
		xmlFree (type);
		return;
	}
	for (; child ; child = child->next) {
		if (xmlIsBlankNode (node)) {
			continue;
		}
		if (!xmlStrcmp ("name", child->name)) {
			name = xmlNodeGetContent (child);
		} else if (!xmlStrcmp ("value", child->name)) {
			value = xmlNodeGetContent (child);
		} else if (!xmlStrcmp ("info", child->name)) {
			info = xmlNodeGetContent (child);
		}
	}
	itype = find_item_type (name);
	if (itype == SKKCONF_ITEM_INVALID) {
		goto err;
	}
	info_e = info ? skk_utils_utf8_to_eucjp (info) : NULL;
	switch (ctype) {
		case SKKCONF_TYPE_BOOL:
			if (!xmlStrcasecmp ("true", value)) {
				value_e = GINT_TO_POINTER (TRUE);
			} else if (!xmlStrcasecmp ("false", value)) {
				value_e = GINT_TO_POINTER (FALSE);
			} else {
				value_e = GINT_TO_POINTER (FALSE);
			}
			break;
		case SKKCONF_TYPE_NUM:
			/* FIXME
			 * need error check ?
			 */
			value_e = GINT_TO_POINTER (atoi (value));
			break;
		case SKKCONF_TYPE_STRING:
			value_e = (gpointer)value ? skk_utils_utf8_to_eucjp (value) : NULL;
			break;
		default:
			g_assert (0); /* not reached */
			break;
	}
	item = skk_conf_item_new_with_value (ctype, itype, name, info_e, value_e);
	skk_conf_add_item (conf, item);
	if (name)
		xmlFree (name);
	if (value)
		xmlFree (value);
	if (type)
		xmlFree (type);
	if (info)
		xmlFree (info);
	if (info_e)
		g_free (info_e);
	if (value_e && ctype == SKKCONF_TYPE_STRING)
		g_free (value_e);
	return;

err:
	if (name)
		xmlFree (name);
	if (value)
		xmlFree (value);
	if (type)
		xmlFree (type);
	if (info)
		xmlFree (info);
	return;
}

static void
parse_rule_delete (SkkConf *conf, xmlDocPtr doc)
{
	xmlXPathContextPtr context;
	xmlXPathObjectPtr object;
	xmlNodeSetPtr nodeset = NULL;
	xmlNodePtr node = NULL;
	xmlChar *key = NULL;
	gint i = 0;
	if (!doc)
		return;
	context = xmlXPathNewContext (doc);
	if (!context)
		return;
	object = xmlXPathEval ("/iiimf-skk/delete-rule/key", context);
	if (!object) {
		xmlXPathFreeContext (context);
		return;
	}
	if (object->type == XPATH_NODESET) {
		nodeset = object->nodesetval;
	}
	/*
	 * NOTICE:
	 * nodesetval is not allocated by libxml2, so don't free after using it
	 */
	if (!nodeset || !nodeset->nodeTab) {
		xmlXPathFreeObject (object);
		return;
	}
	for (node = nodeset->nodeTab[0]; i < nodeset->nodeNr ; node = nodeset->nodeTab[i]) {
		key = xmlNodeGetContent (node);
		if (key) {
			if (*key) 
				delete_rule_emit (conf, key);
			xmlFree (key);
		}
		i++;
	}
	xmlXPathFreeObject (object);
	xmlXPathFreeContext (context);
	return;
}

static void
parse_item (SkkConf *conf, xmlDocPtr doc) 
{
	xmlXPathContextPtr context;
	xmlXPathObjectPtr object;
	xmlNodeSetPtr nodeset = NULL;
	xmlNodePtr node = NULL;
	gint i = 0;
	if (!doc)
		return;
	context = xmlXPathNewContext (doc);
	if (!context)
		return;
	object = xmlXPathEval ("/iiimf-skk/item", context);
	if (!object) {
		xmlXPathFreeContext (context);
		return;
	}
	if (object->type == XPATH_NODESET) {
		nodeset = object->nodesetval;
	}
	/*
	 * NOTICE:
	 * nodesetval is not allocated by libxml2, so don't free after using it
	 */
	if (!nodeset || !nodeset->nodeTab) {
		xmlXPathFreeObject (object);
		return;
	}
	for (node = nodeset->nodeTab[0]; i < nodeset->nodeNr ; node = nodeset->nodeTab[i]) {
		parse_item_one (conf, node);
		i++;
	}
	xmlXPathFreeObject (object);
	xmlXPathFreeContext (context);
	return;
}

static void
parse_rule_one (SkkConf *conf, xmlNodePtr node)
{
	xmlNodePtr child;
	xmlChar *key = NULL;
	xmlChar *hiragana = NULL;
	xmlChar *katakana = NULL;
	xmlChar *append = NULL;
	gchar *hira_e = NULL;
	gchar *kata_e = NULL;
	gchar *append_e = NULL;
	if (!node)
		return;
	child = node->xmlChildrenNode;
	if (!child)
		return;
	/*
	 * NOTICE
	 * we have DTD (<!ELEMENT rule (key, hiragana, katakana, append)>
	 * when any modification is added to DTD, FIXME.
	 * famao: currently, dont't believe DTD.
	 */
	for (; child ; child = child->next) {
		if (xmlIsBlankNode (node)) {
			continue;
		}
		if (!xmlStrcmp ("key", child->name)) {
			key = xmlNodeGetContent (child);
		} else if (!xmlStrcmp ("hiragana", child->name)) {
			hiragana = xmlNodeGetContent (child);
		} else if (!xmlStrcmp ("katakana", child->name)) {
			katakana = xmlNodeGetContent (child);
		} else if (!xmlStrcmp ("append", child->name)) {
			append = xmlNodeGetContent (child);
		}
	}
	/* FIXME add rule */
	hira_e = skk_utils_utf8_to_eucjp (hiragana);
	kata_e = skk_utils_utf8_to_eucjp (katakana);
	append_e = skk_utils_utf8_to_eucjp (append);
	add_rule_emit (conf, key, hira_e, kata_e, append_e);
#if 0
	rule = skk_conv_rule_item_new_with_value (key, hira_e, kata_e, append_e);
	skk_conv_add_rule (rule);
#endif
	if (key)
		xmlFree (key);
	if (hiragana)
		xmlFree (hiragana);
	if (katakana)
		xmlFree (katakana);
	if (append)
		xmlFree (append);
	if (hira_e)
		g_free (hira_e);
	if (kata_e)
		g_free (kata_e);
	if (append_e)
		g_free (append_e);
	return;
}

static void
parse_rule (SkkConf *conf, xmlDocPtr doc)
{
	xmlXPathContextPtr context;
	xmlXPathObjectPtr object;
	xmlNodeSetPtr nodeset = NULL;
	xmlNodePtr node = NULL;
	gint i = 0;
	if (!doc)
		return;
	context = xmlXPathNewContext (doc);
	if (!context)
		return;
	object = xmlXPathEval ("/iiimf-skk/rule", context);
	if (!object) {
		xmlXPathFreeContext (context);
		return;
	}
	if (object->type == XPATH_NODESET) {
		nodeset = object->nodesetval;
	}
	/*
	 * NOTICE:
	 * nodesetval is not allocated by libxml2, so don't free after using it
	 */
	if (!nodeset || !nodeset->nodeTab) {
		xmlXPathFreeObject (object);
		return;
	}
	for (node = nodeset->nodeTab[0]; i < nodeset->nodeNr ; node = nodeset->nodeTab[i]) {
		parse_rule_one (conf, node);
		i++;
	}
	xmlXPathFreeObject (object);
	xmlXPathFreeContext (context);
	return;
}

static void
list_free_func (gpointer data, gpointer user_data)
{
	if (data) {
		skk_conf_item_destroy ((SkkConfItem *)data);
	}
}

static void
clear (SkkConf *conf)
{
	if (conf->conf_num) {
		skk_utils_list_free (conf->conf_num, TRUE, list_free_func, NULL);
		conf->conf_num = NULL;
	}
	if (conf->conf_string) {
		skk_utils_list_free (conf->conf_string, TRUE, list_free_func, NULL);
		conf->conf_string = NULL;
	}
	if (conf->conf_bool) {
		skk_utils_list_free (conf->conf_bool, TRUE, list_free_func, NULL);
		conf->conf_bool = NULL;
	}
	return;
}

static void
load_rc (SkkConf *conf, const char *filename)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	struct stat s;
	if (stat (filename, &s) == -1)
		return;

	/* reset setting */
	if (conf) {
		clear (conf);
		add_defaults (conf);
	}
	doc = xmlParseFile (filename);
	if (!doc)
		return;
	root = xmlDocGetRootElement (doc);
	if (xmlStrcasecmp (root->name, "iiimf-skk") != 0) {
		xmlFreeDoc (doc);
		return;
	}
	parse_rule_delete (conf, doc);
	parse_rule (conf, doc);
	parse_item (conf, doc);
	xmlFreeDoc(doc);
}

void
skk_conf_load_rc (SkkConf *conf, char *user_name)
{
	char fname[FILENAME_MAX];

#ifdef SKKCONF_DEBUG
	strcpy(fname, "sample.config.xml");
#else
	struct passwd *pw;
	if ((pw = getpwnam(user_name)) == NULL)
		return;
	snprintf (fname, FILENAME_MAX, "%s/.iiimf-skk/config.xml", pw->pw_dir);
#endif
	load_rc (conf, fname);
}


SkkConf*
skk_conf_new (void)
{
	SkkConf *ret;
	ret = g_new (SkkConf, 1);
	memset (ret, 0, sizeof (SkkConf));
	ret->clear = clear;
	ret->conf_string = NULL;
	ret->conf_bool = NULL;
	ret->conf_num = NULL;
	ret->conf_add_rule_listener = NULL;
	ret->conf_delete_rule_listener = NULL;
	/* initialize */
	add_defaults (ret);
	skk_conf_ref (ret);
	return ret;
}

void
skk_conf_destroy (SkkConf *conf)
{
	if (!conf)
		return;
	skk_conf_unref (conf);
	if (conf->ref_count > 0) {
		return;
	}
	skk_conf_clear (conf);
	if (conf->conf_add_rule_listener) {
		skk_utils_slist_free (conf->conf_add_rule_listener, TRUE, NULL, NULL);
	}
	if (conf->conf_delete_rule_listener) {
		skk_utils_slist_free (conf->conf_delete_rule_listener, TRUE, NULL, NULL);
	}
	g_free (conf);
}

void
skk_conf_clear (SkkConf *conf)
{
	if (!conf)
		return;
	if (conf->clear)
		conf->clear (conf);
	return;
}

void
skk_conf_ref (SkkConf *conf)
{
	if (!conf)
		return;
	conf->ref_count++;
	return;
}

void
skk_conf_unref (SkkConf *conf)
{
	if (!conf)
		return;
	conf->ref_count--;
	if (!conf->ref_count) {
		skk_conf_destroy (conf);
	}
	return;
}

void
skk_conf_add_item (SkkConf *conf, SkkConfItem *item)
{
	if (!conf || !item)
		return;
	switch (item->type) {
		case SKKCONF_TYPE_BOOL:
			add_item_bool (conf, item);
			break;
		case SKKCONF_TYPE_STRING:
			add_item_string (conf, item);
			break;
		case SKKCONF_TYPE_NUM:
			add_item_num (conf, item);
			break;
		default:
			return;
			break;
	}
	return;
}

static SkkConfItem *
find_item (GList *list, SkkConfItemType type)
{
	GList *ret;
	if (!list)
		return NULL;
	ret = g_list_find_custom (list, GINT_TO_POINTER (type), find_item_func);
	if (!ret)
		return NULL;
	return (SkkConfItem *)ret->data;
}

void
skk_conf_set_bool (SkkConf *conf, SkkConfItemType type, gboolean value)
{
	SkkConfItem *item;
	/* TODO How to */
	if (!conf)
		return;
	item = find_item (conf->conf_bool, type);
	if (!item)
		return;
	item->value = GINT_TO_POINTER (value);
	return;
}

void
skk_conf_set_bool_by_name (SkkConf *conf, const gchar *name, gboolean value)
{
	if (!name)
		return;
	skk_conf_set_bool (conf, find_item_type (name), value);
}

void
skk_conf_set_string (SkkConf *conf, SkkConfItemType type, const gchar *string)
{
	SkkConfItem *item;
	gchar *value = NULL;
	/* TODO How to */
	if (!conf)
		return;
	if (!string)
		return;
	item = find_item (conf->conf_string, type);
	if (!item)
		return;
	if (item->value) {
		value = item->value;
	}
	item->value = g_strdup (string);
	if (value)
		g_free (value);
	return;
}

void
skk_conf_set_string_by_name (SkkConf *conf, const gchar *name, const gchar *string)
{
	if (!name)
		return;
	skk_conf_set_string (conf, find_item_type (name), string);
}

void
skk_conf_set_num (SkkConf *conf, SkkConfItemType type, gint num)
{
	SkkConfItem *item;
	/* TODO How to */
	if (!conf)
		return;
	item = find_item (conf->conf_num, type);
	if (!item)
		return;
	item->value = GINT_TO_POINTER (num);
	return;
}

void
skk_conf_set_num_by_name (SkkConf *conf, const gchar *name, gint num)
{
	if (!name)
		return;
	skk_conf_set_num (conf, find_item_type (name), num);
}

gboolean
skk_conf_get_bool_by_name (SkkConf *conf, const gchar *name)
{
	if (!conf)
		return FALSE;
	if (!name)
		return FALSE;
	return skk_conf_get_bool (conf, find_item_type (name));
}

gboolean
skk_conf_get_bool (SkkConf *conf, SkkConfItemType type)
{
	SkkConfItem *item;
	if (!conf)
		return FALSE;
	item = find_item (conf->conf_bool, type);
	if (!item)
		return FALSE;
	else
		return GPOINTER_TO_INT (item->value);
	return FALSE;
}

gchar *
skk_conf_get_string_by_name (SkkConf *conf, const gchar *name)
{
	if (!conf)
		return NULL;
	if (!name)
		return NULL;
	return skk_conf_get_string (conf, find_item_type (name));
}

gchar *
skk_conf_get_string (SkkConf *conf, SkkConfItemType type)
{
	SkkConfItem *item;
	if (!conf)
		return NULL;
	item = find_item (conf->conf_string, type);
	if (!item || !item->value)
		return NULL;
	else
		return g_strdup ((gchar *)item->value);
	return NULL;
}

gint
skk_conf_get_num_by_name (SkkConf *conf, const gchar *name)
{
	if (!conf)
		return -1;
	if (!name)
		return -1;
	return skk_conf_get_num (conf, find_item_type (name));
}

gint
skk_conf_get_num (SkkConf *conf, SkkConfItemType type)
{
	SkkConfItem *item;
	if (!conf)
		return -1;
	item = find_item (conf->conf_num, type);
	if (!item)
		return -1;
	else
		return GPOINTER_TO_INT (item->value);
	return -1;
}

void
skk_conf_delete_item (SkkConf *conf, SkkConfItem *item)
{
	return;
}

SkkConfItem*
skk_conf_item_new (void)
{
	SkkConfItem *ret;
	ret = g_new0 (SkkConfItem, 1);
	ret->user_defined = TRUE;
	return ret;
}

SkkConfItem*
skk_conf_item_new_with_value (SkkConfType type, SkkConfItemType itemtype, const gchar *name, const gchar *info, gpointer value)
{
	SkkConfItem *ret;
	ret = skk_conf_item_new ();
	ret->type = type;
	ret->itemtype = itemtype;
	ret->name = name ? g_strdup (name) : NULL;
	ret->info = info ? g_strdup (info) : NULL;
	switch (type) {
		case SKKCONF_TYPE_BOOL:
		case SKKCONF_TYPE_NUM:
			ret->value = value;
			break;
		case SKKCONF_TYPE_STRING:
			ret->value = g_strdup ((gchar *)value);
			break;
		default:
			break;
	}
	return ret;
}

void
skk_conf_item_destroy (SkkConfItem *item)
{
	if (!item)
		return;
	if (!item->user_defined) 
		return;
	if (item->name)
		g_free (item->name);
	if (item->info)
		g_free (item->info);
	switch (item->type) {
		case SKKCONF_TYPE_STRING:
			if (item->value)
				g_free (item->value);
			break;
		default:
			break;
	}
	g_free (item);
	return;
}

gboolean
skk_conf_add_rule_listener (SkkConf *conf, SkkConfAddRuleListener listener, gpointer user_data)
{
	SkkConfAddRuleFunc *func;
	if (!conf)
		return FALSE;
	if (!listener)
		return FALSE;
	func = g_new0 (SkkConfAddRuleFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	conf->conf_add_rule_listener = g_slist_append (conf->conf_add_rule_listener, func);
	return TRUE;
}

gboolean
skk_conf_add_rule_delete_listener (SkkConf *conf, SkkConfDeleteRuleListener listener, gpointer user_data)
{
	SkkConfDeleteRuleFunc *func;
	if (!conf)
		return FALSE;
	if (!listener)
		return FALSE;
	func = g_new0 (SkkConfDeleteRuleFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	conf->conf_delete_rule_listener = g_slist_append (conf->conf_delete_rule_listener, func);
	return TRUE;
}

#ifdef SKKCONF_DEBUG

#include "skkconv_kana.h"

void
listener (SkkConf *conf, const gchar *key, const gchar *hira, const gchar *kata, const gchar *append, gpointer user_data)
{
	SkkConvRule *rule = (SkkConvRule *) user_data;
	printf ("add_rule_listener\n");
	printf ("key %s hira %s kata %s append %s\n", key, hira, kata, append);
	skk_conv_add_rule (rule, skk_conv_rule_item_new_with_value (key, hira, kata, append));
	return;
}

int
main (void)
{
	SkkConf *conf;
	SkkConvRule *rule;
	conf = skk_conf_new();
	rule = skk_conv_rule_new ();
	printf("'%s' -> '%s'\n",".", skk_conv_get_hiragana(rule, ".", NULL));
	skk_conf_add_rule_listener (conf, listener, rule);
	skk_conf_load_rc(conf, NULL);
	printf("skk_egg_like_newline = %s\n",
					skk_conf_get_bool (conf, SKKCONF_BOOL_EGG_LIKE_NEWLINE) ? "TRUE" : "FALSE");
	printf("skk_egg_like_newline = %s\n",
					skk_conf_get_bool_by_name (conf, "skk-egg-like-newline") ? "TRUE" : "FALSE");
	printf("'%s' -> '%s'\n",".", skk_conv_get_hiragana(rule, ".", NULL));
	skk_conf_set_bool (conf, SKKCONF_BOOL_EGG_LIKE_NEWLINE, FALSE);
	printf("skk_egg_like_newline = %s\n",
					skk_conf_get_bool_by_name (conf, "skk-egg-like-newline") ? "TRUE" : "FALSE");
	skk_conf_set_bool_by_name (conf, "skk-egg-like-newline", TRUE);
	printf("skk_egg_like_newline = %s\n",
					skk_conf_get_bool_by_name (conf, "skk-egg-like-newline") ? "TRUE" : "FALSE");
	printf ("skk_server_host = %s\n",
			skk_conf_get_string_by_name (conf, "skk-server-host") ? 
			skk_conf_get_string_by_name (conf, "skk-server-host") : "None");
	printf("skk_plugin_use_dict = %s\n",
					skk_conf_get_bool_by_name (conf, "skk-plugin-use-dict") ? "TRUE" : "FALSE");
	return 0;
}
#endif
