/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkkeymapdebug.c,v 1.1.2.7 2002/11/09 09:54:03 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <ctype.h>
#include <string.h>
#include "skkkeysym.h"
#include "skkkeymap.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifndef USE_GTK2

static SkkBuffer *buf = NULL;
static SkkFunc *func = NULL;
static SkkQuery *query = NULL;
static SkkConvRule *rule = NULL;
static GtkWidget *entry = NULL;
static GtkTooltips *tooltips = NULL;

static guint
key_translate (gint keyval)
{
	switch (keyval) {
		case GDK_BackSpace:
			return SKK_VK_BACK_SPACE;
			break;
		default:
			return keyval;
			break;
	}
}

static gint
keypress (GtkWidget *widget, GdkEventKey *event, gpointer user_data)
{
	GtkWidget *text = (GtkWidget *)user_data;
	guint skk_state = 0;
	guint skk_key = 0;
	gint result;

	skk_key = key_translate (event->keyval);
	if (event->state & GDK_CONTROL_MASK) {
		skk_state |= SKK_CONTROL_MASK;
	}
	if (event->state & GDK_MOD1_MASK) {
		skk_state |= SKK_ALT_MASK;
	}
	if (event->state & GDK_SHIFT_MASK) {
		skk_state |= SKK_SHIFT_MASK;
	}
	if (!skk_state) {
		skk_state = SKK_NONE_MASK;
	}
	result = skk_keymap_do_func (buf, func, skk_key , skk_state);
	if (result == SKK_NOTRANS) {
		if (isascii (skk_key)) {
			gchar *value = g_strdup_printf ("%c", skk_key);
			gtk_text_insert (GTK_TEXT (text),
					NULL, NULL, NULL, value, strlen (value));
			g_free (value);
		}
	} else if (result == SKK_INSERT_CHAR) {
		skk_buffer_add_char (buf, skk_key);
	}

	gtk_entry_set_text (GTK_ENTRY (entry), "");
	return TRUE;
}
		
static void
plistener (SkkBuffer *buf, gchar *value, gpointer user_data)
{
	GtkWidget *text = (GtkWidget *)user_data;
	GtkTooltipsData *data;
	gtk_label_set_text (GTK_LABEL (text), value);
	gtk_tooltips_set_tip (tooltips, text,
			value, "hoge");
	gtk_tooltips_enable (tooltips);
	gtk_tooltips_force_window (tooltips);
	return;
}

static void
listener (SkkBuffer *buf, gchar *value, gpointer user_data)
{
	GtkWidget *text = (GtkWidget *)user_data;
	if (!value)
		return;
	gtk_text_insert (GTK_TEXT (text),
			NULL, NULL, NULL, value, strlen (value));
	return;
}

static void
slistener (SkkMode *mode, gint status, gpointer user_data)
{
	GtkWidget *slabel = (GtkWidget *)user_data;
	const gchar *sstring = skk_mode_get_status_string (mode);
	gtk_label_set_text (GTK_LABEL (slabel), sstring);
	return;
}

GtkWidget *
create_widget (void)
{
	GtkWidget *toplevel;
	GtkWidget *result;
	GtkWidget *preedit;
	GtkWidget *status;
	GtkWidget *vbox;
	toplevel = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	result = gtk_text_new (NULL,NULL);
	preedit = gtk_label_new (NULL);
	status = gtk_label_new ("");
	entry = gtk_entry_new ();
	tooltips = gtk_tooltips_new ();
	vbox = gtk_vbox_new (FALSE, FALSE);
	gtk_container_add (GTK_CONTAINER (toplevel), vbox);
	gtk_box_pack_start (GTK_BOX (vbox), entry, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), preedit, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), result, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), status, FALSE, FALSE, 0);
	gtk_signal_connect (GTK_OBJECT (entry), "key_press_event",
			GTK_SIGNAL_FUNC (keypress), (gpointer) result);
	gtk_signal_connect (GTK_OBJECT (toplevel), "delete_event",
			GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
	gtk_tooltips_set_tip (tooltips, result, "fuga","hoge");
	gtk_tooltips_set_delay (tooltips, 0);
	buf = skk_buffer_new ();
	func = skk_func_new ();
	query = skk_query_new_with_path ("./.libs");
	rule = skk_conv_rule_new ();
	skk_query_add_item (query, skk_query_item_new (query, SKKQUERY_LOCAL));
	skk_query_add_item (query, skk_query_item_new (query, SKKQUERY_SERVER));
	skk_query_add_item (query, skk_query_item_new (query, SKKQUERY_LOOK));
	skk_buffer_set_query (buf, query);
	skk_buffer_set_rule (buf, rule);
	skk_buffer_add_preedit_listener (buf, plistener, (gpointer)preedit);
	skk_buffer_add_result_listener (buf, listener, (gpointer)result);
	skk_mode_add_status_listener (skkbuf_mode (buf), slistener, (gpointer) status);
	return toplevel;
}

int
main (int argc, char *argv[])
{
	GtkWidget *toplevel;
#ifdef HAVE_SETENV
	setenv ("LC_ALL","ja_JP.EUC-JP",TRUE);
#elif HAVE_PUTENV
	putenv ("LC_ALL=ja_JP.eucJP");
#endif
	gtk_set_locale ();
	gtk_init (&argc, &argv);
	toplevel = create_widget ();
	gtk_widget_show_all (toplevel);
	gtk_main ();
	return 0;
}
#else
int
main (void)
{
	return 0;
}
#endif
