/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 *
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkkeymap.c,v 1.12.2.18 2002/11/12 10:33:38 famao Exp $ */

/* vi:set ts=4 sw=4: */

#include <ctype.h>
#include "skkkeymap.h"
#include "skkkeysym.h"
#include "skktypes.h"
#include "skkfunc.h"
#include "skkconv_kana.h"

static gint
keymap_jmode (SkkBuffer *buf, SkkFuncResult *result)
{
	SkkJStatus jstatus = skk_buffer_get_j_status (buf);
#ifdef SKKKEYMAP_DEBUG
	g_message ("keymap_jmode");
	g_message ("-----------------------------------------------");
#endif
	switch (result->result) {
		case SKKFUNC_BACKWARD_DELETE:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in BACKWARD_DELETE");
#endif
			if (jstatus == SKK_CHOICE &&
					!skk_conf_get_bool_by_name (skkbuf_conf (buf), "skk-delete-implies-kakutei")) {
				skk_buffer_undo (buf);
				return SKK_NOTHING;
			}
			if (skk_buffer_has_preedit (buf)) {
				skk_buffer_delete_backward (buf);
				if (skk_buffer_can_clear (buf)) {
					skk_buffer_clear (buf);
				}
				return SKK_NOTHING;
			} else {
				return SKK_NOTRANS;
			}
			break;
		case SKKFUNC_COMPLETE_BEGIN:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in COMPLETE_BEGIN");
#endif
			if (jstatus == SKK_HENKAN) {
				skk_buffer_get_completion (buf);
				return SKK_NOTHING;
			} else {
				return SKK_NOTRANS;
			}
			break;
		case SKKFUNC_KAKUTEI:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in KAKUTEI");
#endif
			switch (jstatus) {
				case SKK_NONE:
					skk_buffer_clear (buf);
					return SKK_NOTRANS;
					break;
				case SKK_HENKAN:
				case SKK_OKURI:
					skk_buffer_commit (buf, FALSE);
					skk_buffer_clear (buf);
					break;
				case SKK_CHOICE:
					skk_buffer_commit (buf, TRUE);
					skk_buffer_clear (buf);
					break;
			}
#if 0
			if (result->actual_key == SKK_VK_ENTER) {
				if (skk_conf_get_egg_like_newline (skkbuf_conf (buf)) && jstatus != SKK_NONE) {
					return SKK_NOTHING;
				} else {
					return SKK_NOTRANS;
				}
			}
#else
			if (skk_conf_get_bool_by_name (skkbuf_conf (buf), "skk-egg-like-newline")) {
				return SKK_NOTHING;
			} else {
				return SKK_NOTRANS;
			}
#endif
			return SKK_NOTHING;
			break;
		case SKKFUNC_COMPLETE_TRANSLATE:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in COMPLETE_TRANSLATE");
#endif
			/* reset search plugin */
			skk_query_set_first (skkbuf_query (buf));
			skk_buffer_get_completion (buf);
			skk_buffer_set_j_status (buf, SKK_CHOICE);
			skk_buffer_query_dict (buf);
			/* finish */
			skk_buffer_end_completion (buf);
			return SKK_NOTHING;
			break;
		case SKKFUNC_TRANSLATE_BEGIN:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in TRANSLATE_BEGIN");
#endif
			if (jstatus == SKK_HENKAN) {
				skk_buffer_set_j_status (buf, SKK_CHOICE);
				skk_query_set_first (skkbuf_query (buf));
				skk_buffer_query_dict (buf);
				return SKK_NOTHING;
			} else if (jstatus == SKK_CHOICE) {
				if (skk_buffer_has_next_candidate (buf)) {
					static gint candidate_count = -1;
					if (candidate_count == -1) {
						candidate_count = skk_conf_get_num_by_name (skkbuf_conf (buf), "skk-candidate-count");
					}
					/* TODO */
					if (buf->candidate_count >= candidate_count) {
						buf->candidate_count++;
						skk_buffer_lookup_emit (buf);
					} else {
						skk_buffer_set_next_candidate (buf);
					}
				} else {
					skk_buffer_adddict_emit (buf);
				}
				return SKK_NOTHING;
			}
			return SKK_NOTRANS;
			break;
		case SKKFUNC_PREFIX:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in PREFIX");
#endif
			if (jstatus == SKK_HENKAN) {
				skk_buffer_add_char (buf, result->actual_key);
				skk_buffer_set_j_status (buf, SKK_CHOICE);
				skk_buffer_query_dict (buf);
				return SKK_NOTHING;
			}
			break;
		case SKKFUNC_POSTFIX:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in POSTFIX");
#endif
			if (jstatus == SKK_CHOICE) {
				skk_buffer_commit (buf, TRUE);
				skk_buffer_clear (buf);
				skk_buffer_set_j_status (buf, SKK_HENKAN);
				skk_buffer_add_char (buf, result->actual_key);
				return SKK_NOTHING;
			}
			break;
		case SKKFUNC_COMPLETE_PREV:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in COMPLETE_PREV");
#endif
			if (jstatus == SKK_HENKAN) {
				if (skkbuf_is_completion (buf)) {
					skk_buffer_set_prev_completion (buf);
					return SKK_NOTHING;
				}
			}
			/* We are not in COMPLETE_MODE */
			return SKK_INSERT_CHAR;
			break;
		case SKKFUNC_COMPLETE_NEXT:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in COMPLETE_NEXT");
#endif
			if (jstatus == SKK_HENKAN) {
				if (skkbuf_is_completion (buf)) {
					if (skk_conf_get_bool_by_name (skkbuf_conf (buf), "skk-dabbrev-like-completion")) {
						skk_buffer_get_completion (buf);
					} else {
						skk_buffer_set_next_completion (buf);
					}
					return SKK_NOTHING;
				}
			}
			/* We are not in COMPLETE_MODE */
			return SKK_INSERT_CHAR;
			break;
		case SKKFUNC_CANCEL:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in CANCEL");
#endif
			if (jstatus == SKK_CHOICE) {
				skkbuf_candidate_num (buf) = 0;
				if (skkbuf_okuri (buf)) {
					if (skk_conf_get_bool_by_name (skkbuf_conf (buf), "skk-delete-okuri-when-quit")) {
						/* TODO dirty hack */
						g_free (skkbuf_okuri (buf));
						g_free (skkbuf_okurip (buf));
						skkbuf_okuri (buf) = NULL;
						skkbuf_okurip (buf) = NULL;
					}
					skk_buffer_undo (buf);
				} else {
					skk_buffer_undo (buf);
				}
				return SKK_NOTHING;
			}
			skk_buffer_clear (buf);
			return SKK_NOTHING;
		case SKKFUNC_MODE_CHANGE:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in MODE_CHANGE");
#endif
			if (skk_buffer_has_preedit (buf)) {
				skk_buffer_commit (buf, FALSE);
			}
			skk_buffer_set_jisx0201_katakana (buf, FALSE);
			skk_buffer_set_status (buf, result->actual_key);
			skk_buffer_preedit_emit (buf);
			return SKK_NOTHING;
		case SKKFUNC_CATEGORY_CHANGE:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in CATEGORY_CHANGE");
#endif
			if ((jstatus == SKK_HENKAN) || (jstatus == SKK_OKURI)) {
				skk_buffer_change_category (buf);
				return SKK_NOTHING;
			}
			break;
		case SKKFUNC_KANA_SWITCH:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in KANA_SWITCH");
#endif
			{
				gboolean katakana_state = skk_buffer_get_katakana (buf);
#if 1
				skk_buffer_commit (buf, FALSE);
#endif
				skk_buffer_clear (buf);
				skk_buffer_set_katakana (buf, !katakana_state);
				/* anyway, do */
				skk_buffer_set_jisx0201_katakana (buf, FALSE);
				return SKK_NOTHING;
			}
			break;
		case SKKFUNC_HALFWIDTH_KANA_SWITCH:
			{
				gboolean jisx0201_state = skk_buffer_get_jisx0201_katakana (buf);
				skk_buffer_clear (buf);
				skk_buffer_set_jisx0201_katakana (buf, !jisx0201_state);
				return SKK_NOTHING;
			}
			break;
		case SKKFUNC_UNDO:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in UNDO");
#endif
			if (jstatus == SKK_CHOICE) {
				skk_buffer_undo (buf);
				g_message ("%s", skkbuf_d (buf));
				return SKK_NOTHING;
			}
			break;
		case SKKFUNC_CODETABLE:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in CODETABLE");
#endif
			skk_buffer_codetable_emit (buf);
			/* TODO */
			return SKK_NOTHING;
			break;
		case SKKFUNC_TODAY:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in TODAY");
#endif
			{
				gchar *today_str = skk_gadget_get_jdate ();
				switch (jstatus) {
					case SKK_NONE:
						skk_buffer_clear (buf);
						skk_buffer_set_result (buf, today_str);
						break;
					case SKK_HENKAN:
						skk_buffer_add_preedit (buf, today_str);
						break;
					case SKK_OKURI:
						skk_buffer_add_preedit (buf, today_str);
						break;
					case SKK_CHOICE:
						skk_buffer_commit (buf, TRUE);
						skk_buffer_clear (buf);
						skk_buffer_set_result (buf, today_str);
						break;
				}
				if (today_str)
					g_free (today_str);
			}
			return SKK_NOTHING;
			break;
		case SKKFUNC_CURSOR_LEFT:
			skk_buffer_move_left (buf);
			g_message ("fuge");
			return SKK_NOTHING;
			break;
		case SKKFUNC_CURSOR_RIGHT:
			skk_buffer_move_right (buf);
			return SKK_NOTHING;
			break;
		default:
			break;
	}
	return SKK_NOTRANS;
}

static gint
keymap_abbrev (SkkBuffer *buf, SkkFuncResult *result)
{
	SkkQueryStatus qstatus = skk_buffer_get_query_status (buf);
	switch (result->result) {
		case SKKFUNC_BACKWARD_DELETE:
			skk_buffer_delete_backward (buf);
			if (!skkbuf_p (buf)) {
				skk_buffer_set_status (buf, SKK_J_MODE);
				return SKK_NOTHING;
			}
			return SKK_NOTHING;
			break;
		case SKKFUNC_KAKUTEI:
			if (qstatus == QUERY_NONE) {
				skk_buffer_clear (buf);
				return SKK_NOTRANS;
			}
			skk_buffer_commit (buf, TRUE);
			skk_buffer_clear (buf);
#if 0
			if (result->actual_key == SKK_VK_ENTER) {
				if (skk_conf_get_egg_like_newline (skkbuf_conf (buf))) {
					return SKK_NOTHING;
				} else {
					return SKK_NOTRANS;
				}
			}
#else
			if (skk_conf_get_bool_by_name (skkbuf_conf (buf), "skk-egg-like-newline")) {
				return SKK_NOTHING;
			} else {
				return SKK_NOTRANS;
			}
#endif
			return SKK_NOTHING;
			break;
		case SKKFUNC_CATEGORY_CHANGE:
			if ((qstatus == QUERY_NORMAL)) {
				skk_buffer_change_category (buf);
			}
			return SKK_NOTHING;
			break;
		case SKKFUNC_CANCEL:
			if (qstatus == QUERY_DONE) {
				skkbuf_candidate_num (buf) = 0;
				skk_buffer_undo (buf);
			} else {
				skk_buffer_clear (buf);
			}
			return SKK_NOTHING;
			break;
		case SKKFUNC_COMPLETE_BEGIN:
			skk_buffer_get_completion (buf);
			return SKK_NOTHING;
			break;
		case SKKFUNC_COMPLETE_NEXT:
			if (skkbuf_is_completion (buf)) {
				if (skk_conf_get_bool_by_name (skkbuf_conf (buf), "skk-dabbrev-like-completion")) {
					skk_buffer_get_completion (buf);
				} else {
					skk_buffer_set_next_completion (buf);
				}
				return SKK_NOTHING;
			} else {
				return SKK_INSERT_CHAR;
			}
			break;
		case SKKFUNC_COMPLETE_PREV:
			if (skkbuf_is_completion (buf)) {
				skk_buffer_set_prev_completion (buf);
				return SKK_NOTHING;
			} else {
				return SKK_INSERT_CHAR;
			}
			break;
		case SKKFUNC_COMPLETE_TRANSLATE:
			/* reset search plugin */
			skk_query_set_first (skkbuf_query (buf));
			skk_buffer_get_completion (buf);
			skk_buffer_set_query_status  (buf, QUERY_DONE);
			skk_buffer_query_dict (buf);
			skk_buffer_end_completion (buf);
			return SKK_NOTHING;
			break;
		case SKKFUNC_TRANSLATE_BEGIN:
			if (qstatus == QUERY_NORMAL) {
				skk_buffer_set_query_status (buf, QUERY_DONE);
				skk_query_set_first (skkbuf_query (buf));
				skk_buffer_query_dict (buf);
			} else if (qstatus == QUERY_DONE) {
				static gint candidate_count = -1;
				if (candidate_count == -1) {
					candidate_count = skk_conf_get_num_by_name (skkbuf_conf (buf), "skk-candidate-count");
				}
				if (skk_buffer_has_next_candidate (buf)) {
					if (buf->candidate_count >= candidate_count) {
						buf->candidate_count++;
						skk_buffer_lookup_emit (buf);
					} else {
						skk_buffer_set_next_candidate (buf);
					}
				} else {
					skk_buffer_adddict_emit (buf);
				}
			}
			return SKK_NOTHING;
			break;
		case SKKFUNC_UNDO:
			skk_buffer_undo (buf);
			return SKK_NOTHING;
			break;
		case SKKFUNC_CURSOR_LEFT:
			skk_buffer_move_left (buf);
			return SKK_NOTHING;
			break;
		case SKKFUNC_CURSOR_RIGHT:
			skk_buffer_move_right (buf);
			return SKK_NOTHING;
			break;
		default:
			break;
	}
	return SKK_NOTHING;
}

static gint
keymap_latin (SkkBuffer *buf, SkkFuncResult *result)
{
#ifdef SKKKEYMAP_DEBUG
	g_message ("keymap_latin");
	g_message ("-----------------------------------------------");
#endif
	switch (result->result) {
		case SKKFUNC_MODE_CHANGE:
#ifdef SKKKEYMAP_DEBUG
			g_message ("in MODE_CHANGE");
#endif
			skk_buffer_set_status (buf, result->actual_key);
			return SKK_NOTHING;
			break;
		default:
			break;
	}
	return SKK_NOTHING;
}

static gint
keymap_jisx0208_latin (SkkBuffer *buf, SkkFuncResult *result)
{
	switch (result->result) {
		case SKKFUNC_MODE_CHANGE:
			skk_buffer_set_status (buf, result->actual_key);
			return SKK_NOTHING;
			break;
		default:
			break;
	}
	return SKK_NOTHING;
}

gint
skk_keymap_do_func (SkkBuffer *buf, SkkFunc *func, guint ch, guint mask)
{
	SkkFuncResult *result;
	SkkStatus status;
	SkkQueryStatus qstatus;
	gint ret;
	if (!buf)
		return 0;
	if (!func)
		return 0;
	status = skk_buffer_get_status (buf);
	qstatus = skk_buffer_get_query_status (buf);

	result = skk_func_do_query (func, status, qstatus, ch, mask);
	if (!result) {
		if (status == SKK_LATIN_MODE) {
			return SKK_NOTRANS;
		}
		if (!(mask & (SKK_NONE_MASK | SKK_SHIFT_MASK)))
			return SKK_NOTRANS;
		/* TODO ok ? */
		if (!isascii (ch) || !isprint (ch))
			return SKK_NOTRANS;
		skk_buffer_add_char (buf, ch);
		ret = SKK_NOTHING;
	} else {
		switch (status) {
			case SKK_J_MODE:
				{
					gchar *check;
					if (isalnum (ch) &&
							(mask & (SKK_NONE_MASK | SKK_SHIFT_MASK)) && skkbuf_b (buf)) {
						check = g_strdup_printf ("%s%c", skkbuf_b (buf), ch);
						if (skk_conv_is_exist (buf->rule, check)) {
							ret = SKK_INSERT_CHAR;
							g_free (check);
							break;
						}
					}
					ret = keymap_jmode (buf, result);
				}
				break;
			case SKK_ABBREV_MODE:
				ret = keymap_abbrev (buf, result);
				break;
			case SKK_JISX0208_LATIN_MODE:
				ret = keymap_jisx0208_latin (buf, result);
				break;
			case SKK_LATIN_MODE:
				ret =  keymap_latin (buf, result);
				break;
			default:
				ret = SKK_NOTRANS;
				break;
		}
		g_free (result);
		if (ret == SKK_INSERT_CHAR) {
			skk_buffer_add_char (buf, ch);
			return SKK_NOTHING;
		}
	}
	/* TODO clean up */
	
	/* re-check Status Mode */
	status = skk_buffer_get_status (buf);
	if ((status == SKK_J_MODE)) {
		if (skk_buffer_can_clear (buf)) {
			skk_buffer_clear (buf);
		}
	}
	return ret;
}

#ifdef SKKKEYMAP_DEBUG
#include <stdio.h>

static void
listener (SkkBuffer *buf, gchar *value, gpointer user_data)
{
	g_message ("listener");
	if (value)
		g_message ("%s",value);
	return;
}

int
main (void)
{
	SkkBuffer *buf;
	SkkFunc *func;
	SkkQuery *query;
	SkkQueryItem *item;
	SkkConvRule *rule;
	gchar tmp[8192];
	int i;
	buf = skk_buffer_new ();
	func = skk_func_new ();
	query = skk_query_new_with_path ("./.libs");
	rule = skk_conv_rule_new ();
#if 1
	item = skk_query_item_new (query, SKKQUERY_LOCAL);
	skk_query_add_item (query, item);
	item = skk_query_item_new (query, SKKQUERY_SERVER);
	skk_query_add_item (query, item);
#else
	skk_query_add_item (query, skk_query_item_new (query, SKKQUERY_LOCAL));
	skk_query_add_item (query, skk_query_item_new (query, SKKQUERY_SERVER));
#endif
	skk_query_add_item (query, skk_query_item_new (query, SKKQUERY_LOOK));
	skk_buffer_add_preedit_listener (buf, listener, NULL);
	skk_buffer_add_result_listener (buf, listener, NULL);
	printf ("%p\n", query);
	skk_buffer_set_query (buf, query);
#if 1
	skk_buffer_set_rule (buf, rule);
#endif
	skk_keymap_do_func (buf, func, SKK_VK_q, SKK_CONTROL_MASK);
	while (fgets (tmp, 8192, stdin)) {
		for (i = 0;  tmp[i]; i++) {
#if 0
			skk_keymap_do_func (buf, func, tmp[i], SKK_ALL_MASK);
#else
			skk_keymap_do_func (buf, func, tmp[i], SKK_NONE_MASK);
			printf ("%p\n", buf->query);
#endif
		}
	}
	return 0;
}
#endif
