/*
 * IIIMF-SKK, Japanese Language Engine for 
 *                        IIIMF (Internet/Intranet Input Method Framework)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skk.c,v 1.9.2.3 2002/04/24 22:13:01 famao Exp $ */

/* vi:set ts=4 sw=4: */


/*
Copyright 1990-2001 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <dlfcn.h>
#include <pthread.h>
#include "csconv.h"
#include "skkcomm.h"
#include "skkbuffer.h"
#include "skkconv_jisx0208_latin.h"
#include "skkkeymap.h"
#include "skkkeysym.h"

/*
 * header files for SunIM library
 */
#include "SunIM.h"
#include "SunIMMMan.h"
#include "SunIMProt.h"
#include "SunIMMthd.h"
#include "SunIMPriv.h"
#include "SunIMCore.h"
#include "SunIMPub.h"

#include "skk_xaux.h"
#include "skk.h"

/*
 * define If method prototype
 */
static Bool if_skk_OpenIF (iml_if_t *);
static Bool if_skk_CloseIF (iml_if_t *);
static Bool if_skk_GetIFValues (iml_if_t *, IMArgList, int);
static Bool if_skk_SetIFValues (iml_if_t *, IMArgList, int);
static Bool if_skk_OpenDesktop (iml_desktop_t *, IMArgList, int);
static Bool if_skk_CloseDesktop (iml_desktop_t *);
static Bool if_skk_CreateSC (iml_session_t *, IMArgList, int);
static Bool if_skk_DestroySC (iml_session_t *);
static Bool if_skk_GetSCValues (iml_session_t *, IMArgList, int);
static Bool if_skk_SetSCValues (iml_session_t *, IMArgList, int);
static IMText *if_skk_ResetSC (iml_session_t *);
static void if_skk_SetSCFocus (iml_session_t *);
static void if_skk_UnsetSCFocus (iml_session_t *);
static void if_skk_SendEvent (iml_session_t *, IMInputEvent * ev);

/*
 * define if_methods_t
 */
static if_methods_t if_methods = {
  if_skk_OpenIF,
  if_skk_CloseIF,
  if_skk_GetIFValues,
  if_skk_SetIFValues,
  if_skk_OpenDesktop,
  if_skk_CloseDesktop,
  if_skk_CreateSC,
  if_skk_DestroySC,
  if_skk_GetSCValues,
  if_skk_SetSCValues,
  if_skk_ResetSC,
  if_skk_SetSCFocus,
  if_skk_UnsetSCFocus,
  if_skk_SendEvent
};

/* SKK */
static void skk_commit_key (iml_session_t *s, IMKeyListEvent *kev);
static void skk_process_keyevent_jmode (iml_session_t *s, IMKeyListEvent *kev);
static void skk_process_keyevent_abbrev (iml_session_t *s, IMKeyListEvent *kev);
static void skk_process_keyevent_jisx0208_latin (iml_session_t *s, IMKeyListEvent *kev);
static void skk_process_keyevent (iml_session_t * s, IMKeyListEvent * kev);
static int  skk_translate_keyevent_luc (iml_session_t *s, IMKeyListEvent *kev);
static int  skk_translate_keyevent_jmode (iml_session_t *s, IMKeyListEvent *kev);
static int  skk_translate_keyevent_latin_mode (iml_session_t *s, IMKeyListEvent *kev);
static int  skk_translate_keyevent_jisx0208_latin_mode (iml_session_t *s, IMKeyListEvent *kev);
static int  skk_translate_keyevent_abbrev_mode (iml_session_t *s, IMKeyListEvent *kev);
static int  skk_translate_keyevent (iml_session_t *s, IMKeyListEvent *kev);

/* Event Handling */
static void skk_preedit_listener (SkkBuffer *buf, gchar *value, gpointer user_data);
static void skk_result_listener (SkkBuffer *buf, gchar *value, gpointer user_data);
static void skk_adddict_listener (SkkBuffer *buf, gchar *query_str, gpointer user_data);
static void skk_lookup_listener (SkkBuffer *buf, gchar **label, gchar **candidate, gchar **annotate, gint count, gpointer user_data);
static void skk_codetable_listener (SkkBuffer *buf, gpointer user_data);
static void skk_status_listener (SkkMode *mode, gint status, gpointer user_data);

/* Preedit */
static void skk_preedit_done (iml_session_t * s);
static void skk_preedit_draw (iml_session_t * s, unsigned char *str);
static void skk_preedit_clear (iml_session_t *s);

/* Status */
static void skk_status_draw (iml_session_t * s);
static void skk_status_draw_off (iml_session_t *s);
static void skk_status_done (iml_session_t *s);

/* String */
static void skk_commit_string (iml_session_t *s, unsigned char *str);
static void skk_commit_utf_string (iml_session_t *s, UTFCHAR *ustr);
static void skk_commit_imtext (iml_session_t *s, IMText *text);

/* Start/End IM */
static void skk_make_conversion_on (iml_session_t * s);
static void skk_make_conversion_off (iml_session_t * s);

/* Lookup */
static void skk_start_lookup_choice (iml_session_t *s, iml_inst **prrv, int num);
static void skk_show_lookup_choice (iml_session_t *s);
static void skk_lookup_choice_done (iml_session_t *s);

/* AUX */
static void skk_aux_start (iml_session_t *s, char *class_name);
static void skk_aux_status_start (iml_session_t *s);
static void skk_aux_dict_start (iml_session_t *s);
static void skk_aux_dict_draw (iml_session_t *s, const gchar *query_str);

static void skk_start_conversion (iml_session_t *s);
static void skk_end_conversion (iml_session_t *s);

#define skk_aux_status_done(s) skk_aux_done ((s), SKK_AUX_PALETTE_CLASS_NAME)
#define skk_aux_dict_done(s) skk_aux_done ((s), SKK_AUX_ADD_DICT_CLASS_NAME)
static void skk_aux_done (iml_session_t *s, char *class_name);
static void skk_aux_draw (iml_session_t *s, char *class_name, int int_count, int *integers,
		                  int str_count, UTFCHAR **strings);
static void skk_process_auxevent (iml_session_t *s, IMAuxDrawCallbackStruct *aux);

static void skk_aux_status_draw (iml_session_t *s);
static void skk_aux_status_draw_off (iml_session_t *s);

/* String */
static IMText *skk_utftoimtext (iml_session_t * s, UTFCHAR * p);
static IMText *skk_strtoimtext (iml_session_t * s, int nseg, int *nb,
			 unsigned char **strs, int *feedback_type);
static UTFCHAR *skk_strtoutf (const unsigned char *str);


typedef enum
{
  SKK_Feedback_Input,
  SKK_Feedback_Strong,
  SKK_Feedback_Normal,
}
SKK_Feedback_Types;

static int skk_context_id_counter = 1;

typedef struct _SkkDesktopData SkkDesktopData;
typedef struct _SkkSessionData SkkSessionData;

struct _SkkSessionData
{
	int context_id;
	int conversion_start;
	int status_start;
	int preedit_start;
	int luc_start;
	int luc_count;
	int luc_hasmore;
	int dict_start;
	int dict_aux_ok;
	int palette_aux_ok;
	SkkBuffer *skkbuf;
};

struct _SkkDesktopData
{
	SkkFunc *skkfunc;
	SkkQuery *skkquery;
	SkkConf *skkconf;
};


static SkkSessionData *skk_session_data (iml_session_t *s);
static SkkDesktopData *skk_desktop_data (iml_session_t *s);
static SkkBuffer *skk_session_buf (iml_session_t *s);

static SkkSessionData *
skk_session_data (iml_session_t * s)
{
	return (SkkSessionData *) (s->specific_data);
}

static SkkDesktopData *
skk_desktop_data (iml_session_t *s)
{
	return (SkkDesktopData *) (s->desktop->specific_data);
}

static SkkBuffer *
skk_session_buf (iml_session_t * s)
{
	return (SkkBuffer *) (((SkkSessionData *) (s->specific_data))->skkbuf);
}

static int
skk_session_context (iml_session_t * s)
{
	return ((SkkSessionData *) (s->specific_data))->context_id;
}

/*
  AUX Object:
*/
IMObjectDescriptorStruct *objects = NULL;

/*
 * define Human Readable Name as UTFCHAR for LE
 */
UTFCHAR lename_string[] = { 0x53, 0x4B, 0x4B, 0x0 };
UTFCHAR jahrn_string[] = { 0x65E5, 0x672C, 0x8A9E, 0x0 };
UTFCHAR lookup_choice_title[] = { 0x5019, 0x88DC, 0x9078, 0x629E, 0x0 };

static IMLEName lename = {
  "skk", lename_string
};

static IMLocale locales[] = {
  {"ja", jahrn_string},
  {NULL, NULL},
};

/* CSConv Stuff */

#define CSC_PATH        "/usr/lib/im/csconv/csconv.so"
#define CSC_OPEN_LOCALE "csconv_open_locale"
#define CSC_OPEN        "csconv_open"
#define CSC_CONV        "csconv"
#define CSC_CLOSE       "csconv_close"

#define CSC_FAILED -1
#define CSC_UNLOADED 0
#define CSC_UNOPENED 1
#define CSC_LOADED 2

#define SKK_CODESET_LOCALE "ja_JP.eucJP"

#ifdef DEBUG
#define DEBUG_DO(X) (X)
#else
#define DEBUG_DO(X)
#endif

typedef csconv_t (*csc_open_locale_t) (const char *,
				       const char *, const char *);
typedef csconv_t (*csc_open_t) (const char *, const char *);
typedef size_t (*csc_conv_t) (csconv_t, const char **, size_t *,
			      char **, size_t *);
typedef int (*csc_close_t) (csconv_t);

static void *csc_handle;
static csc_open_locale_t csc_open_locale;
static csc_open_t csc_open;
static csc_conv_t csc_conv;
static csc_close_t csc_close;
static csconv_t cd_euc2utf = NULL;
static csconv_t cd_utf2euc = NULL;

/* static char *current_locale = NULL; */

static void
dlopen_csconv (void)
{
	csc_handle = dlopen (CSC_PATH, RTLD_LAZY);
	if (csc_handle == NULL) {
		csc_handle = (void *) (-1);
		return;
	}
	csc_open_locale = (csc_open_locale_t) dlsym (csc_handle, CSC_OPEN_LOCALE);
	csc_open = (csc_open_t) dlsym (csc_handle, CSC_OPEN);
	csc_conv = (csc_conv_t) dlsym (csc_handle, CSC_CONV);
	csc_close = (csc_close_t) dlsym (csc_handle, CSC_CLOSE);

	if ((csc_open_locale == NULL) || (csc_open == NULL) ||
			(csc_conv == NULL) || (csc_close == NULL)) {
		dlclose (csc_handle);
		csc_handle = (void *) (-1);
		return;
	}
	return;
}

static int
csconv_status ()
{
	if ((csc_handle == (void *) (-1)) || (cd_euc2utf == (csconv_t) (-1))
			|| (cd_utf2euc == (csconv_t)(-1))) {
		return CSC_FAILED;
	} else if (csc_handle == NULL) {
		return CSC_UNLOADED;
	} else if (cd_euc2utf == NULL) {
		return CSC_UNOPENED;
	}
	return CSC_LOADED;
}

static void
csconv_open_conv ()
{
	/* Open UTF-16 => SKK Multibyte Format */
	if (csconv_status () == CSC_UNOPENED) {
		cd_euc2utf = (csc_open_locale) (SKK_CODESET_LOCALE, "UTF-16", "MultiByte");
		cd_utf2euc = (csc_open_locale) (SKK_CODESET_LOCALE, "MultiByte", "UTF-16");
	}
	if (cd_euc2utf <= 0) {
		printf ("skk: Cannot Open csconv - %s\n", SKK_CODESET_LOCALE);
	}
	if (cd_utf2euc <= 0) {
		printf ("skk: Cannot Open csconv - %s\n", "UTF-16");
	}	
	return;
}

static void
setup_csconv ()
{
	if (csconv_status () == CSC_UNLOADED) {
		dlopen_csconv ();
	}
	if (csconv_status () == CSC_UNOPENED) {
		csconv_open_conv ();
	}
	return;
}

static UTFCHAR AuxName[] = { 's', 'k', 'k' };

#ifdef WIN32
#define EXPORT extern __declspec(dllexport)

EXPORT
#endif
void
if_GetIfInfo (IMArgList args, int num_args)
{
	int i;
#ifdef	DEBUG
	printf ("if_GetIfInfo()\n");
	printf ("	This method is invoked when htt_server retrieves\n");
	printf ("	if_method_t method table.\n\n");
#endif
	for (i = 0; i < num_args; i++, args++) {
		switch (args->id) {
			case IF_VERSION:
				args->value = (IMArgVal) "1.2";
				break;
			case IF_METHOD_TABLE:
				args->value = (IMArgVal) & if_methods;
				break;
			case IF_LE_NAME:
				args->value = (IMArgVal) & lename;
				break;
			case IF_SUPPORTED_LOCALES:
				args->value = (IMArgVal) & locales;
				break;
			case IF_SUPPORTED_OBJECTS:
				if (!objects) {
					int obj_id = 0;
					objects = (IMObjectDescriptorStruct *) calloc (4, sizeof (IMObjectDescriptorStruct));
					objects[obj_id].leid = "skk";
					objects[obj_id].type = IM_DOWNLOADINGOBJECT_BINGUI_TYPE;
					objects[obj_id].name = AuxName;
					objects[obj_id].name_length = 3;
					objects[obj_id].domain = "org.kondara";
					objects[obj_id].scope = "skk";
					objects[obj_id].path = "./locale/ja/skk/xaux/xaux.so";
					objects[obj_id].signature = "";
					obj_id++;
				}
				args->value = (IMArgVal) objects;
				break;
			case IF_NEED_THREAD_LOCK:
				args->value = (IMArgVal) True;
				break;
			default:
				break;
		}
	}
}

static Bool
if_skk_OpenIF (iml_if_t * If)
{
	int st;
	st = csconv_status ();
	if (st == CSC_UNLOADED) {
		setup_csconv ();
	} else if (st == CSC_FAILED) {
		return False;
	}
	return True;
}

static Bool
if_skk_CloseIF (iml_if_t * If)
{
	return True;
}

static Bool
if_skk_GetIFValues (iml_if_t * If, IMArgList args, int num_args)
{
#ifdef DEBUG
	printf ("if_skk_GetIFValues()\n");
	printf ("	This method is invoked when the Language Engine\n");
	printf ("	wants to set IF attributes of IM Client.\n\n");
#endif  
	return True;
}

static Bool
if_skk_SetIFValues (iml_if_t * If, IMArgList args, int num_args)
{
#ifdef DEBUG
	printf ("if_skk_SetIFValues()\n");
	printf ("	This method is invoked when the IM client\n");
	printf ("	wants to set IF attributes.\n\n");
#endif
	return True;
}

static Bool
if_skk_OpenDesktop (iml_desktop_t * desktop, IMArgList args, int num_args)
{
	int i;
	SkkDesktopData *d;

#ifdef DEBUG
	printf ("if_skk_OpenDesktop()\n");
	printf ("	This method is invoked when the user start to use\n");
	printf ("	the desktop, which means the first connection.\n");

	printf ("\n");
	printf ("	user [%s@%s] is entered\n",
	 desktop->user_name, desktop->host_name);
	printf ("\n");

	printf ("	The additional info of IM client are\n");
#endif

	for (i = 0; i < num_args; i++, args++) {
		switch (args->id) {
			case UI_USER_NAME:
				if (args->value) {
					printf (" UI_USER_NAME=%s\n", args->value);
				}
				break;
			case UI_HOST_NAME:
				if (args->value) {
					printf (" UI_HOST_NAME=%s\n", args->value);
				}
				break;
			case UI_DISPLAY_ID:
				if (args->value) {
					printf (" UI_DISPLAY_ID=%s\n", args->value);
				}
				break;
			case UI_PROTOCOL_TYPE:
				if (args->value) {
					printf (" UI_PROTOCOL_TYPE=%s\n", args->value);
				}
				break;
			case UI_CLIENT_TYPE:
				if (args->value) {
					printf (" UI_CLIENT_TYPE=%s\n", args->value);
				}
				break;
			case UI_XSERVER_VENDOR:
				if (args->value) {
					printf (" UI_XSERVER_VENDOR=%s\n", args->value);
				}
				break;
			case UI_OS_NAME:
				if (args->value) {
					printf (" UI_OS_NAME=%s\n", args->value);
				}
				break;
			case UI_OS_ARCH:
				if (args->value) {
					printf (" UI_OS_ARCH=%s\n", args->value);
				}
				break;
			case UI_OS_VERSION:
				if (args->value) {
					printf ("UI_OS_VERSION=%s\n", args->value);
				}
				break;
		}
	}
	d = g_new0 (SkkDesktopData, 1);
	d->skkfunc = skk_func_new ();
	d->skkquery = skk_query_new ();
	d->skkconf = skk_conf_new ();
	/* TODO configurable */
	skk_query_add_item (d->skkquery, skk_query_item_new (SKKQUERY_LOCAL));
	skk_query_add_item (d->skkquery, skk_query_item_new (SKKQUERY_SERVER));
	skk_query_add_item (d->skkquery, skk_query_item_new (SKKQUERY_LOOK));
	desktop->specific_data = (void *)d;
	return True;
}

static Bool
if_skk_CloseDesktop (iml_desktop_t * desktop)
{
	SkkDesktopData *d = (SkkDesktopData *)desktop->specific_data;
#ifdef DEBUG
	printf ("if_skk_CloseDesktop()\n");
	printf ("	This method is invoked when the user exits from\n");
	printf ("	the desktop, which means there is no IM client\n");
	printf ("	for the user.\n");

	printf ("\n");
	printf ("	user [%s@%s] exited\n",
			desktop->user_name, desktop->host_name);
	printf ("\n");
#endif
#if 0
		shutdown (skk_cur_sock, 2);
		skk_comm_bye (skk_cur_sock);
		skk_comm_bye (skk_cur_sock);
		skk_cur_sock = -1;
	}
#endif
	/*
	skk_aux_status_done (s);
	skk_aux_dict_done (s);
	*/
	if (d->skkquery) {
		skk_query_destroy (d->skkquery);
	}
	if (d->skkfunc) {
		skk_func_destroy (d->skkfunc);
	}
	if (d->skkconf) {
		skk_conf_destroy (d->skkconf);
	}
	g_free (d);
	return True;
}

static void
skk_result_listener (SkkBuffer *buf, gchar *value, gpointer user_data)
{
	iml_session_t *s = (iml_session_t *) user_data;
	if (!value) {
		return;
	}
	skk_commit_string (s, value);
	return;
}

static void
skk_preedit_listener (SkkBuffer *buf, gchar *value, gpointer user_data)
{
	iml_session_t *s = (iml_session_t *) user_data;
	if (!value)
		skk_preedit_clear (s);
	else {
		skk_preedit_draw (s, value);
	}
	return;
}

static void
skk_status_listener (SkkMode *buf, gint status, gpointer user_data)
{
	iml_session_t *s = (iml_session_t *) user_data;
	skk_status_draw (s);
}

static void
skk_lookup_listener (SkkBuffer *buf, gchar **label, gchar **candidate, gchar **annotate, gint count, gpointer user_data)
{
	int i;
	int max_len = 0;
	gint len[1];
	IMText *pvt;
	IMText *plt;
	IMLookupDrawCallbackStruct *draw;
	iml_inst *lp;
	iml_inst *rrv = NULL;
	iml_session_t *s = (iml_session_t *) user_data;

	if (count < 1) {
		/* no more choice */
		skk_lookup_choice_done (s);
		/* show AddDictAux */
		skk_buffer_adddict_emit (buf);
		return;
	}
	draw = ((IMLookupDrawCallbackStruct *)
			s->If->m->iml_new (s, sizeof (IMLookupDrawCallbackStruct)));
	memset (draw, 0, sizeof (IMLookupDrawCallbackStruct));
	draw->title = skk_utftoimtext (s, lookup_choice_title);
	draw->index_of_first_candidate = 0;
	draw->index_of_last_candidate = count - 1;
	draw->index_of_current_candidate = 0;
	draw->n_choices = count;
	draw->choices = ((IMChoiceObject *)s->If->m->iml_new (s, count * sizeof (IMChoiceObject)));
	memset (draw->choices, 0, count * sizeof (IMChoiceObject));
	for (i = 0; i < count; i++) {
		len[0] = strlen (label[i]);
		plt = draw->choices[i].label
			= skk_strtoimtext (s, 1, len, (unsigned char **)&label[i], NULL);
		len[0] = strlen (candidate[i]);
		pvt = draw->choices[i].value
			= skk_strtoimtext (s, 1, len, (unsigned char **)&candidate[i], NULL);
		if (max_len < pvt->char_length)
			max_len = pvt->char_length;
		if (max_len < plt->char_length)
			max_len = plt->char_length;
	}
	draw->max_len = max_len;
	skk_start_lookup_choice (s, &rrv, count);
	lp = s->If->m->iml_make_lookup_draw_inst (s, draw);
	s->If->m->iml_link_inst_tail (&rrv, lp);
	s->If->m->iml_execute (s, &rrv);
	return;
}

static void
skk_adddict_listener (SkkBuffer *buf, gchar *query_str, gpointer user_data)
{
	iml_session_t *s = (iml_session_t *) user_data;
	skk_aux_dict_draw (s, query_str);
	return;
}

static void
skk_codetable_listener (SkkBuffer *buf, gpointer user_data)
{
	iml_session_t *s = (iml_session_t *) user_data;
	int status[1];
	status[0] = SKK_AUX_PALETTE_CODE;
	skk_aux_draw (s, SKK_AUX_PALETTE_CLASS_NAME, 1, status, 0, NULL);
	return;
}

Bool
if_skk_CreateSC (iml_session_t * s, IMArgList args, int num_args)
{
	SkkDesktopData *d = skk_desktop_data (s);
	SkkSessionData *ses = g_new0 (SkkSessionData, 1);
	SkkBuffer *buf = skk_buffer_new ();
	skk_buffer_set_status (buf,SKK_LATIN_MODE);
	if (!ses)
		return False;
	ses->conversion_start = False;
	ses->status_start = False;
	ses->luc_start = False;
	ses->preedit_start = False;
	ses->dict_start = False;
	ses->palette_aux_ok = False;
	ses->dict_aux_ok = False;
	ses->luc_count = 0;
	ses->luc_hasmore = True;
	ses->context_id = skk_context_id_counter++;
	ses->skkbuf = buf;
	s->specific_data = (void *) ses;
	skk_buffer_set_query (buf, d->skkquery);
	skk_buffer_set_conf (buf, d->skkconf);
	skk_buffer_add_result_listener (buf, skk_result_listener, (gpointer) s);
	skk_buffer_add_preedit_listener (buf, skk_preedit_listener, (gpointer) s);
	skk_mode_add_status_listener (skkbuf_mode (buf), skk_status_listener, (gpointer) s);
	skk_mode_add_j_status_listener (skkbuf_mode (buf), skk_status_listener, (gpointer) s);
	skk_buffer_add_adddict_listener (buf, skk_adddict_listener, (gpointer) s);
	skk_buffer_add_lookup_listener (buf, skk_lookup_listener, (gpointer) s);
	skk_buffer_add_codetable_listener (buf, skk_codetable_listener, (gpointer) s);
	return True;
}

Bool
if_skk_DestroySC (iml_session_t * s)
{
	SkkSessionData *ses;
	ses = skk_session_data (s);
	skk_buffer_destroy (ses->skkbuf);
	g_free (ses);
	return True;
}

Bool
if_skk_GetSCValues (iml_session_t * s, IMArgList args, int num_args)
{
	int i;
	IMArg *p = args;

	static int charsubset[] = {
	    67,				/* LATIN */
	    47,				/* HIRAGANA */
	    48,				/* KATAKANA */
	    71,				/* KANJI */
	    0
	};

	for (i = 0; i < num_args; i++, p++) {
		switch (p->id) {
			case SC_SUPPORTED_CHARACTER_SUBSETS:
				/* specify CHARACTER_SUBSETS */
				p->value = (IMArgVal) charsubset;
				break;
			default:
				break;
		}
	}
	return True;
}

Bool
if_skk_SetSCValues (iml_session_t * s, IMArgList args, int num_args)
{
	int i;
	IMArg *p = args;

#ifdef DEBUG
	printf ("if_skk_SetSCValues(s=0x%x)\n", s);
	printf ("	This method is invoked when the IM client\n");
	printf ("	wants to set SC attributes.\n\n");
#endif

	for (i = 0; i < num_args; i++,p++) {
		switch (p->id) {
			case SC_TRIGGER_ON_NOTIFY:
				skk_make_conversion_on (s);
				break;
			case SC_TRIGGER_OFF_NOTIFY:
				skk_make_conversion_off (s);
				break;
			case SC_REALIZE:
				if (s->desktop->session_count == 1) {
#ifdef DEBUG_XAUX
					fprintf (stderr, "SC_REALIZE: just after OpenDesktop\n");
#endif
					skk_aux_status_start (s);
					skk_aux_dict_start (s);
				} else {
#ifdef DEBUG_XAUX
					printf ("	SC_REALIZE: received\n");
					skk_status_draw (s);
#endif
				}
				break;
			default:
				break;
		}
	}
#ifdef DEBUG_XAUX
	printf ("\n");
#endif
	return True;
}

IMText *
if_skk_ResetSC (iml_session_t *s)
{
	iml_inst *lp;
	lp = s->If->m->iml_make_preedit_erase_inst (s);
	s->If->m->iml_execute (s, &lp);
	return (IMText *) NULL;
}

void
if_skk_SetSCFocus (iml_session_t *s)
{
	skk_status_draw (s);
	return;
}

void
if_skk_UnsetSCFocus (iml_session_t * s)
{
#ifdef DEBUG
	printf ("if_skk_UnsetSCFocus(s=0x%x)\n", s);
	printf ("	This method is invoked when the IM client\n");
	printf ("	loses the input focus.\n\n");
#endif  
}

void
if_skk_SendEvent (iml_session_t *s, IMInputEvent *ev)
{
#ifdef DEBUG
	printf ("if_skk_SendEvent(s=0x%x,ev=%x)\n",s,ev);
	printf ("This method is invoked when any event occurs\n");
	printf ("on the IM client\n\n");
#endif
	if (ev) {
		switch (ev->type) {
			case IM_EventKeyList:
				skk_process_keyevent (s, (IMKeyListEvent *)ev);
				break;
			case IM_EventAux:
				printf ("IM_EventAux\n");
				{
					IMAuxDrawCallbackStruct *aux_data = ((IMAuxEvent *) ev)->aux;
					skk_process_auxevent (s, aux_data);
				}
				break;
			case IM_EventString:
				printf ("IM_EventString\n");
				break;
			case IM_EventText:
				printf ("IM_EventText\n");
			default:
				break;
		}
	}
	return;
}

/* */

static void
skk_make_conversion_on (iml_session_t *s)
{
	SkkSessionData *ses = skk_session_data (s);
	SkkBuffer *buf = skk_session_buf (s);
	if (ses->conversion_start == False) {
		ses->conversion_start = True;
		skk_buffer_set_katakana (buf, FALSE);
		skk_buffer_set_jisx0201_katakana (buf, FALSE);
		skk_buffer_set_status (buf, SKK_J_MODE);
	}
	return;
}

static void
skk_make_conversion_off (iml_session_t *s)
{
	SkkSessionData *ses = skk_session_data (s);
	if (ses->conversion_start == True) {
		ses->conversion_start = False;
		skk_status_draw (s);
		skk_status_done (s);
		skk_end_conversion (s);
	}
}

/* commit string */
static void
skk_commit_imtext (iml_session_t *s, IMText *text)
{
	iml_inst *lp;
	SkkSessionData *ses = skk_session_data (s);
	if (!text)
		return;

	/* erase preedit area first */
	if (ses->preedit_start == True) {
		skk_preedit_clear (s);
		ses->preedit_start = False;
	}
	/* commit string */
	lp = s->If->m->iml_make_commit_inst (s, text);
	s->If->m->iml_execute (s, &lp);
}

static void
skk_commit_utf_string (iml_session_t *s, UTFCHAR *ustr)
{
	IMText   *p;
	p = skk_utftoimtext (s, ustr);
	skk_commit_imtext (s, p);
}

static void
skk_commit_string (iml_session_t *s,  unsigned char *str)
{
	UTFCHAR *ustr;
	ustr = skk_strtoutf (str);
	skk_commit_utf_string (s, ustr);
	g_free (ustr);
}

/* modify AUX event to keyevent */
IMInputEvent *
modify_event (iml_session_t *s, IMInputEvent *ev)
{
	static IMKeyEventStruct akey;
	static IMKeyListEvent   keyev;
	if (ev->type == IM_EventAux) {
		IMAuxDrawCallbackStruct *aux_data = ((IMAuxEvent *) ev)->aux;
		if (aux_data->count_integer_values == 4 &&
				aux_data->integer_values[0] == SKK_AUXSETVAL_KEYPRESS)
		{
			akey.keyCode = aux_data->integer_values[1];
			akey.keyChar = aux_data->integer_values[2];
			akey.modifier = aux_data->integer_values[3];
			keyev.type = IM_EventKeyList;
			keyev.keylist = (IMKeyList) & akey;
			return (IMInputEvent *) & keyev;
		}
	}
	return ev;
}

/* translate keyevent */
/* if -1, then we don't need to handle it */
static int
skk_translate_keyevent (iml_session_t *s, IMKeyListEvent * kev)
{
  SkkBuffer *buf = skk_session_buf (s);
  SkkSessionData *ses = skk_session_data (s);
#if 0
  IMKeyEventStruct *k = (IMKeyEventStruct *) kev->keylist;
  printf ("iml_session_t() keycode=%x,keychar=%x,state=%x\n",
	  k->keyCode, k->keyChar, k->modifier);
  printf ("translate_keyevent status %d\n",skk_buffer_get_status (buf));
#endif
  if (ses->dict_start && ses->dict_aux_ok) {
	  return SKK_NOTHING;
  }
  if (ses->luc_start == True) {
	  return skk_translate_keyevent_luc (s, kev);
  }
  switch (skk_buffer_get_status (buf)) {
	  case SKK_J_MODE:
		  return skk_translate_keyevent_jmode (s, kev);
	  case SKK_LATIN_MODE:
		  return skk_translate_keyevent_latin_mode (s, kev);
	  case SKK_ABBREV_MODE:
		  return skk_translate_keyevent_abbrev_mode (s, kev);
	  case SKK_JISX0208_LATIN_MODE:
		  return skk_translate_keyevent_jisx0208_latin_mode (s, kev);
	  default:
		  return -1;
  }
}

static int
skk_translate_keyevent_jmode (iml_session_t *s, IMKeyListEvent *kev)
{
	IMKeyEventStruct *k = (IMKeyEventStruct *)kev->keylist;
	SkkBuffer *buf = skk_session_buf (s);

	switch (k->keyCode) {
		case IM_VK_BACK_SPACE:
			return SKK_VK_BACK_SPACE;
			break;
		case IM_VK_TAB:
			return SKK_VK_TAB;
			break;
		case IM_VK_ENTER:
			return SKK_VK_ENTER;
			break;
		case IM_VK_SPACE:
			if ((k->modifier & (IM_CTRL_MASK | IM_SHIFT_MASK))) {
				skk_buffer_commit (buf, FALSE);
				skk_buffer_clear (buf);
				skk_make_conversion_off (s);
				return SKK_NOTHING;
			}
			/* fall through */
		default:
			if ((k->keyChar > 0) && (k->keyChar < 0xFFFF)) {
				return k->keyChar;
			}
			break;
	}
	return SKK_NOTRANS;
}

static int
skk_translate_keyevent_abbrev_mode (iml_session_t *s, IMKeyListEvent *kev)
{
	IMKeyEventStruct *k = (IMKeyEventStruct *)kev->keylist;
	SkkBuffer *buf = skk_session_buf (s);
	switch (k->keyCode) {
		case IM_VK_BACK_SPACE:
			return SKK_VK_BACK_SPACE;
			break;
		case IM_VK_ENTER:
			return SKK_VK_ENTER;
			break;
		case IM_VK_TAB:
			return SKK_VK_TAB;
			break;
		case IM_VK_SPACE:
			if ((k->modifier & (IM_CTRL_MASK | IM_SHIFT_MASK))) {
				skk_buffer_commit (buf, FALSE);
				skk_buffer_clear (buf);
				skk_make_conversion_off (s);
				return SKK_NOTHING;
			}
			/* fall through */
		default:
			if ((k->keyChar > 0) && (k->keyChar < 0xFFFF)) {
				return k->keyChar;
			}
			break;
	}
	return 0;
}

static int
skk_translate_keyevent_latin_mode (iml_session_t *s, IMKeyListEvent *kev)
{
	IMKeyEventStruct *k = (IMKeyEventStruct *)kev->keylist;
	switch (k->keyCode) {
		case IM_VK_SPACE:
			if ((k->modifier & (IM_CTRL_MASK | IM_SHIFT_MASK))) {
				skk_make_conversion_off (s);
				return SKK_NOTHING;
			}
			/* fall through */
		default:
			if ((k->keyChar > 0) && (k->keyChar < 0xFFFF)) {
				return k->keyChar;
			}
			break;
	}
	return 0;
}

static int
skk_translate_keyevent_luc (iml_session_t *s, IMKeyListEvent *kev)
{
	IMKeyEventStruct *k = (IMKeyEventStruct *)kev->keylist;
	SkkBuffer *buf = skk_session_buf (s);
	gchar *candidate;
	gint offset = 7;
	gint num = -1;

	/* SKK uses 7 candidates at once */
	if (buf->candidate_max - buf->candidate_count < 7) {
		offset = buf->candidate_max - buf->candidate_count;
	}
	switch (k->keyCode) {
		case IM_VK_A:
			num = 0;
			goto trans;
			break;
		case IM_VK_S:
			num = 1;
			goto trans;
			break;
		case IM_VK_D:
			num = 2;
			goto trans;
			break;
		case IM_VK_F:
			num = 3;
			goto trans;
			break;
		case IM_VK_J:
			num = 4;
			goto trans;
			break;
		case IM_VK_K:
			num = 5;
			goto trans;
			break;
		case IM_VK_L:
			num = 6;
			goto trans;
			break;
		case IM_VK_G:
			if (k->modifier & IM_CTRL_MASK) {
				buf->candidate_count = 0;
				skk_lookup_choice_done (s);
				skk_buffer_undo (buf);
				return SKK_TRANS;
			}
			break;
		case IM_VK_X:
			if ( (buf->candidate_count - 7)  < 0) {
				/* we have less than 7 candidates */
				/* so we do
				 *   finished luc.
				 *   undo buffer
				 */
				skk_lookup_choice_done (s);
				skk_buffer_undo (buf);
				return SKK_NOTHING;
			} else {
				/* decrement */
				buf->candidate_count -= 7;
				skk_buffer_lookup_emit (buf);
				return SKK_NOTHING;
			}
			break;
		case IM_VK_SPACE:
			buf->candidate_count += offset;
			skk_buffer_lookup_emit (buf);
			return SKK_NOTHING;
			break;
		default:
			return SKK_NOTHING;
			break;
	}
	return SKK_NOTHING;

trans:
			if (num > offset) {
				/* Offset holds our candidates number */
				/* In this case, we have less than 7 candididates, */
				/* and pressed key indicates number we don't have */
				return SKK_NOTHING;
			}
			skk_lookup_choice_done (s);
			candidate = skk_buffer_get_nth_candidate (buf, buf->candidate_count + num);
			if (candidate) {
				skk_commit_string (s, candidate);
				skk_buffer_add_dict_with_value (buf, candidate);
				skk_buffer_clear (buf);
				g_free (candidate);
			}
			return SKK_NOTHING;
}

static int
skk_translate_keyevent_jisx0208_latin_mode (iml_session_t *s, IMKeyListEvent *kev)
{
	IMKeyEventStruct *k = (IMKeyEventStruct *)kev->keylist;
	switch (k->keyCode) {
		case IM_VK_SPACE:
			if ((k->modifier & (IM_CTRL_MASK | IM_SHIFT_MASK))) {
				skk_make_conversion_off (s);
				return SKK_NOTHING;
			}
			/* fall through */
		default:
			if ((k->keyChar > 0) && (k->keyChar < 0xFFFF)) {
				return k->keyChar;
			}
			break;
	}
	return SKK_NOTRANS;
}

static void
skk_process_keyevent_abbrev (iml_session_t *s, IMKeyListEvent *kev)
{
	SkkBuffer *buf = skk_session_buf (s);
	if (skk_buffer_can_commit (buf)) {
		skk_commit_string (s, skkbuf_r (buf));
	}
	if (skk_buffer_has_preedit (buf)) {
		gchar *preedit = skk_buffer_get_preedit_string (buf);
		skk_preedit_clear (s);
		skk_preedit_draw (s, preedit);
		g_free (preedit);
	} else {
		skk_preedit_clear (s);
	}
	if (skk_buffer_can_clear (buf)) {
		skk_buffer_clear (buf);
		skk_buffer_set_status (buf, SKK_J_MODE); /* return to default */
	}
	return;
}

static void
skk_process_keyevent_jisx0208_latin (iml_session_t *s, IMKeyListEvent *kev)
{
	SkkBuffer *buf = skk_session_buf (s);
	if (skk_buffer_can_clear (buf)) {
		skk_buffer_clear (buf);
		skk_buffer_set_status (buf, SKK_JISX0208_LATIN_MODE);
	}
	return;
}

static void
skk_process_keyevent_jmode (iml_session_t *s, IMKeyListEvent *kev)
{
	SkkBuffer *buf = skk_session_buf (s);
	DEBUG_DO (printf ("skk_process_keyevent_jmode\n"));
	DEBUG_DO (g_message ("skkbuf result (%s)\n", skkbuf_r (buf)));
	DEBUG_DO (g_message ("skkbuf preedit (%s)\n", skkbuf_p (buf)));
	DEBUG_DO (g_message ("skkbuf direction_word (%s)\n", skkbuf_d (buf)));
	DEBUG_DO (g_message ("skkbuf in_buf (%s)\n", skkbuf_b (buf)));
	#if 0
	if (skk_buffer_can_commit (buf)) {
		skk_commit_string (s, skkbuf_r (buf));
		/* once commit */
	}
	if (skk_buffer_has_preedit (buf)) {
		gchar *preedit = skk_buffer_get_preedit_string (buf);
		skk_preedit_clear (s);
		skk_preedit_draw (s,preedit);
		g_free (preedit);
	} else {
		skk_preedit_clear (s);
	}
	#endif
	if (skk_buffer_can_clear (buf)) {
		skk_buffer_clear (buf);
	}
	return;
}

static void
skk_commit_key (iml_session_t *s, IMKeyListEvent *kev)
{
	iml_inst *lp;
	lp = s->If->m->iml_make_keypress_inst (s, ((IMKeyEventStruct *)kev->keylist));
	s->If->m->iml_execute (s,&lp);
	return;
}

static void
skk_process_keyevent (iml_session_t * s, IMKeyListEvent * kev)
{
	int ch;
	int status;
	guint skk_status = 0;
	SkkDesktopData *d = skk_desktop_data (s);
	SkkBuffer *buf = skk_session_buf (s);
	IMKeyEventStruct *k = (IMKeyEventStruct *)kev->keylist;
	ch = skk_translate_keyevent (s, kev);
	if (ch == SKK_NOTRANS) {
		skk_commit_key (s, kev);
		return;
	} else if (ch == SKK_NOTHING) {
		return;
	}
	/* FIXME switch is OK ? ex) pressing Alt + Shift */
	if ((k->modifier & IM_ALT_MASK)) {
		skk_status |= SKK_ALT_MASK;
	}
	if ((k->modifier & IM_SHIFT_MASK)) {
		skk_status |= SKK_SHIFT_MASK;
	}
	if ((k->modifier & IM_CTRL_MASK)) {
		skk_status |= SKK_CONTROL_MASK;
	}
	if (!skk_status) {
		skk_status = SKK_NONE_MASK;
	}
	status = skk_keymap_do_func (buf, d->skkfunc, ch, skk_status);
	switch (status) {
		case SKK_NOTHING:
			return;
			break;
		case SKK_NOTRANS:
			skk_commit_key (s, kev);
			return;
			break;
		case SKK_TRANS:
			break;
	}
	switch (skk_buffer_get_status (buf)) {
		case SKK_J_MODE:
			skk_process_keyevent_jmode (s,kev);
			break;
		case SKK_JISX0208_LATIN_MODE:
			skk_process_keyevent_jisx0208_latin (s,kev);
			break;
		case SKK_ABBREV_MODE:
			skk_process_keyevent_abbrev (s, kev);
			break;
		case SKK_LATIN_MODE:
			skk_commit_key (s,kev);
			break;
	}
	return;
}

/* FEEDBACK */

static IMFeedbackList *create_feedback (iml_session_t * s, int size);
static IMFeedbackList *create_feedback2 (iml_session_t * s, int size);
static int get_feedback (IMFeedbackList * fbl);
static void set_feedback_private (IMFeedbackList * fbl,
			   int normalfeedback, int fg, int bg, int underline);
static void set_skk_feedback_1 (IMFeedbackList * fbl, int feedback_type);
static void set_skk_feedback (IMFeedbackList * fbl, int feedback_type, int st,
		       int end);
static IMFeedbackList *create_skk_feedback (iml_session_t * s, int size,
				     int normalfeedback, int fg, int bg);


static IMFeedbackList*
create_feedback (iml_session_t *s, int size)
{
	int i;
	IMFeedbackList *feedback;
	IMFeedback *fb;
	if (!s)
		return NULL;
	feedback = ((IMFeedbackList *)s->If->m->iml_new (s,sizeof (IMFeedbackList) *size));
	for (i = 0; i < size; i++) {
		IMFeedbackList *fbl = &feedback[i];
		fbl->count_feedbacks = 1;
		fb = ((IMFeedback *) s->If->m->iml_new (s, sizeof (IMFeedback) * 4));
		fbl->feedbacks = fb;
		memset (fbl->feedbacks, 0, sizeof (IMFeedback) * 4);
	}
	return feedback;
}

static IMFeedbackList *
create_feedback2 (iml_session_t * s, int size)
{
  int i;
  IMFeedbackList *feedback;
  IMFeedback *fb;

  if (!s)
    return NULL;

  feedback = ((IMFeedbackList *)
	      s->If->m->iml_new2 (s, sizeof (IMFeedbackList) * size));
  for (i = 0; i < size; i++)
    {
      IMFeedbackList *fbl = &feedback[i];
      fbl->count_feedbacks = 1;
      fb = ((IMFeedback *) s->If->m->iml_new2 (s, sizeof (IMFeedback) * 4));
      fbl->feedbacks = fb;
      memset (fbl->feedbacks, 0, sizeof (IMFeedback) * 4);
    }
  return feedback;
}

static int
get_feedback (IMFeedbackList *fbl)
{
	/* returns IM_DECORATION_FEEDBACK */
	IMFeedback *fb = &fbl->feedbacks[0];
	return IM_FEEDBACK_VALUE (fb);
}

static void
set_feedback_private (IMFeedbackList * fbl,
		      int normalfeedback, int fg, int bg, int underline)
{
  int count = 0;
  IMFeedback *fb;

  fb = &fbl->feedbacks[count];
  IM_FEEDBACK_TYPE (fb) = IM_DECORATION_FEEDBACK;
  IM_FEEDBACK_VALUE (fb) = normalfeedback;
  count++;

#ifdef  USE_COLOR_FEEDBACK

  if (fg != -1)
    {
      fb = &fbl->feedbacks[count];
      IM_FEEDBACK_TYPE (fb) = IM_FOREGROUND_RGB_FEEDBACK;
      IM_FEEDBACK_VALUE (fb) = fg;
      count++;
    }
  if (bg != -1)
    {
      fb = &fbl->feedbacks[count];
      IM_FEEDBACK_TYPE (fb) = IM_BACKGROUND_RGB_FEEDBACK;
      IM_FEEDBACK_VALUE (fb) = bg;
      count++;
    }
  if (underline != -1)
    {
      fb = &fbl->feedbacks[count];
      IM_FEEDBACK_TYPE (fb) = IM_UNDERLINE_RGB_FEEDBACK;
      IM_FEEDBACK_VALUE (fb) = underline;
      count++;
    }

#endif
  IM_FEEDBACK_COUNT (fbl) = count;
}

static void
set_skk_feedback_1 (IMFeedbackList * fbl, int feedback_type)
{

  switch (feedback_type)
    {
    case SKK_Feedback_Input:
      set_feedback_private (fbl, IMUnderline, IM_RGB_COLOR (0, 0, 255),	/* FG: blue */
			    IM_RGB_COLOR (255, 255, 255),	/* BG: white */
			    0);	/* Underline */
      break;
    case SKK_Feedback_Strong:
      set_feedback_private (fbl, IMReverse, IM_RGB_COLOR (255, 255, 255),	/* FG: white */
			    IM_RGB_COLOR (0, 0, 255),	/* BG: blue */
			    -1);
      break;
    case SKK_Feedback_Normal:
    default:
      set_feedback_private (fbl, IMNormal, -1, -1, -1);
      break;
    }
  return;

}

static void
set_skk_feedback (IMFeedbackList *fbl, int feedback_type, int st, int end)
{
	for (; st < end; st++)
		set_skk_feedback_1 ((fbl + st), feedback_type);
	return;
}

static IMFeedbackList*
create_skk_feedback (iml_session_t *s,
		int size, int normalfeedback, int fg, int bg)
{
	int i;
	IMFeedbackList *feedback;
	feedback = (IMFeedbackList *) create_feedback (s, size);
	for (i = 0; i < size; i++) {
		IMFeedbackList *fbl = &feedback[i];
		set_feedback_private (fbl, normalfeedback, fg, bg, -1);
	}
	return feedback;
}

/* IMText */

static IMText*
skk_create_imtext (iml_session_t *s, int len)
{
	IMText *ret;
	if (!s)
		return NULL;
	ret = (IMText *) s->If->m->iml_new (s, sizeof (IMText));
	memset (ret, 0, sizeof (IMText));
	ret->encoding = UTF16_CODESET;
	ret->text.utf_chars = ((UTFCHAR *)s->If->m->iml_new (s, sizeof (UTFCHAR) * (len + 1)));
	ret->text.utf_chars[len+1] = 0;
	ret->char_length = len;
	ret->feedback = create_feedback (s, len);
	
	return ret;
}

static IMText*
skk_utftoimtext (iml_session_t *s, UTFCHAR *ustr)
{
	IMText *ret;
	UTFCHAR *p = ustr;
	int utflen;
	for (utflen = 0; *p; p++)
		utflen++;
	ret = skk_create_imtext (s, utflen);
	if (!ret) 
		return NULL;
	memcpy (ret->text.utf_chars, ustr, (utflen + 1) * sizeof (UTFCHAR));
	return ret;
}

/*
  String conversion:
*/
static int
skk_utflen (int skk_str_len)
{
	return skk_str_len * sizeof (UTFCHAR);
}

/* Convert String to UTFCHAR */
static UTFCHAR*
skk_strtoutf (const unsigned char *str)
{
	int ret, strlength, ustrlength;
	UTFCHAR *p,*ustr = NULL;
	strlength = strlen (str);
	ustrlength = skk_utflen (strlength + 1);
	p = ustr = (UTFCHAR *) malloc (ustrlength);
	ret = csc_conv (cd_euc2utf,
			(const char **) &str, &strlength,
			(char **) &p, &ustrlength);
	*p = 0;
	return ustr;
}

/* Convert UTFCHAR to String */
static char*
skk_utftostr (UTFCHAR *ustr)
{
	int ret;
	int str_len;
	int ustr_len;
	int ustr_bytes;
	UTFCHAR *up;
	char *str;
	char *p;
	for (ustr_len = 0, up = ustr; *up; up++) {
		ustr_len++;
	}
	str_len = (ustr_len * 3) + 1;
	str = p = (char* )malloc (str_len);
	ustr_bytes = ((ustr_len + 1) * sizeof (UTFCHAR));
	ret = csc_conv (cd_utf2euc,
			(const char **) &ustr, &ustr_bytes,
			(char **)&p, &str_len);
	*p = 0;
	return str;
}

/* Convert String to IMText */
static IMText*
skk_strtoimtext (iml_session_t *s,
		int nseg, int *nb,
		unsigned char **strs,
		int *feedback_type)
{
	UTFCHAR *ustr, *p;
	IMText *pit;
	gchar *from;
	int from_size;
	int i, ret, clen, ulen, uidx;
	int *idices;

	clen = 0;
	for (i = 0; i < nseg; i++) {
		clen += nb[i];
	}
	ulen = skk_utflen (clen + 1);
	ustr = p = (UTFCHAR *) alloca (ulen);
	idices = (int *) alloca (sizeof (int) * (nseg + 1));

	for (i = 0; i < nseg; i++) {
		from = strs[i];
		from_size = nb[i];
		idices[i] = p -ustr;
		ret = csc_conv (cd_euc2utf,
				(const char **) &from, &from_size, (char **) &p, &ulen);
	}
	*p = 0;
	uidx = p - ustr;
	idices[nseg] = uidx;
	pit = skk_create_imtext (s, uidx);
	if (!pit)
		return NULL;
	memcpy (pit->text.utf_chars, ustr, (p - ustr + 1) * sizeof (UTFCHAR));
	if (feedback_type) {
		/* set feedback */
		for (i = 0; i < nseg; i++) {
			set_skk_feedback (pit->feedback, feedback_type[i],
					idices[i], idices[i + 1]);
		}
	}
	return pit;
}

/* initialize */


Bool
skk_init (gchar *if_path)
{
	/*
  gchar **warning = NULL;
  gchar *init_filename;
  */

#if 0
  init_filename = canna_init_filename (if_path);
  if (init_filename)
    {
      free (init_filename);
    }
#endif
  return True;
}


/* Draw Preedit Area */
static void
skk_preedit_draw (iml_session_t *s, unsigned char *str)
{
	iml_inst *lp;
	IMText *p;
	SkkSessionData *sses = skk_session_data (s);

	if (sses->preedit_start == False) {
		lp = s->If->m->iml_make_preedit_start_inst (s);
		s->If->m->iml_execute (s, &lp);
		sses->preedit_start = True;
	}
	{
		/* Create IMText with feedbacks */
		int nb[1],fts[1];
		unsigned char *strs[1];
		nb[0] = strlen(str);
		fts[0] = SKK_Feedback_Input;
		strs[0] = str;
		p = skk_strtoimtext (s, 1, nb, strs, fts);
	}
	lp = s->If->m->iml_make_preedit_erase_inst (s);
	s->If->m->iml_execute (s,&lp);
	lp = s->If->m->iml_make_preedit_draw_inst (s,p);
	s->If->m->iml_execute (s, &lp);
}

/* PreEditing Done */
static void
skk_preedit_done (iml_session_t *s)
{
	SkkSessionData *ses = skk_session_data (s);
	if (ses->preedit_start == True) {
		iml_inst *lp;
		lp = s->If->m->iml_make_preedit_done_inst (s);
		s->If->m->iml_execute (s, &lp);
		ses->preedit_start = False;
	}
}

static void
skk_preedit_clear (iml_session_t *s)
{
	SkkSessionData *ses = skk_session_data (s);
	if (ses->preedit_start == True) {
		iml_inst *lp;
		lp = s->If->m->iml_make_preedit_erase_inst (s);
		s->If->m->iml_execute (s,&lp);
		lp = s->If->m->iml_make_preedit_done_inst (s);
		s->If->m->iml_execute (s,&lp);
		ses->preedit_start = False;
	}
}

/* Draw Status Area */
static void
skk_aux_status_draw_off (iml_session_t *s)
{
	int send_status[1];
	UTFCHAR *send_string[1];
	SkkSessionData *ses = skk_session_data (s);
	if (ses->palette_aux_ok == False) {
		skk_aux_start (s, SKK_AUX_PALETTE_CLASS_NAME);
		/* Send Initialize Request */
		send_status[0] = SKK_AUX_PALETTE_INIT;
		skk_aux_draw (s, SKK_AUX_PALETTE_CLASS_NAME,
				1, send_status, 0, NULL);
	} /* TODO */
	send_status[0] = SKK_MODE_OFF;
	send_string[0] = skk_strtoutf ("OFF");
	skk_aux_draw (s, SKK_AUX_PALETTE_CLASS_NAME, 1,
			send_status, 1, send_string);
	free (send_string[0]);
	return;
}

static void
skk_aux_status_draw (iml_session_t *s)
{
	int send_status[1];
	gchar *str;
	UTFCHAR *ustr;
	UTFCHAR *send_string[1];
	SkkSessionData *ses = skk_session_data (s);
	SkkBuffer *buf = skk_session_buf (s);
	SkkStatus status = skk_buffer_get_status (buf);

	if (ses->palette_aux_ok == False)  {
		skk_aux_start (s, SKK_AUX_PALETTE_CLASS_NAME);
		/* send initialize request */
		send_status[0] = SKK_AUX_PALETTE_INIT;
		skk_aux_draw (s, SKK_AUX_PALETTE_CLASS_NAME,
				1, send_status, 0, NULL);
	} /* TODO */
	switch (status) {
		case SKK_J_MODE:
			/* check we are in JISX0201_KANA mode */
			if (skk_buffer_get_jisx0201_katakana (buf)) {
				send_status[0] = SKK_MODE_JISX0201_KANA;
				break;
			}
			/* check we are in KATAKANA mode */
			if (skk_buffer_get_katakana (buf)) {
				send_status[0] = SKK_MODE_JMODE_KATA;
			} else {
				send_status[0] = SKK_MODE_JMODE_HIRA;
			}
			break;
		case SKK_ABBREV_MODE:
			send_status[0] = SKK_MODE_ABBREV;
			break;
		case SKK_JISX0208_LATIN_MODE:
			send_status[0] = SKK_MODE_JISX0208_LATIN;
			break;
		case SKK_LATIN_MODE:
			send_status[0] = SKK_MODE_LATIN;
			break;
		default:
			break;
	}
	str = (gchar*) skk_buffer_get_status_string (buf);
	ustr = skk_strtoutf (str);
	send_string[0] = ustr;
	skk_aux_draw (s, SKK_AUX_PALETTE_CLASS_NAME, 1, send_status, 1, send_string);
	g_free (ustr);
	return;
}

static void
skk_status_draw_off (iml_session_t *s)
{
	iml_inst *lp;
	iml_inst *rrv = NULL;
	IMText *p;
	UTFCHAR *ustr;
	SkkSessionData *ses = skk_session_data (s);
	ustr = skk_strtoutf ("SKK(OFF)");
	p = skk_utftoimtext (s, ustr);
	if (ses->status_start == False) {
		lp = s->If->m->iml_make_status_start_inst (s);
		s->If->m->iml_link_inst_tail (&rrv, lp);
		ses->status_start = True;
	}
	lp = s->If->m->iml_make_status_draw_inst (s, p);
	s->If->m->iml_link_inst_tail (&rrv, lp);
	s->If->m->iml_execute (s, &lp);
	g_free (ustr);

	skk_aux_status_draw_off (s);
	return;
}

static void
skk_status_draw (iml_session_t *s)
{
	iml_inst *lp;
	iml_inst *rrv = NULL;
	IMText *p;
	UTFCHAR *ustr;
	gchar *str;
	SkkSessionData *ses = skk_session_data (s);
	SkkBuffer *buf = skk_session_buf (s);

	if (ses->conversion_start == False) {
		skk_status_draw_off (s);
		return;
	}
	str = (gchar*)skk_buffer_get_status_string (buf);
	ustr = skk_strtoutf (str);
	p = skk_utftoimtext (s,ustr);
	if (ses->status_start == False) {
		lp = s->If->m->iml_make_status_start_inst (s);
		s->If->m->iml_link_inst_tail (&rrv,lp);
		ses->status_start = True;
	}
	lp = s->If->m->iml_make_status_draw_inst (s,p);
	s->If->m->iml_link_inst_tail (&rrv,lp);
	s->If->m->iml_execute (s, &rrv);
	g_free (ustr);

	/* Send Message to AUX */
	skk_aux_status_draw (s);
	return;
}

static void
skk_start_lookup_choice (iml_session_t *s, iml_inst **prrv, int num)
{
	SkkSessionData *ses = skk_session_data (s);
	if (ses->luc_start == False) {
		iml_inst *lp;
		IMLookupStartCallbackStruct *start;
		start = ((IMLookupStartCallbackStruct *)
				s->If->m->iml_new (s, sizeof (IMLookupStartCallbackStruct)));
		start->whoIsMaster = IMIsMaster;
		start->IMPreference = (LayoutInfo *) s->If->m->iml_new (s, sizeof (LayoutInfo));
		memset (start->IMPreference, 0, sizeof (LayoutInfo));
		start->IMPreference->choice_per_window = strlen ("asdfjkl");
		start->IMPreference->ncolumns = strlen ("asdfjkl");
		start->IMPreference->drawUpDirection = DrawUpVertically;
		start->IMPreference->whoOwnsLabel = IMOwnsLabel;
		start->CBPreference = NULL;

		lp = s->If->m->iml_make_lookup_start_inst (s,start);
		s->If->m->iml_link_inst_tail (prrv,lp);
		ses->luc_start = True;
	}
}

static void
skk_show_lookup_choice (iml_session_t *s)
{
	int num;
	int actual_count = 0;
	int cur_candidate_count;
	iml_inst *lp;
	iml_inst *rrv = NULL;
	IMLookupDrawCallbackStruct *draw;
	SkkBuffer *buf = skk_session_buf (s);

	/* Save Current Candidate's Number */
	cur_candidate_count = buf->candidate_count;
	/* SKK shows 7 candidates at once */
	num = strlen ("asdfjkl");
	/* Check we have 7 or more candidates */
	if ( (buf->candidate_max - buf->candidate_count) < 7) {
		/* don't have */
		num = buf->candidate_max - buf->candidate_count;
		if (num < 1) {
			/* no more choice */
			skk_lookup_choice_done (s);
			/* show AddDictAux */
			skk_buffer_adddict_emit (buf);
			return;
		}
	}

	/* Create IMLookupDrawCallbackStruct */
	draw = ((IMLookupDrawCallbackStruct *)
			s->If->m->iml_new(s,sizeof (IMLookupDrawCallbackStruct)));
	memset (draw, 0, sizeof (IMLookupDrawCallbackStruct));
	draw->title = skk_utftoimtext (s, lookup_choice_title);
	draw->index_of_first_candidate = 0;
	draw->index_of_last_candidate = num - 1;
	draw->index_of_current_candidate = 0;
	draw->n_choices = num;
	draw->choices = ((IMChoiceObject *)s->If->m->iml_new (s, num * sizeof (IMChoiceObject)));
	memset (draw->choices, 0, num * sizeof (IMChoiceObject));

	/* set choices */
	{
		int i;
		int len[1];
		IMText *pvt;
		IMText *plt;
		int max_len = 0;
		gchar *tmp[1];
		gchar *label[7] = {"A:","S:","D:","F:","J:","K:","L:"};
		for (i = 0, tmp[0] = skk_buffer_get_cur_candidate (buf);
				i < num && tmp[0] && *tmp[0] ; tmp[0] = skk_buffer_get_cur_candidate (buf))
		{
			printf ("%s\n",tmp[0]);
			len[0] = 2;
			plt = draw->choices[actual_count].label
				= skk_strtoimtext (s,1,len,(unsigned char**)&label[actual_count],NULL);
			len[0] = strlen (tmp[0]);
			pvt = draw->choices[actual_count].value 
				= skk_strtoimtext (s,1,len,(unsigned char**)tmp,NULL);
			if (max_len < pvt->char_length)
				max_len = pvt->char_length;
			if (max_len < plt->char_length)
				max_len = plt->char_length;
			actual_count++;
			i++;
			g_free (tmp[0]);
			if (!skk_buffer_has_next_candidate (buf)) {
				break;
			} else {
				buf->candidate_count++;
			}
		}
		draw->max_len = max_len;
	}
	skk_start_lookup_choice (s, &rrv, actual_count);
	lp = s->If->m->iml_make_lookup_draw_inst (s,draw);
	s->If->m->iml_link_inst_tail (&rrv,lp);
	s->If->m->iml_execute (s, &rrv);
	/* reset */
	buf->candidate_count = cur_candidate_count;
	return;
}

static void
skk_lookup_choice_done (iml_session_t *s)
{
	SkkSessionData *ses = skk_session_data (s);
	if (ses->luc_start == True) {
		iml_inst *lp;
		lp = s->If->m->iml_make_lookup_done_inst (s);
		s->If->m->iml_execute (s, &lp);
		ses->luc_start = False;
		ses->luc_count = 0;
	}
}

/* Aux */
static void
skk_aux_start (iml_session_t *s, char *class_name)
{
	iml_inst *lp;
	IMAuxStartCallbackStruct *aux = (IMAuxStartCallbackStruct *)
		s->If->m->iml_new (s, sizeof (IMAuxStartCallbackStruct));
	aux->aux_name = class_name;
	lp = s->If->m->iml_make_aux_start_inst (s, aux);
	s->If->m->iml_execute (s, &lp);
	return;
}

static void
skk_aux_done (iml_session_t *s, char *class_name)
{
	iml_inst *lp;
	IMAuxDoneCallbackStruct *aux = (IMAuxDoneCallbackStruct *)
		s->If->m->iml_new (s, sizeof (IMAuxDoneCallbackStruct));
	aux->aux_name = class_name;
	lp = s->If->m->iml_make_aux_done_inst (s, aux);
	s->If->m->iml_execute (s, &lp);
	return;
}

static void
skk_aux_status_start (iml_session_t *s)
{
	skk_aux_start (s, SKK_AUX_PALETTE_CLASS_NAME);
	return;
}

static void
skk_aux_dict_start (iml_session_t *s)
{
	skk_aux_start (s, SKK_AUX_ADD_DICT_CLASS_NAME);
	return;
}

static void
skk_aux_draw (iml_session_t *s, char *class_name, int int_count, int *integers,
		int str_count, UTFCHAR **strings)
{
	iml_inst *lp;
	int i,len;
	IMText *lts, *lt;
	IMAuxDrawCallbackStruct *aux;
	aux = (IMAuxDrawCallbackStruct *) s->If->m->iml_new (s, sizeof (IMAuxDrawCallbackStruct));
	memset (aux, 0, sizeof (IMAuxDrawCallbackStruct));
	aux->aux_name = class_name;
	aux->count_integer_values = int_count;
	if (aux->count_integer_values) {
		aux->integer_values = (int *) s->If->m->iml_new (s, sizeof (int) * aux->count_integer_values);
		memset (aux->integer_values, 0, sizeof (int) * aux->count_integer_values);

		for (i = 0; i < aux->count_integer_values; i++) {
			aux->integer_values[i] = integers[i];
		}
	}
	aux->count_string_values = str_count;
	if (aux->count_string_values) {
		aux->string_values = lts = (IMText *) s->If->m->iml_new (s, sizeof (IMText) * aux->count_string_values);
		memset (aux->string_values, 0, sizeof (IMText) * aux->count_string_values);
		for (i = 0, lt = lts; i < aux->count_string_values; i++, lt++) {
			UTFCHAR *p = strings[i];
			for (len = 0; *p; p++)
				len++;
			lt->text.utf_chars = (UTFCHAR *) s->If->m->iml_new (s, sizeof (UTFCHAR) * len);
			lt->char_length = len;
			memcpy (lt->text.utf_chars, strings[i], sizeof (UTFCHAR) * len);
		}
	}
	lp = s->If->m->iml_make_aux_draw_inst (s, aux);
	s->If->m->iml_execute (s, &lp);
	return;
}

static void
skk_process_auxevent (iml_session_t *s, IMAuxDrawCallbackStruct *aux)
{
	int i;
	SkkSessionData *ses = skk_session_data (s);
	SkkBuffer *buf = skk_session_buf (s);
	for (i = 0; i < aux->count_integer_values; i++) {
		switch (aux->integer_values[i]) {
			case SKK_MODE_JMODE_HIRA:
				skk_buffer_clear (buf);
				skk_buffer_set_katakana (buf, FALSE);
				skk_status_draw (s);
				break;
			case SKK_MODE_JMODE_KATA:
				skk_buffer_clear (buf);
				skk_buffer_set_katakana (buf, TRUE);
				skk_status_draw (s);
				break;
			case SKK_MODE_LATIN:
				skk_buffer_clear (buf);
				skk_buffer_set_status (buf, SKK_LATIN_MODE);
				break;
			case SKK_MODE_JISX0201_KANA:
				skk_buffer_clear (buf);
				skk_buffer_set_jisx0201_katakana (buf, TRUE);
				skk_status_draw (s);
				break;
			case SKK_MODE_JISX0208_LATIN:
				skk_buffer_clear (buf);
				skk_buffer_set_status (buf, SKK_JISX0208_LATIN_MODE);
				break;
			case SKK_MODE_ABBREV:
				skk_buffer_clear (buf);
				skk_buffer_set_status (buf, SKK_ABBREV_MODE);
				break;
			case SKK_AUX_CODE_DEFINED:
				{
					IMText *lt = NULL;
					lt = aux->string_values;
					skk_commit_imtext (s, lt);
				}
				break;
			case SKK_AUX_DICT_ADD:
				{
					/* TODO annotation */
					IMText *lt = NULL;
					int type;
					i++;
					type = aux->integer_values[i];
					lt = aux->string_values;
					if (type == 0) {
						gchar *value;
						skk_commit_imtext (s, lt);
						value = skk_utftostr (lt->text.utf_chars);
						skk_buffer_add_dict_with_value (buf, value);
						g_free (value);
						if (ses->dict_start == True) {
							skk_buffer_clear (buf);
							ses->dict_start = False;
						}
					} else if (type == 1) {
						/* Undo */
						buf->candidate_count = 0;
						skk_buffer_undo (buf);
						if (ses->dict_start == True) {
							ses->dict_start = False;
						}
					}
				}
				break;
			case SKK_AUX_PALETTE_OK:
				DEBUG_DO (printf ("palette initialized\n"));
				ses->palette_aux_ok = True;
				break;
			case SKK_AUX_DICT_OK:
				DEBUG_DO (printf ("dict initialized\n"));
				ses->dict_aux_ok = True;
				break;
			default:
				break;
		}
	}
	return;
}

static void
skk_aux_dict_draw (iml_session_t *s, const gchar *query_str)
{
	SkkSessionData *ses = skk_session_data (s);
	int status[1];
	UTFCHAR *send_string[1];
	if (!query_str)
		return;
	if (ses->dict_aux_ok == False) {
		skk_aux_start (s, SKK_AUX_ADD_DICT_CLASS_NAME);
		/* send initialize request */
		status[0] = SKK_AUX_DICT_INIT;
		skk_aux_draw (s, SKK_AUX_ADD_DICT_CLASS_NAME,
				1, status, 0, NULL);
		ses->dict_aux_ok++;
	} /* TODO */
	if (ses->dict_start == False) {
		status[0] = SKK_AUX_DICT_ADD;
		send_string[0] = skk_strtoutf (query_str);
		skk_aux_draw (s, SKK_AUX_ADD_DICT_CLASS_NAME,
				1, status, 1, send_string);
		g_free (send_string[0]);
		ses->dict_start = True;
	}
	return;
}

static void
skk_end_conversion (iml_session_t *s)
{
	iml_inst *lp;
	lp = s->If->m->iml_make_end_conversion_inst (s);
	s->If->m->iml_execute (s, &lp);
	return;
}

static void
skk_start_conversion (iml_session_t *s)
{
	iml_inst *lp;
	lp = s->If->m->iml_make_start_conversion_inst (s);
	s->If->m->iml_execute (s, &lp);
	return;
}

static void
skk_status_done (iml_session_t *s)
{
	SkkSessionData *ses = skk_session_data (s);
	iml_inst *lp;
	if (ses->status_start == True) {
		lp = s->If->m->iml_make_status_done_inst (s);
		s->If->m->iml_execute (s, &lp);
		ses->status_start = False;
	}
	return;
}
