/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkconv_num.c,v 1.9 2002/03/21 21:56:00 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <glib.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include "skkconv_jisx0208_latin.h"
#include "skkdebug.h"

static const gchar *kanji_num_table[] =  {
	"","","","","","","ϻ","","Ȭ","",
};
static const gchar *kanji_num_table_old[] = {
	"","","","","","","ϻ","","Ȭ","",
};
/* TODO */
/* Dirty Hack Table */
static const gchar *kanji_num_tanni[] = {
	NULL, "","","","",
};

static const gchar *kanji_num_tanni_old[] = {
	NULL, "","","","",
};

static const gchar *kanji_num_tanni_sub[] = {
	"","ɴ","",
};

static const gchar *kanji_num_tanni_sub_old[] = {
	"","ɴ","",
};

#define TANNI_NUM ((sizeof (kanji_num_tanni)) / (sizeof (kanji_num_tanni[0])))

static gchar * 
add_val (gchar *item, gchar *append)
{
	gchar *ret;
	DEBUG_DO (printf ("skkconv_num.c : Entering add_val\n"));
	if (!append)
		return NULL;
	if (!item) {
		ret = g_strdup (append);
	} else {
		ret = g_strconcat (item,append,NULL);
	}
	return ret;
}

static gchar*
jisx0208_latin (const gchar *number)
{
	const gchar *jisx0208_latin;
	gchar *ret = NULL;
	gint i;
	DEBUG_DO (printf ("skkconv_num.c : Entering jisx0208_latin\n"));
	if (!number)
		return NULL;
	for ( i = 0; number[i] && isdigit (number[i]); i++) {
		jisx0208_latin = skkconv_get_jisx0208_latin (number[i]);
		if (ret) {
			gchar *be_free = ret;
			ret = g_strconcat (ret, jisx0208_latin, NULL);
			g_free (be_free);
		} else {
			ret = g_strdup (jisx0208_latin);
		}
	}
	return ret;
}

static gchar *
num_recompute (const gchar *number)
{
	gchar *ret = NULL;
	DEBUG_DO (printf ("skkconv_num.c : Entering num_recompute\n"));
	if (!number)
		return NULL;
	ret = g_strdup (number);
	return ret;
}

static gchar *
num_type_5 (const gchar *number)
{
	const gchar *tmp_buf = NULL;
	gchar *ret = NULL;
	gchar *tmp2 = NULL;
	gchar *tmp = NULL;
	gchar *tmp_result;
	gint suffix;
	gint offset;
	gint num = 0;
	gint len;
	gint i;
	gint count;

	if (!number)
		return NULL;
	len = strlen (number);
	suffix = (len >> 2);
	offset = (len % 4);
	for (i = 0; i < len; i++) {
		tmp = g_strdup_printf ("%c", number[i]);
		num = atoi (tmp);
		count = (abs (offset -i - 1) % 4);
		if (!count && (suffix < TANNI_NUM)) {
			tmp_buf = kanji_num_tanni_old[suffix];
			suffix--;
			if (!tmp_buf) {
				if (num == 0) {
					tmp_buf = NULL;
				} else {
					tmp_buf = kanji_num_table_old[num];
				}
			} else {
				if (num != 0) {
					tmp2 = g_strconcat (kanji_num_table_old[num], tmp_buf, NULL);
					tmp_buf = tmp2;
				}
			}
		} else {
			if (num == 0) {
				tmp_buf = NULL;
#if 0
			} else if (num == 1) {
				tmp2 = g_strconcat (kanji_num_tanni_sub_old[count -1], NULL);
				tmp_buf = tmp2;
#endif
			} else {
				tmp2 = g_strconcat (kanji_num_table_old[num],
						kanji_num_tanni_sub_old[count - 1], NULL);
				tmp_buf = tmp2;
			}
		}
		if (ret) {
			if (tmp_buf) {
				tmp_result = ret;
				ret = g_strconcat (ret, tmp_buf, NULL);
				g_free (tmp_result);
			}
		} else {
			if (tmp_buf) {
				ret = g_strdup (tmp_buf);
			}
		}
		g_free (tmp);
		if (tmp2) {
			g_free (tmp2);
			tmp2 = NULL;
		}
	}
	return ret;
}

static gchar *
num_type_3 (const gchar *number)
{
	const gchar *tmp_buf = NULL;
	gchar *ret = NULL;
	gchar *tmp2 = NULL;
	gchar *tmp = NULL;
	gchar *tmp_result;
	gint suffix;
	gint offset;
	gint num = 0;
	gint len;
	gint i;
	gint count;

	DEBUG_DO (printf ("skkconv_num.c : Entering num_type_3\n"));
	if (!number)
		return NULL;
	len = strlen (number);
	suffix = (len >> 2);
	offset = (len % 4);
#ifdef SKKCONV_NUM_DEBUG
	printf ("suffix %d offset %d\n", suffix, offset);
#endif
	for (i = 0; i < len ; i++) {
		tmp = g_strdup_printf ("%c", number[i]);
		num = atoi (tmp);
		count = (abs (offset - i - 1) % 4);
		if (!count && (suffix < TANNI_NUM)) {
			tmp_buf = kanji_num_tanni[suffix];
			suffix--;
			if (!tmp_buf) {
				if (num == 0) {
					tmp_buf = NULL;
				} else {
					tmp_buf = kanji_num_table[num];
				}
			} else {
				if (num != 0) {
					tmp2 = g_strconcat (kanji_num_table[num], tmp_buf, NULL);
					tmp_buf = tmp2;
				}
			}
		} else {
			if (num == 0) {
				tmp_buf = NULL;
			} else if (num == 1) {
				tmp2 = g_strdup (kanji_num_tanni_sub[count - 1]);
				tmp_buf = tmp2;
			} else {
				tmp2 = g_strconcat (kanji_num_table[num],kanji_num_tanni_sub[count - 1],NULL);
				tmp_buf = tmp2;
			}
		}
#ifdef SKKCONV_NUM_DEBUG
		if (tmp_buf)
			printf ("tmp_buf %s\n", tmp_buf);
#endif
		if (ret) {
			if (tmp_buf) {
				tmp_result = ret;
				ret = g_strconcat (ret, tmp_buf, NULL);
				g_free (tmp_result);
			}
		} else {
			if (tmp_buf)
				ret = g_strdup (tmp_buf);
		}
		g_free (tmp);
		if (tmp2) {
			g_free (tmp2);
			tmp2 = NULL;
		}
	}
	return ret;
}

static gchar *
num_type_2 (const gchar *number)
{
	gchar *ret = NULL;
	const gchar *tmp_buf;
	gchar *tmp;
	gchar *tmp_result;
	gint   num = 0;
	gint   i;
	DEBUG_DO (printf ("skkconv_num.c : Entering num_type_2\n"));
	if (!number)
		return NULL;
	for (i = 0; number[i]; i++) {
		tmp = g_strndup (&number[i],1);
		num = atoi (tmp);
		tmp_buf = kanji_num_table[num];
		if (ret) {
			tmp_result = ret;
			ret = g_strconcat (ret, tmp_buf, NULL);
			g_free (tmp_result);
		} else {
			ret = g_strdup (tmp_buf);
		}
		g_free (tmp);
	}
	return ret;
}

static gchar *
num_type_9 (const gchar *number)
{
	gchar *ret = NULL;
	gint len;
	gint val;
	const gchar *res1,*res2;
	DEBUG_DO (printf ("skkconv_num.c : Entering num_type_9\n"));
	/* FIXME better handling is needed if len != 2 */
	if (!number)
		return NULL;
	len = strlen (number);
	if (len != 2) {
		return strdup (number);
	}
	val = atoi (&number[1]);
	res1 = skkconv_get_jisx0208_latin (number[0]);
	res2 = kanji_num_table[val];
	ret = g_strconcat (res1,res2,NULL);
	return ret;
}

static gchar *
num_identity (const gchar *number)
{
	gchar *ret = NULL;
	DEBUG_DO (printf ("skkconv_num.c : Entering num_identity\n"));
	if (!number)
		return NULL;
	ret = g_strdup (number);
	return ret;
}

gchar *
skk_num_jisx0208_latin (const gchar *direction_word, const gchar *number)
{
	gchar *sep;
	gchar *copy;
	gchar *trans = NULL;
	const gchar *jisx0208_latin;
	gchar *ret = NULL;
	int i;
	DEBUG_DO (printf ("skkconv_num.c : Entering skk_num_jisx0208_latin\n"));
	if (!direction_word || !number)
		return NULL;
	copy = g_strdup (direction_word);
	sep = strstr (copy, "#1");
	if (!sep) {
		g_free (copy);
		return NULL;
	}
	*sep = '\0';
	sep += 2;
	for (i = 0; number[i] && isdigit (number[i]); i++) {
		jisx0208_latin = skkconv_get_jisx0208_latin (number[i]);
		if (trans) {
			gchar *be_free = trans;
			trans = g_strconcat (trans, jisx0208_latin, NULL);
			g_free (be_free);
		} else {
			trans = g_strdup (jisx0208_latin);
		}
	}
	ret = g_strconcat (copy, trans, NULL);
	g_free (trans);
	return ret;
}

gchar *
skk_num_identity (const gchar *direction_word, const gchar *number)
{
	gchar *sep;
	gchar *copy;
	gchar *ret = NULL;
	DEBUG_DO (printf ("skkconv_num.c : Entering skk_num_identity\n"));
	if (!direction_word || !number)
		return NULL;
	copy = g_strdup (direction_word);
	sep = strstr (copy, "#0");
	if (!sep) {
		g_free (copy);
		return NULL;
	}
	*sep = '\0';
	sep += 2;
	ret = g_strconcat (copy, number, sep, NULL);
	g_free (copy);
	return ret;
}

gchar *
skk_num_recompute (const gchar *direction_word, const gchar *number)
{
	gchar *sep;
	gchar *copy;
	gchar *ret = NULL;
	DEBUG_DO (printf ("skkconv_num.c : Entering skk_num_recompute\n"));
	if (!direction_word || !number)
		return NULL;
	copy = g_strdup (direction_word);
	sep = strstr (copy, "#4");
	if (!sep) {
		g_free (copy);
		return NULL;
	}
	*sep = '\0';
	sep += 2;
	ret = g_strconcat (copy, number, sep, NULL);
	g_free (copy);
	return ret;
}

gchar *
skk_num_translate (const gchar *direction_word, const GList *number_list, gint list_count, gint *type)
{
	gchar *sep;
	gchar *sep2;
	gchar *copy;
	gchar *copy2;
	gchar *tmp_result;
	gchar *ret = NULL;
	guint  found_num = 0;
	DEBUG_DO (printf ("skkconv_num.c : Entering skk_num_translate\n"));
	if (!direction_word || !number_list)
		return NULL;
	copy = copy2 = g_strdup (direction_word);
	sep = sep2 = strchr (copy, '#');
	if (!sep) {
		g_free (copy);
		return NULL;
	}
	while (found_num < list_count ) {
		DEBUG_DO (printf ("%s\n",sep2));
		while (TRUE) {
			sep2++;
			if (!isdigit (*sep2)) {
			    sep = sep2 = strchr (sep2, '#');
				/* TODO */
				if (!sep) {
					g_free (copy);
					return NULL;
				}
			} else {
				DEBUG_DO (printf ("skk_num_translate: found separator\n"));
				DEBUG_DO (printf ("%s\n",sep2));
				*sep = '\0';
				tmp_result = add_val (ret,copy2);
				if (ret) {
					g_free (ret);
					ret = tmp_result;
				}
				copy2 = sep2 + 1;
				found_num++;
				break;
			}
		}
		tmp_result = NULL;
		if (type) {
			*type = atoi (sep2);
		}
		switch (*sep2) {
			case '0':
				tmp_result = num_identity ( (const gchar*)g_list_nth_data ((GList*)number_list, found_num - 1));
				break;
			case '1':
				tmp_result = jisx0208_latin ( (const gchar*) g_list_nth_data ((GList*)number_list, found_num - 1));
				break;
			case '2':
				tmp_result = num_type_2 ( (const gchar*) g_list_nth_data ((GList*)number_list, found_num - 1));
				break;
			case '3':
				tmp_result = num_type_3 ( (const gchar*) g_list_nth_data ((GList*)number_list, found_num - 1));
				break;
			case '4':
				tmp_result = num_recompute ( (const gchar*) g_list_nth_data ((GList*)number_list, found_num - 1));
				break;
			case '5':
				tmp_result = num_type_5 ( (const gchar*) g_list_nth_data ((GList*)number_list, found_num - 1));
				break;
			case '9':
				tmp_result = num_type_9 ( (const gchar*) g_list_nth_data ((GList*)number_list, found_num - 1));
				break;
			default:
				/* TODO */
				tmp_result = num_identity ((const gchar*)g_list_nth_data ((GList*)number_list, found_num - 1));
				break;
		}
		if (!tmp_result) {
			if (ret) {
				g_free (ret);
			}
			g_free (copy);
			return NULL;
		}
		DEBUG_DO (printf ("skk_num_translate: tmp_result %s\n",tmp_result));
		if (ret) {
			gchar *be_free = ret;
			ret = g_strconcat (ret,tmp_result,NULL);
			g_free (be_free);
		} else {
			ret = g_strdup (tmp_result);
		}
	}
	if (copy2) {
		gchar *be_free = ret;
		ret = g_strconcat (ret, copy2, NULL);
		g_free (be_free);
	}
	return ret;
}

#ifdef SKKCONV_NUM_DEBUG
int
main (void)
{
	GList *test = NULL;
	gchar *string = "Ϥ#0#0ؤ#1#4ۤ#9#2#1#3#5#3|#3"; 
	test = g_list_append (test,"1234");
	test = g_list_append (test,"5678");
	test = g_list_append (test,"3347");
	test = g_list_append (test,"3004426");
	test = g_list_append (test,"12");
	test = g_list_append (test,"22589");
	test = g_list_append (test,"22589");
	test = g_list_append (test,"98765432123456789");
	test = g_list_append (test,"111111110000");
	test = g_list_append (test,"3004426");
	test = g_list_append (test,"111111110000");
	g_message ("%s\n",skk_num_translate (string,test,11,NULL));
	string = skk_num_identity ("#0#0#1#4","1234");
	g_message ("%s\n",string);
	string = skk_num_identity (string,"5678");
	g_message ("%s\n",string);
	string = skk_num_jisx0208_latin (string,"3347");
	g_message ("%s\n",string);
	return 0;
}
#endif
