/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkbuffer.c,v 1.14.2.4 2002/04/24 00:27:55 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include "skkbuffer.h"
#include "skkcomm.h"
#include "skkconv_kana.h"
#include "skkconv_num.h"
#include "skkconv_jisx0201_kana.h"
#include "skkconv_jisx0208_latin.h"
#include "skkutils.h"
#include "skkdictitem.h"
#include "skkdebug.h"

/* buffer operation */
static gchar* get_buf (SkkBuffer *buf, gint ch, gboolean lower);
static gchar* get_translate_buf (SkkBuffer *buf);
static gchar* create_candidate (SkkBuffer *buf);

static gboolean delete_backward_okurigana (SkkBuffer *buf);
static gboolean delete_backward_buf (SkkBuffer *buf);
static gboolean delete_backward_preedit (SkkBuffer *buf);
static gboolean delete_backward_direction_word (SkkBuffer *buf);
static void delete_backward (SkkBuffer *buf);

static gboolean undo (SkkBuffer *buf);

/* SkkMode */
static gboolean add_char (SkkBuffer *buf, int ch);
static gboolean add_char_jmode (SkkBuffer *buf, int ch);           /* SKK_J_MODE */
static gboolean add_char_jisx0208_latin (SkkBuffer *buf, int ch);  /* SKK_JISX0208_LATIN_MODE */
static gboolean add_char_latin (SkkBuffer *buf, int ch);           /* SKK_LATIN_MODE */
static gboolean add_char_abbrev (SkkBuffer *buf, int ch);          /* SKK_ABBREV_MODE */

/* for SKK_J_MODE */
static void translate_buffer_before_okuri (SkkBuffer *buf);
static gboolean add_buffer_normal (SkkBuffer *buf, int ch); /* SKK_NONE */
static gboolean add_buffer_henkan (SkkBuffer *buf, int ch); /* SKK_HENKAN */
static gboolean add_buffer_okuri (SkkBuffer *buf, int ch);  /* SKK_OKURI */
static gboolean add_buffer_choice (SkkBuffer *buf, int ch); /* SKK_CHOICE */

static void set_okurigana (SkkBuffer *buf, const gchar *value);
static void set_okuri_prefix (SkkBuffer *buf, const gchar *value);
static void set_result_buf (SkkBuffer *buf, const gchar *value);
static void set_preedit_buf (SkkBuffer *buf, const gchar *value);
static void set_buf (SkkBuffer *buf, const gchar *value);
static void set_direction_word (SkkBuffer *buf, const gchar *value);
static void set_lasth (SkkBuffer *buf);

static void add_okurigana (SkkBuffer *buf, const gchar *append);
static void add_preedit_buf (SkkBuffer *buf, const gchar *append);
static void add_direction_word (SkkBuffer *buf, const gchar *append);
static void add_direction_word_int (SkkBuffer *buf, const gchar *append);
static void add_result_buf (SkkBuffer *buf, const gchar *append);

/* change category */
static void change_category (SkkBuffer *buf);

/* check function */
static gboolean can_clear (SkkBuffer *buf);
static gboolean can_commit (SkkBuffer *buf);
static gboolean has_preedit (SkkBuffer *buf);

typedef struct _SkkBufFunc {
	SkkBufListener listener;
	gpointer user_data;
}SkkBufFunc;

typedef struct _SkkAddDictFunc {
	SkkAddDictListener listener;
	gpointer user_data;
}SkkAddDictFunc;

typedef struct _SkkCodeTableFunc {
	SkkCodeTableListener listener;
	gpointer user_data;
}SkkCodeTableFunc;

typedef struct _SkkLookupFunc {
	SkkLookupListener listener;
	gpointer user_data;
}SkkLookupFunc;

static void
lookup_emit (SkkBuffer *buf)
{
	GList *tmp_list;
	SkkLookupFunc *func;
	gchar *annotate[7] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL};
	gchar *candidate[7] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL};
	gchar *label[7] = { "A:", "S:", "D:", "F:", "J:", "K:", "L:"};  /* TODO */
	gint count = 0;
	gint i;
	gint restore;
	restore = buf->candidate_count;
	for (i = 0; i < 7; i++) {
		candidate[i] = skk_buffer_get_cur_candidate (buf);
		if (candidate[i]) {
			count++;
		}
		if (skk_buffer_has_next_candidate (buf)) {
			buf->candidate_count++;
		} else {
			break;
		}
	}
	for (tmp_list = buf->lookup_listener; tmp_list; tmp_list = g_list_next (tmp_list)) {
		func = (SkkLookupFunc *)tmp_list->data;
		if (func) {
			func->listener (buf, label, candidate, annotate, count, func->user_data);
		}
	}
	for (i = 0; i < 7; i++) {
		/* Free */
		if (candidate[i])
			g_free (candidate[i]);
		if (annotate[i])
			g_free (annotate[i]);
	}
	/* Restore */
	buf->candidate_count = restore;
	return;
}

static void
adddict_emit (SkkBuffer *buf)
{
	GList *tmp_list;
	SkkAddDictFunc *func;
	gchar *query_str;
	if (!buf)
		return;
	if (skkbuf_okuri (buf)) {
		query_str = g_strdup_printf ("%s*%s", skkbuf_d (buf), skkbuf_okuri (buf));
	} else {
		query_str = g_strdup (skkbuf_d (buf));
	}
	for (tmp_list = buf->adddict_listener; tmp_list; tmp_list = g_list_next (tmp_list)) {
		func = (SkkAddDictFunc *)tmp_list->data;
		if (func) {
			func->listener (buf, query_str, func->user_data);
		}
	}
	g_free (query_str);
	return;
}

static void
codetable_emit (SkkBuffer *buf)
{
	GList *tmp_list;
	SkkCodeTableFunc *func;
	if (!buf)
		return;
	for (tmp_list = buf->codetable_listener; tmp_list; tmp_list = g_list_next (tmp_list)) {
		func = (SkkCodeTableFunc *)tmp_list->data;
		if (func) {
			func->listener (buf, func->user_data);
		}
	}
	return;
}

static void
preedit_emit (SkkBuffer *buf)
{
	GList* tmp_list;
	SkkBufFunc *func;
	gchar *preedit;
	if (!buf)
		return;
	preedit = skk_buffer_get_preedit_string (buf);
	if (preedit && *preedit == '\0') {
		g_free (preedit);
		preedit = NULL;
	}
	for (tmp_list = buf->preedit_listener; tmp_list; tmp_list = g_list_next (tmp_list)) {
		func = (SkkBufFunc *)tmp_list->data;
		if (func) {
			func->listener (buf, preedit, func->user_data);
		}
	}
	if (preedit) {
		g_free (preedit);
	}
	return;
}

static void
result_emit (SkkBuffer *buf)
{
	GList *tmp_list;
	SkkBufFunc *func;
	if (!buf)
		return;
	for (tmp_list = buf->result_listener; tmp_list; tmp_list = g_list_next (tmp_list)) {
		func = (SkkBufFunc *)tmp_list->data;
		if (func) {
			func->listener (buf, skkbuf_r (buf), func->user_data);
		}
	}
}

/* Undo Buffer */
static gboolean
undo (SkkBuffer *buf)
{
	if (!buf)
		return FALSE;
	/* If we are not in QUERY_DONE, we cannot undo */
	if (skk_buffer_get_query_status (buf) != QUERY_DONE)
		return FALSE;
	if (skk_buffer_set_prev_candidate (buf))
		return TRUE;
	set_preedit_buf (buf, skkbuf_lasth (buf));
	if (skkbuf_okuri (buf)) {
		add_preedit_buf (buf, skkbuf_okuri (buf));
		add_direction_word (buf, skkbuf_okuri (buf));
		set_okurigana (buf, NULL);
		set_okuri_prefix (buf, NULL);
	}
	/* TODO */
	buf->candidate_count = 0;
	/* reset query_item */
	skk_query_set_first (buf->query);
	skk_buffer_set_query_status (buf, QUERY_NORMAL);
	if (skk_buffer_get_status (buf) == SKK_J_MODE) {
		skk_buffer_set_j_status (buf, SKK_HENKAN);
	}
	preedit_emit (buf);
	return TRUE;
}

static void
change_category (SkkBuffer *buf)
{
	/* HIRAGANA <-> KATAKANA */
	/* LATIN <-> JISX0208_LATIN */
	SkkStatus status;
	gchar *result = NULL;
	if (!buf)
		return;
	status = skk_buffer_get_status (buf);
	switch (status) {
		case SKK_J_MODE:
			/* if we don't have preedit, nothing to do */
			if (!skkbuf_p (buf)) {
				skk_buffer_clear (buf);
				return;
			}
			/* if hiragana mode, then translate to katakana,
			 * otherwise katakana to hiragana
			 */
			if (skk_buffer_get_katakana (buf)) {
				result = skkconv_katakana_to_hiragana (skkbuf_p (buf));
			} else {
				result = skkconv_hiragana_to_katakana (skkbuf_p (buf));
			}
			/* now we have a translated value, so can clear */
			skk_buffer_clear (buf);
			/* set result */
			set_result_buf (buf, result);
			/* emit signal */
			result_emit (buf);
			preedit_emit (buf);
			break;
		case SKK_LATIN_MODE:
			/* not reached here */
			break;
		case SKK_JISX0208_LATIN_MODE:
			break;
		case SKK_ABBREV_MODE:
			/* if we don't have preedit, nothing to do */
			if (!skkbuf_p (buf)) {
				skk_buffer_clear (buf);
				return;
			}
			/* translate preedit_buf to JISX0208_LATIN */
			result = skkconv_get_jisx0208_latin_from_str (skkbuf_p (buf), TRUE);
			/* now we have a translated value, so can clear */
			skk_buffer_clear (buf);
			/* set result */
			set_result_buf (buf, result);
			/* emit signal */
			result_emit (buf);
			preedit_emit (buf);
			break;
	}
	if (result)
		g_free (result);
	return;
}

static void
query_num (SkkBuffer *buf)
{
	GList *num = NULL;
	GList *tmp = NULL;
	gchar *candidate;
	gint list_num;
	gint type;
	gchar *ret;

	if (!buf->candidate_list)
		return;
	candidate = skkbuf_cur_candidate (buf);
	if (!candidate)
		return;
	ret = skk_num_translate (candidate, buf->number_list, buf->has_num, &type);
	if (ret) {
		if (type == 4) {
			/* in the case of #4, retry */
			/* reset query item */
			skk_query_set_first (skkbuf_query (buf));
			/* query to dict */
			num = skk_query_do_query (buf->query, ret, skkbuf_b (buf), &list_num);
			if (!num) {
				while (skk_query_set_next (buf->query)) {
					num = skk_query_do_query (buf->query, ret, skkbuf_b (buf), &list_num);
					if (num)
						break;
				}
			}
			tmp = g_list_nth (buf->candidate_list, buf->candidate_count);
			buf->candidate_list = g_list_remove_link (buf->candidate_list, tmp);
			skk_dict_item_destroy ((SkkDictItem *)tmp->data);
			g_list_free (tmp);
			/* found */
			if (num) {
				buf->candidate_list = skk_dict_item_merge_list (buf->candidate_list, num);
				buf->candidate_max = g_list_length (buf->candidate_list);
#if 0
				for ( tmp = num , count = buf->candidate_count
						; tmp ;
						tmp = g_list_next (tmp)) {
					/* FIXME */
					buf->candidate_list =
						g_list_insert (buf->candidate_list, tmp->data , count);
					g_message ("candidate %s", ((SkkDictItem *)tmp->data)->candidate);
					count++;
				}
				buf->candidate_max += list_num;
				g_list_free (num);
				/*
				query_num (buf);
				*/
#endif
			}
			g_free (ret);
		} else {
			tmp = g_list_nth (buf->candidate_list, buf->candidate_count);
			skk_dict_item_destroy ((SkkDictItem *)tmp->data);
			tmp->data = skk_dict_item_new_with_value (ret, NULL);
			g_free (ret);
		}
	}
	return;
}

static void
query_dict (SkkBuffer *buf)
{
	gchar *preedit;
	DEBUG_DO (printf ("skkbuffer.c : Entering query_dict\n"));
	if (!buf)
		return;
	preedit = create_candidate (buf);
	if (preedit) {
		set_preedit_buf (buf, preedit);
		g_free (preedit);
	} else {
		set_preedit_buf (buf, "");
		adddict_emit (buf);
	}
	preedit_emit (buf);	
	return;
}

static gboolean
can_clear (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering can_clear\n"));
	if (!buf)
		return FALSE;
	if ((!skkbuf_b (buf)) && (!skkbuf_p (buf))) {
		return TRUE;
	} else {
		return FALSE;
	}
}

static gboolean
can_commit (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering can_commit\n"));
	if (!buf)
		return FALSE;
	if (skkbuf_r (buf)) {
		return TRUE;
	} else {
		return FALSE;
	}
}

static gboolean
has_buf (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering has_buf\n"));
	if (!buf)
		return FALSE;
	if (skkbuf_b (buf)) {
		return TRUE;
	} else {
		return FALSE;
	}
}

static gboolean
has_preedit (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering has_preedit\n"));
	if (!buf)
		return FALSE;
	if (skkbuf_p (buf) || skkbuf_b (buf)) {
		return TRUE;
	} else {
		return FALSE;
	}
}

static void
set_result_buf (SkkBuffer *buf, const gchar *value)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering set_result_buf\n"));
	if (!buf)
		return;
   if (value && skkbuf_r (buf)) {
		if (!strcmp (skkbuf_r (buf), value)) {
			return;
		}
	} else if (!value && !skkbuf_r (buf)) {
		return;
	}
	if (skkbuf_r (buf)) {
		g_free (skkbuf_r (buf));
		skkbuf_r (buf) = NULL;
		skkbuf_rlen (buf) = 0;
	}
	if (value) {
		skkbuf_r (buf) = g_strdup (value);
		skkbuf_rlen (buf) = strlen (skkbuf_r (buf));
	}
	return;
}

static void
set_lasth (SkkBuffer *buf)
{
	if (!buf)
		return;
	if (skkbuf_lasth (buf)) {
		g_free (skkbuf_lasth (buf));
		skkbuf_lasth (buf) = NULL;
	}
	if (skkbuf_p (buf)) {
		skkbuf_lasth (buf) = g_strdup (skkbuf_p (buf));
	}
	return;
}

static void
set_okurigana (SkkBuffer *buf, const gchar *value)
{
	/* skkbuf_okuri (buf) == buf->henkan_okurigana */
	if (!buf)
		return;
	if (skkbuf_okuri (buf)) {
		g_free (skkbuf_okuri (buf));
		skkbuf_okuri (buf) = NULL;
	}
	if (!value)
		return;
	skkbuf_okuri (buf) = g_strdup (value);
	return;
}

static void
set_okuri_prefix (SkkBuffer *buf, const gchar *value)
{
	/* skkbuf_okurip (buf) == buf->henkan_okuri_char */
	if (!buf)
		return;
	if (skkbuf_okurip (buf)) {
		g_free (skkbuf_okurip (buf));
		skkbuf_okurip (buf) = NULL;
	}
	if (!value)
		return;
	skkbuf_okurip (buf) = g_strdup (value);
	return;
}

static void
set_preedit_buf (SkkBuffer *buf, const gchar *value)
{
	/* skkbuf_p (buf) == buf->preedit_buf */
	DEBUG_DO (printf ("skkbuffer.c : Entering set_preedit_buf\n"));
	if (!buf)
		return;
	if (value && skkbuf_p (buf)) {
		if (!strcmp (skkbuf_p (buf), value)) {
			return;
		}
	} else if (!value && !skkbuf_p (buf)) {
		return;
	}	
	if (skkbuf_p (buf)) {
		g_free (skkbuf_p (buf));
		skkbuf_p (buf) = NULL;
		skkbuf_plen (buf) = 0;
	}
	if (value) {
		skkbuf_p (buf) = g_strdup (value);
		skkbuf_plen (buf) = strlen (skkbuf_p (buf));
	}
	return;
}

static void
set_direction_word (SkkBuffer *buf, const gchar *value)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering set_direction_word\n"));
	if (!buf)
		return;
	if (skkbuf_d (buf)) {
		g_free (skkbuf_d (buf));
		skkbuf_d (buf) = NULL;
		skkbuf_dlen (buf) = 0;
	}
	if (!value)
		return;
	skkbuf_d (buf) = g_strdup (value);
	skkbuf_dlen (buf) = strlen (skkbuf_d (buf));
	return;
}

static void
set_buf (SkkBuffer *buf, const gchar *value)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering set_buf\n"));
	if (!buf)
		return;
	if (value && skkbuf_b (buf)) {
		if (!strcmp (skkbuf_b (buf), value)) {
			return;
		}
	} else if (!value && !skkbuf_b (buf)) {
		return;
	}	
	if (skkbuf_b (buf)) {
		g_free (skkbuf_b (buf));
		skkbuf_b (buf) = NULL;
		skkbuf_blen (buf) = 0;
	}
	if (value) {
		skkbuf_b (buf) = g_strdup (value);
		skkbuf_blen (buf) = strlen (skkbuf_b (buf));
	}
	return;
}

static gchar*
get_buf (SkkBuffer *buf, int ch, gboolean lower)
{
	gchar *ret;
	DEBUG_DO (printf ("skkbuffer.c : Entering get_buf\n"));
	if (!buf)
		return NULL;
	/* check ch is valid ascii char */
	if (!isascii (ch)) {
		return NULL;
	}
	if (skkbuf_b (buf)) {
		ret = g_strdup_printf ("%s%c",skkbuf_b (buf), lower ? tolower (ch) : ch);
	} else {
		ret = g_strdup_printf ("%c", lower ? tolower (ch) : ch);
	}
	return ret;
}

static void
translate_buffer_before_okuri (SkkBuffer *buf)
{
	gchar *tmpbuf;

	tmpbuf = get_translate_buf (buf);
	if (tmpbuf) {
		add_preedit_buf (buf, tmpbuf);
		add_direction_word (buf, tmpbuf);
		g_free (tmpbuf);
	}
}

static gboolean
add_buffer_normal (SkkBuffer *buf, int ch)
{
	/* TODO clean up */
//	gint buf_exist = 0;
	gint buf_is_exist = 0;
	gchar *tmp_in_buf; /* temporary buffer */
	gchar *result;     /* result buf */
	DEBUG_DO (printf ("skkbuffer.c : Entering add_buffer_normal\n"));
	tmp_in_buf = get_buf (buf, ch, TRUE);
	if (!tmp_in_buf)
		return FALSE;

	/* Check whether we can translate buf to KANA */
	buf_is_exist = skkconv_is_exist (tmp_in_buf);

	if (!buf_is_exist) {
		/* Can not translate to KANA */
		/* FAMAO */
		/* */
		if (buf->check_before) {
			gchar *tmp_preedit;
			tmp_preedit = get_translate_buf (buf);
			buf->check_before = FALSE;
			if (tmp_preedit) {
				g_free (tmp_in_buf);
				set_buf (buf, NULL);
				set_result_buf (buf, tmp_preedit);
				g_free (tmp_preedit);
				result_emit (buf);
				preedit_emit (buf);
				return add_buffer_normal (buf, ch);
#if 0
				tmp_in_buf = get_buf (buf, ch, TRUE);
				set_buf (buf, tmp_in_buf);
				set_result_buf (buf, tmp_preedit);
				g_free (tmp_preedit);
				result_emit (buf);
				preedit_emit (buf);
				return TRUE;
#endif
			}
		}
		if ( !isalpha (ch) && isascii (ch)) {
			/* Not [a-zA-Z] */
			set_result_buf (buf, tmp_in_buf);
			set_buf (buf,NULL);
			g_free (tmp_in_buf);
			result_emit (buf);
			preedit_emit (buf);
			return TRUE;
		}
		buf->in_buf[buf->buf_len - 1] = tolower (ch);
		skkbuf_blen (buf) = strlen (skkbuf_b (buf));
		g_free (tmp_in_buf);
		preedit_emit (buf);
		return TRUE;
	} else {
		set_buf (buf,tmp_in_buf);
		if (buf_is_exist > 1) {
			/* we have more candidates than one */
			buf->check_before = TRUE;
			g_free (tmp_in_buf);
			preedit_emit (buf);
			return TRUE;
		}
#if 0
		/* TODO */
		preedit_emit (buf);
#endif
		/* Try to translate to KANA */
		result = get_translate_buf (buf);
		/* add buffer */
		set_result_buf (buf, result);
		g_free (tmp_in_buf);
		if (result)
			g_free (result);
		result_emit (buf);
		preedit_emit (buf);
		return TRUE;
	}
	return FALSE;
}

static gboolean
add_buffer_henkan (SkkBuffer *buf, int ch)
{
	gint buf_is_exist = 0;
	gchar *tmp_in_buf;
	gchar *tmp_preedit;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_buffer_henkan\n"));

	tmp_in_buf = get_buf (buf, ch, TRUE);
	if (!tmp_in_buf)
		return FALSE;
	
	/* Check whether we can translate buf to KANA */
	buf_is_exist = skkconv_is_exist (tmp_in_buf);

	if (!buf_is_exist) {
		/* FAMAO NOW */
		if (buf->check_before) {
			tmp_preedit = get_translate_buf (buf);
			buf->check_before = FALSE;
			if (tmp_preedit) {
				g_free (tmp_in_buf);
				set_buf (buf, NULL);
				add_preedit_buf (buf, tmp_preedit);
				add_direction_word (buf, tmp_preedit);
				g_free (tmp_preedit);
				preedit_emit (buf);
				return add_buffer_henkan (buf, ch);
			}
		}
		/* Can not translate to KANA */
		if ( !isalpha (ch) && isascii (ch)) { 
			/* Not [a-zA-Z] */
			if (isdigit (ch)) {
				/* [0-9] */
				/* we must translate Number to '#' */
				add_direction_word_int (buf, tmp_in_buf);
			} else {
				/* Not [0-9] nor [a-zA-Z] */
				/* ex.) !"#$%&'()~=|... */
				add_direction_word (buf, tmp_in_buf);
			}
			add_preedit_buf (buf, tmp_in_buf);
			set_buf (buf, NULL);
			g_free (tmp_in_buf);
			preedit_emit (buf);
			return TRUE;
		}
		/* Change Last Buffer */
		/* ex) k + p -> p */
		buf->in_buf[buf->buf_len -1] = tolower (ch);
		skkbuf_blen (buf) = strlen (skkbuf_b (buf));
		g_free (tmp_in_buf);
		preedit_emit (buf);
		return TRUE;
	} else {
		set_buf (buf,tmp_in_buf);
		if (buf_is_exist > 1) {
			/* we have more candidates than one */
			buf->check_before = TRUE;
			g_free (tmp_in_buf);
			preedit_emit (buf);
			return TRUE;
		}
		/* try to translate to KANA */
		tmp_preedit = get_translate_buf (buf);
		if (tmp_preedit) {
			add_preedit_buf (buf, tmp_preedit);
			/* FIXME */
			add_direction_word (buf, tmp_preedit);
			g_free (tmp_preedit);
		}
		g_free (tmp_in_buf);
		preedit_emit (buf);
	}
	return FALSE;
}

static gboolean
add_buffer_okuri (SkkBuffer *buf, int ch)
{
	gint buf_is_exist = 0;
	gchar *tmp_in_buf;
	gchar *tmp_okurigana;
	gchar *preedit;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_buffer_okuri\n"));
	if (!buf)
		return FALSE;

	tmp_in_buf = get_buf (buf, ch, TRUE);
	if (!tmp_in_buf)
		return FALSE;

	/* Check whether we can translate buf to KANA */
	buf_is_exist = skkconv_is_exist (tmp_in_buf);

	if (!buf_is_exist) {
		/* Can not translate to KANA */
		if (!isalpha (ch) && isascii (ch)) {
			g_free (tmp_in_buf);
			return FALSE;
		}
		if (buf->check_before) {
			tmp_okurigana = get_translate_buf (buf);
			buf->check_before = FALSE;
			if (tmp_okurigana) {
				if (skkbuf_b (buf)) {
					if (skkbuf_okurip (buf)) {
						add_okurigana (buf, tmp_okurigana);
					} else {
						set_okurigana (buf, tmp_okurigana);
						set_okuri_prefix (buf, tmp_in_buf);
					}
					g_free (tmp_in_buf);
					g_free (tmp_okurigana);
					preedit_emit (buf);
					return TRUE;
				}
				skk_buffer_set_j_status (buf, SKK_CHOICE);
				skk_buffer_set_query_status (buf, QUERY_DONE);
				add_okurigana (buf, tmp_okurigana);
				if (!skkbuf_okurip (buf))
					set_okuri_prefix (buf, skkbuf_b (buf));
				preedit = create_candidate (buf);
				if (preedit) {
					set_preedit_buf (buf, preedit);
					g_free (preedit);
				}
				/* add_preedit_buf (buf, skkbuf_okuri (buf)); */
				g_free (tmp_in_buf);
				g_free (tmp_okurigana);
				preedit_emit (buf);
				return TRUE;
			}
		}
		buf->in_buf[buf->buf_len -1] = tolower (ch);
		skkbuf_blen (buf) = strlen (skkbuf_b (buf));
		g_free (tmp_in_buf);
		preedit_emit (buf);
		return TRUE;
	} else {
		set_buf (buf,tmp_in_buf);
		if (buf_is_exist > 1) {
			/* we have more candidates than one */
			buf->check_before = TRUE;
			g_free (tmp_in_buf);
			preedit_emit (buf);
			return TRUE;
		}
		tmp_okurigana = get_translate_buf (buf);
		if (!tmp_okurigana) {
			g_free (tmp_in_buf);
			preedit_emit (buf);
			return TRUE;
		} else {
			if (skkbuf_b (buf)) {
				if (skkbuf_okurip (buf)) {
					add_okurigana (buf, tmp_okurigana);
				} else {
					set_okurigana (buf, tmp_okurigana);
					set_okuri_prefix (buf, tmp_in_buf);
				}
				g_free (tmp_in_buf);
				g_free (tmp_okurigana);
				preedit_emit (buf);
				return TRUE;
			}
			/* TODO */
			/* prepare stop_emit */
			skk_buffer_set_j_status (buf, SKK_CHOICE);
			skk_buffer_set_query_status (buf, QUERY_DONE);
			add_okurigana (buf, tmp_okurigana);
			if (!skkbuf_okurip (buf))
				set_okuri_prefix (buf, tmp_in_buf);
			preedit = create_candidate (buf);
			if (preedit) {
				set_preedit_buf (buf, preedit);
				g_free (preedit);
			}
			/* add_preedit_buf (buf, skkbuf_okuri (buf)); */
			g_free (tmp_in_buf);
			g_free (tmp_okurigana);
			preedit_emit (buf);
			return TRUE;
		}
	}
	return FALSE;
}

static gboolean
add_buffer_choice (SkkBuffer *buf, int ch)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering add_buffer_choice\n"));
	if (!buf)
		return FALSE;
	skk_buffer_commit (buf, TRUE);
	skk_buffer_clear (buf);
	if (isupper (ch)) {
		skk_buffer_set_j_status (buf, SKK_HENKAN);
		return add_buffer_henkan (buf, ch);
	} else {
		return add_buffer_normal (buf, ch);
	}
	return TRUE;
}

static void
add_mode_check (SkkBuffer *buf, int ch)
{
	SkkJStatus jstatus;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_mode_check\n"));
	if (!buf)
		return;
	jstatus = skk_buffer_get_j_status (buf);
	if (isupper (ch)) {
		switch (jstatus) {
			case SKK_NONE:
				/* HENKAN */
				skk_buffer_set_j_status (buf, SKK_HENKAN);
				break;
			case SKK_HENKAN:
				if (skkbuf_p (buf)) {
					skk_buffer_set_j_status (buf, SKK_OKURI);
					translate_buffer_before_okuri (buf);
				}
				break;
			case SKK_OKURI:
				/* nothing to do */
				break;
			case SKK_CHOICE:
				/* nothing to do */
				/* HENKAN */
#if 0
				buf->store_result = TRUE;
				skk_buffer_set_j_status (buf, SKK_HENKAN);
#endif
				break;
			default:
				break;
		}
	} else {
		switch (jstatus) {
			case SKK_NONE:
				/* nothing to do */
				break;
			case SKK_HENKAN:
				/* nothing to do */
				break;
			case SKK_OKURI:
				/* nothing to do */
				break;
			case SKK_CHOICE:
				/* normal mode */
#if 0
				buf->store_result = TRUE;
				skk_buffer_set_j_status (buf, SKK_NONE);
#endif
				break;
			default:
				break;
		}
	}
	return;
}

static gboolean
add_char_jmode (SkkBuffer *buf, int ch)
{
	SkkJStatus jstatus;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_char_jmode\n"));
	if (!buf)
		return FALSE;
	add_mode_check (buf, ch);
	/* again */
	jstatus = skk_buffer_get_j_status (buf);
	switch (jstatus) {
		case SKK_NONE:
			return add_buffer_normal (buf, ch);
			break;
		case SKK_HENKAN:
			return add_buffer_henkan (buf, ch);
			break;
		case SKK_OKURI:
			return add_buffer_okuri (buf, ch);
			break;
		case SKK_CHOICE:
			return add_buffer_choice (buf, ch);
			break;
		default:
			break;
	}
	return TRUE;
}

static gboolean
add_char_latin (SkkBuffer *buf, int ch)
{
	/* nothing to do */
	return TRUE;
}

static gboolean
add_char_jisx0208_latin (SkkBuffer *buf, int ch)
{
	const gchar *result;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_char_jisx0208_latin\n"));
	result = skkconv_get_jisx0208_latin (ch);
	set_result_buf (buf,result);
	result_emit (buf);
	if (result)
		return TRUE;
	else
		return FALSE;
}

static gboolean
add_char_abbrev (SkkBuffer *buf, int ch)
{
	gchar *preedit;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_char_abbrev\n"));
	preedit = get_buf (buf, ch, FALSE);
	if (preedit) {
		add_preedit_buf (buf, preedit);
		if (isdigit (ch)) {
			add_direction_word_int (buf, preedit);
		} else {
			add_direction_word (buf, preedit);
		}
		g_free (preedit);
		preedit_emit (buf);
		return TRUE;
	} else {
		return FALSE;
	}
	return FALSE;
}

static gboolean
add_char (SkkBuffer *buf, int ch)
{
	SkkStatus status;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_char\n"));
	if (!buf)
		return FALSE;
	status = skk_buffer_get_status (buf);
	/* FIXME */
	switch (status) {
		case SKK_J_MODE:
			return add_char_jmode (buf, ch);
			break;
		case SKK_LATIN_MODE:
			return add_char_latin (buf, ch);
			break;
		case SKK_JISX0208_LATIN_MODE:
			return add_char_jisx0208_latin (buf, ch);
			break;
		case SKK_ABBREV_MODE:
			return add_char_abbrev (buf, ch);
			break;
		default:
			break;
	}
	return FALSE;
}

static gboolean
delete_backward_okurigana (SkkBuffer *buf)
{
    gint len;
    if (!buf)
        return FALSE;
    if (!skkbuf_okuri (buf))
        return FALSE;
    len = skk_utils_last_charbytes (skkbuf_okuri (buf)); 
    buf->henkan_okurigana[strlen (skkbuf_okuri (buf)) - len] = '\0'; 
    if (strlen (skkbuf_okuri (buf)) == 0) { 
        set_okurigana (buf, NULL);
		set_okuri_prefix (buf, NULL);
#if 0
		if (skkbuf_lasth (buf))
			set_preedit_buf (buf, skkbuf_lasth (buf));
		skk_buffer_set_j_status (buf, SKK_HENKAN);
#endif
    }
	skk_buffer_commit (buf, FALSE); /* don't add to Dict */
    return TRUE;
}

static gboolean
delete_backward_buf (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering delete_backward_buf\n"));
	if (!buf)
		return FALSE;
	if (skk_buffer_has_buf (buf)) {
		buf->in_buf[buf->buf_len - 1] = '\0';
		buf->buf_len--;
		if (skkbuf_blen(buf) == 0) {
			set_buf (buf, NULL);
			if (skk_buffer_get_j_status (buf) == SKK_HENKAN &&
					!skkbuf_p (buf)) {
					skk_buffer_clear (buf);
			} else if (skk_buffer_get_j_status (buf) == SKK_OKURI) {
				if (!skkbuf_okuri (buf))
					skk_buffer_set_j_status (buf, SKK_HENKAN);
			}	
		}	
		return TRUE;
	}
	return FALSE;
}

static gboolean
delete_backward_direction_word (SkkBuffer *buf)
{
	gint len;
	DEBUG_DO (printf ("skkbuffer.c : Entering delete_backward_direction_word\n"));
	if (!buf)
		return FALSE;
	if (!skkbuf_d (buf)) /* happen ? */
		return FALSE;
	len = skk_utils_last_charbytes (skkbuf_d (buf));
	/* TODO */
	if (buf->has_num && 
			len == 1 && 
			buf->direction_word[skkbuf_dlen (buf) - 1] == '#')
	{
		gint num_len;
		gchar *last_num = skkbuf_last_num_list (buf);
		num_len = strlen (last_num);
		last_num[num_len - 1] = '\0';
		if ((num_len - 1) == 0) {
			/* free deleted list */
			GList *free_list;
			free_list = g_list_last (buf->number_list);
			buf->number_list = g_list_remove_link (buf->number_list, free_list);
			g_free (last_num);
			g_list_free (free_list);
			/* delete unused '#' */
			buf->direction_word[skkbuf_dlen (buf) - 1] = '\0';
			skkbuf_dlen (buf) -= len;
		}
	} else {
		buf->direction_word[skkbuf_dlen (buf) - len] = '\0';
		skkbuf_dlen (buf) -= len;
	}
	if (skkbuf_dlen (buf) == 0) {
		g_free (skkbuf_d (buf));
		skkbuf_d (buf) = NULL;
	}
	return TRUE;
}

static gboolean
delete_backward_preedit (SkkBuffer *buf)
{
	gint len;
	DEBUG_DO (printf ("skkbuffer.c : Entering delete_backward_preedit\n"));
	if (!buf)
		return FALSE;
	if (!skkbuf_p (buf)) /* happen ? */
		return FALSE;
	len = skk_utils_last_charbytes (skkbuf_p (buf));
	buf->preedit_buf[skkbuf_plen (buf) - len] = '\0';
	skkbuf_plen (buf) -= len;
	if (skkbuf_plen (buf) == 0) {
		skk_buffer_set_query_status (buf, QUERY_NONE);
		set_preedit_buf (buf, NULL);
	} else {
		preedit_emit (buf);
	}	
	return TRUE;
}

static void
delete_backward (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering delete_backward\n"));
	if (!buf)
		return;
	/* if in SKK_NONE or SKK_HENKAN , then we must remove result */
	if (skk_buffer_get_j_status (buf) == SKK_NONE
	    || skk_buffer_get_j_status (buf) == SKK_HENKAN) {
		set_result_buf (buf, NULL);
	}
	if (delete_backward_buf (buf))
		goto finished;
	if (delete_backward_okurigana (buf))
		goto finished;
	if (!delete_backward_preedit (buf)) {
		DEBUG_DO (printf ("skkbuffer.c : delete_backward (delete preedit failed)\n"));
	}
	if (!delete_backward_direction_word (buf)) {
		DEBUG_DO (printf ("skkbuffer.c : delte_backward (delete direction_word failed\n"));
	}
	if (skk_buffer_get_query_status (buf) == QUERY_DONE) {
		skk_buffer_commit (buf, FALSE);
		skk_buffer_clear (buf);
	}
	/* fall through */

finished:
	preedit_emit (buf);
	return;
}

static void
add_okurigana (SkkBuffer *buf, const gchar *append)
{
	/* skkbuf_okuri (buf) == buf->henkan_okurigana */
	gchar *append_buf;
	if (!buf)
		return;
	if (!append)
		return;
	if (skkbuf_okuri (buf)) {
		append_buf = g_strdup_printf ("%s%s", skkbuf_okuri (buf), append);
		g_free (skkbuf_okuri (buf));
		skkbuf_okuri (buf) = append_buf;
	} else {
		skkbuf_okuri (buf) = g_strdup (append);
	}
	return;
}

/*
 * add strings to preedit_buf
 */
static void
add_preedit_buf (SkkBuffer *buf, const gchar *append)
{
	gchar *append_buf;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_preedit_buf\n"));
	if (!append) return;
	if (skkbuf_p (buf)) {
		append_buf = g_strdup_printf ("%s%s", skkbuf_p (buf), append);
		g_free (skkbuf_p (buf));
		skkbuf_p (buf) = append_buf;
	} else {
		skkbuf_p (buf) = g_strdup (append);
	}
	skkbuf_plen (buf) = strlen (skkbuf_p (buf));
	DEBUG_DO (printf ("add_preedit %s\n",buf->preedit_buf));
	return;
}

/* add strings to result_buf */
static void
add_result_buf (SkkBuffer *buf, const gchar *append)
{
	gchar *append_buf;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_result_buf\n"));
	if (!append) return;
	if (skkbuf_r (buf)) {
		append_buf = g_strdup_printf ("%s%s",skkbuf_r (buf), append);
		g_free (skkbuf_r (buf));
		skkbuf_r (buf) = append_buf;
	} else {
		skkbuf_r (buf) = g_strdup (append);
	}
	skkbuf_rlen (buf) = strlen (skkbuf_r (buf));
	return;
}

/*
 * add strings to direction_word
 */
static void
add_direction_word (SkkBuffer *buf, const gchar *append)
{
	/* FIXME */
	gchar *append_buf;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_direction_word\n"));
	if (!append) return;
	if (skkbuf_d (buf)) {
		append_buf = g_strdup_printf ("%s%s", skkbuf_d (buf), append);
		g_free (skkbuf_d (buf));
		skkbuf_d (buf) = append_buf;
	} else {
		skkbuf_d (buf) = g_strdup (append);
	}
	skkbuf_dlen (buf) = strlen (skkbuf_d (buf));
	buf->in_number = FALSE;
	DEBUG_DO (printf ("add_henkan %s\n",buf->direction_word));
	return;
}

/*
 * add int to direction_word
 */
static void
add_direction_word_int (SkkBuffer *buf, const gchar *append)
{
	gchar *append_buf;
	DEBUG_DO (printf ("skkbuffer.c : Entering add_direction_word_int\n"));
	if (!append)
		return;
	if (!buf->in_number) {
		if (buf->direction_word) {
			append_buf = g_strconcat (buf->direction_word,"#",NULL);
		} else {
			append_buf = g_strdup ("#");
		}
		buf->number_list = g_list_append (buf->number_list,g_strdup (append));
		buf->has_num++;
		buf->in_number = TRUE;
		buf->direction_word = append_buf;
	} else {
		GList *last;
		gchar *last_buf;
		last = g_list_last (buf->number_list);
		last_buf = (gchar*)last->data;
		append_buf = g_strconcat (last_buf,append,NULL);
		last->data = append_buf;
		g_free (last_buf);
	}
	buf->henkan_len = strlen (buf->direction_word);
}

static void
set_status (SkkBuffer *buf,SkkStatus status)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering set_status\n"));
	if (!buf) return;
	skk_buffer_clear (buf);
	skk_mode_set_status (buf->mode,status);
}

static void
set_j_status (SkkBuffer *buf,SkkJStatus jstatus)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering set_j_status\n"));
	if (!buf) return;
	skk_mode_set_j_status (buf->mode,jstatus);
}

static void
set_query_status (SkkBuffer *buf,SkkQueryStatus status)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering set_query_status\n"));
	if (!buf) return;
	skk_mode_set_query_status (buf->mode,status);
}

static void
clear (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering clear\n"));
	if (!buf) return;
	if (buf->mode) {
		skk_mode_clear (buf->mode);
	}
	if (skkbuf_b (buf))
		set_buf (buf, NULL);
	if (skkbuf_p (buf))
		set_preedit_buf (buf, NULL);
	if (skkbuf_r (buf))
		set_result_buf (buf, NULL);
	if (skkbuf_d (buf))
		set_direction_word (buf, NULL);
	if (buf->candidate_list) {
		skk_dict_item_destroy_all (buf->candidate_list);
		buf->candidate_list = NULL;
	}
	if (buf->number_list) {
		skk_utils_list_free (buf->number_list, TRUE, NULL, NULL);
		buf->number_list = NULL;
		buf->in_number = FALSE;
		buf->has_num = 0;
	}
	if (buf->completion_list) {
		skk_utils_list_free (buf->completion_list, TRUE, NULL, NULL);
		buf->completion_list = NULL;
		buf->completion_count = 0;
	}
	if (skkbuf_okurip (buf)) {
		set_okuri_prefix (buf, NULL);
	}
	if (skkbuf_okuri (buf)) {
		set_okurigana (buf, NULL);
	}
	if (skkbuf_lasth (buf)) {
		set_lasth (buf);
	}
	buf->candidate_count = 0;
	buf->candidate_max = 0;
	buf->in_completion = FALSE;
	skk_query_set_first (buf->query);
	preedit_emit (buf);
}

static void
translate_henkan (SkkBuffer *buf)
{
	gchar *tmp_buf;
	SkkQueryStatus qstatus;
	DEBUG_DO (printf ("skkbuffer.c : Entering translate_henkan\n"));
	qstatus = skk_buffer_get_query_status (buf);
	skk_mode_prepare_mark (buf->mode);
	if (qstatus == QUERY_NORMAL) { 
		/* ֢ʡ */
		tmp_buf = get_translate_buf (buf);
		if (!tmp_buf) return; /* cannot find string */
		add_preedit_buf (buf,tmp_buf);
		add_direction_word (buf,tmp_buf);
		g_free (tmp_buf);
	} else if (qstatus == QUERY_DONE) {
		/* ֢ʡ */
		buf->preedit_buf = create_candidate (buf);
		/*  ⡼ɤ˰ܹ */
		skk_buffer_set_j_status (buf,SKK_CHOICE);
	}
}

static void
translate_okuri (SkkBuffer *buf)
{
	gchar *preedit;
	DEBUG_DO (printf ("skkbuffer.c : Entering translate_okuri\n"));
	skk_buffer_set_query_status (buf,QUERY_OKURI);
	buf->henkan_okurigana = get_translate_buf (buf);
	if (!buf->henkan_okurigana) return;
	/* FIXME */
	skk_buffer_set_query_status (buf,QUERY_DONE);
	preedit = create_candidate (buf);
	/* FIXME put to enable undo */
	if (buf->preedit_buf)
		g_free (buf->preedit_buf);
	buf->preedit_buf = g_strdup_printf ("%s%s",
			preedit,buf->henkan_okurigana);
	g_free (preedit);
	skk_buffer_set_j_status (buf, SKK_CHOICE);
}

static gchar*
get_preedit_string (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering get_preedit_string\n"));
	if (!buf) return NULL;
	if (skk_conf_get_echo (skkbuf_conf (buf))) {
		return g_strdup_printf ("%s%s%s%s%s",
				skk_mode_get_query_status_prefix (buf->mode) ?
				skk_mode_get_query_status_prefix (buf->mode) : "",
				skkbuf_p (buf) ? skkbuf_p (buf) : "",
				skk_mode_get_query_status_append (buf->mode) ?
				skk_mode_get_query_status_append (buf->mode) : "",
				skkbuf_okuri (buf) ? skkbuf_okuri (buf) :  "",
				skkbuf_b (buf) ? skkbuf_b (buf) : "");
	} else {
		return g_strdup_printf ("%s%s%s%s",
				skk_mode_get_query_status_prefix (buf->mode) ?
				skk_mode_get_query_status_prefix (buf->mode) : "",
				skkbuf_p (buf) ? skkbuf_p (buf) : "",
				skk_mode_get_query_status_append (buf->mode) ?
				skk_mode_get_query_status_append (buf->mode) : "",
				skkbuf_okuri (buf) ? skkbuf_okuri (buf) : "");
	}
}

static void
commit (SkkBuffer *buf, gboolean regist_dict)
{
	if (!buf)
		return;
	if (skkbuf_p (buf)) {
		set_result_buf (buf, skkbuf_p (buf));
	}	
	/* TODO */
	/* add dict */
	if (regist_dict)
		skk_buffer_add_dict (buf);

	if (skkbuf_okuri (buf)) {
		add_result_buf (buf, skkbuf_okuri (buf));
	}
	skk_buffer_set_j_status (buf, SKK_NONE);
	/* clear unused buffer */
	set_okurigana (buf, NULL);
	set_okuri_prefix (buf, NULL);
	set_preedit_buf (buf, NULL);
	set_lasth (buf);
	set_buf (buf, NULL);
	set_direction_word (buf, NULL);
	result_emit (buf);
}

static SkkJStatus
get_j_status (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering get_j_status\n"));
	if (!buf)
		return SKK_NONE;
	return skk_mode_get_j_status (buf->mode);
}

static SkkStatus
get_status (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering get_status\n"));
	if (!buf) return SKK_J_MODE;
	return skk_mode_get_status (buf->mode);
}

static gchar*
get_status_string (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering get_status_string\n"));
	return skk_mode_get_status_string (buf->mode);
}

static gchar*
get_translate_buf (SkkBuffer *buf)
{
	gchar *append = NULL;
	gchar *ret;
	gchar *jisx0201 = NULL;
	gboolean katakana_state = FALSE;
	DEBUG_DO (printf ("skkbuffer.c : Entering get_translate_buf\n"));
	if (!buf) 
		return NULL;
	if (!skkbuf_b (buf))
		return NULL;
	katakana_state = skk_mode_get_katakana (skkbuf_mode (buf));
	ret = (gchar *) katakana_state ?
			skkconv_get_katakana (skkbuf_b (buf), &append) :
			skkconv_get_hiragana (skkbuf_b (buf), &append);
	if (ret) {
		set_buf (buf,NULL);
		if (append) {
			set_buf (buf, append);
			g_free (append);
		}
	}
	if (skk_buffer_get_jisx0201_katakana (buf)) {
		jisx0201 = katakana_state ?
			skkconv_katakana_to_jisx0201_kana (ret) :
			skkconv_hiragana_to_jisx0201_kana (ret);

		if (jisx0201) {
			g_free (ret);
			return jisx0201;
		}
	}
	return ret;
}

static gchar*
create_candidate (SkkBuffer *buf)
{
	gchar *okuri = NULL;
	gchar *kana;
	DEBUG_DO (printf ("skkbuffer.c : Entering create_candidate\n"));

	if (!buf)
		return NULL;

	/* if we already have a candidate_list, free it */
	if (buf->candidate_list) {
		skk_dict_item_destroy_all (buf->candidate_list);
		buf->candidate_list = NULL;
	}
	/* if we have a hankan_okuri_char, use it (SKK_OKURI -> SKK_CHOICE) */
	if (skkbuf_okurip (buf)) {
		okuri = skkbuf_okurip (buf);
	} else {
		/* try to translate last buf to char (SKK_HENKAN -> SKK_CHOICE) */
		if (skkbuf_b (buf)) {
			if (skkconv_is_exist (skkbuf_b (buf)))  {
				kana = get_translate_buf (buf);
				if (kana) {
					/*  if successfully translated,
					 *  store preedit_buf, and direction_word.
					 *  but not use skkbuf_b (buf)
					 */ 
					add_preedit_buf (buf, kana);
					add_direction_word (buf, kana);
					g_free (kana);
				}
				/* Drop skkbuf_b (buf) */
				set_buf (buf, NULL);
				okuri = NULL;
			} else {
				/* may not reached */
				okuri = skkbuf_b (buf);
			}
		}
	}
	/* store preedit_buf for UNDO */
	set_lasth (buf);

	/* if we use KATAKANA, translate to HIRAGANA */
	if (skk_buffer_get_katakana (buf) || skk_buffer_get_jisx0201_katakana (buf)) {
		gchar *query = NULL;
		gchar *jisx0201 = NULL;
		if (skk_mode_get_jisx0201_katakana (skkbuf_mode (buf))) {
			jisx0201 = skkconv_jisx0201_kana_to_katakana (skkbuf_d (buf));
		}
		query = skkconv_katakana_to_hiragana (jisx0201 ? jisx0201 : skkbuf_d (buf));
		if (jisx0201)
			g_free (jisx0201);
		buf->candidate_list = skk_query_do_query (buf->query, query, okuri, &buf->candidate_max);
		if (!buf->candidate_list) {
			while (skk_query_set_next (buf->query)) {
				buf->candidate_list = skk_query_do_query (buf->query, query, okuri, &buf->candidate_max);
				if (buf->candidate_list)
					break;
			}		
		}
		g_free (query);
	} else {
		buf->candidate_list = skk_query_do_query (buf->query, skkbuf_d (buf), okuri, &buf->candidate_max);
		if (!buf->candidate_list) {
			while (skk_query_set_next (buf->query)) {
				buf->candidate_list = skk_query_do_query (buf->query, skkbuf_d (buf), okuri, &buf->candidate_max);
				if (buf->candidate_list)
					break;
			}
		}	
	}
	DEBUG_DO (printf ("skkbuffer.c : create_candidate buf->candidate_max %d\n", buf->candidate_max));
	/* Reset current count */
	buf->candidate_count = 0;

	/* when we have an number, translate */
	if (buf->has_num) {
		query_num (buf);
	}

	/* return string */
	if (buf->candidate_list) 
		return g_strdup (((SkkDictItem *)buf->candidate_list->data)->candidate);
	else
		return NULL;
}

void
skk_buffer_translate (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_translate\n"));
	if (buf->result_buf) {
		g_free (buf->result_buf);
		buf->result_buf = NULL;
	}
	switch (skk_mode_get_j_status (buf->mode)) {
		case SKK_NONE:
			DEBUG_DO (printf ("direction_word SKK_NONE\n"));
			skk_mode_prepare_mark (buf->mode);
			buf->result_buf = get_translate_buf (buf);
			break;
		case SKK_HENKAN:
			translate_henkan (buf);
			break;
		case SKK_OKURI:
			translate_okuri (buf);
			/* NOT YET */
			break;
		default:
			break;
	}
	return;
}

SkkBuffer*
skk_buffer_new (void)
{
	SkkBuffer* ret;
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_new\n"));
	ret = g_new (SkkBuffer,1);
	memset (ret,0,sizeof (SkkBuffer));
	ret->mode = skk_mode_new ();
	ret->conf = NULL;
	ret->query = NULL;
	/* set default functions. */
	ret->clear = clear;
	ret->set_status = set_status;
	ret->set_query_status = set_query_status;
	ret->set_j_status = set_j_status;
	ret->get_status = get_status;
	ret->get_query_status = skk_buffer_get_query_status;
	ret->get_j_status = get_j_status;
	ret->get_preedit_string = get_preedit_string;
	ret->get_status_string = get_status_string;
	ret->delete_backward = delete_backward;
	ret->has_preedit = has_preedit;
	ret->has_buf = has_buf;
	ret->can_commit = can_commit;
	ret->can_clear = can_clear;
	ret->query_dict = query_dict;
	ret->change_category = change_category;
	ret->undo = undo;
	ret->commit = commit;
	return ret;
}

void
skk_buffer_destroy (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_destroy\n"));
	skk_buffer_clear (buf);
	skk_mode_destroy (buf->mode);
	skk_conf_destroy (buf->conf);
	if (buf->preedit_listener) {
		skk_utils_list_free (buf->preedit_listener, TRUE, NULL, NULL);
		buf->preedit_listener = NULL;
	}
	if (buf->result_listener) {
		skk_utils_list_free (buf->result_listener, TRUE, NULL, NULL);
		buf->result_listener = NULL;
	}
	if (buf->lookup_listener) {
		skk_utils_list_free (buf->lookup_listener, TRUE, NULL, NULL);
		buf->lookup_listener = NULL;
	}
	if (buf->adddict_listener) {
		skk_utils_list_free (buf->adddict_listener, TRUE, NULL, NULL);
		buf->adddict_listener = NULL;
	}
	if (buf->codetable_listener) {
		skk_utils_list_free (buf->adddict_listener, TRUE, NULL, NULL);
		buf->codetable_listener = NULL;
	}
	g_free (buf);
	return;
}

gboolean
skk_buffer_add_char (SkkBuffer *buf, int ch)
{
	SkkStatus status;
	if (!buf) 
		return FALSE;
	status = skk_buffer_get_status (buf);	

	/* FIXME */
	switch (status) {
		case SKK_J_MODE:
			return add_char (buf, ch);
			break;
		case SKK_LATIN_MODE:
			/* Nothing to do */
			return TRUE;
			break;
		case SKK_JISX0208_LATIN_MODE:
			return add_char (buf, ch);
			break;
		case SKK_ABBREV_MODE:
			return add_char (buf, ch);
			break;
		default:
			break;
	}
	return FALSE;
}

void
skk_buffer_set_katakana (SkkBuffer *buf, gboolean state)
{
	gboolean cur_state;
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_set_katakana\n"));
	if (!buf)
		return;
	cur_state = skk_mode_get_katakana (skkbuf_mode (buf));
	if (state != cur_state) {
		/* reset status */
		skk_buffer_clear (buf);
		skk_mode_set_status (skkbuf_mode (buf), SKK_J_MODE);
		skk_mode_set_katakana (skkbuf_mode (buf), state);
	}
	return;
}

void
skk_buffer_set_jisx0201_katakana (SkkBuffer *buf, gboolean state)
{
	gboolean cur_state;
	if (!buf)
		return;
	cur_state = skk_mode_get_jisx0201_katakana (skkbuf_mode (buf));
	if (state != cur_state) {
		/* reset status */
		skk_buffer_clear (buf);
		skk_mode_set_status (skkbuf_mode (buf), SKK_J_MODE);
		skk_mode_set_jisx0201_katakana (skkbuf_mode (buf), state);
	}
	return;
}

void
skk_buffer_set_query_status (SkkBuffer *buf, SkkQueryStatus status)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_set_query_status\n"));
	if (!buf) 
		return;
	if (buf->set_query_status) {
		buf->set_query_status (buf,status);
	}
	return;
}

SkkQueryStatus
skk_buffer_get_query_status (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_query_status\n"));
	if (!buf) 
		return QUERY_NONE;
	return skk_mode_get_query_status (buf->mode);
}

void
skk_buffer_set_status (SkkBuffer *buf, SkkStatus status)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_set_status\n"));
	if (!buf)
		return;
	if (buf->set_status) {
		buf->set_status (buf,status);
	}
}

SkkStatus
skk_buffer_get_status (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_status\n"));
	if (!buf) 
		return SKK_J_MODE;
	if (buf->get_status) {
		return buf->get_status (buf);
	} else {
		return SKK_J_MODE;
	}
}

void
skk_buffer_set_j_status (SkkBuffer *buf, SkkJStatus jstatus)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_set_j_status\n"));
	if (!buf) 
		return;
	if (buf->set_j_status) {
		buf->set_j_status (buf,jstatus);
	}
}

SkkJStatus
skk_buffer_get_j_status (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_j_status\n"));
	if (!buf) 
		return SKK_NONE;
	if (buf->get_j_status) {
		return buf->get_j_status (buf);
	} else {
		return SKK_NONE;
	}
}

void
skk_buffer_clear (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_clear\n"));
	if (!buf) 
		return;
	if (buf->clear) {
		buf->clear (buf);
	}
}

const gchar *
skk_buffer_get_first_candidate (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_first_candidate\n"));
	if (!buf) return NULL;
	if (!buf->candidate_list) return NULL;
	return (const gchar *)buf->candidate_list->data;
}

const gchar *
skk_buffer_candidate (SkkBuffer *buf)
{
	GList *last_list;
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_candidate\n"));
	if (!buf) return NULL;
	if (!buf->candidate_list) return NULL;
	last_list = g_list_last (buf->candidate_list);
	return (const gchar *)last_list->data;
}

gboolean
skk_buffer_has_next_candidate (SkkBuffer *buf)
{
	GList *next_candidate;
	next_candidate = g_list_next (skkbuf_cur_candidate_list (buf));
	if (!next_candidate) {
		GList *append_list = NULL;
		while (skk_query_set_next (buf->query)) {
			append_list = skk_query_do_query (buf->query, skkbuf_d (buf), skkbuf_okurip (buf), NULL);
			if (append_list) {
				buf->candidate_list = skk_dict_item_merge_list (buf->candidate_list, append_list);
				buf->candidate_max = g_list_length (buf->candidate_list);
				break;
			}
		}
		if (!buf->candidate_list)
			return FALSE;
		next_candidate = g_list_next (skkbuf_cur_candidate_list (buf));
		if (!next_candidate)
			return FALSE;
	}	
	if (next_candidate->data)
		return TRUE;
	return FALSE;
}

gboolean
skk_buffer_set_next_candidate (SkkBuffer *buf)
{
	gchar *candidate;
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_set_next_candidate\n"));
	if (!buf)
		return FALSE;
	if (!buf->candidate_list) {
		if (!skk_buffer_has_next_candidate (buf)) {
			return FALSE;
		}
	}
	buf->candidate_count++;
	if (buf->has_num) {
		query_num (buf);
	}
	candidate = skkbuf_cur_candidate (buf);
	if (candidate) {
		set_preedit_buf (buf,candidate);
		preedit_emit (buf);
		return TRUE;
	}
	return FALSE;
}

gboolean
skk_buffer_set_prev_candidate (SkkBuffer *buf)
{
	gchar *candidate;
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_set_prev_candidate\n"));
	if (!buf)
		return FALSE;
	if (!buf->candidate_list)
		return FALSE;
	if (buf->candidate_count < 1)
		return FALSE;
	buf->candidate_count--;
	candidate = skkbuf_cur_candidate (buf);
	if (!candidate)
		return FALSE;
	if (candidate) {
		set_preedit_buf (buf,candidate);
		preedit_emit (buf);
		return TRUE;
	}
}

gchar *
skk_buffer_get_nth_candidate (SkkBuffer *buf, gint num)
{
	gchar *candidate;
	gint   cur_count;
	/* TODO */
	if (!buf)
		return NULL;
	if (!buf->candidate_list)
		return NULL;
	cur_count = buf->candidate_count;
	buf->candidate_count = num;
	if (buf->has_num) {
		query_num (buf);
	}
	candidate = skkbuf_cur_candidate (buf);
	buf->candidate_count = cur_count;
	if (!candidate)
		return NULL;
	if (skkbuf_okuri (buf)) {
		return g_strconcat (candidate, skkbuf_okuri (buf), NULL);
	} else {
		return g_strdup (candidate);
	}
}

gchar *
skk_buffer_get_cur_candidate (SkkBuffer *buf)
{
	gchar *candidate;
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_cur_candidate\n"));
	if (!buf)
		return NULL;
	if (!buf->candidate_list)
		return NULL;
	if (buf->candidate_count == buf->candidate_max)
		return NULL;
	if (buf->has_num) {
		query_num (buf);
	}
	candidate = skkbuf_cur_candidate (buf);
	if (!candidate)
		return NULL;
	if (skkbuf_okuri (buf)) {
		return g_strconcat (candidate, skkbuf_okuri (buf), NULL);
	} else {
		return g_strdup (candidate);
	}
}

gchar *
skk_buffer_get_next_candidate (SkkBuffer *buf) 
{
	gchar *candidate;
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_next_candidate\n"));
	if (!buf)
		return NULL;
	if (!buf->candidate_list) 
		return NULL;
	buf->candidate_count++;
	if (buf->has_num) {
		query_num (buf);
	}
	candidate = skkbuf_cur_candidate (buf);
	if (!candidate)
		return NULL;
	if (skkbuf_okuri (buf)) {
		return g_strconcat (candidate, skkbuf_okuri (buf), NULL);
	} else {
		return g_strdup (candidate);
	}
}

gchar *
skk_buffer_get_prev_candidate (SkkBuffer *buf) {
	gchar *candidate;
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_prev_candidate\n"));
	if (!buf) 
		return NULL;
	if (!buf->candidate_list) 
		return NULL;
	if (buf->candidate_count < 1) 
		return NULL;
	buf->candidate_count--;
	candidate = skkbuf_cur_candidate (buf);
	if (!candidate)
		return NULL;
	if (skkbuf_okuri (buf)) {
		return g_strconcat (candidate, skkbuf_okuri (buf), NULL);
	} else {
		return g_strdup (candidate);
	}
}

gchar*
skk_buffer_get_preedit_string (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_preedit_string\n"));
	if (!buf) 
		return NULL;
	if (buf->get_preedit_string) {
		return buf->get_preedit_string (buf);
	} else {
		return NULL;
	}
}

const gchar*
skk_buffer_get_status_string (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_status_string\n"));
	/* FIXME OK ? */
	if (!buf) 
		return g_strdup("Unknown");
	if (buf->get_status_string) {
		return buf->get_status_string (buf);
	} else {
		return g_strdup ("Unknown");
	}
}

gboolean
skk_buffer_get_katakana (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_get_katakana\n"));
	if (!buf) 
		return FALSE;
	return skk_mode_get_katakana (buf->mode);
}

gboolean
skk_buffer_get_jisx0201_katakana (SkkBuffer *buf)
{
	if (!buf)
		return FALSE;
	return skk_mode_get_jisx0201_katakana (skkbuf_mode (buf));
}

/**
 * skk_buffer_delete_backward :
 * @buf: SkkBuffer
 **/
void
skk_buffer_delete_backward (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_delete_backward\n"));
	if (!buf) 
		return;
	if (buf->delete_backward) {
		buf->delete_backward (buf);
	}
	return;
}

/**
 * skk_buffer_can_commit :
 * @buf: SkkBuffer
 *
 * Check SkkBuffer has commitable buffer.
 *
 * Returns: True if commitable.
 **/
gboolean
skk_buffer_can_commit (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_can_commit\n"));
	if (!buf) 
		return FALSE;
	if (buf->can_commit) {
		return buf->can_commit (buf);
	}
	return FALSE;
}

/**
 * skk_buffer_has_preedit :
 * @buf: SkkBuffer
 *
 * Check SkkBuffer has preedit-area.
 *
 * Returns: True if SkkBuffer has preedit_area.
 **/
gboolean
skk_buffer_has_preedit (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_has_preedit\n"));
	if (!buf) 
		return FALSE;
	if (buf->has_preedit) {
		return buf->has_preedit (buf);
	}
	return FALSE;
}

/**
 * skk_buffer_has_buf :
 * @buf: SkkBuffer
 *
 * Check SkkBuffer has editted-buffer.
 *
 * Returns: True if SkkBuffer has editted-buffer.
 **/
gboolean
skk_buffer_has_buf (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_has_buf\n"));
	if (!buf)
		return FALSE;
	if (buf->has_buf) {
		return buf->has_buf (buf);
	}
	return FALSE;
}

/**
 * skk_buffer_can_clear :
 * @buf: SkkBuffer
 *
 * Returns : TRUE if SkkBuffer can be clear. 
 **/
gboolean
skk_buffer_can_clear (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_can_clear\n"));
	if (!buf)
		return FALSE;
	if (buf->can_clear) {
		return buf->can_clear (buf);
	}
	return FALSE;
}

/**
 * skk_buffer_query_dict :
 * @buf: SkkBuffer
 *
 **/
void
skk_buffer_query_dict (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_query_dict\n"));
	if (!buf)
		return;
	if (buf->query_dict) {
		buf->query_dict (buf);
	}
	return;
}

/**
 * skk_buffer_undo :
 * @buf: SkkBuffer
 *
 * Returns: TRUE on sucess, otherwise FALSE
 **/
gboolean
skk_buffer_undo (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_undo\n"));
	if (!buf)
		return FALSE;
	if (buf->undo)
		return buf->undo (buf);
	return FALSE;
}

/**
 * skk_buffer_change_category :
 * @buf: SkkBuffer
 *
 * If status is SKK_J_MODE,
 * translate KATAKANA to HIRAGANA, or HIRAGANA to KATAKANA
 * If status is ABBREV_MODE,
 * translate JISX0208_LATIN to LATIN
 * If status is LATIN_MODE,
 * translate LATIN to JISX0208_LATIN
 * 
 **/
void
skk_buffer_change_category (SkkBuffer *buf)
{
	DEBUG_DO (printf ("skkbuffer.c : Entering skk_buffer_change_category\n"));
	if (!buf)
		return;
	if (buf->change_category)
		buf->change_category (buf);
	return;
}

/**
 * skk_buffer_commit :
 * @buf: SkkBuffer
 **/
void
skk_buffer_commit (SkkBuffer *buf, gboolean regist_dict)
{
	if (!buf)
		return;
	if (buf->commit) {
		buf->commit (buf, regist_dict);
	}
	return;
}

void
skk_buffer_set_query (SkkBuffer *buf, SkkQuery *query)
{
	if (!buf)
		return;
	if (buf->query) {
		skk_query_destroy (buf->query);
		buf->query = NULL;
	}
	buf->query = query;
	skk_query_ref (buf->query);
	return;
}

void
skk_buffer_set_conf (SkkBuffer *buf, SkkConf *conf)
{
	if (!buf)
		return;
	if (buf->conf) {
		skk_conf_destroy (buf->conf);
		buf->conf = NULL;
	}
	buf->conf = conf;
	skk_conf_ref (buf->conf);
	return;
}

void
skk_buffer_add_codetable_listener (SkkBuffer *buf, SkkCodeTableListener listener, gpointer user_data)
{
	SkkCodeTableFunc *func;
	if (!buf)
		return;
	func = g_new (SkkCodeTableFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	buf->codetable_listener = g_list_append (buf->codetable_listener, func);
	return;
}

void
skk_buffer_add_lookup_listener (SkkBuffer *buf, SkkLookupListener listener, gpointer user_data)
{
	SkkLookupFunc *func;
	if (!buf)
		return;
	func = g_new (SkkLookupFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	buf->lookup_listener = g_list_append (buf->lookup_listener, func);
	return;
}

void
skk_buffer_add_adddict_listener (SkkBuffer *buf, SkkAddDictListener listener, gpointer user_data)
{
	SkkAddDictFunc *func;
	if (!buf)
		return;
	func = g_new (SkkAddDictFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	buf->adddict_listener = g_list_append (buf->adddict_listener, func);
	return;
}

void
skk_buffer_add_preedit_listener (SkkBuffer *buf, SkkBufListener listener, gpointer user_data)
{
	SkkBufFunc *func;
	if (!buf)
		return;
	func = g_new (SkkBufFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	buf->preedit_listener = g_list_append (buf->preedit_listener, func);
	return;
}

void
skk_buffer_add_result_listener (SkkBuffer *buf, SkkBufListener listener, gpointer user_data)
{
	SkkBufFunc *func;
	if (!buf)
		return;
	func = g_new (SkkBufFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	buf->result_listener = g_list_append (buf->result_listener, func);
	return;
}

gboolean
skk_buffer_remove_lookup_listener (SkkBuffer *buf, SkkLookupListener listener)
{
	return FALSE;
}

gboolean
skk_buffer_remove_preedit_listener (SkkBuffer *buf, SkkBufListener listener)
{
	return FALSE;
}

gboolean
skk_buffer_remove_result_listener (SkkBuffer *buf, SkkBufListener listener)
{
	return FALSE;
}

gboolean
skk_buffer_remove_adddict_listener (SkkBuffer *buf, SkkAddDictListener listener)
{
	return FALSE;
}

gboolean
skk_buffer_remove_codetable_listener (SkkBuffer *buf, SkkCodeTableListener listener)
{
	return FALSE;
}

void
skk_buffer_add_dict (SkkBuffer *buf)
{
	if (!buf)
		return;
	if (!skkbuf_r (buf))
		return;
	if (!skkbuf_d (buf))
		return;
	skk_query_set_with_type (buf->query, SKKQUERY_LOCAL);
	skk_query_add (buf->query, skkbuf_d (buf), skkbuf_okurip (buf), skkbuf_r (buf));
	return;
}

void
skk_buffer_add_dict_with_value (SkkBuffer *buf, gchar *value)
{
	if (!buf)
		return;
	if (!value)
		return;
	if (!skkbuf_d (buf))
		return;
	skk_query_set_with_type (buf->query, SKKQUERY_LOCAL);
	skk_query_add (buf->query, skkbuf_d (buf), skkbuf_okurip (buf), value);
	return;
}

void
skk_buffer_get_completion (SkkBuffer *buf)
{
	gchar *value;
	GList *append = NULL;
	if (!buf)
		return;
	if (buf->completion_list) {
		skk_utils_list_free (buf->completion_list, TRUE, NULL, NULL);
		buf->completion_count = 0;
		buf->completion_list = NULL;
	}
	if (!skkbuf_d (buf))
		return;
	/* initialized before completion */
	skk_query_set_first (buf->query);
	buf->completion_list = skk_query_do_completion (buf->query, skkbuf_d (buf));
	while (skk_query_set_next (buf->query)) {
		append = skk_query_do_completion (buf->query, skkbuf_d (buf));
		if (append)
			buf->completion_list = skk_utils_list_merge_string (buf->completion_list, append);
	}
	/* re-initialized */
	skk_query_set_first (buf->query);
	if (!buf->completion_list)
		return;
	value = (gchar *)g_list_nth_data (buf->completion_list, 0);
	if (!value)
		return;
	set_direction_word (buf, value);
	set_preedit_buf (buf, value);
	buf->in_completion = TRUE;
	preedit_emit (buf);
	/* TODO */
	/* Translate num? */
}

void
skk_buffer_set_next_completion (SkkBuffer *buf)
{
	gchar *value;
	if (!buf)
		return;
	if (!buf->completion_list)
		return;
	buf->completion_count++;
	value = g_list_nth_data (buf->completion_list, buf->completion_count);
	if (!value) {
		buf->completion_count--;
		return;
	}
	set_direction_word (buf, value);
	set_preedit_buf (buf, value);
	preedit_emit (buf);
}

void
skk_buffer_set_prev_completion (SkkBuffer *buf)
{
	gchar *value;
	if (!buf)
		return;
	if (!skkbuf_comp_list (buf))
		return;
	if (skkbuf_comp_count (buf) < 1)
		return;
	buf->completion_count--;
	value = (gchar *)g_list_nth_data (buf->completion_list, buf->completion_count);
	if (!value) /* may not reached */
		return;
	set_direction_word (buf, value);
	set_preedit_buf (buf, value);
	preedit_emit (buf);
}

void
skk_buffer_preedit_emit (SkkBuffer *buf)
{
	if (!buf)
		return;
	preedit_emit (buf);
	return;
}

void
skk_buffer_result_emit (SkkBuffer *buf)
{
	if (!buf)
		return;
	result_emit (buf);
	return;
}

void
skk_buffer_adddict_emit (SkkBuffer *buf)
{
	if (!buf)
		return;
	adddict_emit (buf);
	return;
}

void
skk_buffer_codetable_emit (SkkBuffer *buf)
{
	if (!buf)
		return;
	codetable_emit (buf);
	return;
}

void
skk_buffer_lookup_emit (SkkBuffer *buf)
{
	if (!buf)
		return;
	lookup_emit (buf);
	return;
}

void
skk_buffer_set_preedit (SkkBuffer *buf, const gchar *value)
{
	if (!buf)
		return;
	set_preedit_buf (buf, value);
	preedit_emit (buf);
	return;
}

void
skk_buffer_add_preedit (SkkBuffer *buf, const gchar *value)
{
	if (!buf)
		return;
	add_preedit_buf (buf, value);
	preedit_emit (buf);
	return;
}

void
skk_buffer_set_result (SkkBuffer *buf, const gchar *value)
{
	if (!buf)
		return;
	set_result_buf (buf, value);
	result_emit (buf);
	return;
}

void
skk_buffer_add_result (SkkBuffer *buf, const gchar *value)
{
	if (!buf)
		return;
	add_result_buf (buf, value);
	result_emit (buf);
	return;
}
