/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkmode.c,v 1.3 2002/04/11 02:04:14 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <stdio.h>
#include <string.h>
#include "skkmode.h"
#include "skkutils.h"
#include "skkdebug.h"

static gchar* get_status_string (SkkMode *mode);
static void clear (SkkMode *mode);
static void set_status (SkkMode *mode, SkkStatus status);
static SkkStatus get_status (SkkMode *mode);
static void set_j_status (SkkMode *mode, SkkJStatus jstatus);
static SkkJStatus get_j_status (SkkMode *mode);
static void set_query_status (SkkMode *mode, SkkQueryStatus status);

typedef struct _SkkModeFunc {
	SkkModeListener listener;
	gpointer user_data;
}SkkModeFunc;

static void
status_emit (SkkMode *mode)
{
	GList *tmp_list;
	SkkModeFunc *func;
	if (!mode)
		return;
	for (tmp_list = mode->status_listener; tmp_list; tmp_list = g_list_next (tmp_list)) {
		func = (SkkModeFunc *)tmp_list->data;
		if (func) {
			func->listener (mode, skk_mode_get_status (mode), func->user_data);
		}
	}
	return;
}

static void
j_status_emit (SkkMode *mode)
{
	GList *tmp_list;
	SkkModeFunc *func;
	if (!mode)
		return;
	for (tmp_list = mode->j_status_listener; tmp_list; tmp_list = g_list_next (tmp_list)) {
		func = (SkkModeFunc *)tmp_list->data;
		if (func) {
			func->listener (mode, skk_mode_get_j_status (mode), func->user_data);
		}
	}
	return;
}

static void
query_status_emit (SkkMode *mode)
{
	GList *tmp_list;
	SkkModeFunc *func;
	if (!mode)
		return;
	for (tmp_list = mode->query_status_listener; tmp_list; tmp_list = g_list_next (tmp_list)) {
		func = (SkkModeFunc *)tmp_list->data;
		if (func)
			func->listener (mode, skk_mode_get_query_status (mode), func->user_data);
	}
	return;
}

static gchar *
get_status_string (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering get_status_string\n"));
	if (!mode) return NULL;
	if (mode->status_buf) g_free (mode->status_buf);
	switch (mode->status)
	{
		case SKK_J_MODE:
			if (mode->jisx0201_kana) {
				mode->status_buf = g_strdup("Ⱦ");
				break;
			}
			if (mode->katakana)
				mode->status_buf = g_strdup ("");
			else
				mode->status_buf = g_strdup ("");
			break;
		case SKK_LATIN_MODE:
			mode->status_buf = g_strdup ("SKK");
			break;
		case SKK_JISX0208_LATIN_MODE:
			mode->status_buf = g_strdup ("");
			break;
		case SKK_ABBREV_MODE:
			mode->status_buf = g_strdup ("a");
			break;
		default:
			mode->status_buf = g_strdup ("Unknown");
			break;
	}
	return mode->status_buf;
}

static void
clear (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering clear\n"));
	if (!mode) return;
	if (mode->query_prefix) {
		g_free (mode->query_prefix);
		mode->query_prefix = NULL;
	}
	if (mode->query_suffix) {
		g_free (mode->query_suffix);
		mode->query_suffix = NULL;
	}
	if (mode->status_buf) {
		g_free (mode->status_buf);
		mode->status_buf = NULL;
	}
	// mode->katakana = katakana;
	if (mode->status != SKK_J_MODE)
		skk_mode_set_status (mode, SKK_J_MODE);
	if (mode->jstatus != SKK_NONE)
		skk_mode_set_j_status (mode, SKK_NONE);
	if (mode->query_status != QUERY_NONE)
		skk_mode_set_query_status (mode, QUERY_NONE);
	return;
}

static void
set_status (SkkMode *mode, SkkStatus status)
{
	DEBUG_DO (printf ("skkmode.c : Entering set_status\n"));
	if (!mode) return;
	mode->status = status;
	skk_mode_prepare_mark (mode);
	if (status == SKK_ABBREV_MODE) {
		skk_mode_set_query_status (mode, QUERY_NORMAL);
	}
	status_emit (mode);
	return;
}

static void
set_query_status (SkkMode *mode, SkkQueryStatus status)
{
	DEBUG_DO (printf ("skkmode.c : Entering set_query_status\n"));
	if (!mode)
		return;
	mode->query_status = status;
	skk_mode_prepare_mark (mode);
	query_status_emit (mode);
	return;
}	

static void
set_j_status (SkkMode *mode, SkkJStatus jstatus)
{
	DEBUG_DO (printf ("skkmode.c : Entering set_j_status\n"));
	if (!mode) return;
	mode->jstatus = jstatus;
	switch (jstatus) {
		case SKK_NONE:
			skk_mode_set_query_status (mode, QUERY_NONE);
			break;
		case SKK_HENKAN:
			skk_mode_set_query_status (mode, QUERY_NORMAL);
			break;
		case SKK_OKURI:
			skk_mode_set_query_status (mode, QUERY_OKURI);
			break;
		case SKK_CHOICE:
			skk_mode_set_query_status (mode, QUERY_DONE);
			break;
	}
	j_status_emit (mode);
	return;
}

static SkkJStatus
get_j_status (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering get_j_status\n"));
	if (!mode) return SKK_NONE;
	return mode->jstatus;
}

static SkkStatus
get_status (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering get_status\n"));
	if (!mode) return SKK_J_MODE;
	return mode->status;
}

SkkMode *
skk_mode_new (void)
{
	SkkMode *ret;
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_new\n"));
	ret = g_new (SkkMode,1);
	memset (ret,0,sizeof (SkkMode));
	ret->status = SKK_J_MODE;
	ret->query_status = QUERY_NONE;
	ret->jstatus = SKK_NONE;
	ret->clear = clear;
	ret->set_status = set_status;
	ret->set_j_status = set_j_status;
	ret->set_query_status = set_query_status;
	ret->get_status = get_status;
	ret->get_status_string = get_status_string;
	ret->get_query_status = skk_mode_get_query_status;
	ret->get_j_status = get_j_status;
	return ret;
}

void
skk_mode_destroy (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_destroy\n"));
	if (!mode) return;
	skk_mode_clear (mode);
	if (mode->status_listener) {
		skk_utils_list_free (mode->status_listener, TRUE, NULL, NULL);
		mode->status_listener = NULL;
	}
	if (mode->j_status_listener) {
		skk_utils_list_free (mode->j_status_listener, TRUE, NULL, NULL);
		mode->j_status_listener = NULL;
	}
	if (mode->query_status_listener) {
		skk_utils_list_free (mode->query_status_listener, TRUE, NULL, NULL);
		mode->query_status_listener = NULL;
	}	
	g_free (mode);
	return;
}

void
skk_mode_set_status (SkkMode *mode,SkkStatus status)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_set_status\n"));
	if (!mode) return;
	if (mode->set_status) {
		mode->set_status (mode,status);
	}
	return;
}

void
skk_mode_set_jisx0201_katakana (SkkMode *mode, gboolean val)
{
	if (!mode)
		return;
	g_message ("set jisx0201");
	if (mode->jisx0201_kana != val) {
		mode->jisx0201_kana = val;
		status_emit (mode);
	}
}

gboolean
skk_mode_get_jisx0201_katakana (SkkMode *mode)
{
	if (!mode)
		return FALSE;
	return mode->jisx0201_kana;
}

void
skk_mode_set_katakana (SkkMode* mode,gboolean val)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_set_katakana\n"));
	if (!mode) return;
	if (mode->katakana != val) {
		mode->katakana = val;
		status_emit (mode);
	}
}

gboolean
skk_mode_get_katakana (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_katakana\n"));
	if (!mode) return FALSE;
	return mode->katakana;
}

void
skk_mode_prepare_mark (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_prepare_mark\n"));
	if (!mode) return;
	switch (mode->query_status) {
		case QUERY_NONE:
			if (mode->query_prefix) {
				g_free (mode->query_prefix);
				mode->query_prefix = NULL;
			}
			if (mode->query_suffix) {
				g_free (mode->query_suffix);
				mode->query_suffix = NULL;
			}
			break;
		case QUERY_NORMAL:
			if (mode->query_prefix) {
				g_free (mode->query_prefix);
			}
			mode->query_prefix = g_strdup ("");
			if (mode->query_suffix) {
				g_free (mode->query_suffix);
				mode->query_suffix = NULL;
			}
			break;
		case QUERY_OKURI:
			if (mode->query_prefix) {
				g_free (mode->query_prefix);
			}
			mode->query_prefix = g_strdup ("");
			if (mode->query_suffix) {
				g_free (mode->query_suffix);
			}
			mode->query_suffix = g_strdup ("*");
			break;
		case QUERY_DONE:
			if (mode->query_prefix) {
				g_free (mode->query_prefix);
			}
			mode->query_prefix = g_strdup ("");
			if (mode->query_suffix) {
				g_free (mode->query_suffix);
				mode->query_suffix = NULL;
			}
			break;
		default:
			break;
	}
	return;
}

void
skk_mode_set_query_status (SkkMode *mode, SkkQueryStatus status)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_set_query_status\n"));
	if (!mode)
		return;
	if (mode->set_query_status) {
		mode->set_query_status (mode, status);
	}
	return;
}

void
skk_mode_get_query_status_strings (SkkMode *mode, const gchar **prefix, const gchar **append)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_query_status_strings\n"));
	if (!mode) return;
	if (prefix)
		*prefix = mode->query_prefix;
	if (append)
		*append = mode->query_suffix;
	return;
}

SkkQueryStatus
skk_mode_get_query_status (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_query_status\n"));
	if (!mode) return QUERY_NONE;
	return mode->query_status;
}

const gchar*
skk_mode_get_query_status_prefix (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_query_status_prefix\n"));
	if (!mode) return NULL;
	return mode->query_prefix;
}

const gchar *
skk_mode_get_query_status_append (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_query_status_append\n"));
	if (!mode) return NULL;
	return mode->query_suffix;
}
		

SkkJStatus
skk_mode_get_j_status (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_j_status\n"));
	if (!mode) return SKK_NONE;
	if (mode->get_j_status) {
		return mode->get_j_status (mode);
	} else {
		return SKK_NONE;
	}
}

void
skk_mode_set_j_status (SkkMode *mode, SkkJStatus jstatus)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_set_j_status\n"));
	if (!mode) return;
	if (mode->set_j_status) {
		mode->set_j_status (mode,jstatus);
	}
	return;
}

SkkStatus
skk_mode_get_status (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_status\n"));
	if (!mode) return SKK_J_MODE;
	if (mode->get_status) {
		return mode->get_status (mode);
	} else {
		return SKK_J_MODE;
	}
}

void
skk_mode_clear (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_clear\n"));
	if (!mode) return;
	if (mode->clear) {
		mode->clear (mode);
	}
}

gchar*
skk_mode_get_status_string (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_status_string\n"));
	if (!mode) 
		return g_strdup ("Unknown");
	if (mode->get_status_string) {
		return mode->get_status_string (mode);
	} else {
		return g_strdup ("Unknown");
	}
}

void
skk_mode_add_status_listener (SkkMode *mode, SkkModeListener listener, gpointer user_data)
{
	SkkModeFunc *func;
	if (!mode)
		return;
	func = g_new (SkkModeFunc, 1);
	func->listener =listener;
	func->user_data = user_data;
	mode->status_listener = g_list_append (mode->status_listener, func);
}

void
skk_mode_add_j_status_listener (SkkMode *mode, SkkModeListener listener, gpointer user_data)
{
	SkkModeFunc *func;
	if (!mode)
		return;
	func = g_new (SkkModeFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	mode->j_status_listener = g_list_append (mode->j_status_listener, func);
	return;
}

void
skk_mode_add_query_status_listener (SkkMode *mode, SkkModeListener listener, gpointer user_data)
{
	SkkModeFunc *func;
	if (!mode)
		return;
	func = g_new (SkkModeFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	mode->query_status_listener = g_list_append (mode->query_status_listener, func);
	return;
}

gboolean
skk_mode_remove_status_listener (SkkMode *mode, SkkModeListener listener)
{
	return FALSE;
}

gboolean
skk_mode_remove_j_status_listener (SkkMode *mode, SkkModeListener listener)
{
	return FALSE;
}

gboolean
skk_mode_remove_query_status_listener (SkkMode *mode, SkkModeListener listener)
{
	return FALSE;
}	

#ifdef SKKMODE_DEBUG
static void
status_listener (SkkMode *mode, gint status, gpointer user_data)
{
	g_message ("status changed to %d", status);
	g_message ("mode string is %s", skk_mode_get_status_string (mode));
}

static void
j_status_listener (SkkMode *mode, gint status, gpointer user_data)
{
	g_message ("j_status changed to %d", status);
}

static void
query_status_listener (SkkMode *mode, gint status, gpointer user_data)
{
	g_message ("query_status changed to %d", status);
}

int
main (void)
{
	SkkMode *mode;
	mode = skk_mode_new ();
	/* add listener */
	skk_mode_add_status_listener (mode, status_listener, NULL);
	skk_mode_add_j_status_listener (mode, j_status_listener, NULL);
	skk_mode_add_query_status_listener (mode, query_status_listener, NULL);
	skk_mode_set_status (mode, SKK_J_MODE);
	skk_mode_set_j_status (mode, SKK_CHOICE);
	skk_mode_set_query_status (mode, QUERY_DONE);
	return 0;
}
#endif
