/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkfunc.c,v 1.8 2002/11/09 16:56:49 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <glib.h>
#include <string.h>
#include "skkfunc.h"
#include "skkkeysym.h"
#include "skkfunc_private.h"
#include "skkutils.h"

SkkFunc *
skk_func_new (void)
{
	SkkFunc *ret;
	int i;
	ret = g_new (SkkFunc, 1);
	memset (ret, 0, sizeof (SkkFunc));
	for (i = 0; i < DefaultItemSize; i++) {
		skk_func_add_item (ret, &(DefaultItem[i]));
	}
	ret->ref_count++;
	return ret;
}

static void
item_free_func (gpointer data, gpointer user_data)
{
	if (data) {
		skk_func_item_destroy ((SkkFuncItem *)data);
		data = NULL;
	}
}

void
skk_func_destroy (SkkFunc *func)
{
	if (!func)
		return;
	func->ref_count--;
	if (func->ref_count > 0) {
		return;
	}
	if (func->jmode_func_list) {
		skk_utils_list_free (func->jmode_func_list, TRUE, item_free_func, NULL);
		func->jmode_func_list = NULL;
	}
	if (func->abbrev_func_list) {
		skk_utils_list_free (func->abbrev_func_list, TRUE, item_free_func, NULL);
		func->abbrev_func_list = NULL;
	}
	if (func->latin_func_list) {
		skk_utils_list_free (func->latin_func_list, TRUE, item_free_func, NULL);
		func->latin_func_list = NULL;
	}
	if (func->jisx0208_latin_func_list) {
		skk_utils_list_free (func->jisx0208_latin_func_list, TRUE, item_free_func, NULL);
		func->jisx0208_latin_func_list = NULL;
	}
	g_free (func);
	return;
}

void
skk_func_ref (SkkFunc *func)
{
	if (!func)
		return;
	func->ref_count++;
	return;
}

void
skk_func_unref (SkkFunc *func)
{
	if (!func)
		return;
	func->ref_count--;
	return;
}

void
skk_func_add_item (SkkFunc *func, SkkFuncItem *item)
{
	if (!func)
		return;
	if (!item)
		return;
	if (item->mode & SKK_FUNC_JMODE) {
		func->jmode_func_list = g_list_append (func->jmode_func_list, item);
	}
	if (item->mode & SKK_FUNC_ABBREV) {
		func->abbrev_func_list = g_list_append (func->abbrev_func_list, item);
	}
	if (item->mode & SKK_FUNC_LATIN) {
		func->latin_func_list = g_list_append (func->latin_func_list, item);
	}
	if (item->mode & SKK_FUNC_JISX0208_LATIN) {
		func->jisx0208_latin_func_list = g_list_append (func->jisx0208_latin_func_list, item);
	}
	return;
}

static gboolean
find_item_custom (gconstpointer p1, gconstpointer p2) 
{
	SkkFuncItem *item1;
	SkkFuncItem *item2;
	if (!p1)
		return TRUE;
	if (!p2)
		return TRUE;
	item1 = (SkkFuncItem *)p1;
	item2 = (SkkFuncItem *)p2;

	if ( (item1->key == item2->key) &&
			(item1->submode & item2->submode) &&
			(item1->mask & item2->mask)) {
		return FALSE;
	}
	return TRUE;
}

static SkkFuncItem*
find_item (SkkFunc *func, GList *list, gint submode, gint key, gint mask)
{
	SkkFuncItem query;
	GList *ret;
	if (!func)
		return NULL;
	if (!list)
		return NULL;
	memset (&query, 0, sizeof (SkkFuncItem));
	query.submode = submode;
	query.key = key;
	query.mask = mask;
	ret = g_list_find_custom (list, (gpointer)&query, find_item_custom);
	if (!ret)
		return NULL;
	return (SkkFuncItem *) ret->data;
}

static SkkFuncResult*
do_query_abbrev (SkkFunc *func, gint submode, gint key, gint mask)
{
	SkkFuncItem *item;
	SkkFuncResult *ret;
	if (!func)
		return NULL;
	item = find_item (func, func->abbrev_func_list, submode, key, mask);
	if (!item)
		return NULL;
	ret = g_new (SkkFuncResult, 1);
	ret->result = item->result;
	ret->actual_key = item->actual_key;
	return ret;
}

static SkkFuncResult*
do_query_latin (SkkFunc *func, gint submode, gint key, gint mask)
{
	SkkFuncItem *item;
	SkkFuncResult *ret;
	if (!func)
		return NULL;
	item = find_item (func, func->latin_func_list, submode, key, mask);
	if (!item)
		return NULL;
	ret = g_new (SkkFuncResult, 1);
	ret->result = item->result;
	ret->actual_key = item->actual_key;
	return ret;
}

static SkkFuncResult*
do_query_jmode (SkkFunc *func, gint submode, gint key, gint mask)
{
	SkkFuncItem *item;
	SkkFuncResult *ret;
	if (!func)
		return NULL;
	item = find_item (func, func->jmode_func_list, submode, key, mask);
	if (!item)
		return NULL;
	ret = g_new (SkkFuncResult, 1);
	ret->result = item->result;
	ret->actual_key = item->actual_key;
	return ret;
}

static SkkFuncResult *
do_query_jisx0208_latin (SkkFunc *func, gint submode, gint key, gint mask)
{
	SkkFuncItem *item;
	SkkFuncResult *ret;
	if (!func)
		return NULL;
	item = find_item (func, func->jisx0208_latin_func_list, submode, key, mask);
	if (!item)
		return NULL;
	ret = g_new (SkkFuncResult, 1);
	ret->result = item->result;
	ret->actual_key = item->actual_key;
	return ret;
}

SkkFuncResult*
skk_func_do_query (SkkFunc *func, guint mode, guint submode, guint key, guint mask)
{
	SkkFuncResult *ret = NULL;
	if (!func)
		return NULL;
	switch (mode) {
		case SKK_FUNC_JMODE:
			ret = do_query_jmode (func, submode, key, mask);
			break;
		case SKK_FUNC_ABBREV:
			ret = do_query_abbrev (func, submode, key, mask);
			break;
		case SKK_FUNC_LATIN:
			ret = do_query_latin (func, submode, key, mask);
			break;
		case SKK_FUNC_JISX0208_LATIN:
			ret = do_query_jisx0208_latin (func, submode, key, mask);
			break;
		default:
			break;
	}
	return ret;
}

SkkFuncItem *
skk_func_item_new (guint mode, guint submode, guint key, guint mask, guint actual_key, guint result)
{
	SkkFuncItem *ret;
	ret = g_new (SkkFuncItem, 1);
	ret->mode = mode;
	ret->submode = submode;
	ret->key = key;
	ret->mask = mask;
	ret->actual_key = actual_key;
	ret->user_defined = TRUE;
	ret->result = result;
	return ret;
}

void
skk_func_item_destroy (SkkFuncItem *item)
{
	if (!item)
		return;
	if (!item->user_defined)  {
		return;
	}
	g_free (item);
	return;
}

#ifdef SKKFUNC_DEBUG
int
main (void)
{
	SkkFunc *func;
	SkkFuncResult *result;
	SkkFuncItem *item;
	func = skk_func_new ();
	item = skk_func_item_new (SKK_FUNC_JMODE, QUERY_NONE, 'a', SKK_CONTROL_MASK, 'b', 0);
	skk_func_add_item (func, item);
	result = skk_func_do_query (func, SKK_FUNC_JMODE, QUERY_NONE, 'a', SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 1 %c result %d", result->actual_key, result->result);
	}
	result = skk_func_do_query (func, SKK_FUNC_ABBREV, QUERY_NORMAL, SKK_VK_SPACE, SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 2 %d result %d", result->actual_key, result->result);
	}
	result = skk_func_do_query (func, SKK_FUNC_ABBREV, QUERY_ALL, SKK_VK_h, SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 3 %d result %d", result->actual_key, result->result);
	}
	result = skk_func_do_query (func, SKK_FUNC_JMODE, QUERY_ALL , SKK_VK_q, SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 4 %d result %d", result->actual_key, result->result);
	}
	result = skk_func_do_query (func, SKK_FUNC_JMODE, QUERY_NORMAL , SKK_VK_q, SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 4 %d result %d", result->actual_key, result->result);
	}
	return 0;
}
#endif
