/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkdictitem.c,v 1.3 2002/11/09 16:56:49 famao Exp $ */

/* vi:set ts=4 sw=4: */

#include <string.h>
#include "skkdictitem.h"
#include "skkutils.h"

SkkDictItem*
skk_dict_item_new (void)
{
	SkkDictItem *ret;
	ret = g_new0 (SkkDictItem, 1);
	ret->candidate = NULL;
	ret->annotation = NULL;
	return ret;
}

SkkDictItem*
skk_dict_item_new_with_value (const gchar *candidate, const gchar *annotation)
{
	SkkDictItem *ret;
	ret = g_new0 (SkkDictItem, 1);
	ret->candidate = candidate ? g_strdup (candidate) : NULL;
	ret->annotation = annotation ? g_strdup (annotation) : NULL;
	return ret;
}

SkkDictItem *
skk_dict_item_new_with_string (const gchar *string, const gchar *delim)
{
	gchar *sep;
	gchar *copy;
	SkkDictItem *ret;
	if (!string)
		return skk_dict_item_new ();
	if (!delim) {
		return skk_dict_item_new_with_value (string, NULL);
	}
	copy = g_strdup (string);
	sep = strstr (copy, delim);
	if (!sep) {
		g_free (copy);
		return skk_dict_item_new_with_value (string, NULL);
	} else {
		*sep = '\0';
		sep += strlen (delim);
	}
	ret = skk_dict_item_new_with_value ((const gchar*) copy, sep);
	g_free (copy);
	return ret;
}

void
skk_dict_item_destroy (SkkDictItem *item)
{
	if (!item)
		return;
	if (item->candidate) {
		g_free (item->candidate);
	}
	if (item->annotation) {
		g_free (item->annotation);
	}
	g_free (item);
	return;
}

static void
destroy_item (gpointer p1, gpointer user_data)
{
	skk_dict_item_destroy ((SkkDictItem*)p1);
	return;
}

void
skk_dict_item_destroy_all (GList *list)
{
	if (!list)
		return;
	skk_utils_list_free (list, TRUE, destroy_item, NULL);
}

gchar *
skk_dict_item_to_string (SkkDictItem *item, const gchar *delim)
{
	if (!item)
		return NULL;
	if (!item->candidate)
		return NULL;
	/* when we have annotation, must define delim */
	if (item->annotation && delim) {
		return g_strconcat (item->candidate, delim, item->annotation, NULL);
	} else {
		return g_strconcat (item->candidate, NULL);
	}
}

gchar *
skk_dict_item_to_string_all (GList *list, const gchar *sep, const gchar *delim)
{
	gchar *ret = NULL;
	gchar *tmp_result;
	gchar *word;
	if (!list)
		return NULL;
	if (sep) {
		ret = g_strdup (sep);
	}
	while (list) {
		if (list->data) {
			word = skk_dict_item_to_string ((SkkDictItem*)list->data, delim);
			if (word) {
				if (ret) {
					tmp_result = ret;
					if (sep) {
						ret = g_strconcat (ret, word, sep, NULL);
					} else {
						ret = g_strconcat (ret, word, NULL);
					}
					g_free (tmp_result);
				} else {
					ret = g_strdup (word);
				}
				g_free (word);
			}
		}
		list = g_list_next (list);
	}
	return ret;
}

static gint
find_item (gconstpointer p1, gconstpointer p2)
{
	if (!p1)
		return 1;
	if (!p2)
		return 1;
	if (!strcmp (((SkkDictItem *)p1)->candidate, ((SkkDictItem *)p2)->candidate)) {
		return 0;
	}
	return 1;

}

GList *
skk_dict_item_merge_list (GList *list1, GList *list2)
{
	GList *found;
	GList *ret = NULL;
	GList *first = list1;
	for (; list1; list1 = g_list_next (list1)) {
		if (list1->data) {
			found = g_list_find_custom (list2, list1->data, find_item);
			if (found) {
				list2 = g_list_remove_link (list2, found);
				g_list_free_1 (found);
			}
		}
	}
	ret = g_list_concat (first, list2);
	return ret;
}
