/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkconv_jisx0201_kana.c,v 1.12 2002/11/09 16:56:49 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "skkutils.h"
#include "skkconv_kana.h"
#include "skkconv_jisx0201_kana.h"
#include "skkconv_jisx0201_kana_private.h"

static const gchar *
translate_to_jisx0201_kana (const gchar *str)
{
	gint i;
	if (!str)
		return NULL;
	for (i = 0; i < SKKCONV_JISX0201_SIZE ; i++) {
#if 1
		/* Katakana is always 2bytes */
		if (!strncmp (jisx0201_list[i].katakana , str, 2)) {
#else
		if (!strncmp (jisx0201_list[i].katakana , str, strlen (jisx0201_list[i].katakana))) {
#endif
			return (jisx0201_list[i]).jisx0201_kana;
		}
	}
	return NULL;
}

gchar *
skk_conv_katakana_to_jisx0201_kana (const gchar *katakana)
{
	gchar *ret, *buf;
	const gchar *trans;
	gint len;
	gint actual_len = 0;
	if (!katakana)
		return NULL;
	ret = buf = g_new (gchar, (strlen (katakana) * 2) + 1);
	while (katakana && *katakana) {
		len = skk_utils_charbytes (katakana);
		trans = translate_to_jisx0201_kana (katakana);
		if (trans) {
			strcpy (buf, trans);
			buf += strlen (trans);
			actual_len += strlen (trans);
		} else {
			strncpy (buf, katakana, len);
			buf += len;
			actual_len += len;
		}
		katakana += len;
	}
	ret = g_realloc (ret, actual_len + 1);
	ret[actual_len] = '\0';
	return ret;
}

gchar*
skk_conv_hiragana_to_jisx0201_kana (const gchar *hiragana)
{
	/* TODO */
	gchar *katakana;
	gchar *ret;
	if (!hiragana)
		return NULL;
	katakana = skk_conv_hiragana_to_katakana (hiragana);
	ret = skk_conv_katakana_to_jisx0201_kana (katakana);
	g_free (katakana);
	return ret;
}

static gint
find_kana (gconstpointer p1, gconstpointer p2)
{
	return strcmp ((const gchar *)p1, ((jisx0201_rule_list *)p2)->jisx0201_kana);
}

gchar*
skk_conv_jisx0201_kana_to_katakana (const gchar *jisx0201_kana)
{
	jisx0201_rule_list *list;
	const gchar *copy;
	gchar *word;
	gchar *ret = NULL;
	gchar *tmp_ret;
	if (!jisx0201_kana)
		return NULL;
	copy  = jisx0201_kana;
	while (copy && *copy) {
		word = skk_utils_get_char (copy, TRUE);
		if (!word) {
			break;
		}
redo:
		list = bsearch (word, jisx0201_list, SKKCONV_JISX0201_SIZE, 
				sizeof (jisx0201_rule_list), find_kana);
		if (list) {
			if (ret) {
				tmp_ret = ret;
				ret = g_strconcat (ret, list->katakana, NULL);
				g_free (tmp_ret);
			} else {
				ret = g_strdup (list->katakana);
			}
			copy += strlen (word);
		}  else {
			if (strlen (word) == 4) {
				g_free (word);
				word = skk_utils_get_char (copy, FALSE);
				goto redo;
			}
			/* FIXME dirty hack */
			if (ret) {
				tmp_ret = ret;
				ret = g_strconcat (ret, word, NULL);
				g_free (tmp_ret);
			} else {
				ret = word;
			}
			copy += strlen (word);
		}
		g_free (word);
	}
	return ret;
}

#ifdef SKKCONV_JISX0201_KANA_DEBUG
int
main (void)
{
	int i = 0;
	for (; i < 10000; i++) {
		printf ("from Hiragana %s\n", skk_conv_hiragana_to_jisx0201_kana ("1"));
		printf ("from Katakana %s\n", skk_conv_katakana_to_jisx0201_kana (""));
		printf ("from Katakana %s\n", skk_conv_katakana_to_jisx0201_kana ("Хӥ֥٥ܥѥԥץڥ"));
		printf ("from Katakana %s\n", skk_conv_katakana_to_jisx0201_kana (""));
		printf ("a %s\n", skk_conv_jisx0201_kana_to_katakana ("1"));
		printf ("a %s\n", skk_conv_jisx0201_kana_to_katakana ("ގގގގގʎߎˎߎ̎ߎ͎ߎΎ"));
		printf ("a %s\n", skk_conv_jisx0201_kana_to_katakana ("ގގގގގ"));
		printf ("from Katakana %s\n", skk_conv_katakana_to_jisx0201_kana (""));
	}
	return 0;
}
#endif
