/*
 * IIIMF-SKK, Japanese Language Engine for 
 *                        IIIMF (Internet/Intranet Input Method Framework)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: palette.c,v 1.9 2002/11/09 16:56:47 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <locale.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xmd.h>

#include "xaux_common.h"
#include "xaux_ext_common.h"
#include "skk_xaux.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "pixmaps/hira.xpm"
#include "pixmaps/kata.xpm"
#include "pixmaps/jisx0208.xpm"
#include "pixmaps/jisx0201.xpm"
#include "pixmaps/ascii.xpm"
#ifndef notdef
#include "pixmaps/code.xpm"
#endif

#include "codetable.h"
#include "support.h"

xaux_class_t xaux_class = {
	"org.kondara.skk.PaletteAux", 1,
	 NULL, (Window) 0, (Window) 0,
	(Atom) 0, (Atom) 0, (Atom) 0,
	{(Atom) 0,}, 0, {(Atom) 0,} , 
	0, NULL,
};

static GtkWidget *toplevel = NULL;
static GtkWidget *status_pixmaps[6] = { NULL, NULL, NULL, NULL, NULL, NULL};
static gboolean is_slit = FALSE;
static xaux_class_t *xc_ = 0;
static int im_ = 0;
static int ic_ = 0;

/* Position */
static void palette_reposition (GtkWidget *widget);

/* Event */
static GdkFilterReturn ext_event_handler (GdkXEvent *gdk_xevent, GdkEvent *event, gpointer user_data);
static void status_changed (gchar *text, gint type);
void skk_aux_palette_send_engine (gint int_count, gint *integers, gint str_count, gchar **strings);
static void set_menu (gint type);

/* Menu Detach */
static void menu_detach_func (GtkWidget *attach_widget, GtkMenu *menu);

/* Signal */
static void code_button_clicked (GtkWidget *widget, gpointer user_data);
static void status_button_clicked (GtkWidget *widget, gpointer user_data);
static void status_menu_activate (GtkWidget *widget, gpointer user_data);

static void show_codetable (void);
static GtkWidget *palette_new (void);
static GtkWidget *status_menu_new (void);
static GtkWidget *pixmap_new (GtkWidget *parent, gint type);

/* WM Spec */
static void set_palette_window_docked (GdkWindow *window);
static void set_palette_window_sticky (void);
static void set_palette_window_sticky_gnome (GdkWindow *window);

static gboolean is_blackbox_wm (void);

/* Error */
static int palette_x_error (Display *d, XErrorEvent *error);

static int
palette_x_error (Display *d, XErrorEvent *error)
{
	if (error->error_code) {
		char buf[64];
		XGetErrorText (d, error->error_code, buf, 63);
		fprintf (stderr, 
				"IIIMF-SKK (PaletteAux) **: %s\n serial %ld error_code %d request_code %d minor_code %d\n",
				buf,
				error->serial,
				error->error_code,
				error->request_code,
				error->minor_code);
	}
	return 0;
}

static void
menu_detach_func (GtkWidget *attach_widget, GtkMenu *menu)
{
	/* nothing to do */
	return;
}

#ifndef notdef
static GtkWidget*
pixmap_new_with_xpm (GtkWidget *parent, gchar **xpm)
{
	GdkColormap *colormap;
	GdkPixmap *gdkpixmap;
	GdkBitmap *mask;
	GtkWidget *pixmap;

	colormap = gtk_widget_get_colormap (parent);
	gdkpixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap, &mask,
			NULL, xpm);
	if (gdkpixmap == NULL)
		return NULL;
	pixmap = gtk_pixmap_new (gdkpixmap, mask);
	gdk_pixmap_unref (gdkpixmap);
	gdk_bitmap_unref (mask);
	return pixmap;
}
#endif

static GtkWidget* 
pixmap_new (GtkWidget *parent, gint type)
{
	GdkColormap *colormap;
	GdkPixmap *gdkpixmap;
	GdkBitmap *mask;
	GtkWidget *pixmap;
	gchar **xpm = NULL;

	colormap = gtk_widget_get_colormap (parent);
	switch (type) {
		case 0:
		case 2:
			xpm = hira_xpm;
			break;
		case 1:
			xpm = kata_xpm;
			break;
		case 3:
			xpm = ascii_xpm;
			break;
		case 4:
			xpm = jisx0208_xpm;
			break;
		case 5:
			xpm = jisx0201_xpm;
			break;
	}
	gdkpixmap = gdk_pixmap_colormap_create_from_xpm_d (NULL, colormap, &mask,
			NULL, xpm);
	if (gdkpixmap == NULL)
		return NULL;
#ifndef USE_GTK2
	pixmap = gtk_pixmap_new (gdkpixmap, mask);
#else
	pixmap = gtk_image_new_from_pixmap (gdkpixmap, mask);
#endif
	gdk_pixmap_unref (gdkpixmap);
	gdk_bitmap_unref (mask);
	return pixmap;
}

static GtkWidget *
create_palette_button (gchar *text, GtkPixmap *icon)
{
	GtkWidget *button;
	GtkWidget *label;
	GtkWidget *hbox;

	button = gtk_button_new ();
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (button), hbox);
	gtk_widget_show (hbox);
	if (text) {
		label = gtk_label_new (text);
		gtk_box_pack_end (GTK_BOX (hbox), label, FALSE, FALSE, 0);
		gtk_object_set_data (GTK_OBJECT (button), "label", (gpointer) label);
		gtk_widget_show (label);
	}
	if (icon) {
		gtk_box_pack_end (GTK_BOX (hbox), GTK_WIDGET (icon), FALSE, FALSE, 0);
		gtk_object_set_data (GTK_OBJECT (button), "icon", (gpointer) icon);
		gtk_widget_show (GTK_WIDGET (icon));
	}
	return button;
}

static void
set_palette_window_docked (GdkWindow *window)
{
	Atom atoms[2] = { None, None };
	atoms[0] = XInternAtom (GDK_WINDOW_XDISPLAY (window), "_NET_WM_WINDOW_TYPE_DOCK", False);
	XChangeProperty (GDK_WINDOW_XDISPLAY (window),
			GDK_WINDOW_XWINDOW (window),
			XInternAtom (GDK_WINDOW_XDISPLAY (window), "_NET_WM_WINDOW_TYPE", False),
			XA_ATOM, 32, PropModeReplace, (guchar *)atoms, 1);
	return;
}

static gboolean
is_blackbox_wm (void)
{
	long *value;
	Atom type_r;
	int format_r;
	unsigned long nitems_r, bytes_after;
	/* 
	 * BlackBox-based WindowManager sets _BLACKBOX_PID Property
	 * to RootWindow
	 */
	Atom isbb = XInternAtom (GDK_DISPLAY (), "_BLACKBOX_PID", False);
	if (XGetWindowProperty (GDK_DISPLAY (), GDK_ROOT_WINDOW (), 
			isbb, 0, 128, False, XA_CARDINAL, &type_r, &format_r,
			&nitems_r, &bytes_after,
			(unsigned char **)&value) == Success) {
		XFree (value);
		if (nitems_r > 0 && type_r != None) {
			return TRUE;
		} else {
			return FALSE;
		}
	}
	return FALSE;
}

static GtkWidget *
palette_new (void)
{
	GtkWidget *palette;
	GtkWidget *handlebox;
	GtkWidget *vbox;
	GtkWidget *toolbar;
	GtkWidget *icon;
	GtkWidget *button;

	palette = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_realize (palette);
	vbox = gtk_vbox_new (FALSE, FALSE);
	gtk_window_set_policy (GTK_WINDOW (palette), FALSE, FALSE, TRUE);
	handlebox = gtk_handle_box_new ();
	gtk_widget_show (handlebox);
	gtk_container_add (GTK_CONTAINER (palette), vbox);
	gtk_widget_show (vbox);
	gtk_box_pack_start (GTK_BOX (vbox), handlebox, FALSE, FALSE, 0);
#ifndef USE_GTK2 
	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,
			GTK_TOOLBAR_ICONS);
#else
	toolbar = gtk_toolbar_new ();
	gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), GTK_TOOLBAR_BOTH_HORIZ); 
#endif
	gtk_widget_show (toolbar);
	gtk_container_add (GTK_CONTAINER (handlebox), toolbar);
	icon = pixmap_new (palette, 0);

#ifndef USE_GTK2
	button = create_palette_button ("SKK", GTK_PIXMAP (icon));
	gtk_container_set_resize_mode (GTK_CONTAINER (button), GTK_RESIZE_PARENT);
	gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
			GTK_TOOLBAR_CHILD_WIDGET, button, NULL, NULL, NULL, NULL, NULL, NULL);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
#else
	button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
			GTK_TOOLBAR_CHILD_BUTTON, NULL, "SKK", NULL, NULL, icon,
			NULL, NULL);
#endif
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (status_button_clicked), NULL);
	gtk_object_set_data (GTK_OBJECT (palette), "status_button",
			(gpointer) button);
	gtk_widget_show (button);

	gtk_object_set_data (GTK_OBJECT (palette), "toolbar", (gpointer) toolbar);

#ifndef notdef
	icon = pixmap_new_with_xpm (palette, code_xpm);
	button = create_palette_button (NULL, GTK_PIXMAP (icon));
#else
	icon = gtk_image_new_from_stock (GTK_STOCK_INDEX, GTK_ICON_SIZE_BUTTON);
#endif
#if 0
	button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
			GTK_TOOLBAR_CHILD_BUTTON, NULL, NULL, "Code Table", NULL, icon, 
			GTK_SIGNAL_FUNC (code_button_clicked), NULL);
#endif
	button = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
			GTK_TOOLBAR_CHILD_WIDGET, button, NULL, "Code Table", NULL, NULL,
			NULL, NULL);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (code_button_clicked), NULL);
	gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
	gtk_widget_show (button);		
	return palette;
}

static void
palette_reposition (GtkWidget *widget)
{
	GtkRequisition sizereq;
	gint width,height;
	if (is_slit)
		return;
	if (!GTK_WIDGET_REALIZED (widget))
		return;
	if (!GTK_IS_WINDOW (widget))
		return;
	width = gdk_screen_width ();
	height = gdk_screen_height ();
	gtk_widget_size_request (widget, &sizereq);
#ifdef USE_GTK2
	gtk_window_move (GTK_WINDOW (toplevel), width - sizereq.width,
			height - sizereq.height);
#else
	gtk_window_reposition (GTK_WINDOW (toplevel), width - sizereq.width,
			height - sizereq.height);
#endif
	return;
}

static GdkFilterReturn
ext_event_handler (GdkXEvent *gdk_xevent, GdkEvent *event, gpointer user_data)
{
    XEvent *xevent;
    xevent = (XEvent *) gdk_xevent;
    if (xevent->type == ClientMessage) {
	    xaux_ext_process_client_message(GDK_DISPLAY(), (XClientMessageEvent *) xevent);
	    return GDK_FILTER_REMOVE;
    } 
	else if (xevent->type == KeyRelease) {
	 //   key_event_handler(0, 0, xevent, 0);
		    return GDK_FILTER_REMOVE;
    }
    return GDK_FILTER_CONTINUE;
}

static void
status_menu_activate (GtkWidget *widget, gpointer user_data)
{
	gint send_status[1];
	send_status[0] = GPOINTER_TO_INT (user_data);
	skk_aux_palette_send_engine (1, send_status, 0, NULL);
}

static void
status_button_clicked (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *menu;
	menu = gtk_object_get_data (GTK_OBJECT (toplevel), "menu");
	if (!menu)
		return;
#if 0
	gtk_menu_popup ( GTK_MENU (menu), NULL, NULL, NULL, widget, 0, 0);
#else
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, toplevel, 0, 0);
#endif
	return;
}

static void
code_button_clicked (GtkWidget *widget, gpointer user_data)
{
	GtkWidget *code;
	code = gtk_object_get_data (GTK_OBJECT (toplevel), "code");
	if (!code)
		return;
	gtk_widget_show (code);
	return;
}

static void
set_menu (gint type)
{
	GtkWidget *menu;
	GtkWidget *menuitem;
	menu = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (toplevel), "menu");
	if (!menu)
		return;
	switch (type) {
		case SKK_MODE_JMODE_HIRA:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "hiragana");
			break;
		case SKK_MODE_JMODE_KATA:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "katakana");
			break;
		case SKK_MODE_ABBREV:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "abbrev");
			break;
		case SKK_MODE_LATIN:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "latin");
			break;
		case SKK_MODE_JISX0208_LATIN:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "jisx0208_latin");
			break;
		case SKK_MODE_JISX0201_KANA:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "jisx0201_kana");
			break;
		default:
			menuitem = NULL;
			break;
	}
	if (!menuitem)
		return;
#if 0
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem), TRUE);
#endif
	return;

}
static void
status_changed (gchar *text, gint type)
{
	GtkWidget *toolbar;
	GtkToolbarChild *child;
	GdkBitmap *mask;
	GdkPixmap *pix;
	GtkWidget *item;
#ifdef USE_GTK2
	gchar *ustr;
#endif
	if (!text)
		return;

	toolbar = gtk_object_get_data (GTK_OBJECT (toplevel), "toolbar");
	if (!toolbar)
		return;
#if 0
	child = (GtkToolbarChild *) g_list_nth_data (GTK_TOOLBAR (toolbar)->children, 1);
	if (!child)
		return;
	gtk_label_set_text (GTK_LABEL (child->label), text);
#endif
	child = (GtkToolbarChild *) g_list_first (GTK_TOOLBAR (toolbar)->children)->data;
	if (!child)
		return;
	/* Icon */
	if (type < 0 || type > 5)
		return;
	if (child->widget) {
		/* label */
#ifdef USE_GTK2
		item = child->label;
#else
		item = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (child->widget), "label");
#endif
		if (item) {
#ifdef USE_GTK2
			ustr = g_locale_to_utf8 (text, -1, NULL, NULL, NULL);
			gtk_label_set_text (GTK_LABEL (item), ustr);
			g_free (ustr);
#else
			gtk_label_set_text (GTK_LABEL (item), text);
#endif
		}
#ifdef USE_GTK2
		item = child->icon;
#else
		item = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (child->widget), "icon");
#endif
		if (item) {
			if (status_pixmaps[type] == NULL) {
				status_pixmaps[type] = pixmap_new (toplevel, type);
			}
#ifndef USE_GTK2
			gtk_pixmap_get (GTK_PIXMAP (status_pixmaps[type]), &pix, &mask);
			gtk_pixmap_set (GTK_PIXMAP (item), pix, mask);
#else
			gtk_image_get_pixmap (GTK_IMAGE (status_pixmaps[type]), &pix, &mask);
			gtk_image_set_from_pixmap (GTK_IMAGE (item), pix, mask);
#endif
		}
	}
	/* RePositioning */
	palette_reposition (toplevel);
	return;
}

static void
show_codetable (void)
{
	GtkWidget *code;
	code = (GtkWidget *) gtk_object_get_data (GTK_OBJECT (toplevel), "code");
	if (!code)
		return;
	/* 
	  in the case of having codetable in DISPLAY,
	  raise window
	 */ 
	if (GTK_WIDGET_VISIBLE (code)) {
		gdk_window_raise (code->window);
		return;
	}
	gtk_widget_show (code);
	/* TODO need it ? */
	gdk_window_raise (code->window);
	return;
}

static void
reply_initialize (void)
{
	gint status[1];
	status[0] = SKK_AUX_PALETTE_OK;
	skk_aux_palette_send_engine (1, status, 0, NULL);
	return;
}

Bool
xaux_ext_Start (xaux_class_t *xc, aux_ext_data_t *aux_ext_data)
{
	gtk_widget_show (toplevel);
	gdk_window_raise (toplevel->window);
	return True;
}

Bool
xaux_ext_Done (xaux_class_t *xc, aux_ext_data_t *aux_ext_data)
{
	return True;
}

Bool
xaux_ext_Draw (xaux_class_t *xc, aux_ext_data_t *aux_ext_data)
{
	gint i;

	im_ = aux_ext_data->im;
	ic_ = aux_ext_data->ic;
	xc_ = xc;

#if 0
	g_message ("fuga %d\n", aux_ext_data->integer_count);
	for (i = 0; i < aux_ext_data->string_count; i++) {
		printf ("\tsv[%02d]=%s\n",i, aux_ext_data->string_list[i].ptr);
	}
#endif
	for (i = 0; i < aux_ext_data->integer_count; i++) {
		switch (aux_ext_data->integer_list[i]) {
			/* Initialize Request */
			case SKK_AUX_PALETTE_INIT:
				reply_initialize ();
				break;
			case SKK_AUX_PALETTE_CODE:
				show_codetable ();
				break;
			/* Mode */
			case SKK_MODE_JMODE_HIRA:
				status_changed (aux_ext_data->string_list[i].ptr, 0);
				set_menu (SKK_MODE_JMODE_HIRA);
				break;
			case SKK_MODE_JMODE_KATA:
				status_changed (aux_ext_data->string_list[i].ptr, 1);
				set_menu (SKK_MODE_JMODE_KATA);
				break;
			case SKK_MODE_ABBREV:
				status_changed (aux_ext_data->string_list[i].ptr, 2);
				set_menu (SKK_MODE_ABBREV);
				break;
			case SKK_MODE_LATIN:
				status_changed (aux_ext_data->string_list[i].ptr, 3);
				set_menu (SKK_MODE_LATIN);
				break;
			case SKK_MODE_JISX0201_KANA:
				status_changed (aux_ext_data->string_list[i].ptr, 5);
				set_menu (SKK_MODE_JISX0201_KANA);
				break;
			case SKK_MODE_OFF:
#if 1
				status_changed (aux_ext_data->string_list[i].ptr, 3);
#else
				/* TODO use SkkConf */
				gtk_widget_hide (toplevel);
#endif				
				break;
			case SKK_MODE_JISX0208_LATIN:
				status_changed (aux_ext_data->string_list[i].ptr, 4);
				set_menu (SKK_MODE_JISX0208_LATIN);
				break;
			default:
				break;
		}
	}
	return True;
}


static GtkWidget *
status_menu_new (void)
{
	GtkWidget *menu;
	GtkAccelGroup *accel;
	GSList *status_group = NULL;
	GtkWidget *hiragana;
	GtkWidget *katakana;
	GtkWidget *jisx0201_kana;
	GtkWidget *jisx0208_latin;
	GtkWidget *latin;
	GtkWidget *abbrev;

	menu = gtk_menu_new ();
#if 0
	accel = gtk_menu_ensure_uline_accel_group (GTK_MENU (menu));
#endif
	accel = gtk_menu_get_accel_group (GTK_MENU (menu));
	hiragana = gtk_radio_menu_item_new_with_label
		(status_group, (_("hiragana")));
	gtk_object_set_data (GTK_OBJECT (hiragana), "status", GINT_TO_POINTER (SKK_MODE_JMODE_HIRA));
	gtk_object_set_data (GTK_OBJECT (menu), "hiragana", (gpointer) hiragana);
	status_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (hiragana));

	katakana = gtk_radio_menu_item_new_with_label
		(status_group, (_("katakana")));
	gtk_object_set_data (GTK_OBJECT (katakana), "status", GINT_TO_POINTER (SKK_MODE_JMODE_KATA));
	gtk_object_set_data (GTK_OBJECT (menu), "katakana", (gpointer) katakana);
	status_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (katakana));

	jisx0201_kana = gtk_radio_menu_item_new_with_label
		(status_group, (_("jisx0201_kana")));
	gtk_object_set_data (GTK_OBJECT (jisx0201_kana), "status", GINT_TO_POINTER (SKK_MODE_JISX0201_KANA));
	gtk_object_set_data (GTK_OBJECT (menu), "jisx0201_kana", (gpointer) jisx0201_kana);
	status_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (jisx0201_kana));

	jisx0208_latin = gtk_radio_menu_item_new_with_label
		(status_group, (_("jisx0208_latin")));
	gtk_object_set_data (GTK_OBJECT (jisx0208_latin), "status", GINT_TO_POINTER (SKK_MODE_JISX0208_LATIN));
	gtk_object_set_data (GTK_OBJECT (menu), "jisx0208_latin", (gpointer) jisx0208_latin);
	status_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (jisx0208_latin));

	latin = gtk_radio_menu_item_new_with_label
		(status_group, (_("latin")));
	gtk_object_set_data (GTK_OBJECT (latin), "status", GINT_TO_POINTER (SKK_MODE_LATIN));
	gtk_object_set_data (GTK_OBJECT (menu), "latin", (gpointer) latin);
	status_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (latin));

	abbrev = gtk_radio_menu_item_new_with_label
		(status_group, (_("abbrev")));
	gtk_object_set_data (GTK_OBJECT (abbrev), "status", GINT_TO_POINTER (SKK_MODE_ABBREV));
	gtk_object_set_data (GTK_OBJECT (menu), "abbrev", (gpointer) abbrev);
	status_group = gtk_radio_menu_item_group (GTK_RADIO_MENU_ITEM (abbrev));

	gtk_container_add (GTK_CONTAINER (menu), hiragana);
	gtk_container_add (GTK_CONTAINER (menu), katakana);
	gtk_container_add (GTK_CONTAINER (menu), jisx0201_kana);
	gtk_container_add (GTK_CONTAINER (menu), jisx0208_latin);
	gtk_container_add (GTK_CONTAINER (menu), latin);
	gtk_container_add (GTK_CONTAINER (menu), abbrev);

	gtk_signal_connect (GTK_OBJECT (hiragana), "activate", 
			GTK_SIGNAL_FUNC (status_menu_activate), GINT_TO_POINTER (SKK_MODE_JMODE_HIRA));
	gtk_signal_connect (GTK_OBJECT (katakana), "activate",
			GTK_SIGNAL_FUNC (status_menu_activate), GINT_TO_POINTER (SKK_MODE_JMODE_KATA));
	gtk_signal_connect (GTK_OBJECT (jisx0201_kana), "activate",
			GTK_SIGNAL_FUNC (status_menu_activate), GINT_TO_POINTER (SKK_MODE_JISX0201_KANA));
	gtk_signal_connect (GTK_OBJECT (jisx0208_latin), "activate",
			GTK_SIGNAL_FUNC (status_menu_activate), GINT_TO_POINTER (SKK_MODE_JISX0208_LATIN));
	gtk_signal_connect (GTK_OBJECT (latin), "activate",
			GTK_SIGNAL_FUNC (status_menu_activate), GINT_TO_POINTER (SKK_MODE_LATIN));
	gtk_signal_connect (GTK_OBJECT (abbrev), "activate",
			GTK_SIGNAL_FUNC (status_menu_activate), GINT_TO_POINTER (SKK_MODE_ABBREV));
	gtk_widget_show (hiragana);
	gtk_widget_show (katakana);
	gtk_widget_show (jisx0201_kana);
	gtk_widget_show (jisx0208_latin);
	gtk_widget_show (latin);
	gtk_widget_show (abbrev);
	return menu;
}

void
skk_aux_palette_send_engine (gint int_count, gint *integers, gint str_count, gchar **strings) 
{
	gint i;
	aux_ext_data_t *aux_data;

	if (!im_ || !ic_ || !xc_)
		return;
	aux_data = (aux_ext_data_t *) g_new0 (aux_ext_data_t,1);
	aux_data->im = im_;
	aux_data->ic = ic_;
	aux_data->integer_count = int_count;
	aux_data->string_count = str_count;

	if (aux_data->integer_count) {
		aux_data->integer_list = (int *) g_new0 (int, aux_data->integer_count);
		for (i = 0; i < aux_data->integer_count; i++) {
			aux_data->integer_list[i] = integers[i];
		}
	}
	if (aux_data->string_count) {
		aux_data->string_list = (aux_ext_string_t *) g_new0 (aux_ext_string_t, aux_data->string_count);
		for (i = 0; i < aux_data->string_count; i++) {
			aux_ext_string_t *p = &aux_data->string_list[i];
			p->length = strlen (strings[i]);
			p->ptr = (guchar *) g_strdup (strings[i]);
		}
	}
	xaux_ext_SetValue (GDK_DISPLAY(), xc_, aux_data);

	if (aux_data->integer_list) {
		g_free (aux_data->integer_list);
	}
	if (aux_data->string_list) {
		for (i = 0; i < aux_data->string_count; i++) {
			aux_ext_string_t *p = &aux_data->string_list[i];
			if (p && p->ptr)
				g_free (p->ptr);
		}
		g_free (aux_data->string_list);
	}
	g_free (aux_data);
	return;
}

static void
set_palette_window_sticky (void)
{
	/*
	 * Sticky Support
	 * See http://www.freedesktop.org/standards/wm-spec/x186.html for more detail.
	 */
	Atom wm_state = XInternAtom (GDK_DISPLAY (), "_NET_WM_STATE", FALSE);
	Atom actual_state_sticky = XInternAtom (GDK_DISPLAY (), "_NET_WM_STATE_STICKY", FALSE);
	Atom actual_state_skip_taskbar = XInternAtom (GDK_DISPLAY (), "_NET_WM_STATE_SKIP_TASKBAR", FALSE);
	Atom actual_state_skip_pager = XInternAtom (GDK_DISPLAY (), "_NET_WM_STATE_SKIP_PAGER", FALSE);
	XClientMessageEvent xev;

	/* sending wm_state */
	xev.type = ClientMessage;
	xev.window = GDK_WINDOW_XWINDOW (toplevel->window);
	xev.message_type = (Atom)wm_state;
	xev.format = 32;
	xev.data.l[0] = 1; /* _NET_WM_STATE_ADD */
	xev.data.l[1] = actual_state_sticky;
	xev.data.l[2] = actual_state_skip_taskbar;
	xev.data.l[3] = actual_state_skip_pager;
	xev.data.l[4] = 0;
	XSendEvent (GDK_DISPLAY (), GDK_ROOT_WINDOW (), False, SubstructureNotifyMask, (XEvent *) &xev);

	return;
	
}

static void
set_palette_window_sticky_gnome (GdkWindow *window)
{
	/* WIN_STATE_STICKY */
	long state_flags = (1<<0);
	/* WIN_HINTS_SKIP_FOCUS | WIN_HINTS_SKIP_TASKBAR */
	long hints_flags = (1<<1) | (1<<2);
	Atom gnome_sticky = XInternAtom (GDK_DISPLAY (), "_WIN_STATE", FALSE);
	Atom gnome_hints = XInternAtom (GDK_DISPLAY (), "_WIN_HINTS", FALSE);
	if (!window)
		return;
	/* Gnome Support */
	XChangeProperty (GDK_DISPLAY (), GDK_WINDOW_XWINDOW (window),
			gnome_sticky, 
			XA_CARDINAL,
			32, 
			PropModeReplace, (unsigned char *)&state_flags, 1);

	XChangeProperty (GDK_DISPLAY (), GDK_WINDOW_XWINDOW (window),
			gnome_hints,
			XA_CARDINAL,
			32,
			PropModeReplace, (unsigned char *)&hints_flags, 1);

}

int
main (int argc, char *argv[])
{
	GtkWidget *menu;
	GtkWidget *code;
	/* force to EUC-JP */
#ifdef HAVE_SETENV
	setenv ("LC_ALL","ja_JP.eucJP",True);
#elif HAVE_PUTENV
	putenv ("LC_ALL=ja_JP.eucJP");
#endif
	gtk_set_locale ();

#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
#ifdef USE_GTK2
	bind_textdomain_codeset (PACKAGE, "UTF-8");
#endif
	textdomain (PACKAGE);
#endif

	gtk_init (&argc, &argv);
	toplevel = palette_new ();
	gdk_window_set_decorations (toplevel->window,0);
	/* Dock Support */
	if (getenv ("IIIMF_SKK_WITH_SLIT") || is_blackbox_wm ()) {
		is_slit = TRUE;
	}	
	if (is_slit) {
		XWMHints hints;
		hints.flags = StateHint;
		hints.initial_state = WithdrawnState;
		XSetWMHints (GDK_DISPLAY (), GDK_WINDOW_XWINDOW (toplevel->window), &hints);
	} else {
		set_palette_window_docked (toplevel->window);
		set_palette_window_sticky ();
		set_palette_window_sticky_gnome (toplevel->window);
	}	
	gtk_widget_show (toplevel);
	palette_reposition (toplevel);
	menu = status_menu_new ();
	gtk_menu_attach_to_widget (GTK_MENU (menu), toplevel, menu_detach_func);
	gtk_object_set_data (GTK_OBJECT (toplevel), "menu", (gpointer)menu);
	code = skk_aux_code_new ();
	gtk_object_set_data (GTK_OBJECT (toplevel), "code", (gpointer)code);
	gdk_window_set_transient_for (toplevel->window, code->window);
	gdk_window_add_filter (toplevel->window, ext_event_handler, 0);
	gtk_widget_hide (toplevel);
	if (xaux_ext_init_classes (GDK_DISPLAY (),
				&xaux_class, GDK_WINDOW_XWINDOW (toplevel->window)) == False) {
		exit (1);
	}
	/* set Error Handler */
	XSetErrorHandler (palette_x_error);
	gtk_main ();
	return 0;
}
