package net.reduls.igo.dictionary;

import java.io.IOException;
import java.io.EOFException;
import java.text.ParseException;
import net.reduls.igo.util.ReadLine;
import net.reduls.igo.util.FileMappedInputStream;
import net.reduls.igo.util.FileMappedOutputStream;

public final class Matrix {
    public static void build(String inputDir, String outputDir) throws IOException, ParseException {
	final ReadLine rl = new ReadLine(inputDir+"/matrix.def", "UTF-8"); 
	try {
	    // 一行目はサイズ: [左文脈IDの数] [右文脈IDの数]
	    String s = rl.readEof();
	    final int leftNum = Integer.valueOf(s.substring(0,s.indexOf(' ')));
	    final int rightNum= Integer.valueOf(s.substring(s.indexOf(' ')+1));
	    final FileMappedOutputStream fmos =
		new FileMappedOutputStream(outputDir+"/matrix.bin", 4*2+leftNum*rightNum*2);
	    try {
		fmos.putInt(leftNum);
		fmos.putInt(rightNum);
		
		// 二行目以降はデータ: [左文脈ID] [右文脈ID] [連接コスト]
		final short[] tmpMatrix = new short[leftNum*rightNum];
		for(int i=0; i < leftNum; i++)
		    for(int j=0; j < rightNum; j++) {
			s = rl.readEof();
			final int p1 = s.indexOf(' ');
			final int p2 = s.indexOf(' ',p1+1);
			
			final int   lftID = Integer.valueOf(s.substring(0,    p1));
			final int   rgtID = Integer.valueOf(s.substring(p1+1, p2));
			final short cost  =   Short.valueOf(s.substring(p2+1));
			
			if(i != lftID) throw new ParseException
					   ("Unexpected left context ID. ID="+lftID+", expedted="+i, rl.lineNumber());
			if(j != rgtID) throw new ParseException
					   ("Unexpected right context ID. ID="+rgtID+", expedted="+j, rl.lineNumber());
			
			// NOTE: tmpMatrixという一時配列を用いている理由
			// 
			// この段階で、fmos.putShort(cost)、などとしてファイルに書き出した場合、
			// matrix[leftId][rightId]=cost、といった配列ができる。
			//
			// それでも特に問題はないのだが、今回のケースでは、
			// 「rightIdが固定で、leftIdのみが変動する」といったようなメモリアクセスパターンが多い。
			//
			// そのためtmpMatrix配列を用いて、コスト値の並び順を変更し、
			// matrix[rightId][leftId]とったように、rightIdが第一添字になるようにした方が
			// メモリアクセスの局所性が高まり、若干だが処理速度が向上する。
			tmpMatrix[j*rightNum + i] = cost;  
		    }
		for(short cost : tmpMatrix)
		    fmos.putShort(cost);
	    } finally {
		fmos.close();
	    }
	} catch (NumberFormatException e) {
	    throw new ParseException("matrix.bin: Parse number failed. "+e.getMessage(), rl.lineNumber());
	} catch (EOFException e) {
	    throw new ParseException("matrix.bin: End of file reached", rl.lineNumber());
	} finally {
	    rl.close();
	}
    }

    private final int     leftSize;
    private final int     rightSize;
    private final short[] matrix;

    public Matrix(String dataDir) throws IOException {
	final FileMappedInputStream fmis = new FileMappedInputStream(dataDir+"/matrix.bin");
	try {
	    leftSize = fmis.getInt();
	    rightSize= fmis.getInt();
	    matrix   = fmis.getShortArray(leftSize*rightSize);
	} finally {
	    fmis.close();
	}
    }

    /**
     * 形態素同士の連接コストを求める
     */
    public short linkCost(int leftId, int rightId) {
	return matrix[rightId*rightSize + leftId];
    }
}