/*
Copyright (C) 2015 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.xcloud.aws.ui.dialogs;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.clustercontrol.ws.xcloud.AWSOptionEndpoint;
import com.clustercontrol.ws.xcloud.AvailabilityZone;
import com.clustercontrol.ws.xcloud.CloudEndpoint;
import com.clustercontrol.ws.xcloud.CloudManagerException;
import com.clustercontrol.ws.xcloud.Filter;
import com.clustercontrol.ws.xcloud.Image;
import com.clustercontrol.ws.xcloud.Instance;
import com.clustercontrol.ws.xcloud.InvalidRole_Exception;
import com.clustercontrol.ws.xcloud.InvalidUserPass_Exception;
import com.clustercontrol.ws.xcloud.KeyPair;
import com.clustercontrol.ws.xcloud.NetworkInterface;
import com.clustercontrol.ws.xcloud.SecurityGroup;
import com.clustercontrol.ws.xcloud.Snapshot;
import com.clustercontrol.ws.xcloud.Vpc;
import com.clustercontrol.xcloud.model.CloudModelException;
import com.clustercontrol.xcloud.model.cloud.ILocation;
import com.clustercontrol.xcloud.util.HinemosUtil;

public class AWSResourceProvider {
	protected ILocation location;
	protected CloudEndpoint endpoint;
	protected AWSOptionEndpoint awsEndpoint;
	
	protected List<Image> images;
	protected List<String> instanceTypes;
	protected List<String> shutdownBehaviorTypes;
	protected List<String> volumeTypes;
	protected List<KeyPair> keyNames;
	protected List<SecurityGroup> sgs;
	protected List<Vpc> vpcs;
	protected List<Instance> instances;
	protected Map<String, List<NetworkInterface>> networkInterfaceMap = new HashMap<>();
	
	public AWSResourceProvider(ILocation location) {
		this.location = location;
	}
	
	public List<AvailabilityZone> getZone() {
		try {
			return getAWSOptionEndpoint().getAvailabilityZones(location.getCloudScope().getId(), location.getId());
		} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
			throw new CloudModelException(e);
		}
	}
	public List<Image> getImages(List<Filter> filters) {
		try {
			return getAWSOptionEndpoint().getImagesWithFilter(location.getCloudScope().getId(), location.getId(), filters);
		} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
			throw new CloudModelException(e);
		}
	}
	/**
	 * @return
	 */
	public List<Snapshot> getSnapshots(boolean all) {
		try {
			return getAWSOptionEndpoint().getSnapshotsWithFilter(
					location.getCloudScope().getId(),
					location.getId(),
					all ? HinemosUtil.emptyList(String.class): Arrays.asList("self"),
					HinemosUtil.emptyList(Filter.class));
		} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
			throw new CloudModelException(e);
		}
	}
	public List<SecurityGroup> getSecurityGroups() {
		if (sgs == null) {
			try {
				sgs = getAWSOptionEndpoint().getSecurityGroups(location.getCloudScope().getId(), location.getId());
			} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
				throw new CloudModelException(e);
			}
		}
		return sgs;
	}
	public List<String> getInstanceTypes() {
		if (instanceTypes == null) {
			try {
				instanceTypes = getAWSOptionEndpoint().getInstanceTypes();
			} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
				throw new CloudModelException(e);
			}
		}
		return instanceTypes;
	}
	public List<String> getVolumeType() {
		if (volumeTypes == null) {
			try {
				volumeTypes = getAWSOptionEndpoint().getVolumeTypes();
			} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
				throw new CloudModelException(e);
			}
		}
		return volumeTypes;
	}
	public List<KeyPair> getKeyNames() {
		if (keyNames == null) {
			try {
				keyNames = getAWSOptionEndpoint().getKeyNames(location.getCloudScope().getId(), location.getId());
			} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
				throw new CloudModelException(e);
			}
		}
		return keyNames;
	}
	public List<Vpc> getVpcs() {
		if (vpcs == null) {
			try {
				vpcs = getAWSOptionEndpoint().getVpcs(location.getCloudScope().getId(), location.getId());
			} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
				throw new CloudModelException(e);
			}
		}
		return vpcs;
	}
	public List<String> getInstanceInitiatedShutdownBehaviorTypes() {
		if (shutdownBehaviorTypes == null) {
			try {
				shutdownBehaviorTypes = getAWSOptionEndpoint().getInstanceInitiatedShutdownBehaviorTypes();
			} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
				throw new CloudModelException(e);
			}
		}
		return shutdownBehaviorTypes;
	}
	public List<NetworkInterface> getNetworkInterfaces(String networkId, boolean refresh) {
		if (refresh)
			networkInterfaceMap.remove(networkId);
		
		List<NetworkInterface> nics = networkInterfaceMap.get(networkId);
		if (nics == null) {
			try {
				nics = getAWSOptionEndpoint().getNetworkInterfaces(location.getCloudScope().getId(), location.getId(), networkId);
				networkInterfaceMap.put(networkId, nics);
			} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
				throw new CloudModelException(e);
			}
		}
		return nics;
	}
	public List<Instance> getInstances() {
		if (instances == null) {
			try {
				instances = getCloudEndpoint().getInstances(
						location.getCloudScope().getId(), location.getId(),
						HinemosUtil.emptyList(String.class));
			} catch (CloudManagerException | InvalidRole_Exception | InvalidUserPass_Exception e) {
				throw new CloudModelException(e);
			}
		}
		return instances;
	}
	protected CloudEndpoint getCloudEndpoint() {
		if (endpoint == null) {
			endpoint = location.getCounterScope().getCloudScopeScope().getHinemosManager().getEndpoint(CloudEndpoint.class);
		}
		return endpoint;
	}
	
	protected AWSOptionEndpoint getAWSOptionEndpoint() {
		if (awsEndpoint == null) {
			awsEndpoint = location.getCounterScope().getCloudScopeScope().getHinemosManager().getEndpoint(AWSOptionEndpoint.class);
		}
		return awsEndpoint;
	}
}