/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmp.factory;

import java.net.InetAddress;
import java.net.UnknownHostException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.monitor.run.factory.RunMonitor;
import com.clustercontrol.monitor.run.factory.RunMonitorStringValueType;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.snmp.model.MonitorSnmpInfoEntity;
import com.clustercontrol.snmp.util.QueryUtil;
import com.clustercontrol.snmp.util.RequestSnmp;
import com.clustercontrol.util.Messages;

/**
 * SNMP監視 文字列監視を実行するファクトリークラス<BR>
 *
 * @version 4.0.0
 * @since 2.1.0
 */
public class RunMonitorSnmpString extends RunMonitorStringValueType {

	private static Log m_log = LogFactory.getLog( RunMonitorSnmpString.class );

	private static final String MESSAGE_ID_INFO = "001";
	private static final String MESSAGE_ID_WARNING = "002";
	private static final String MESSAGE_ID_CRITICAL = "003";
	private static final String MESSAGE_ID_UNKNOWN = "004";

	/** SNMP監視情報 */
	private MonitorSnmpInfoEntity m_snmp = null;

	/** コミュニティ名 */
	private String m_communityName = null;

	/** ポート番号 */
	private int m_snmpPort = 161;

	/** タイムアウト（ms） リポジトリのノード情報から取得 */
	private int m_pollingTimeout = 1000;

	/** リトライ回数 リポジトリのノード情報から取得 */
	private int m_pollingRetries = 3;

	/** OID */
	private String m_snmpOid = null;

	/** バージョン */
	private int m_snmpVersion = 1;

	/** オリジナルメッセージ */
	private String m_messageOrg = null;

	/** メッセージ */
	private String m_message = "";

	/**
	 * コンストラクタ
	 * 
	 */
	public RunMonitorSnmpString() {
		super();
	}

	/**
	 * マルチスレッドを実現するCallableTaskに渡すためのインスタンスを作成するメソッド
	 * 
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#runMonitorInfo()
	 * @see com.clustercontrol.monitor.run.util.CallableTask
	 */
	@Override
	protected RunMonitor createMonitorInstance() {
		return new RunMonitorSnmpString();
	}

	/**
	 * OID値を取得
	 * 
	 * @param facilityId ファシリティID
	 * @return 値取得に成功した場合、true
	 * @throws HinemosUnknown
	 */
	@Override
	public boolean collect(String facilityId) throws HinemosUnknown {

		if (m_now != null) {
			m_nodeDate = m_now.getTime();
		}
		m_value = null;

		// メッセージを設定
		m_message = "";
		m_messageOrg = Messages.getString("oid") + " : " + m_snmpOid;

		try{

			// IPアドレスの取得
			String ipNetworkNumber = null;
			try {
				// ノードの属性取得
				NodeInfo info = new RepositoryControllerBean().getNode(facilityId);
				//バージョンを先に読んでIPv4のIPv6どちらを読みだすかせんたくする。

				Integer ipProtNum = info.getIpAddressVersion();
				//m_log.error((Integer)facilityAttrMap.get(FacilityAttributeConstant.IPPROTOCOLNUMBER));
				if(ipProtNum != null && ipProtNum == 6){
					ipNetworkNumber = info.getIpAddressV6();
				} else {
					ipNetworkNumber = info.getIpAddressV4();
				}

				// SNMPポーリングのコミュニティ名
				String communityName = info.getSnmpCommunity();
				if(communityName != null){
					m_log.debug("communityName   : " + communityName);
					m_communityName = communityName;
				}

				// SNMPポーリングのポート番号
				Integer port = info.getSnmpPort();
				if(port != null){
					m_log.debug("port   : " + port);
					m_snmpPort = port;
				}

				// SNMPポーリングのバージョン
				String versionStr = info.getSnmpVersion();;
				if(versionStr != null){
					m_log.debug("version   : " + versionStr);
					m_snmpVersion = SnmpVersionConstant.stringToType(versionStr);
				}

				// SNMPポーリングのタイムアウト
				Integer timeout = info.getSnmpTimeout();
				if(timeout != null){
					m_log.debug("timeout   : " + timeout);
					m_pollingTimeout = timeout;

				}

				// SNMPポーリングのリトライ回数
				Integer retries = info.getSnmpRetryCount();
				m_log.debug("retries   : " + retries);
				if(retries != null){
					m_pollingRetries = retries;
				}
			}
			catch(FacilityNotFound e){
				m_message = Messages.getString("message.snmp.6");
				m_messageOrg = m_messageOrg + " (" + e.getMessage() + ")";
				return false;
			}

			// SNMP値取得
			RequestSnmp m_request = new RequestSnmp();

			boolean result = m_request.polling(
					InetAddress.getByName(ipNetworkNumber),
					m_communityName,
					m_snmpPort,
					m_snmpOid,
					m_snmpVersion,
					m_pollingTimeout,
					m_pollingRetries);

			if(result){

				m_value = m_request.getValue();
				m_nodeDate = m_request.getDate();

				m_messageOrg = m_messageOrg + ", " + Messages.getString("select.value") + " : " + m_value;
			}
			else{
				m_message = m_request.getMessage();
			}
			return result;
		}
		catch (UnknownHostException e) {
			m_log.info("run() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());

			m_message = Messages.getString("message.snmp.5");
			m_messageOrg = m_messageOrg + " (" + e.getMessage() + ")";
			return false;
		}
	}

	/* (非 Javadoc)
	 * SNMP監視情報を設定
	 * @see com.clustercontrol.monitor.run.factory.OperationNumericValueInfo#setMonitorAdditionInfo()
	 */
	@Override
	protected void setCheckInfo() throws MonitorNotFound {

		// SNMP監視情報を取得
		m_snmp = QueryUtil.getMonitorSnmpInfoPK(m_monitorId);

		// SNMP監視情報を設定
		m_snmpOid = m_snmp.getSnmpOid().trim();
	}

	/* (非 Javadoc)
	 * ノード用メッセージIDを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageId()
	 */
	@Override
	public String getMessageId(int id) {

		String messageId = super.getMessageId(id);
		if(messageId == null || "".equals(messageId)){
			return MESSAGE_ID_UNKNOWN;
		}
		return messageId;
	}

	/* (非 Javadoc)
	 * ノード用メッセージを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessage(int)
	 */
	@Override
	public String getMessage(int id) {

		String message = super.getMessage(id);
		if(message == null || "".equals(message)){
			return m_message;
		}
		return message;
	}

	/* (非 Javadoc)
	 * ノード用オリジナルメッセージを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageOrg(int)
	 */
	@Override
	public String getMessageOrg(int id) {
		return m_messageOrg;
	}

	/* (非 Javadoc)
	 * スコープ用メッセージIDを取得
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#getMessageIdForScope(int)
	 */
	@Override
	protected String getMessageIdForScope(int priority) {

		if(priority == PriorityConstant.TYPE_INFO){
			return MESSAGE_ID_INFO;
		}
		else if(priority == PriorityConstant.TYPE_WARNING){
			return MESSAGE_ID_WARNING;
		}
		else if(priority == PriorityConstant.TYPE_CRITICAL){
			return MESSAGE_ID_CRITICAL;
		}
		else{
			return MESSAGE_ID_UNKNOWN;
		}
	}
}
