/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.commons.util;

import java.util.ArrayList;
import java.util.List;

import javax.persistence.EntityExistsException;
import javax.persistence.EntityTransaction;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.ObjectPrivilegeMode;
import com.clustercontrol.accesscontrol.util.ObjectPrivilegeCallback;
import com.clustercontrol.fault.HinemosUnknown;
/*
 * JPA用のトランザクション制御機能
 */
public class JpaTransactionManager {

	private static Log m_log = LogFactory.getLog(JpaTransactionManager.class);

	public final static String EM = "entityManager";
	public final static String CALLBACK = "callbackImpl";
	// HinemosEntityManager
	private HinemosEntityManager em = null;
	// HinemosEntityManagerが既に起動されているか
	private boolean nestedEm = false;
	// EntityTransaction
	private EntityTransaction tx = null;
	// EntityTransactionが既に起動されているか
	private boolean nestedTx = false;

	/**
	 * コンストラクタ
	 */
	public JpaTransactionManager() {
		// EntityManager生成
		em = (HinemosEntityManager)HinemosSessionContext.instance().getProperty(EM);
		if (em == null) {
			em = new HinemosEntityManager();
			((HinemosEntityManager)em).setEntityManager(JpaPersistenceConfig.getHinemosEMFactory().createEntityManager());
			HinemosSessionContext.instance().setProperty(EM, em);

			clearCallback();
		} else {
			nestedEm = true;
		}
	}

	/**
	 * トランザクション開始
	 * 
	 * @param abortIfTxBegined trueかつ既にトランザクションが開始されていればException
	 */
	public void begin(boolean abortIfTxBegined) throws HinemosUnknown {
		// EntityTransaction開始
		tx = em.getTransaction();
		nestedTx = tx.isActive();

		if (!nestedTx) {
			// オブジェクト権限のコールバックメソッド追加
			addCallback(new ObjectPrivilegeCallback());

			List<JpaTransactionCallback> callbacks = getCallback();
			for (JpaTransactionCallback callback : callbacks) {
				if (m_log.isDebugEnabled()) {
					m_log.debug("executing callback preBegin : " + callback.getClass().getName());
				}
				try {
					callback.preBegin();
				} catch (Throwable t) {
					m_log.warn("callback execution failure : " + callback.getClass().getName(), t);
				}
			}

			tx.begin();

			for (JpaTransactionCallback callback : callbacks) {
				if (m_log.isDebugEnabled()) {
					m_log.debug("executing callback postBegin : " + callback.getClass().getName());
				}
				try {
					callback.postBegin();
				} catch (Throwable t) {
					m_log.warn("callback execution failure : " + callback.getClass().getName(), t);
				}
			}
		} else {
			if (abortIfTxBegined) {
				HinemosUnknown e = new HinemosUnknown("transaction has already started.");
				m_log.info("begin() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			}
		}
	}

	/**
	 * トランザクション開始
	 * (トランザクションを引き継ぐ)
	 */
	public void begin() {
		try {
			begin(false);
		} catch (HinemosUnknown e) { }
	}

	/**
	 * コミット処理
	 */
	public void commit() {
		if (!nestedTx) {
			List<JpaTransactionCallback> callbacks = getCallback();
			for (JpaTransactionCallback callback : callbacks) {
				if (m_log.isDebugEnabled()) {
					m_log.debug("executing callback preCommit : " + callback.getClass().getName());
				}
				try {
					callback.preCommit();
				} catch (Throwable t) {
					m_log.warn("callback execution failure : " + callback.getClass().getName());
					throw t;
				}
			}

			tx.commit();

			for (JpaTransactionCallback callback : callbacks) {
				if (m_log.isDebugEnabled()) {
					m_log.debug("executing callback postCommit : " + callback.getClass().getName());
				}
				try {
					callback.postCommit();
				} catch (Throwable t) {
					m_log.warn("callback execution failure : " + callback.getClass().getName());
					throw t;
				}
			}
		}
	}

	/**
	 * フラッシュ処理
	 * (JPAではクエリの順序性が保証されないため、INSERT -> DELETEという順序性が求められる場合、
	 * INSERT(persist) -> flush -> DELETE(remove)という処理が必要である。
	 * flushしないと、DELETE -> INSERTという順序となり、SQLExceptionが生じる可能性がある。
	 */
	public void flush() {
		List<JpaTransactionCallback> callbacks = getCallback();
		for (JpaTransactionCallback callback : callbacks) {
			if (m_log.isDebugEnabled()) {
				m_log.debug("executing callback preFlush : " + callback.getClass().getName());
			}
			try {
				callback.preFlush();
			} catch (Throwable t) {
				m_log.warn("callback execution failure : " + callback.getClass().getName(), t);
			}
		}

		em.flush();

		for (JpaTransactionCallback callback : callbacks) {
			if (m_log.isDebugEnabled()) {
				m_log.debug("executing callback postFlush : " + callback.getClass().getName());
			}
			try {
				callback.postFlush();
			} catch (Throwable t) {
				m_log.warn("callback execution failure : " + callback.getClass().getName(), t);
			}
		}
	}

	/**
	 * ロールバック処理
	 */
	public void rollback() {
		if (!nestedTx) {
			if (tx != null && tx.isActive()) {
				m_log.debug("session is rollback.");
				List<JpaTransactionCallback> callbacks = getCallback();
				for (JpaTransactionCallback callback : callbacks) {
					if (m_log.isDebugEnabled()) {
						m_log.debug("executing callback preRollback : " + callback.getClass().getName());
					}
					try {
						callback.preRollback();
					} catch (Throwable t) {
						m_log.warn("callback execution failure : " + callback.getClass().getName(), t);
					}
				}

				tx.rollback();

				for (JpaTransactionCallback callback : callbacks) {
					if (m_log.isDebugEnabled()) {
						m_log.debug("executing callback postRollback : " + callback.getClass().getName());
					}
					try {
						callback.postRollback();
					} catch (Throwable t) {
						m_log.warn("callback execution failure : " + callback.getClass().getName(), t);
					}
				}
			}
		}
	}

	/**
	 * クローズ処理
	 */
	public void close() {
		if(!nestedEm && em != null) {
			if(em.isOpen()) {
				List<JpaTransactionCallback> callbacks = getCallback();
				for (JpaTransactionCallback callback : callbacks) {
					if (m_log.isDebugEnabled()) {
						m_log.debug("executing callback preClose : " + callback.getClass().getName());
					}
					try {
						callback.preClose();
					} catch (Throwable t) {
						m_log.warn("callback execution failure : " + callback.getClass().getName(), t);
					}
				}

				try {
					em.close();

					for (JpaTransactionCallback callback : callbacks) {
						if (m_log.isDebugEnabled()) {
							m_log.debug("executing callback postClose : " + callback.getClass().getName());
						}
						try {
							callback.postClose();
						} catch (Throwable t) {
							m_log.warn("callback execution failure : " + callback.getClass().getName(), t);
						}
					}
				} finally {
					HinemosSessionContext.instance().setProperty(JpaTransactionManager.EM, null);
				}
			}
			HinemosSessionContext.instance().setProperty(EM, null);
		}
	}

	/**
	 * EntityManager取得
	 */
	public HinemosEntityManager getEntityManager() {
		return em;
	}

	/**
	 * トランザクションが既に開始されているか
	 */
	public boolean isNestedEm() {
		return nestedEm;
	}

	/**
	 * 重複チェック
	 *   重複エラーの場合、EntityExistsException発生
	 * 
	 *  Eclipselink(2.4.1以前)では、persist()時にEntityExistsExceptionが
	 *  発生しないため、本メソッドを使用し重複チェックをする。
	 * 
	 *  Eclipselink(2.4.1以前)では、In-memory上でCascade.removeを行うと、
	 *  DB・キャッシュ間で差異が発生するため、ヒント句を設定している。
	 *
	 * @param clazz 検索対象のEntityクラス
	 * @param primaryKey 検索対象のPrimaryKey
	 * @throws EntityExistsException
	 */
	public <T> void checkEntityExists(Class<T> clazz, Object primaryKey) throws EntityExistsException {
		String strPk = "";
		if (primaryKey instanceof String) {
			strPk = "primaryKey = " + primaryKey;
		} else {
			strPk = primaryKey.toString();
		}
		Object obj = ((HinemosEntityManager)em).
				find(clazz, primaryKey, JpaPersistenceConfig.JPA_EXISTS_CHECK_HINT_MAP, ObjectPrivilegeMode.NONE);
		if (obj != null) {
			// 重複エラー
			EntityExistsException e = new EntityExistsException(clazz.getSimpleName()
					+ ", " + strPk);
			throw e;
		}
	}

	/**
	 * トランザクションAPIの前処理・後処理に関するcallbackクラスを追加する。<br />
	 * ただし、追加されたcallbackクラスの実行順序は保証されない。<br />
	 * 
	 * @param callback 追加するcallbackクラス
	 */
	public void addCallback(JpaTransactionCallback callback) {
		List<JpaTransactionCallback> callbacks = getCallback();
		callbacks.add(callback);
		if (m_log.isDebugEnabled()) {
			m_log.debug("adding callback : " + callback.getClass().getName());
		}
		em.setProperty(CALLBACK, callbacks);
	}

	/**
	 * トランザクションAPIの前処理・後処理に関するcallbackクラス一覧を取得する。<br />
	 * @return callbackクラスのリスト
	 */
	private List<JpaTransactionCallback> getCallback() {
		List<JpaTransactionCallback> callbacks = new ArrayList<JpaTransactionCallback>();

		if (em.getProperties().containsKey(CALLBACK)) {
			Object list = em.getProperties().get(CALLBACK);
			if (list instanceof List) {
				for (Object element : (List)list) {
					if (element instanceof JpaTransactionCallback) {
						callbacks.add((JpaTransactionCallback)element);
					}
				}
			}
		}

		return callbacks;
	}

	/**
	 * EntityManagerとcallbackクラスの関連を消去する。<br/>
	 */
	private void clearCallback() {
		em.setProperty(CALLBACK, new ArrayList<JpaTransactionCallback>());
	}

}
