/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.bean;

import java.io.Serializable;
import java.util.ArrayList;

import javax.xml.bind.annotation.XmlType;

/**
 * 一括制御のコマンド情報をツリー構造として保持するためのクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
@XmlType(namespace = "http://collectiverun.ws.clustercontrol.com")
public class TreeItem implements Serializable, Cloneable {
	private static final long serialVersionUID = -8195124905953255470L;

	/** 親 */
	private TreeItem parent = null;

	/** 情報オブジェクト */
	private CommandInfo commandInfo = null;
	private CommandParameterInfo commandParameterInfo = null;
	private ParameterInfo parameterInfo = null;
	private ParameterSelectInfo parameterSelectInfo = null;
	private TypeInfo typeInfo = null;

	/** 子の格納リスト */
	private ArrayList<TreeItem> children = new ArrayList<TreeItem>();

	public TreeItem(){
		super();
	}

	/**
	 * 引数の情報を保持したインスタンスを返します。
	 * 
	 * @param parent
	 *            親のオブジェクト
	 * @param data
	 *            スコープ情報オブジェクト
	 */
	public TreeItem(TreeItem parent, CommandInfo info) {

		this.setParent(parent);
		this.setCommandInfo(info);

		if (parent != null) {
			parent.addChildren(this);
		}
	}

	/**
	 * 引数の情報を保持したインスタンスを返します。
	 * 
	 * @param parent
	 *            親のオブジェクト
	 * @param data
	 *            スコープ情報オブジェクト
	 */
	public TreeItem(TreeItem parent, CommandParameterInfo info) {

		this.setParent(parent);
		this.setCommandParameterInfo(info);

		if (parent != null) {
			parent.addChildren(this);
		}
	}

	/**
	 * 引数の情報を保持したインスタンスを返します。
	 * 
	 * @param parent
	 *            親のオブジェクト
	 * @param data
	 *            スコープ情報オブジェクト
	 */
	public TreeItem(TreeItem parent, ParameterInfo info) {

		this.setParent(parent);
		this.setParameterInfo(info);

		if (parent != null) {
			parent.addChildren(this);
		}
	}

	/**
	 * 引数の情報を保持したインスタンスを返します。
	 * 
	 * @param parent
	 *            親のオブジェクト
	 * @param data
	 *            スコープ情報オブジェクト
	 */
	public TreeItem(TreeItem parent, ParameterSelectInfo info) {

		this.setParent(parent);
		this.setParameterSelectInfo(info);

		if (parent != null) {
			parent.addChildren(this);
		}
	}

	/**
	 * 引数の情報を保持したインスタンスを返します。
	 * 
	 * @param parent
	 *            親のオブジェクト
	 * @param data
	 *            スコープ情報オブジェクト
	 */
	public TreeItem(TreeItem parent, TypeInfo info) {

		this.setParent(parent);
		this.setTypeInfo(info);

		if (parent != null) {
			parent.addChildren(this);
		}
	}

	// ----- instance フィールド ----- //
	public CommandInfo getCommandInfo() {
		return commandInfo;
	}

	public void setCommandInfo(CommandInfo commandInfo) {
		this.commandInfo = commandInfo;
	}

	public CommandParameterInfo getCommandParameterInfo() {
		return commandParameterInfo;
	}

	public void setCommandParameterInfo(CommandParameterInfo commandParameterInfo) {
		this.commandParameterInfo = commandParameterInfo;
	}

	public ParameterInfo getParameterInfo() {
		return parameterInfo;
	}

	public void setParameterInfo(ParameterInfo parameterInfo) {
		this.parameterInfo = parameterInfo;
	}

	public ParameterSelectInfo getParameterSelectInfo() {
		return parameterSelectInfo;
	}

	public void setParameterSelectInfo(ParameterSelectInfo parameterSelectInfo) {
		this.parameterSelectInfo = parameterSelectInfo;
	}

	public TypeInfo getTypeInfo() {
		return typeInfo;
	}

	public void setTypeInfo(TypeInfo typeInfo) {
		this.typeInfo = typeInfo;
	}

	/**
	 * 親を返します。
	 * 
	 * @return 親
	 */
	public TreeItem getParent() {
		return this.parent;
	}

	/**
	 * 親を設定します。
	 * <p>
	 * 
	 * インスタンス化の際に親へ関係付けているため、子を削除するメソッドを実装した 後に可視性(スコープ)を拡大して下さい。 <br>
	 * また、新しい親への関係付けも行うように実装して下さい。
	 * 
	 * @return 親
	 */
	public void setParent(TreeItem parent) {
		// DTOがループすると、webサービスが動作しないので、parentはsetしない。
		// クライアントでsetする。
		//		this.parent = parent;
	}

	/**
	 * 子を追加します。
	 * <p>
	 * 
	 * この際、childの親はこのオブジェクトとして設定されます。
	 * 
	 * @param child 子
	 */
	public void addChildren(TreeItem child) {
		child.setParent(this);
		children.add(child);
	}

	/**
	 * 子を削除します。
	 * 
	 * @param child
	 */
	public void removeChildren(TreeItem child) {
		for (int i = 0; i < children.size(); i++) {
			if (child.equals(children.get(i))) {
				children.remove(i);
				break;
			}
		}
	}

	/**
	 * 子の数を返します。
	 * 
	 * @return 子の数
	 */
	public int size() {
		return children.size();
	}

	/**
	 * 全ての子を返します。
	 * <p>
	 * 
	 * 並び順は、追加された順となっています。
	 * 
	 * @return 全ての子。
	 */
	public ArrayList<TreeItem> getChildren() {
		return children;
	}

	public void setChildren(ArrayList<TreeItem> children) {
		this.children = children;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#clone()
	 */
	@Override
	public Object clone() {
		CommandInfo cloneCommandInfo = (CommandInfo)this.getCommandInfo().clone();
		CommandParameterInfo cloneCommandParameterInfo = (CommandParameterInfo)this.getCommandParameterInfo().clone();
		ParameterInfo cloneParameterInfo = (ParameterInfo)this.getParameterInfo().clone();
		ParameterSelectInfo cloneParameterSelectInfo = (ParameterSelectInfo)this.getParameterSelectInfo().clone();
		TypeInfo cloneTypeInfo = (TypeInfo)this.getTypeInfo().clone();

		//FIXME super.clone()を呼び出し、clone()メソッドの規約を満たすようにインスタンスを生成する
		//        TreeItem clone = new TreeItem(null, cloneInfo);
		if(cloneCommandInfo != null){
			TreeItem clone;
			try {
				clone = (TreeItem) super.clone();
				clone.setParent(null);
				clone.setCommandInfo(cloneCommandInfo);
				return clone;
			} catch (CloneNotSupportedException e) {
				return null;
			}
		}
		if(cloneCommandParameterInfo != null){
			TreeItem clone;
			try {
				clone = (TreeItem) super.clone();
				clone.setParent(null);
				clone.setCommandParameterInfo(cloneCommandParameterInfo);
				return clone;
			} catch (CloneNotSupportedException e) {
				return null;
			}
		}
		if(cloneParameterInfo != null){
			TreeItem clone;
			try {
				clone = (TreeItem) super.clone();
				clone.setParent(null);
				clone.setParameterInfo(cloneParameterInfo);
				return clone;
			} catch (CloneNotSupportedException e) {
				return null;
			}
		}
		if(cloneParameterSelectInfo != null){
			TreeItem clone;
			try {
				clone = (TreeItem) super.clone();
				clone.setParent(null);
				clone.setParameterSelectInfo(cloneParameterSelectInfo);
				return clone;
			} catch (CloneNotSupportedException e) {
				return null;
			}
		}
		if(cloneTypeInfo != null){
			TreeItem clone;
			try {
				clone = (TreeItem) super.clone();
				clone.setParent(null);
				clone.setTypeInfo(cloneTypeInfo);
				return clone;
			} catch (CloneNotSupportedException e) {
				return null;
			}
		}

		return null;
	}
}