/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.util.apllog;


import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.monitor.bean.EventConfirmConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.session.NotifyControllerBean;
import com.clustercontrol.notify.util.NotifyUtil;
import com.clustercontrol.notify.util.SendSyslog;
import com.clustercontrol.util.CommandCreator;
import com.clustercontrol.util.CommandCreator.PlatformType;
import com.clustercontrol.util.CommandExecutor;
import com.clustercontrol.util.CommandExecutor.CommandResult;
import com.clustercontrol.util.StringBinder;

/**
 * 
 * Hinemosの内部ログ（HinemosApp.log）の出力を行うクラス<BR>
 * 
 * Hinemos内部で発生する事象をログやHinemosのイベントとして
 * 処理します。
 * 
 */
public class AplLogger {

	private static final String RESOURCE_BUNDLE= "com.clustercontrol.util.apllog.apllog";
	private static ResourceBundle m_bundle = ResourceBundle.getBundle(RESOURCE_BUNDLE);

	private static final String INTERNAL_SCOPE="INTERNAL";
	private static final String INTERNAL_SCOPE_TEXT="Hinemos_Internal";

	private static final String PRIORITY_UNKNOWN = "unknown";
	private static final String PRIORITY_INFO = "info";
	private static final String PRIORITY_WARNING = "warning";
	private static final String PRIORITY_CRITICAL = "critical";
	private String m_pluginID;
	private String m_aplName;

	private static ConcurrentHashMap<String, String> m_priorityMap = null;

	// Syslog
	private static boolean isSyslog = false;
	private static int syslogLevel = PriorityConstant.TYPE_INFO;
	private static String[] syslogHostList = {"127.0.0.1"};
	private static int syslogPort = 514;
	private static String syslogFacility = "daemon";
	private static String syslogSeverity = "alert";

	// event
	private static boolean isEvent = true;
	private static int eventLevel = PriorityConstant.TYPE_INFO;

	// file
	private static boolean isFile = true;
	private static int fileLevel = PriorityConstant.TYPE_INFO;

	// mail
	private static boolean isMail = true;
	private static int mailLevel = PriorityConstant.TYPE_INFO;
	private static String[] mailAddress = {""};

	// command
	private static boolean isCommand = true;
	private static int commandLevel = PriorityConstant.TYPE_INFO;
	private static String commandUser = "root";
	private static String commandLine = "";
	private static int commandTimeout = 15000;

	private static Log FILE_LOGGER = LogFactory.getLog("HinemosInternal");
	private static Log log = LogFactory.getLog(AplLogger.class);

	static {
		// hinemos.propertiesから設定値の取得
		initialParameter();
	}

	/**
	 * hinemos.propertiesから設定値の取得
	 */
	private static void initialParameter(){

		try {
			/////
			// 設定値取得(common.internal.syslog)
			////
			String syslog = HinemosProperties.getProperty("common.internal.syslog");
			if(syslog != null && syslog.equals("true")){
				isSyslog = true;
			}
			String hosts = HinemosProperties.getProperty("common.internal.syslog.host");
			syslogHostList = hosts.split(",");
			syslogPort = Integer.parseInt(HinemosProperties.getProperty("common.internal.syslog.port"));
			syslogFacility = HinemosProperties.getProperty("common.internal.syslog.facility");
			syslogSeverity = HinemosProperties.getProperty("common.internal.syslog.severity");
			syslogLevel = getPriority(HinemosProperties.getProperty("common.internal.syslog.priority"));

			/////
			// 設定値取得(common.internal.event)
			////
			String event = HinemosProperties.getProperty("common.internal.event");
			if(event != null && event.equals("false")){
				isEvent = false;
			}
			eventLevel = getPriority(HinemosProperties.getProperty("common.internal.event.priority"));

			/////
			// 設定値取得(common.internal.file)
			////
			String file = HinemosProperties.getProperty("common.internal.file");
			if(file != null && file.equals("false")){
				isFile = false;
			}
			fileLevel = getPriority(HinemosProperties.getProperty("common.internal.file.priority"));

			/////
			// 設定値取得(common.internal.mail)
			////
			String mail = HinemosProperties.getProperty("common.internal.mail");
			if(mail != null && mail.equals("false")){
				isMail = false;
			}
			mailLevel = getPriority(HinemosProperties.getProperty("common.internal.mail.priority"));
			String addr = HinemosProperties.getProperty("common.internal.mail.address");
			try {
				mailAddress = addr.split(",");
			} catch (Exception e) {}

			/////
			// 設定値取得(common.internal.command)
			////
			String command = HinemosProperties.getProperty("common.internal.command");
			if(command != null && command.equals("false")){
				isCommand = false;
			}
			commandLevel = getPriority(HinemosProperties.getProperty("common.internal.command.priority"));
			commandUser = HinemosProperties.getProperty("common.internal.command.user");
			commandLine = HinemosProperties.getProperty("common.internal.command.commandline");
			try {
				commandTimeout = Integer.parseInt(HinemosProperties.getProperty("common.internal.command.timeout"));
			} catch (Exception e) {}

		} catch (Exception e) {
			log.error("initialParameter()",e);
		}

		//重要度文字列取得
		m_priorityMap = new ConcurrentHashMap<String, String>();
		m_priorityMap.put(PRIORITY_CRITICAL, getString("priority.critical"));
		m_priorityMap.put(PRIORITY_WARNING, getString("priority.warning"));
		m_priorityMap.put(PRIORITY_INFO, getString("priority.info"));
		m_priorityMap.put(PRIORITY_UNKNOWN, getString("priority.unknown"));

		// debug用出力
		if(log.isDebugEnabled()){
			log.debug("initialParameter() isEvent=" + isEvent + ", eventLevel=" + eventLevel);
			log.debug("initialParameter() isFile=" + isFile + ", fileLevel=" + fileLevel);
			log.debug("initialParameter() isMail=" + isMail + ", mailLevel=" + mailLevel);
			log.debug("initialParameter() isCommand=" + isCommand + ", commandLevel=" + commandLevel);
			log.debug("initialParameter() isSyslog=" + isSyslog + ", syslogLevel=" + syslogLevel);
			if (isSyslog) {
				String syslogHostStr = "";
				if(syslogHostList != null){
					for (int i = 0; i < syslogHostList.length; i++) {
						if (i != 0) {
							syslogHostStr += ",";
						}
						syslogHostStr += syslogHostList[i];
					}
				}
				log.debug("initialParameter() syslogPort=" + syslogPort +
						", syslogFacility=" + syslogFacility +
						", syslogSeverity=" + syslogSeverity +
						", syslogHost=" + syslogHostStr);
			}
		}
	}

	/**
	 * コンストラクタ<BR>
	 * 
	 * @param pluginID	プラグインID
	 * @param aplId		アプリケーションID
	 */
	public AplLogger(String pluginID, String aplId) {
		m_pluginID = pluginID;

		//アプリケーション名取得
		m_aplName = getString(pluginID + "." + aplId);
	}

	/**
	 * メインクラス
	 * @param args
	 */
	public static void main(String[] args) {
		AplLogger apllog = new AplLogger("REP","rep");
		apllog.put("SYS","001");
		apllog.put("USR","001");
		apllog.put("USR","002");
		apllog.put("USR","003");
	}

	/**
	 * ログを出力します。<BR>
	 * 
	 * @param moniterID		監視項目ID
	 * @param msgID			メッセージID
	 * @since
	 */
	public void put(String moniterID, String msgID) {
		put(moniterID, msgID,null,null);
	}

	/**
	 * 
	 * ログを出力します。<BR>
	 * @param moniterID		監視項目ID
	 * @param msgID			メッセージID
	 * @param msgArgs		メッセージ置換項目
	 * @since
	 */
	public void put(String moniterID, String msgID, Object[] msgArgs) {
		put(moniterID, msgID,msgArgs,null);
	}

	/**
	 * 
	 * ログを出力します。<BR>
	 * 
	 * @param monitorId		監視項目ID
	 * @param msgID			メッセージID
	 * @param msgArgs		メッセージ置換項目
	 * @param detailMsg		詳細メッセージ
	 * @since
	 */
	public void put(String monitorId, String msgID, Object[] msgArgs, String detailMsg) {
		//現在日時取得
		Date nowDate = new Date();

		//監視項目IDとプラグインIDでイベントメッセージ（リソースファイル）から
		String keyBase = m_pluginID + "." + monitorId + "." + msgID;

		//重要度を取得
		String keyPriority = keyBase + "." + "priority";

		String priority = getString(keyPriority);

		//メッセージを取得する。
		String keyMsg = keyBase + "." + "message";

		//メッセージ項目値が指定されている場合、メッセージの項目を置換
		String msg = null;
		if(msgArgs != null && msgArgs.length != 0){
			msg = getString(keyMsg,msgArgs);
		}else{
			msg = getString(keyMsg);
		}

		//    	・発生日時				:現在日時※メソッド呼び出し時に取得したもの
		//    	・出力日時				:現在日時※イベント出力直前に取得したもの

		//メッセージ情報作成
		OutputBasicInfo output = new OutputBasicInfo();
		output.setPluginId(m_pluginID);
		output.setMonitorId(monitorId);
		output.setFacilityId(INTERNAL_SCOPE);
		output.setScopeText(INTERNAL_SCOPE_TEXT);
		output.setApplication(m_aplName);
		output.setMessageId(msgID);
		output.setMessage(msg);
		output.setMessageOrg(detailMsg == null ? "":detailMsg);
		output.setPriority(getPriority(priority));
		if (nowDate != null) {
			output.setGenerationDate(nowDate.getTime());
		}

		if(isEvent && isOutput(eventLevel, getPriority(priority))){
			putEvent(output);
		}
		if(isFile && isOutput(fileLevel, getPriority(priority))){
			putFile(output);
		}
		if (isSyslog && isOutput(syslogLevel, getPriority(priority))){
			putSyslog(output);
		}
		if (isMail && isOutput(mailLevel, getPriority(priority))) {
			putMail(output);
		}
		if (isCommand && isOutput(commandLevel, getPriority(priority))) {
			putCommand(output);
		}
	}

	private void putFile(OutputBasicInfo notifyInfo) {
		/**	ログファイル出力用フォーマット「日付  プラグインID,アプリケーション,監視項目ID,メッセージID,ファシリティID,メッセージ,詳細メッセージ」 */
		MessageFormat logfmt = new MessageFormat("{0,date,yyyy/MM/dd HH:mm:ss}  {1},{2},{3},{4},{5},{6}");
		//メッセージを編集
		Object[] args ={notifyInfo.getGenerationDate(),m_pluginID, m_aplName,
				notifyInfo.getMonitorId(), notifyInfo.getMessageId(), notifyInfo.getPriority(),
				notifyInfo.getMessage(), notifyInfo.getMessageOrg()};
		String logmsg = logfmt.format(args);
		//ファイル出力
		log.debug("putFile() logmsg = " + logmsg);
		FILE_LOGGER.info(logmsg);
	}

	private void putSyslog(OutputBasicInfo notifyInfo) {
		/**	syslog出力用フォーマット
		 * 「日付  プラグインID,アプリケーション,監視項目ID,メッセージID,ファシリティID,メッセージ,詳細メッセージ」 */
		MessageFormat syslogfmt = new MessageFormat("hinemos: {0},{1},{2},{3},{4},{5}");

		// メッセージを編集
		Object[] args ={m_pluginID, m_aplName, notifyInfo.getMonitorId(), notifyInfo.getMessageId(),
				notifyInfo.getPluginId(), notifyInfo.getMessage(), notifyInfo.getMessageOrg()};
		String logmsg = syslogfmt.format(args);

		// 送信時刻をセット
		SimpleDateFormat sdf = new SimpleDateFormat(SendSyslog.HEADER_DATE_FORMAT, Locale.US);
		String timeStamp = sdf.format(new Date());
		
		if(syslogHostList == null){
			return;
		}
		for (String syslogHost : syslogHostList) {
			log.debug("putSyslog() syslogHost = " + syslogHost + ", syslogPort = " + syslogPort +
					", syslogFacility = " + syslogFacility + ", syslogSeverity = " + syslogSeverity +
					", logmsg = " + logmsg + ", timeStamp = " + timeStamp);

			try {
				new NotifyControllerBean().sendAfterConvertHostname(syslogHost, syslogPort, syslogFacility,
						syslogSeverity, INTERNAL_SCOPE, logmsg, timeStamp);
			} catch (InvalidRole e) {
				log.warn("fail putSyslog monitorId=" + notifyInfo.getMonitorId() + ", message=" + notifyInfo.getMessage());
			} catch (HinemosUnknown e) {
				log.warn("fail putSyslog monitorId=" + notifyInfo.getMonitorId() + ", message=" + notifyInfo.getMessage());
			}
		}
	}




	private boolean putEvent(OutputBasicInfo notifyInfo) {
		try {
			new NotifyControllerBean().insertEventLog(notifyInfo, EventConfirmConstant.TYPE_UNCONFIRMED);
			return true;
		} catch (HinemosUnknown e) {
			log.warn("fail putEvent monitorId=" + notifyInfo.getMonitorId() + ", message=" + notifyInfo.getMessage());
			return false;
		} catch (InvalidRole e) {
			log.warn("fail putEvent monitorId=" + notifyInfo.getMonitorId() + ", message=" + notifyInfo.getMessage());
			return false;
		}
	}

	private void putMail(OutputBasicInfo notifyInfo) {
		// メール通知（デフォルトテンプレート）
		try {
			new NotifyControllerBean().sendMail(mailAddress, notifyInfo);
		} catch (Exception e) {
			log.warn("fail putMail monitorId=" + notifyInfo.getMonitorId() + ", message=" + notifyInfo.getMessage());
		}
	}

	private void putCommand(OutputBasicInfo notifyInfo) {
		// コマンド通知
		try {
			Map<String, String> param = NotifyUtil.createParameter(notifyInfo, null);
			StringBinder binder = new StringBinder(param);
			String command = binder.bindParam(commandLine);

			log.info("excuting command. (effectiveUser = " + commandUser + ", command = " + command + ", mode = " + PlatformType.UNIX + ", timeout = " + commandTimeout + ")");
			String[] cmd = CommandCreator.createCommand(commandUser, command, PlatformType.UNIX);
			CommandExecutor cmdExec = new CommandExecutor(cmd, commandTimeout);
			cmdExec.execute();
			CommandResult ret = cmdExec.getResult();

			if (ret != null) {
				log.info("executed command. (exitCode = " + ret.exitCode + ", stdout = " + ret.stdout + ", stderr = " + ret.stderr + ")");
			}

		} catch (Exception e) {
			log.warn("fail putCommand monitorId=" + notifyInfo.getMonitorId() + ", message=" + notifyInfo.getMessage());
			return;
		}
	}


	/**
	 * 
	 * 文字列から、Priority区分を取得します。<BR>
	 * 
	 * @param priority
	 * @since
	 */
	private static int getPriority(String priority) {
		int ret = PriorityConstant.TYPE_UNKNOWN;
		if(priority.equals(PRIORITY_CRITICAL)){
			ret = PriorityConstant.TYPE_CRITICAL;
		}else if(priority.equals(PRIORITY_WARNING)){
			ret = PriorityConstant.TYPE_WARNING;
		}else if(priority.equals(PRIORITY_INFO)){
			ret = PriorityConstant.TYPE_INFO;
		}
		return ret;
	}

	/**
	 * 送信を行うかのPriority毎の判定を行う
	 */
	private static boolean isOutput(int level, int priority){
		if (priority == PriorityConstant.TYPE_CRITICAL) {
			if (level == PriorityConstant.TYPE_CRITICAL ||
					level == PriorityConstant.TYPE_UNKNOWN ||
					level == PriorityConstant.TYPE_WARNING ||
					level == PriorityConstant.TYPE_INFO) {
				return true;
			} else {
				return false;
			}
		}
		if (priority == PriorityConstant.TYPE_UNKNOWN) {
			if (level == PriorityConstant.TYPE_UNKNOWN ||
					level == PriorityConstant.TYPE_WARNING ||
					level == PriorityConstant.TYPE_INFO) {
				return true;
			} else {
				return false;
			}
		}
		if (priority == PriorityConstant.TYPE_WARNING) {
			if (level == PriorityConstant.TYPE_WARNING ||
					level == PriorityConstant.TYPE_INFO) {
				return true;
			} else {
				return false;
			}
		}
		if (priority == PriorityConstant.TYPE_INFO) {
			if (level == PriorityConstant.TYPE_INFO) {
				return true;
			} else {
				return false;
			}
		}
		return false;
	}


	/**
	 * メッセージを置換します<BR>
	 * 
	 * @param key リソース名
	 * @param args メッセージの引数
	 * @return the メッセージ
	 */
	private static String getString(String key, Object[] args) {
		MessageFormat messageFormat = new MessageFormat(getString(key));
		return messageFormat.format(args);
	}

	/**
	 * メッセージを置換します<BR>
	 * 
	 * @param key リソース名
	 * @return the メッセージ
	 */
	private static String getString(String key) {
		try {
			return m_bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}
}
