/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.plugin.HinemosPluginService;
import com.clustercontrol.util.StdOutErrLog;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * Hinemos ManagerのMainクラス<br/>
 * @author takahatat
 */
public class HinemosManagerMain {

	public static final Log log = LogFactory.getLog("HinemosManagerMain");

	// shutdownHookが呼ばれるまでmainスレッドを待機させるためのLockオブジェクトおよびフラグ
	private static final Object shutdownLock = new Object();
	private static boolean shutdown = false;

	// 起動モード(この値を参照して、各HinemosPluginは起動状態をswitchする)
	public enum StartupMode { NORMAL, MAINTENANCE };
	public static final StartupMode _startupMode;

	static {
		// java引数からパラメータを読み込む
		String startupModeStr = System.getenv("STARTUP_MODE");
		StartupMode startupMode = StartupMode.NORMAL;

		try {
			startupMode = StartupMode.valueOf(startupModeStr);
		} catch (IllegalArgumentException e) {
			log.warn("System environment value is not correct. (STARTUP_MODE = " + startupModeStr);
		} finally {
			_startupMode = startupMode;
		}

		// 標準出力
		StdOutErrLog.initialize();
	}

	/**
	 * Hinemos Managerのmainメソッド<br/>
	 * @param args
	 * @throws Exception
	 */
	public static void main(String args[]) {

		long bootTime = System.currentTimeMillis();
		log.info("Hinemos Manager is starting. (startupMode = " + _startupMode + ")");

		// 参照可能なHinemosPluginを全て生成(create)する
		HinemosPluginService.getInstance().create();

		// 参照可能なHinemosPluginを全て活性化(activate)する
		HinemosPluginService.getInstance().activate();

		// Hinemos Mangerの停止処理を定義する
		Runtime.getRuntime().addShutdownHook(
				new Thread() {
					@Override
					public void run() {
						log.info("shutdown hook called.");
						synchronized (shutdownLock) {
							// Hinemos Managerの停止開始を通知する
							AplLogger apllogShutdown = new AplLogger("MNG", "mng");
							String[] msgArgsShutdown = {"shutdown"};
							apllogShutdown.put("SYS", "002", msgArgsShutdown);

							// 参照可能なHinemosPluginを全て非活性化(deactivate)する
							HinemosPluginService.getInstance().deactivate();

							// 参照可能なHinemosPluginを全て破棄(destroy)する
							HinemosPluginService.getInstance().destroy();

							log.info("Hinemos Manager is stopped.");

							shutdown = true;
							shutdownLock.notify();
						}
					}
				});

		// 起動所要時間をログ出力する
		long startupTime = System.currentTimeMillis();
		long initializeSec = (startupTime - bootTime) / 1000;
		long initializeMSec = (startupTime - bootTime) % 1000;
		log.info("Hinemos Manager Started in " + initializeSec + "s:" + initializeMSec + "ms");

		// Hinemos Managerの起動完了を通知する
		AplLogger apllogStart = new AplLogger("MNG", "mng");
		String[] msgArgsStart = {"start"};
		apllogStart.put("SYS", "001", msgArgsStart);

		// Hinemos Managerの停止が完了するまで待機する
		synchronized (shutdownLock) {
			while (! shutdown) {
				try {
					shutdownLock.wait();
				} catch (InterruptedException e) {
					log.warn("shutdown lock interrupted.", e);
					try {
						Thread.sleep(1000);
					} catch (InterruptedException sleepE) { };
				}
			}
		}

		System.exit(0);
	}

}
