/*

 Copyright (C) 2011 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.agent.log;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.ws.WebServiceException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.Agent;
import com.clustercontrol.agent.AgentEndPointWrapper;
import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.util.CalendarWSUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.agent.HinemosUnknown_Exception;
import com.clustercontrol.ws.agent.InvalidRole_Exception;
import com.clustercontrol.ws.agent.InvalidUserPass_Exception;
import com.clustercontrol.ws.agent.MessageInfo;
import com.clustercontrol.ws.monitor.MonitorInfo;
import com.clustercontrol.ws.monitor.MonitorStringValueInfo;

/**
 * ログファイル監視<BR>
 */
public class LogfileMonitor {

	// Syslog転送用ロガー
	private LoggerSyslog m_syslog = null;

	// ロガー
	private static Log m_log = LogFactory.getLog(LogfileMonitor.class);

	private static final String UNCHANGED_STATS_PERIOD = "unchanged.stats.period";

	private static final String FIRST_PART_DATA_CHECK_PERIOD = "first.part.data.check.period";

	private static final String FIRST_PART_DATA_CHECK_SIZE = "first.part.data.check.size";

	private static final String FILE_MAX_SIZE = "file.max.size";

	private static final String LOG_ENCODING = "log.file.encoding";

	private static final String PROGRAM = "log.msg.program";

	private static final String FIRST_HEAD_OPEN = "log.first.head.open";

	private static final String MESSAGE_ID_INFO = "001";

	private static final String MESSAGE_ID_WARNING = "002";

	// private static final String MESSAGE_ID_CRITICAL = "003";

	// private static final String MESSAGE_ID_UNKNOWN = "004";

	private static final String HINEMOS_LOG_AGENT = "hinemos_log_agent";

	private LogfileMonitorManager m_logfileMonitorManager;

	/**
	 * このオブジェクトはログファイルに紐づいている。
	 * ログファイルのオープンに失敗したときは、最初のみ「ファイルがありません」というinternalイベントを発生させる。
	 */
	private boolean m_initFlag = true;

	/** ログファイル名 */
	private String m_filePath;

	/** 動作間隔（ミリ秒） */
	private int m_runInterval;

	/** 初回ファイルチェック時にファイルが存在せず、初めてファイルを読み込んだ際に先頭から読むか否か */
	private boolean m_firstHeadOpen = false;

	/** ファイル先頭から読むか否かのステータス情報を保持するフラグ */
	private boolean m_firstHeadOpenStatus = false;
	
	/** ログローテートを判定するフラグ */
	private boolean m_logrotate = false;
	
	/** ファイル変更チェック期間設定（ミリ秒） */
	private int m_unchangedStatsPeriod = 0;

	/** ファイル変更詳細チェック（冒頭データ比較）期間（ミリ秒） */
	private int m_firstPartDataCheckPeriod = 0;

	/** ファイル変更詳細チェック（冒頭データ比較）サイズ（byte） */
	private int m_firstPartDataCheckSize = 0;

	/** 上限ファイルサイズ設定（byte） */
	private long m_fileMaxSize = 0L;

	/** ログ先頭に定義するプログラム名 */
	private String m_program = HINEMOS_LOG_AGENT;

	/** ログ転送停止フラグ */
	private boolean m_stopFlg = false;

	// ログファイルのエンコーディング
	private String m_logEncoding = "UTF-8";

	private long m_filesize = 0;
	private long m_tmp_filesize = 0;
	private long n_unchanged_stats = 0; // ファイルチェック時に、ファイルに変更がなかった回数
	// String carryOver="";
	private byte[] m_cbuf = new byte[1024]; // ファイルからの読み出し分を保持するバッファ
	private byte[] m_carryOverBuf = new byte[0]; // 文字化け対策用に繰り越すバッファ
	private byte[] m_appendedBuf = null; // 繰り越し分にファイルからの読み出し分を追加したバッファ

	private long m_lastDataCheck = System.currentTimeMillis(); // 最終詳細チェック（冒頭データ比較）実行時刻

	private RandomAccessFileWrapper m_fr = null;

	private ArrayList<MonitorInfo> m_monitorInfoList = new ArrayList<MonitorInfo>();

	//オリジナルメッセージのサイズ上限（Byte）
	private int m_limit_logmsg = 1024;

	/**
	 * コンストラクタ
	 * 
	 * @param queue
	 * @param props
	 * @param path
	 *            転送対象ログファイル
	 * @param flg
	 *            最初にファイルをチェック
	 */
	public LogfileMonitor(LogfileMonitorManager logfileManager,
			String path, int runInterval) {
		m_logfileMonitorManager = logfileManager;
		m_filePath = path;
		m_runInterval = runInterval;

		// ファイル変更チェック期間（秒）
		String sleepInterval = AgentProperties.getProperty(UNCHANGED_STATS_PERIOD, "5");
		m_log.info(UNCHANGED_STATS_PERIOD + " = " + sleepInterval + " sec");
		try {
			this.m_unchangedStatsPeriod = Integer.parseInt(sleepInterval) * 1000;
		} catch (NumberFormatException e) {
			m_log.error("LogfileManager() : " + UNCHANGED_STATS_PERIOD, e);
		}
		m_log.debug(UNCHANGED_STATS_PERIOD + " = " + m_unchangedStatsPeriod);


		// ファイル変更詳細チェック（冒頭データ比較）期間（秒）
		String firstPartDataCheckPeriod = AgentProperties.getProperty(FIRST_PART_DATA_CHECK_PERIOD, "300");
		try {
			this.m_firstPartDataCheckPeriod = Integer.parseInt(firstPartDataCheckPeriod) * 1000;
		} catch (NumberFormatException e) {
			m_log.error("LogfileManager() : " + FIRST_PART_DATA_CHECK_PERIOD, e);
		}
		m_log.debug(FIRST_PART_DATA_CHECK_PERIOD + " = " + m_firstPartDataCheckPeriod);


		// ファイル変更詳細チェック（冒頭データ比較）サイズ（byte）
		String firstPartDataCheckSize = AgentProperties.getProperty(FIRST_PART_DATA_CHECK_SIZE, "256");
		try {
			this.m_firstPartDataCheckSize = Integer.parseInt(firstPartDataCheckSize);
		} catch (NumberFormatException e) {
			m_log.error("LogfileManager() : " + FIRST_PART_DATA_CHECK_SIZE, e);
		}
		m_log.debug(FIRST_PART_DATA_CHECK_SIZE + " = " + m_firstPartDataCheckSize);

		// 上限ファイルサイズ（byte）
		String fileMaxSize = AgentProperties.getProperty(FILE_MAX_SIZE, "2147483648");
		m_log.info(FILE_MAX_SIZE + " = " + fileMaxSize + " byte");
		try {
			this.m_fileMaxSize = Long.parseLong(fileMaxSize);
		} catch (NumberFormatException e) {
			m_log.error("LogfileManager() : " + FILE_MAX_SIZE, e);
		}
		m_log.debug(FILE_MAX_SIZE + " = " + m_fileMaxSize);

		// プログラム名を設定
		m_program= AgentProperties.getProperty(PROGRAM, HINEMOS_LOG_AGENT);
		if ("".equals(m_program)) {
			m_program = HINEMOS_LOG_AGENT;
		}
		m_log.debug(PROGRAM + " = " + m_program);

		// 初回読み込み時にログファイルが存在せず、初めにファイルを読み込んだ際に先頭からファイルを読むか否か
		String firstHeadOpenString = AgentProperties.getProperty(FIRST_HEAD_OPEN, "false");
		if("true".equals(firstHeadOpenString)){
			m_firstHeadOpen = true;
		}
		m_firstHeadOpenStatus = m_firstHeadOpen;
		m_log.debug(FIRST_HEAD_OPEN + " = " + m_firstHeadOpen);
		
		// ログファイルのエンコーディングを設定
		m_logEncoding =  AgentProperties.getProperty(LOG_ENCODING);

		// プロパティファイルに設定がない場合
		if(m_logEncoding == null){
			m_logEncoding = System.getProperty("file.encoding");
		}
		m_log.info(LOG_ENCODING + " = " + m_logEncoding);

		m_syslog = new LoggerSyslog();

		// 1行のメッセージ上限を定める
		String limit_logmsg = AgentProperties.getProperty("limit.size.logmsg");
		if (limit_logmsg != null) {
			try {
				m_limit_logmsg = Integer.parseInt(limit_logmsg);
				m_log.info("limit.size.logmsg = " + m_limit_logmsg);
			} catch (NumberFormatException e) {
				m_log.error("limit.logmsg",e);
			}
		}
	}

	public void setMonitor(ArrayList<MonitorInfo> monitorInfoList) {
		this.m_monitorInfoList = monitorInfoList;
	}

	public boolean clean() {
		if (m_monitorInfoList == null || m_monitorInfoList.size() == 0) {
			m_log.info("clean " + m_filePath);
			closeFile(m_fr);
			return true;
		}
		return false;
	}

	public void printOffset() {
		if (m_fr == null) {
			m_log.info("printOffset : " + m_filePath + " is null");
			return;
		}

		try {
			m_log.info("printOffset : " + m_filePath + " offset = " + m_fr.getFilePointer());
		} catch (IOException e) {
			m_log.info("printOffset IOException(" + m_filePath + ") : " + e.getMessage());
		}

	}

	/**
	 * スレッドの動作メソッド<BR>
	 * 
	 */
	/*
	 * (非 Javadoc)
	 * 
	 * @see java.lang.Thread#run()
	 */
	public void run() {
		m_log.debug("monitor start.  logfile : " + m_filePath
				+ "  interval : " + m_runInterval + "ms"
				+ "  logfile encoding : " + m_logEncoding
				+ "  syslog encoding : " + System.getProperty("file.encoding"));
		// ファイルオープン
		File file = new File(m_filePath);

		if (m_fr == null) {
			// ログローテートまたは、初回読み込みでなく、firstHeadOpenフラグがtrueかつFileNotFoundが前回おきた場合は先頭から読む
			if(!m_initFlag && m_firstHeadOpenStatus && m_firstHeadOpen){
				m_log.info("run() : openFile=" + file + ", init=false");
				m_log.debug("run() : m_initFlag=" + m_initFlag);
				m_log.debug("run() : m_firstHeadOpenStatus=" + m_firstHeadOpenStatus);
				m_log.debug("run() : m_firstHeadOpen=" + m_firstHeadOpen);
				m_fr = openFile(file, false);
			}
			// 初回読み込みは必ず末尾から読む
			// 2回目以降かつ、firstHeadOpenフラグがfalseの場合は末尾から読む
			else{
				m_log.info("run() : openFile=" + file + ", init=true");
				m_fr = openFile(file, true);
			}
			
			// オープンできないと終了
			if (m_fr == null) {
				return;
			}
		}

		String logPrefix = m_program + "(" + m_filePath + "):";

		if (this.m_stopFlg) {
			closeFile(m_fr);
			return;
		}

		boolean readSuccessFlg = true; // 増加分読み込み成功フラグ 
		m_logrotate = false;			// ローテートフラグ

		try {
			m_tmp_filesize = m_fr.length(); // 現在監視しているファイのサイズを取得・・・（１）

			if (m_filesize == m_tmp_filesize) {

				// ファイルサイズがm_unchangedStatsPeriod秒間以上変わらなかったら、ファイル切り替わりチェック
				if ((++n_unchanged_stats * this.m_runInterval) >= this.m_unchangedStatsPeriod) {
					m_log.debug("run() : " + m_filePath + " check log rotation");

					// ログローテートされているかチェックする
					// 従来の判定ロジック：
					// 現在監視しているファイルのサイズと本来監視すべきファイルのサイズが異なっている場合、
					// mv方式によりローテートされたと判断する。
					if (m_fr.length() != file.length()) { // ・・・（２）
						m_log.debug("run() : " + m_filePath + " file size not match");
						m_logrotate = true;

						m_log.debug("run() : m_logrotate set true .1");// rmしたとき、このルートでrotateしたと判断される！！
						m_log.debug("run() : m_fr.length()=" + m_fr.length());
						m_log.debug("run() : file.length()=" + file.length());
						
					} else if (m_tmp_filesize > 0 && m_firstPartDataCheckPeriod > 0 &&
							(System.currentTimeMillis() - m_lastDataCheck) > m_firstPartDataCheckPeriod){
						m_log.debug("run() : " + m_filePath + " check first part of file");

						// 追加された判定ロジック：
						// 現在監視しているファイルのサイズと本来監視すべきファイルのサイズが同じであっても、
						// mv方式によりローテートされた可能性がある。
						// ファイルの冒頭部分を確認することで、ローテートされたことを確認する。
						byte[] refFirstPartOfFile = new byte[m_firstPartDataCheckSize];
						Arrays.fill(refFirstPartOfFile, (byte)0);  // 全ての要素を0で初期化
						byte[] newFirstPartOfFile = new byte[m_firstPartDataCheckSize];
						Arrays.fill(newFirstPartOfFile, (byte)0);  // 全ての要素を0で初期化

						// ファイル名指定で新たにオープンするファイル
						RandomAccessFileWrapper newFile = null;

						// 現在監視しているファイルの冒頭データを取得
						long bufSeek = 0;
						try {
							bufSeek = m_fr.getFilePointer();

							// ファイル冒頭に移動する
							m_fr.seek(0);
							if (m_fr.read(refFirstPartOfFile) > 0) {
								if(m_log.isDebugEnabled()){
									try {
										m_log.debug("run() : " + m_filePath + " refFirstPartOfFile : "
												+ new String(refFirstPartOfFile,
														0,
														m_tmp_filesize < refFirstPartOfFile.length ? (int)m_tmp_filesize : refFirstPartOfFile.length,
																m_logEncoding));
									} catch (Exception e) {
										m_log.error("run() : " + m_filePath + " " + e.getMessage(), e);
									}
								}

								// 再度ファイル名指定でファイルを開く
								newFile = new RandomAccessFileWrapper(m_filePath, "r");
								if (newFile.read(newFirstPartOfFile) > 0) {
									if(m_log.isDebugEnabled()){
										try {
											m_log.debug("run() : " + m_filePath + " newFirstPartOfFile : "
													+ new String(newFirstPartOfFile,
															0,
															m_tmp_filesize < newFirstPartOfFile.length ? (int)m_tmp_filesize : newFirstPartOfFile.length,
																	m_logEncoding));
										} catch (Exception e) {
											m_log.error("run() : " + m_filePath + " " + e.getMessage(), e);
										}
									}

									// ファイルの冒頭部分が異なれば別ファイルと判定
									if (!Arrays.equals(refFirstPartOfFile, newFirstPartOfFile)) {
										m_log.debug("run() : " + m_filePath + " log rotation detected");
										m_logrotate = true;
										m_log.debug("run() : m_logrotate set true .2");
									}
								}
							}
						} catch (Exception e) {
							m_log.error("run() : " + m_filePath + " " + e.getMessage(), e);
						} finally {
							// ファイルのオフセットを元に戻す
							if (bufSeek != 0) {
								m_fr.seek(bufSeek);
							}

							// オープンしたファイルをクローズ
							if(newFile != null){
								try {
									newFile.close();
								} catch (Exception e) {
									m_log.error("run() : " + m_filePath + " " + e.getMessage(), e);
								}
							}
						}
						// 最終詳細チェック（ファイルデータ比較）実行時刻を設定
						m_lastDataCheck = System.currentTimeMillis();
					}

					// ログローテートされたと判定された場合
					if(m_logrotate){
						// 再度ファイルサイズの増加を確認する。
						// ローテートされたと判定されたが、実は、ローテートされておらず、
						//  （１）の時点では最終ログ読み込み時からログが出力されていないためサイズ（filesize）は同一であったが、
						//  （２）の判定直前にログが出力された場合は、ローテートされたと誤検知するため、
						// 再度サイズ比較を行う。
						if (m_tmp_filesize == m_fr.length()) {
							m_log.info(m_filePath + " : file changed");
							closeFile(m_fr);

							// ファイルオープン
							m_fr = openFile(file, false);
							if (m_fr == null) {
								return;
							}

							m_filesize = 0;
							// carryOver = "";
							m_carryOverBuf = new byte[0];
						}

					}
					n_unchanged_stats = 0;
				}
				return;
			}

			n_unchanged_stats = 0;

			if (m_filesize < m_tmp_filesize) {
				// デバッグログ
				m_log.debug("run() : " + m_filePath +
						" filesize " + m_filesize + " tmp_filesize " + m_tmp_filesize);

				while (true) {
					readSuccessFlg = false;
					int read = m_fr.read(m_cbuf);
					readSuccessFlg = true;
					if (read == -1) {
						break;
					}

					// //
					// UTF-8を含むログで文字化けが発生するため修正
					//
					// 最後の改行コードを検索し、最後の改行コード以降は次回に繰越て処理する。
					// carryOverStartには改行コードの次のコードのインデックスが格納される。
					int carryOverStart = read;
					boolean returnCode = false;
					for (int i = read - 1; i >= 0; i--) {
						if (m_cbuf[i] == '\n' || m_cbuf[i] == '\r') {
							carryOverStart = i + 1;
							returnCode = true;
							break;
						}
					}

					// デバッグログ
					m_log.debug("run() : " + m_filePath + " read " + read +
							"    carryOverStart " + carryOverStart);

					// 今回出力処理する分のバッファを作成
					// 前回の繰越分と今回ファイルから読み出したデータのうち
					// 最後の改行までのデータをマージする。
					m_appendedBuf = new byte[m_carryOverBuf.length + carryOverStart];
					ByteBuffer.wrap(m_carryOverBuf).get(m_appendedBuf, 0,
							m_carryOverBuf.length);
					ByteBuffer.wrap(m_cbuf).get(m_appendedBuf,
							m_carryOverBuf.length, carryOverStart);

					// デバッグログ
					m_log.debug("run() : " + m_filePath + " appendedBuf size " + m_appendedBuf.length);

					// 改行コードが含まれない場合
					if (!returnCode) {
						// 今回ファイルから読み込んだものを含めて全て次回へ繰り越す。
						// 出力処理は実施しない。
						m_log.debug("run() : " + m_filePath
								+ " return code is not exist");
						m_carryOverBuf = new byte[m_appendedBuf.length];
						ByteBuffer.wrap(m_appendedBuf).get(m_carryOverBuf);
						continue;
					}

					// 繰越データ用バッファを作成
					m_carryOverBuf = new byte[(read - carryOverStart)];
					try {
						// 最後が改行コード以降にデータがある場合は、次回の処理にまわす
						if (read > carryOverStart) {
							// デバッグログ
							if (m_log.isDebugEnabled()) {
								m_log.debug("run() : " + m_filePath
										+ " carryOverBuf size "
										+ m_carryOverBuf.length);
							}
							ByteBuffer.wrap(m_cbuf, carryOverStart,
									m_carryOverBuf.length).get(m_carryOverBuf);
						}
					} catch (Exception e) {
						m_log.error("run() : " + e.getMessage(), e);
					}

					try {
						// 加分読み込み
						String tmpString = new String(m_appendedBuf, 0,
								m_appendedBuf.length, m_logEncoding);
						String[] result = tmpString.split("\\n");
						// デバッグログ
						if (m_log.isDebugEnabled()) {
							m_log.debug("run() : " + m_filePath + " " + tmpString);
							m_log.debug("run() : " + m_filePath + " size "
									+ tmpString.length());
							m_log.debug("run() : " + m_filePath + " result size "
									+ result.length);
						}

						// 読み込み文字列のサイズが0でない場合は処理する
						if (tmpString.length() != 0) {
							for (String res : result) {
								// 旧バージョンとの互換性のため、syslogでも飛ばせるようにする。
								if (m_syslog.isValid()) {
									// v3.2 mode
									m_syslog.log(logPrefix + res);
								} else {
									// v4.0 mode
									patternMatch(res);
								}
							}
						}

					} catch (UnsupportedEncodingException e) {
						m_log.error("run() : " + m_filePath + " "
								+ e.getMessage());
					}
				}

				m_filesize = m_tmp_filesize;
				if(m_log.isDebugEnabled()){
					m_log.debug("run() : " + m_filePath + " filesize = " + m_filesize + ", FilePointer = " + m_fr.getFilePointer());
				}

			} else if (m_filesize > m_tmp_filesize) {
				// 切詰
				m_log.info(this.m_filePath + " : file size becomes small");
				m_fr.seek(0);
				m_filesize = 0;
				m_carryOverBuf = new byte[0];
			}

		} catch (IOException e) {
			if (readSuccessFlg) {
				m_log.error("run() : " + e.getMessage());
				Object[] args = { this.m_filePath };
				sendMessage(PriorityConstant.TYPE_WARNING,
						Messages.getString("log.agent"),
						MESSAGE_ID_WARNING,
						Messages.getString("message.log.agent.4"),
						Messages.getString("message.log.agent.1", args) + "\n" + e.getMessage());

				// エラーが発生したのでファイルクローズ
				closeFile(m_fr);
			} else {
				m_log.warn("run() : " + e.getMessage());
				// 増加分読み込みの部分でエラーが起きた場合は、ファイルポインタを進める
				try {
					m_fr.seek(m_filesize);
				} catch (IOException e1) {
					m_log.error("run() set file-pointer error : " + e1.getMessage());
				}
			}
		}
	}

	private void patternMatch(String line) {
		// CR-LFの場合は\rが残ってしまうので、ここで削除する。
		line = line.replace("\r", "");
		if (line.length() > m_limit_logmsg) {
			m_log.info("log line is too long");
			line = line.substring(0, m_limit_logmsg);
		}

		for (MonitorInfo monitorInfo : m_monitorInfoList) {
			if (monitorInfo.getCalendarDTO() != null &&
					! CalendarWSUtil.isRunnable(monitorInfo.getCalendarDTO())) {
				m_log.debug("patternMatch is skipped because of calendar");
				continue;
			}
			if (monitorInfo.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
				break;
			}
			for(MonitorStringValueInfo stringInfo : monitorInfo.getStringValueInfo()) {
				if(m_log.isDebugEnabled()){
					m_log.debug("patternMatch() line = " + line
							+ ", monitorId = " + stringInfo.getMonitorId()
							+ ", orderNo = " + stringInfo.getOrderNo()
							+ ", pattern = " + stringInfo.getPattern());
				}
				if (!stringInfo.isValidFlg()) {
					continue;
				}
				String patternText = stringInfo.getPattern();
				String message = "log.line=" + line + "\npattern=" + patternText;
				m_log.trace("check " + message);

				Pattern pattern = null;
				// 大文字・小文字を区別しない場合
				if(stringInfo.isCaseSensitivityFlg()){
					pattern = Pattern.compile(patternText, Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
				}
				// 大文字・小文字を区別する場合
				else{
					pattern = Pattern.compile(patternText, Pattern.DOTALL);
				}
				Matcher matcher = pattern.matcher(line);

				if (matcher.matches()) {
					m_log.debug("match " + message);
					MessageInfo logmsg = new MessageInfo();
					logmsg.setMessage(line);
					logmsg.setGenerationDate(new Date().getTime());
					logmsg.setHostName(Agent.getAgentInfo().getHostname());
					try {
						if (stringInfo.getProcessType() == ProcessConstant.TYPE_YES) {
							sendNotifyRetry(message, logmsg, monitorInfo, stringInfo);
						}
					} catch (HinemosUnknown_Exception e) {
						m_log.warn("patternMatch (" + e.getClass().getSimpleName() +
								") : " + e.getMessage());
					} catch (InvalidRole_Exception e) {
						m_log.warn("patternMatch (" + e.getClass().getSimpleName() +
								") : " + e.getMessage());
					} catch (InvalidUserPass_Exception e) {
						m_log.warn("patternMatch (" + e.getClass().getSimpleName() +
								") : " + e.getMessage());
					}
					break;
				}
			}
		}
	}

	private static int notifyRetryCount = -1;
	private static int notifyRetrySleep = -1;

	/**
	 * 通知に失敗したら、1分後にリトライするメソッド。
	 * @param message
	 * @param logmsg
	 * @param monitorInfo
	 * @param stringInfo
	 * @throws HinemosUnknown_Exception
	 * @throws InvalidRole_Exception
	 * @throws InvalidUserPass_Exception
	 */
	private void sendNotifyRetry(String message, MessageInfo logmsg,
			MonitorInfo monitorInfo, MonitorStringValueInfo stringInfo)
	throws HinemosUnknown_Exception, InvalidRole_Exception, InvalidUserPass_Exception {
		boolean flag = false;

		if (notifyRetryCount < 0) {
			String str = AgentProperties.getProperty("notify.retry.count", "15");
			notifyRetryCount = Integer.parseInt(str);
			m_log.info("notify.retry.count=" + notifyRetryCount);
		}
		if (notifyRetrySleep < 0) {
			String str = AgentProperties.getProperty("notify.retry.sleep", "60000");
			notifyRetrySleep = Integer.parseInt(str);
			m_log.info("notify.retry.sleep=" + notifyRetrySleep);
		}

		for (int i = 0; i < notifyRetryCount; i++) {
			try {
				AgentEndPointWrapper.sendNotify(message, logmsg, monitorInfo, stringInfo);
				flag = true;
			} catch (WebServiceException e) {
				try {
					Thread.sleep(notifyRetrySleep);
				} catch (InterruptedException e1) {
					m_log.warn("sendNotifyRetry " + e1.getMessage());
				}
			}
			if (flag) {
				return;
			}
		}
		m_log.warn("give up notifying. Maybe, manager is down");
	}

	/**
	 * ログ転送を停止します。<BR>
	 * 
	 */
	public void requestStop() {
		this.m_stopFlg = true;

		m_log.info("monitor stop.   logfile : " + m_filePath
				+ "  interval : " + m_runInterval + "ms"
				+ "  logfile encoding : " + m_logEncoding
				+ "  syslog encoding : " + System.getProperty("file.encoding"));
	}

	/**
	 * 監視管理情報へ通知
	 * 
	 * @param priority
	 *            重要度
	 * @param app
	 *            アプリケーション
	 * @param msgId
	 *            メッセージID
	 * @param msg
	 *            メッセージ
	 * @param msgOrg
	 *            オリジナルメッセージ
	 */
	private void sendMessage(int priority, String app, String msgId,
			String msg, String msgOrg) {

		String monitorId = "";
		for (MonitorInfo info : m_monitorInfoList) {
			if (!"".equals(monitorId)) {
				monitorId += ",";
			}
			monitorId += info.getMonitorId();
		}

		m_logfileMonitorManager.sendMessage(m_filePath, priority, app, msgId, msg, msgOrg, monitorId);

	}

	/**
	 * 転送対象ログファイルクローズ
	 * 
	 */
	private void closeFile(RandomAccessFileWrapper fr) {

		if (fr != null) {
			try {
				fr.close();
			} catch (IOException e) {
				m_log.debug("run() : " + e.getMessage());
			}
		}
	}

	/**
	 * 転送対象ログファイルオープン
	 */
	private RandomAccessFileWrapper openFile(File name, boolean init) {
		m_log.info("openFile : filename=" + name + ", init=" + init);

		RandomAccessFileWrapper fr = null;
		// ファイルオープン
		try {
			fr = new RandomAccessFileWrapper(name, "r");
			long filesize = fr.length();
			if (filesize > this.m_fileMaxSize) {
				// ファイルサイズが大きい場合、監視管理へ通知
				Object[] args1 = { this.m_filePath };
				Object[] args2 = { filesize };
				sendMessage(PriorityConstant.TYPE_INFO,
						Messages.getString("agent"),
						MESSAGE_ID_INFO,
						Messages.getString("message.log.agent.3"),
						Messages.getString("message.log.agent.1", args1) + ", "
						+ Messages.getString("message.log.agent.5",args2));
			}

			// ファイルポインタの設定
			// 初回openはinit=trueで途中から読む。
			// ローテーションの再openはinit=falseで最初から読む。
			if (init) {
				fr.seek(filesize);
			}
			
			// 一度以上ファイルオープンに成功したら、フラグを無効化する
			m_firstHeadOpenStatus = false;
		} catch (FileNotFoundException e) {
			m_log.info("openFile : " + e.getMessage());
			if (m_initFlag) {
				// 最初にファイルをチェックする場合、監視管理へ通知
				Object[] args = { this.m_filePath };
				sendMessage(PriorityConstant.TYPE_INFO,
						Messages.getString("agent"),
						MESSAGE_ID_INFO,
						Messages.getString("message.log.agent.2"),
						Messages.getString("message.log.agent.1", args));
			}
			
			/*
			// 一度ファイルが見えなくなったら、次回オープン時は先頭から(仕様変更となるため実施しない)
			m_filesize = 0;
			m_firstHeadOpenStatus = true;
			*/
		} catch (SecurityException e) {
			m_log.info("openFile : " + e.getMessage());
			if (m_initFlag) {
				// 監視管理へ通知
				Object[] args = { this.m_filePath };
				sendMessage(PriorityConstant.TYPE_WARNING,
						Messages.getString("agent"),
						MESSAGE_ID_WARNING,
						Messages.getString("message.log.agent.4"),
						Messages.getString("message.log.agent.1", args) + "\n" + e.getMessage());
			}
		} catch (IOException e) {
			m_log.info("openFile : " + e.getMessage());
			if (m_initFlag) {
				// 監視管理へ通知
				Object[] args = { this.m_filePath };
				sendMessage(PriorityConstant.TYPE_INFO,
						Messages.getString("agent"),
						MESSAGE_ID_INFO,
						Messages.getString("message.log.agent.4"),
						Messages.getString("message.log.agent.1", args));
			}
			closeFile(fr);
		}
		m_initFlag = false;
		return fr;
	}
}